/*  cat.c
 *
 *  Transceiver CAT control functions for the xdemorse application
 */

/*
 *  xdemorse: An application to decode and display
 *  Morse code signals using the computer's sound card.
 *
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "cat.h"
#include "shared.h"

static char
  /* Command codes for Yaesu FT847/FT857 CAT */
  FT847_CAT_ON[]	= { 0, 0, 0, 0, (char)0x00 }, /*  Enable CAT */
  FT847_CAT_OFF[]	= { 0, 0, 0, 0, (char)0x80 }, /* Disable CAT */
  FT847_RX_STATUS[]	= { 0, 0, 0, 0, (char)0xE7 }, /* Request RX status  */
  FT847_MAIN_VFO[]	= { 0, 0, 0, 0, (char)0x03 }, /* Read main VFO freq */
  FT847_SET_VFO[]	= { 0, 0, 0, 0, (char)0x01 }, /* Set main VFO freq  */

  /* Command codes for Elecraft K3 */
  K3_K31_S[] = "K31;",				/* Set K3 CAT mode */
  K3_K31_G[] = "K3;",				/* Request K3 CAT mode */
  K3_FA_S[]	 = "FAnnnnnnnnnnn;",	/* Set VFO A frequency */
  K3_FA_G[]  = "FA;";				/* Get VFO A frequency */

/* Serial port File descriptor */
static int tcvr_serial_fd = 0;

/* Original serial port options */
static struct termios tcvr_serial_old_options;

/*------------------------------------------------------------------------*/

/*  Open_Tcvr_Serial()
 *
 *  Opens Tcvr's Serial Port device, returns the file
 *  descriptor tcvr_serial_fd on success or exits on error
 */

  gboolean
Open_Tcvr_Serial( void )
{
  struct termios new_options; /* New serial port options */
  struct flock lockinfo;      /* File lock information   */
  int error = 0;

  /* Abort if serial already open */
  if( isFlagSet(CAT_SETUP) ) return( TRUE );

  /* Open Serial device, exit on error */
  tcvr_serial_fd = open( rc_data.cat_serial, O_RDWR | O_NOCTTY );
  if( tcvr_serial_fd < 0 )
  {
	char message[128];
	perror( rc_data.cat_serial );
	snprintf( message, 127,
		_("Failed to open %s\n"\
		  "Transceiver CAT not setup"), rc_data.cat_serial );
	Error_Dialog( message );
	tcvr_serial_fd = 0;
	return( FALSE );
  }

  /* Attempt to lock entire Serial port device file */
  lockinfo.l_type   = F_WRLCK;
  lockinfo.l_whence = SEEK_SET;
  lockinfo.l_start  = 0;
  lockinfo.l_len    = 0;

  /* If Serial device is already locked, abort */
  error = fcntl( tcvr_serial_fd, F_SETLK, &lockinfo );
  if( error == -1 )
  {
	char message[128];
	error = fcntl( tcvr_serial_fd, F_GETLK, &lockinfo );
	if( error != -1 )
	{
	  snprintf( message, 127,
		  _("Failed to lock %s\n"\
			"Device locked by pid %d\n"\
			"Transceiver CAT not setup"),
		  rc_data.cat_serial, lockinfo.l_pid );
	  Error_Dialog( message );
	}
	close( tcvr_serial_fd );
	tcvr_serial_fd = 0;
	return( FALSE );
  }

  /* Save the current serial port options */
  error |= tcgetattr( tcvr_serial_fd, &tcvr_serial_old_options );

  /* Read the current serial port options */
  error |= tcgetattr( tcvr_serial_fd, &new_options );

  /* Set the i/o baud rates */
  switch( rc_data.tcvr_type )
  {
	case FT847:
	  error |= cfsetspeed( &new_options, B57600 );
	  break;

	case FT857: case K2: case K3:
	  error |= cfsetspeed( &new_options, B38400 );
	  break;
  }

  /* Set options for 'raw' I/O mode */
  cfmakeraw( &new_options );

  /* Setup read() timeout to .5 sec */
  new_options.c_cc[ VMIN ]  = 0;
  new_options.c_cc[ VTIME ] = 5;

  /* Setup the new options for the port */
  error |= tcsetattr( tcvr_serial_fd, TCSAFLUSH, &new_options );
  if( error ) return( FALSE );

  /* Enable CAT by testing comms with transceiver */
  switch( rc_data.tcvr_type )
  {
	case FT847:
	  {
		char test;
		if( Write_Tcvr_Serial(FT847_CAT_ON, sizeof(FT847_CAT_ON)) &&
			Write_Tcvr_Serial(FT847_RX_STATUS, sizeof(FT847_RX_STATUS)) &&
			Read_Tcvr_Serial(&test, 1) )
		  Set_Flag( CAT_SETUP );
		else
		{
		  Write_Tcvr_Serial( FT847_CAT_OFF, sizeof(FT847_CAT_OFF) );
		  tcsetattr( tcvr_serial_fd, TCSANOW, &tcvr_serial_old_options );
		  close( tcvr_serial_fd );
		  tcvr_serial_fd = 0;
		  return( FALSE );
		}
	  }
	  break;

  case FT857:
	  {
		char test;
		if( Write_Tcvr_Serial(FT847_RX_STATUS, sizeof(FT847_RX_STATUS)) &&
			Read_Tcvr_Serial(&test, 1) )
		  Set_Flag( CAT_SETUP );
		else
		{
		  tcsetattr( tcvr_serial_fd, TCSANOW, &tcvr_serial_old_options );
		  close( tcvr_serial_fd );
		  tcvr_serial_fd = 0;
		  return( FALSE );
		}
	  }
	  break;

	case K2:
	  if( Write_Tcvr_Serial(K3_FA_G, sizeof(K3_FA_G)-1) &&
		  Read_Tcvr_Serial(K3_FA_S, sizeof(K3_FA_S)-1) &&
		  strncmp( K3_FA_S, "K31;", sizeof(K3_FA_S)-1) == 0 )
		Set_Flag( CAT_SETUP );
	  else
	  {
		tcsetattr( tcvr_serial_fd, TCSANOW, &tcvr_serial_old_options );
		close( tcvr_serial_fd );
		tcvr_serial_fd = 0;
		return( FALSE );
	  }
	break;

	case K3:
	  if( Write_Tcvr_Serial(K3_K31_S, sizeof(K3_K31_S)-1) &&
		  Write_Tcvr_Serial(K3_K31_G, sizeof(K3_K31_G)-1) &&
		  Read_Tcvr_Serial(K3_K31_S, sizeof(K3_K31_S)-1)  &&
		  strncmp( K3_K31_S, "K31;", sizeof(K3_K31_S)-1) == 0 )
		Set_Flag( CAT_SETUP );
	  else
	  {
		tcsetattr( tcvr_serial_fd, TCSANOW, &tcvr_serial_old_options );
		close( tcvr_serial_fd );
		tcvr_serial_fd = 0;
		return( FALSE );
	  }
	  break;
  } /* switch( rc_data.tcvr_type ) */

  return( TRUE );
} /* Open_Tcvr_Serial() */

/*-------------------------------------------------------------------------*/

/*  Close_Tcvr_Serial()
 *
 *  Restore old options and close the Serial port
 */

  void
Close_Tcvr_Serial( void )
{
  if( isFlagSet(CAT_SETUP) )
  {
	/* Disable CAT for FT847 */
	if( rc_data.tcvr_type == FT847 )
	  Write_Tcvr_Serial( FT847_CAT_OFF, sizeof(FT847_CAT_OFF) );

	/* Restore serial port and close */
	if( tcvr_serial_fd > 0 )
	{
	  tcsetattr( tcvr_serial_fd, TCSANOW, &tcvr_serial_old_options );
	  close( tcvr_serial_fd );
	}
	tcvr_serial_fd = 0;
  }

  Clear_Flag( CAT_SETUP );
} /* End of Close_Tcvr_Serial() */

/*-------------------------------------------------------------------------*/

/*  Read_Tcvr_Serial()
 *
 *  Reads Data from the Transceiver
 */

  gboolean
Read_Tcvr_Serial( char *data, size_t len )
{
  ssize_t ret = read( tcvr_serial_fd, data, len );

  /* Error condition */
  if( (size_t)ret != len )
  {
	tcflush( tcvr_serial_fd, TCIFLUSH );
	fprintf( stderr,
		"xdemorse: read() return value wrong: %d/%d\n",
		(int)ret, (int)len );
	Error_Dialog( _("Serial port read() I/O error") );
	return( FALSE );
  }

  return( TRUE );
} /* End of Read_Tcvr_Serial() */

/*-------------------------------------------------------------------------*/

/*  Write_Tcvr_Serial()
 *
 *  Writes a command to the Tranceiver
 */

  gboolean
Write_Tcvr_Serial( char *data, size_t len )
{
  tcflush( tcvr_serial_fd, TCIOFLUSH );
  ssize_t ret = write( tcvr_serial_fd, data, len );

  /* Error condition */
  if( (size_t)ret != len )
  {
	tcflush( tcvr_serial_fd, TCOFLUSH );
	fprintf( stderr,
		"xdemorse: write() return value wrong: %d/%d\n",
		(int)ret, (int)len );
	Error_Dialog( _("Serial port write() I/O error") );
	return( FALSE );
  }

  /* Give time to CAT to do its thing */
  usleep(50000);

  return( TRUE );
} /* End of Write_Tcvr_Serial() */

/*-------------------------------------------------------------------------*/

/*  Read_Rx_Freq()
 *
 *  Reads the Rx freq of the Yaesu transceiver.
 */

  gboolean
Read_Rx_Freq( int *freq )
{
  /* A char string for sending and receiving  */
  /* data strings to and from the transceiver */
  char frq[5];

  /* Abort if CAT disabled */
  if( isFlagClear(CAT_SETUP) ) return( TRUE );

  switch( rc_data.tcvr_type )
  {
	case FT847: case FT857:
	/* Read Rx frequency */
	memset( frq, 0, sizeof(frq) );
	if( !Write_Tcvr_Serial(FT847_MAIN_VFO, sizeof(FT847_MAIN_VFO)) ||
		!Read_Tcvr_Serial(frq, sizeof(frq)) )
	  return( FALSE );

	/* Decode frequency data to 10Hz */
	*freq = 0;
	*freq += (frq[0] & 0xF0) >> 4;
	*freq *= 10;
	*freq += frq[0] & 0x0F;
	*freq *= 10;
	*freq += (frq[1] & 0xF0) >> 4;
	*freq *= 10;
	*freq += frq[1] & 0x0F;
	*freq *= 10;
	*freq += (frq[2] & 0xF0) >> 4;
	*freq *= 10;
	*freq += frq[2] & 0x0F;
	*freq *= 10;
	*freq += (frq[3] & 0xF0) >> 4;
	*freq *= 10;
	*freq += frq[3] & 0x0F;
	*freq *= 10;
	break;

	case K2: case K3:
	/* Read Rx frequency */
	if( !Write_Tcvr_Serial(K3_FA_G, sizeof(K3_FA_G)-1) ||
		!Read_Tcvr_Serial(K3_FA_S, sizeof(K3_FA_S)-1) )
	  return( FALSE );
	*freq = atoi( K3_FA_S+2 );
	break;

  } /* switch( rc_data.tcvr_type ) */

  return( TRUE );
} /* End of Read_Rx_Freq() */

/*------------------------------------------------------------------------*/

/*  Write_Rx_Freq()
 *
 *  Writes the Rx freq to the Yaesu FT-847 transceiver.
 */

  gboolean
Write_Rx_Freq( int freq )
{
  /* Buffer used for converting freq. to string */
  char freq_buff[9];
  int idx; /* Index for loops etc */


  /* Abort if CAT disabled */
  if( isFlagClear(CAT_SETUP) ) return( TRUE );

  switch( rc_data.tcvr_type )
  {
	case FT847: case FT857:
	  /* Set Rx frequency */
	  snprintf( freq_buff, sizeof(freq_buff), "%08d", freq/10 );
	  for( idx = 0; idx < 4; idx++ )
	  {
		FT847_SET_VFO[idx]  = (char)((freq_buff[2*idx]   - '0') << 4);
		FT847_SET_VFO[idx] |= (freq_buff[2*idx+1] - '0');
	  }
	  if( !Write_Tcvr_Serial(FT847_SET_VFO, sizeof(FT847_SET_VFO)) )
		return( FALSE );
	  break;

	case K2: case K3:
	  /* Set Rx frequency */
	  snprintf( K3_FA_S+2, 13, "%011d;", freq );
	  if( !Write_Tcvr_Serial(K3_FA_S, sizeof(K3_FA_S)-1) )
		  return( FALSE );
	  break;

  } /* switch( rc_data.tcvr_type ) */

  return( TRUE );
} /* End of Write_Rx_Freq() */

/*-------------------------------------------------------------------------*/

/* Tune_Tcvr()
 *
 * Tunes the transceiver to the frequency of the strongest
 * signal near a mouse click in the waterfall window
 */

  gboolean
Tune_Tcvr( double x )
{
  int
	from, to,	/* Range to scan for a max bin value  */
	bin_max,	/* Max bin value found in this range  */
	max_idx,	/* fft idx at which the max is found */
	fft_idx,	/* Idx used to search fft bin values */
	tcvr_freq,	/* Transceiver's Rx frequency */
	audio_freq;	/* Audio frequency in waterfal window */

  /* Abort if serial already open */
  if( isFlagClear(CAT_SETUP) ) return( TRUE );

  /* Calculate fft index corresponding to pointer x in waterfall window */
  fft_idx = (int)(x - 0.5);

  /* Look above and below click point for max bin val */
  from = fft_idx - 10;
  if( from < 0 ) from = 0;
  to = fft_idx + 10;
  if( to >= gbl_wfall_width ) to = gbl_wfall_width - 1;

  /* Find max bin value around click point */
  bin_max = 0;
  max_idx = fft_idx;
  for( fft_idx = from; fft_idx < to; fft_idx++ )
	if( bin_max < gbl_bin_ave[fft_idx] )
	{
	  bin_max = gbl_bin_ave[fft_idx];
	  max_idx = fft_idx;
	}

  /* Audio freq. corresponding to fft index */
  audio_freq = (2 * max_idx * rc_data.tone_freq) / gbl_wfall_width;

  /* Read current Rx frequency */
  if( !Read_Rx_Freq(&tcvr_freq) ) return( FALSE );

  /* Add correction according to TCVR (reverse or normal CW) */
  switch( rc_data.tcvr_type )
  {
	case FT847: case FT857:
	  tcvr_freq += audio_freq - rc_data.tone_freq;
	  break;

	case K2: case K3:
	  tcvr_freq -= audio_freq - rc_data.tone_freq;
	  break;
 
  } /* switch( rc_data.tcvr_type ) */

  /* Calculate and write Rx freq. to center audio on AUDIO_FREQUENCY */
  if( !Write_Rx_Freq(tcvr_freq) ) return( FALSE );

  return( TRUE );
} /* Tune_Tcvr() */

/*-------------------------------------------------------------------------*/

