/*
 * Hold page for channel notebook.
 *
 * Copyright (C) 2008 Collabora Ltd. <http://www.collabora.co.uk/>
 * Copyright (C) 2008 Nokia Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include "config.h"

#include "page-hold.h"

#include <glade/glade.h>

#include "extensions/extensions.h"

struct _TIPageHoldClass {
    GtkVBoxClass parent;
    gpointer priv;
};


struct _TIPageHoldPrivate {
    TpChannel *channel;

    /* UI for telling the CM what to do */
    GtkToggleButton *hold_toggle;
    GtkButton *request_hold_button;

    /* UI for the CM telling us what's going on */
    GtkLabel *current_status_label;

    /* Event log */
    GtkTextView *event_log;
    GtkTextBuffer *event_buffer;
};


G_DEFINE_TYPE (TIPageHold, ti_page_hold, GTK_TYPE_VBOX);

enum {
    PROP_CHANNEL = 1
};


static void
ti_page_hold_init (TIPageHold *self)
{
  self->priv = G_TYPE_INSTANCE_GET_PRIVATE (self, TI_TYPE_PAGE_HOLD,
      TIPageHoldPrivate);
}


static void
ti_page_hold_dispose (GObject *object)
{
  TIPageHold *self = TI_PAGE_HOLD (object);

  if (self->priv->channel != NULL)
    {
      g_object_unref (self->priv->channel);
      self->priv->channel = NULL;
    }

  G_OBJECT_CLASS (ti_page_hold_parent_class)->dispose (object);
}


static void
ti_page_hold_get_property (GObject *object,
                           guint property_id,
                           GValue *value,
                           GParamSpec *pspec)
{
  TIPageHold *self = TI_PAGE_HOLD (object);

  switch (property_id)
    {
    case PROP_CHANNEL:
      g_value_set_object (value, self->priv->channel);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}


static void
ti_page_hold_set_property (GObject *object,
                           guint property_id,
                           const GValue *value,
                           GParamSpec *pspec)
{
  TIPageHold *self = TI_PAGE_HOLD (object);

  switch (property_id)
    {
    case PROP_CHANNEL:
      g_return_if_fail (self->priv->channel == NULL);
      self->priv->channel = TP_CHANNEL (g_value_dup_object (value));
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}


static void
log_event (TIPageHold *self,
           const gchar *text)
{
  gtk_text_buffer_insert_at_cursor (self->priv->event_buffer, text, -1);
  gtk_text_buffer_insert_at_cursor (self->priv->event_buffer, "\n", -1);
}


static void
requested_hold (TpChannel *channel G_GNUC_UNUSED,
                const GError *error,
                gpointer unused G_GNUC_UNUSED,
                GObject *object)
{
  TIPageHold *self = TI_PAGE_HOLD (object);

  if (error != NULL)
    {
      gchar *text = g_strdup_printf ("RequestHold failed: %s", error->message);
      log_event (self, text);
      g_free (text);
      return;
    }

  log_event (self, "RequestHold succeeded");
}


static void
request_hold_clicked (TIPageHold *self)
{
  gboolean want = gtk_toggle_button_get_active (self->priv->hold_toggle);

  tp_cli_channel_interface_hold_call_request_hold (self->priv->channel, -1,
      want, requested_hold, NULL, NULL, (GObject *) self);
}


static void
hold_state_changed (TpChannel *channel G_GNUC_UNUSED,
                    guint hold_state,
                    guint reason,
                    gpointer unused G_GNUC_UNUSED,
                    GObject *object)
{
  TIPageHold *self = TI_PAGE_HOLD (object);
  gchar *text;
  const gchar *reason_string = "???";
  const gchar *state_string = "???";

  switch (hold_state)
    {
    case TP_LOCAL_HOLD_STATE_UNHELD:
      state_string = "Call active";
      break;
    case TP_LOCAL_HOLD_STATE_HELD:
      state_string = "Local user placed call on hold";
      break;
    case TP_LOCAL_HOLD_STATE_PENDING_HOLD:
      state_string = "Trying to place call on hold";
      break;
    case TP_LOCAL_HOLD_STATE_PENDING_UNHOLD:
      state_string = "Trying to make call active";
      break;
    }

  switch (reason)
    {
    case TP_LOCAL_HOLD_STATE_REASON_NONE:
      reason_string = "no reason";
      break;
    case TP_LOCAL_HOLD_STATE_REASON_REQUESTED:
      reason_string = "by request";
      break;
    case TP_LOCAL_HOLD_STATE_REASON_RESOURCE_NOT_AVAILABLE:
      reason_string = "resource not available";
      break;
    }

  text = g_strdup_printf ("%s (state %u):\n%s (reason %u)",
      state_string, hold_state, reason_string, reason);
  gtk_label_set_text (self->priv->current_status_label, text);
  log_event (self, text);
  g_free (text);
}


static void
got_hold_state (TpChannel *channel,
                guint hold_state,
                guint reason,
                const GError *error,
                gpointer unused G_GNUC_UNUSED,
                GObject *object)
{
  TIPageHold *self = TI_PAGE_HOLD (object);

  if (error == NULL)
    {
      hold_state_changed (channel, hold_state, reason, NULL, object);
    }
  else
    {
      gchar *text = g_strdup_printf ("Error getting initial hold state:\n%s",
          error->message);

      gtk_label_set_text (self->priv->current_status_label, text);
      log_event (self, text);
      g_free (text);
    }
}


static void
ti_page_hold_constructed (GObject *object)
{
  void (*chain_up) (GObject *) =
    G_OBJECT_CLASS (ti_page_hold_parent_class)->constructed;
  TIPageHold *self = TI_PAGE_HOLD (object);
  GtkBox *hbox;
  GtkWidget *widget, *frame;

  if (chain_up != NULL)
    chain_up (object);

  tp_cli_channel_interface_hold_connect_to_hold_state_changed (
      self->priv->channel, hold_state_changed, NULL, NULL, object,
      NULL);

  tp_cli_channel_interface_hold_call_get_hold_state (self->priv->channel, -1,
      got_hold_state, NULL, NULL, object);

  /* The CM telling us what's going on */
  widget = gtk_label_new ("???");
  self->priv->current_status_label = GTK_LABEL (widget);
  frame = gtk_frame_new ("Current status");
  gtk_container_add (GTK_CONTAINER (frame), widget);
  gtk_box_pack_start (GTK_BOX (self), frame, FALSE, FALSE, 0);

  /* Telling the CM what to do */
  hbox = GTK_BOX (gtk_hbox_new (FALSE, 0));

  widget = gtk_check_button_new_with_label ("Put other user on hold");
  self->priv->hold_toggle = GTK_TOGGLE_BUTTON (widget);
  gtk_box_pack_start (hbox, widget, FALSE, FALSE, 0);

  widget = gtk_button_new_with_label ("Call RequestHold");
  self->priv->request_hold_button = GTK_BUTTON (widget);
  gtk_box_pack_end (hbox, widget, FALSE, FALSE, 0);

  g_signal_connect_swapped (self->priv->request_hold_button, "clicked",
      G_CALLBACK (request_hold_clicked), self);

  frame = gtk_frame_new ("Requested status");
  gtk_container_add (GTK_CONTAINER (frame), (GtkWidget *) hbox);
  gtk_box_pack_start (GTK_BOX (self), frame, FALSE, FALSE, 0);

  /* Event log */
  widget = gtk_text_view_new ();
  self->priv->event_log = GTK_TEXT_VIEW (widget);
  self->priv->event_buffer = gtk_text_view_get_buffer (self->priv->event_log);
  gtk_text_view_set_editable (self->priv->event_log, FALSE);
  gtk_text_view_set_cursor_visible (self->priv->event_log, FALSE);
  frame = gtk_frame_new ("Event log");
  gtk_container_add (GTK_CONTAINER (frame), widget);
  gtk_box_pack_start (GTK_BOX (self), frame, TRUE, TRUE, 0);
}


static void
ti_page_hold_class_init (TIPageHoldClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);
  GParamSpec *param_spec;

  object_class->constructed = ti_page_hold_constructed;
  object_class->dispose = ti_page_hold_dispose;
  object_class->get_property = ti_page_hold_get_property;
  object_class->set_property = ti_page_hold_set_property;

  g_type_class_add_private (klass, sizeof (TIPageHoldPrivate));

  param_spec = g_param_spec_object ("channel", "TpChannel",
      "This page's associated channel", TP_TYPE_CHANNEL,
      G_PARAM_CONSTRUCT_ONLY | G_PARAM_READWRITE |
      G_PARAM_STATIC_NAME | G_PARAM_STATIC_BLURB | G_PARAM_STATIC_NICK);
  g_object_class_install_property (object_class, PROP_CHANNEL, param_spec);
}


TIPageHold *
ti_page_hold_new (GtkNotebook *parent_notebook,
                  TpChannel *channel)
{
  TIPageHold *self = NULL;

  g_return_val_if_fail (channel != NULL, NULL);

  self = g_object_new (TI_TYPE_PAGE_HOLD,
      "channel", channel,
      NULL);

  if (parent_notebook != NULL)
    {
      gtk_notebook_append_page (parent_notebook, (GtkWidget *) self,
          gtk_label_new ("Hold"));
    }

  return self;
}
