/*
 * Stellarium
 * Copyright (C) 2018 Fabien Chereau
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Suite 500, Boston, MA  02110-1335, USA.
 */

#include "l12.h"

#include <math.h>

struct term {
    double ampl;
    double phas;
    double freq;
};

struct sat {
    double mu;
    double al[2];

    int a_len;
    int l_len;
    int z_len;
    int zeta_len;

    struct term a[38];
    struct term l[36];
    struct term z[50];
    struct term zeta[25];
};

static struct sat SATS[4];


static int elem2pv(double mu, const double elem[6], double xv[2][3])
{
    double k, h, q, p, a, al, an, ee, ce, se, de, dle, rsam1, asr, phi, psi,
           x1, y1, vx1, vy1, f2, p2, q2, pq;

    k  = elem[2];
    h  = elem[3];
    q  = elem[4];
    p  = elem[5];
    a  = elem[0];
    al = elem[1];

    an = sqrt(mu / pow(a, 3.0));
    ee = al + k * sin(al) - h * cos(al);

    do {
      ce = cos(ee);
      se = sin(ee);
      de = (al - ee + k * se - h * ce) / (1.0 - k * ce - h * se);
      ee = ee + de;
    } while (fabs(de) > 1.0E-12);

    ce = cos(ee);
    se = sin(ee);
    dle = h * ce - k * se;
    rsam1 = -k * ce - h * se;
    asr =1.0 / (1.0 + rsam1);
    phi = sqrt(1.0 - k*k - h*h);
    psi = 1.0 / (1.0 + phi);
    x1 = a * (ce - k - psi * h * dle);
    y1 = a * (se - h + psi * k * dle);
    vx1 = an * asr * a * (-se - psi * h * rsam1);
    vy1 = an * asr * a * ( ce + psi * k * rsam1);
    f2 = 2.0 * sqrt(1.0 - q*q - p*p);
    p2 = 1.0 -2.0 * p*p;
    q2 = 1.0 -2.0 * q*q;
    pq = 2.0 * p * q;
    xv[0][0] = x1 * p2 + y1 * pq;
    xv[0][1] = x1 * pq + y1 * q2;
    xv[0][2] = (q * y1 - x1 * p) * f2;
    xv[1][0] = vx1 * p2 + vy1 * pq;
    xv[1][1] = vx1 * pq + vy1 * q2;
    xv[1][2] = (q * vy1 - vx1 * p) * f2;
    return 0;
}

const double L1toVsop87[9] = {
   9.994327815023905713e-01, 3.039550993390781261e-02,-1.449924943755843383e-02,
  -3.089770442223671880e-02, 9.988822846893227815e-01,-3.577028369016394015e-02,
   1.339578739122566807e-02, 3.619798764705610479e-02, 9.992548516622136737e-01
};

void GetL12Coor(double jd, int ks, double p[3], double v[3])
{
    const struct sat *sat = &SATS[ks];
    double t, arg, s, s1, s2;
    double elem[6];
    double val[5] = {0.0};
    double xv[2][3];
    int k;

    t = jd - 2433282.5;
    s = 0.0;
    for (k = 0; k < sat->a_len; k++) {
        arg = sat->a[k].phas + sat->a[k].freq * t;
        s += sat->a[k].ampl * cos(arg);
    }
    elem[0] = s;

    s = sat->al[0] + sat->al[1] * t;
    for (k = 0; k < sat->l_len; k++) {
        arg = sat->l[k].phas + sat->l[k].freq * t;
        s += sat->l[k].ampl * sin(arg);
    }
    s = fmod(s + val[0], M_PI * 2);
    if (s < 0.0) s += M_PI * 2;
    elem[1] = s;

    s1 = 0.0;
    s2 = 0.0;
    for (k = 0; k < sat->z_len; k++) {
        arg = sat->z[k].phas + sat->z[k].freq * t;
        s1 += sat->z[k].ampl * cos(arg);
        s2 += sat->z[k].ampl * sin(arg);
    }
    elem[2] = s1 + val[1];
    elem[3] = s2 + val[2];

    s1 = 0.0;
    s2 = 0.0;
    for (k = 0; k < sat->zeta_len; k++) {
        arg = sat->zeta[k].phas + sat->zeta[k].freq * t;
        s1 += sat->zeta[k].ampl * cos(arg);
        s2 += sat->zeta[k].ampl * sin(arg);
    }
    elem[4] = s1 + val[3];
    elem[5] = s2 + val[4];

    // computing cartesian coordinates from elements
    elem2pv(sat->mu, elem, xv);

	p[0] = L1toVsop87[0]*xv[0][0]+L1toVsop87[1]*xv[0][1]+L1toVsop87[2]*xv[0][2];
	p[1] = L1toVsop87[3]*xv[0][0]+L1toVsop87[4]*xv[0][1]+L1toVsop87[5]*xv[0][2];
	p[2] = L1toVsop87[6]*xv[0][0]+L1toVsop87[7]*xv[0][1]+L1toVsop87[8]*xv[0][2];
	// GZ Pure guesswork. I hope these make sense...
	v[0] = L1toVsop87[0]*xv[1][0]+L1toVsop87[1]*xv[1][1]+L1toVsop87[2]*xv[1][2];
	v[1] = L1toVsop87[3]*xv[1][0]+L1toVsop87[4]*xv[1][1]+L1toVsop87[5]*xv[1][2];
	v[2] = L1toVsop87[6]*xv[1][0]+L1toVsop87[7]*xv[1][1]+L1toVsop87[8]*xv[1][2];
}


static struct sat SATS[4] = {
    // sat 1 (IO)
    {
        .mu = 0.2824894284338140E-06,
        .al = {0.1446213296021224E+01,  0.3551552286182400E+01},
        .a_len = 38,
        .a = {
            { 0.0028210960212903, 0.00000000000000E+00, 0.00000000000000E+00},
            { 0.0000000762024588, 0.36392902322306E+01, 0.35644591656241E+01},
            { 0.0000000180900324, 0.99554707056522E+00, 0.71289183312483E+01},
            { 0.0000000172337652, 0.18196487820921E+01, 0.17822295777568E+01},
            { 0.0000000101726080, 0.28150559763861E+01, 0.89111478635073E+01},
            { 0.0000000094794086, 0.34760224933239E+01, 0.80200331112799E+01},
            { 0.0000000092196266, 0.46347004953370E+01, 0.10693377436209E+02},
            { 0.0000000058581604, 0.11586746335276E+01, 0.26733443704266E+01},
            {-0.0000000036218148, 0.23173675289588E+01, 0.53466887181044E+01},
            { 0.0000000034892754, 0.17122470613669E+00, 0.12475607079684E+02},
            { 0.0000000030842852, 0.36170311370435E+01, 0.63501320826717E+01},
            { 0.0000000020794650, 0.19906655633153E+01, 0.14257836656755E+02},
            { 0.0000000013655244, 0.49369712857369E+01, 0.13584836518140E-01},
            { 0.0000000011682572, 0.57934065580556E+01, 0.13366721796637E+02},
            {-0.0000000008031976, 0.66879731833041E+00, 0.16040066232595E+02},
            { 0.0000000007309510, 0.56300556878949E+01, 0.17822295806244E+02},
            { 0.0000000007014118, 0.43297377080515E+01, 0.71002044886497E+01},
            { 0.0000000006561624, 0.43188797534991E+01, 0.13034138433510E-01},
            { 0.0000000005753088, 0.54252179509841E+01, 0.95251981240076E+01},
            { 0.0000000004359548, 0.11670110887440E+01, 0.19604525331797E+02},
            { 0.0000000003711992, 0.14936154077537E+01, 0.12938928912340E-01},
            {-0.0000000003412576, 0.26346374300664E+01, 0.15571117257960E-01},
            { 0.0000000003432980, 0.17994723387341E+01, 0.31750663461810E+01},
            { 0.0000000003228344, 0.29861854159944E+01, 0.21386754933987E+02},
            { 0.0000000003014418, 0.19871924348983E+00, 0.24675315510310E-01},
            { 0.0000000001707670, 0.50718778620273E+01, 0.35514255456604E+01},
            { 0.0000000001655832, 0.29783205832994E+01, 0.44555739317536E+01},
            { 0.0000000001612910, 0.48058392680935E+01, 0.23168984521460E+02},
            { 0.0000000001527992, 0.18275651107267E+01, 0.18713410599600E+02},
            { 0.0000000001523312, 0.46323297275220E+01, 0.44686092108182E+01},
            { 0.0000000001449720, 0.19079860214667E+01, 0.30506511533200E-02},
            { 0.0000000001188688, 0.53321680658912E+01, 0.70987549449082E+01},
            { 0.0000000001129258, 0.95031497804420E+00, 0.12700264165343E+02},
            { 0.0000000000986086, 0.34190944178580E+00, 0.24951214111224E+02},
            {-0.0000000000877720, 0.36228267942948E+01, 0.17958145576535E+01},
            { 0.0000000000857194, 0.33682834215727E+01, 0.59736711266730E+01},
            {-0.0000000000545492, 0.19473964103154E+01, 0.22929425718040E-01},
            { 0.0000000000326102, 0.24880420823571E+01, 0.25119610718870E-01},
        },
        .l_len = 32,
        .l = {
            {-0.0001925258348666, 0.49369589722645E+01, 0.13584836583050E-01},
            {-0.0000970803596076, 0.43188796477322E+01, 0.13034138432430E-01},
            {-0.0000898817416500, 0.19080016428617E+01, 0.30506486715800E-02},
            {-0.0000553101050262, 0.14936156681569E+01, 0.12938928911550E-01},
            {-0.0000503584426150, 0.36410196089987E+01, 0.35644591049605E+01},
            {-0.0000444412770116, 0.18196478828985E+01, 0.17822295777568E+01},
            { 0.0000418078870490, 0.26346334480977E+01, 0.15571117221300E-01},
            { 0.0000372356597388, 0.21402440902650E+01, 0.14500977488900E-02},
            {-0.0000234440533016, 0.19871945729267E+00, 0.24675315507400E-01},
            {-0.0000160313164240, 0.28203470990931E+01, 0.95196190000000E-04},
            {-0.0000119049755698, 0.99521552502799E+00, 0.71289183312483E+01},
            {-0.0000109014269320, 0.11586742711973E+01, 0.26733443704266E+01},
            { 0.0000087217118104, 0.22995085327344E+01, 0.44456805185000E-03},
            { 0.0000082229455492, 0.84723690387904E+00, 0.54980078903000E-03},
            { 0.0000075365481720, 0.30644603245150E+01, 0.64826749624000E-03},
            {-0.0000061452803962, 0.28150499448772E+01, 0.89111478635073E+01},
            {-0.0000057575824778, 0.34760236756099E+01, 0.80200331112799E+01},
            {-0.0000053196302672, 0.14952058549171E+01, 0.29001290992300E-02},
            {-0.0000051181206936, 0.46347077042449E+01, 0.10693377436209E+02},
            {-0.0000047817413326, 0.49236512419835E+01, 0.30554833877800E-02},
            { 0.0000045554015322, 0.19585097634352E+01, 0.22928625941210E-01},
            { 0.0000043204134698, 0.15842888614383E+01, 0.15677112478190E-01},
            { 0.0000037684098282, 0.23173652780077E+01, 0.53466887181044E+01},
            {-0.0000031403738248, 0.22184076281042E+01, 0.25155489165510E-01},
            { 0.0000024336535428, 0.85320650238886E+00, 0.25426968834400E-02},
            {-0.0000020289901692, 0.36168998565188E+01, 0.63501320826717E+01},
            { 0.0000018665438704, 0.48458061649481E+01, 0.13589674898130E-01},
            {-0.0000018552431038, 0.17086811529922E+00, 0.12475607079684E+02},
            {-0.0000016229875536, 0.62803206871082E+01, 0.60414171604000E-03},
            {-0.0000013160987604, 0.14718125754925E+01, 0.14358460012320E-01},
            { 0.0000008070729808, 0.38735416148641E+00, 0.37658680379100E-02},
            { 0.0000002602397658, 0.14337589305551E+01, 0.45692429208000E-02},
        },
        .z_len = 23,
        .z = {
            { 0.0041510849668155, 0.40899396355450E+01,-0.12906864146660E-01},
            { 0.0006260521444113, 0.14461888986270E+01, 0.35515522949802E+01},
            { 0.0000352747346169, 0.21256287034578E+01, 0.12727416567000E-03},
            { 0.0000198194483636, 0.55835619926762E+01, 0.32065751140000E-04},
            { 0.0000146399842989, 0.44137212696837E+00, 0.26642533547700E-02},
            { 0.0000098749504021, 0.45076118781320E+00,-0.35773660260022E+01},
            {-0.0000096819265753, 0.59097266550442E+01, 0.17693227079462E+01},
            {-0.0000083063168209, 0.28751474873012E+00, 0.87820791951527E+00},
            { 0.0000059689735869, 0.50740752477871E+01, 0.71160118048918E+01},
            {-0.0000052220588690, 0.27460731023666E+01, 0.67796100936000E-03},
            { 0.0000046538995236, 0.49143203339385E+01,-0.53595956184347E+01},
            { 0.0000045951340101, 0.42533513770304E+01,-0.44684808200578E+01},
            {-0.0000037711061757, 0.54120093562773E+01,-0.17951364445825E+01},
            { 0.0000037405126681, 0.30946737347297E+01,-0.71418251640916E+01},
            { 0.0000022044764663, 0.54360702580001E+01,-0.26491700241240E-01},
            { 0.0000018698303790, 0.41124042914226E+01,-0.27985797954589E+01},
            {-0.0000015410375360, 0.27141931505529E+01, 0.27731236679900E-02},
            { 0.0000013214613496, 0.12750177723530E+01,-0.89240547799787E+01},
            {-0.0000012707585609, 0.51141075152507E+01, 0.37654227378982E+00},
            { 0.0000012193607962, 0.59977053365953E+01,-0.98566169956900E-02},
            {-0.0000011886104747, 0.32658350285168E+01, 0.53337818460633E+01},
            { 0.0000008742035177, 0.23903528311144E+01, 0.25194921818800E-02},
            {-0.0000007689215742, 0.38308837306225E+01,-0.27293225500100E-02},
        },
        .zeta_len = 15,
        .zeta = {
            { 0.0003142172466014, 0.27964219722923E+01,-0.23150960980000E-02},
            { 0.0000904169207946, 0.10477061879627E+01,-0.56920638196000E-03},
            { 0.0000175695395780, 0.24150809680215E+01, 0.00000000000000E+00},
            { 0.0000164452324013, 0.33368861773902E+01,-0.12491307197000E-03},
            { 0.0000055424829091, 0.59720202381027E+01,-0.30561164720000E-04},
            { 0.0000035856270353, 0.84898736841329E+00,-0.25244521900630E-01},
            { 0.0000024180760140, 0.55603770950923E+01, 0.29003681445800E-02},
            {-0.0000008673084930, 0.28496686106299E+00,-0.14500593353200E-02},
            {-0.0000003176227277, 0.53834633036029E+01,-0.23498632298700E-01},
            { 0.0000003152816608, 0.45569499027478E+01, 0.43504654304000E-02},
            { 0.0000002338676726, 0.17633292120047E+01, 0.14501339138600E-02},
            { 0.0000001754553689, 0.48429319984493E+01,-0.25688816532440E-01},
            { 0.0000001286319583, 0.57543347143871E+01,-0.25813660979740E-01},
            { 0.0000000967213304, 0.11503592426900E+01,-0.29001471397800E-02},
            { 0.0000000000692310, 0.40745966852008E+01,-0.32506757319070E-01},
        },
    },
    // sat 2 (Europa)
    {
        .mu = 0.2824832743928930E-06,
        .al = {-0.3735263437471362E+00,  0.1769322711123470E+01},
        .a_len = 38,
        .a = {
            { 0.0044871037804314, 0.00000000000000E+00, 0.00000000000000E+00},
            { 0.0000004324367498, 0.18196456062910E+01, 0.17822295777568E+01},
            { 0.0000001603614750, 0.43002726529577E+01, 0.26733443704266E+01},
            {-0.0000001019146786, 0.54589480865442E+01, 0.53466887181044E+01},
            { 0.0000000924734786, 0.56222139048906E+01, 0.89111478887838E+00},
            {-0.0000000523665800, 0.36392846323417E+01, 0.35644591656241E+01},
            { 0.0000000511509000, 0.29783307371014E+01, 0.44555739317536E+01},
            {-0.0000000311907780, 0.99466557754027E+00, 0.71289183312483E+01},
            {-0.0000000272859938, 0.28144480309092E+01, 0.89111478635073E+01},
            { 0.0000000232225828, 0.62608434364366E+01, 0.27856729211550E+01},
            {-0.0000000181310770, 0.43188692380649E+01, 0.13034138308860E-01},
            { 0.0000000174960544, 0.16563941638726E+01, 0.62378035398422E+01},
            {-0.0000000122874072, 0.46421290370833E+01, 0.10693377254218E+02},
            {-0.0000000095367130, 0.14936536615312E+01, 0.12938928820690E-01},
            {-0.0000000084863836, 0.17146854643555E+00, 0.12475607079684E+02},
            { 0.0000000071939342, 0.49376739095661E+01, 0.13584833017030E-01},
            { 0.0000000069122354, 0.62488746138492E+01, 0.41785094280464E+01},
            { 0.0000000061377568, 0.33434976298081E+00, 0.80200331112799E+01},
            {-0.0000000045343054, 0.19892156959655E+01, 0.14257836656755E+02},
            { 0.0000000044574684, 0.34597804303324E+00, 0.35357692227539E+01},
            { 0.0000000042350072, 0.62719655202169E+01, 0.13928364636651E+01},
            {-0.0000000028783772, 0.38108811302610E+01, 0.16040066232595E+02},
            { 0.0000000024354662, 0.99587190880214E+00, 0.90414989297380E+00},
            { 0.0000000022532940, 0.52958965893939E+01, 0.98022627054664E+01},
            { 0.0000000021573570, 0.62379050559630E+01, 0.55713458670107E+01},
            {-0.0000000016530062, 0.56456686036734E+01, 0.17822294694543E+02},
            { 0.0000000016464798, 0.26346435392424E+01, 0.15571117126760E-01},
            { 0.0000000011589838, 0.32732388195745E+01, 0.17691951440716E+01},
            {-0.0000000010251826, 0.19079858535660E+01, 0.30506497838200E-02},
            {-0.0000000010203510, 0.11692020351116E+01, 0.19604525194172E+02},
            { 0.0000000007614982, 0.16862812414995E+01, 0.35342961443230E+01},
            { 0.0000000007104494, 0.59112717191092E+01, 0.24092214574831E+01},
            {-0.0000000006957184, 0.24879412197796E+01, 0.25119609730670E-01},
            {-0.0000000005817914, 0.19872303312324E+00, 0.24675315511270E-01},
            {-0.0000000003792178, 0.15765189821595E+01, 0.25244441830200E-01},
            { 0.0000000003397378, 0.58126953372535E+01, 0.25973067138760E-01},
            { 0.0000000003159492, 0.23545476741301E+01, 0.26068277099550E-01},
            { 0.0000000002538154, 0.19471441186087E+01, 0.22929424919760E-01},
        },
        .l_len = 36,
        .l = {
            { 0.0008576433172936, 0.43188693178264E+01, 0.13034138308050E-01},
            { 0.0004549582875086, 0.14936531751079E+01, 0.12938928819620E-01},
            { 0.0003248939825174, 0.18196494533458E+01, 0.17822295777568E+01},
            {-0.0003074250079334, 0.49377037005911E+01, 0.13584832867240E-01},
            { 0.0001982386144784, 0.19079869054760E+01, 0.30510121286900E-02},
            { 0.0001834063551804, 0.21402853388529E+01, 0.14500978933800E-02},
            {-0.0001434383188452, 0.56222140366630E+01, 0.89111478887838E+00},
            {-0.0000771939140944, 0.43002724372350E+01, 0.26733443704266E+01},
            {-0.0000632289777196, 0.26346392822098E+01, 0.15571117084700E-01},
            { 0.0000446766477388, 0.54589448561143E+01, 0.53466887181044E+01},
            { 0.0000436574731410, 0.36392908617709E+01, 0.35644591656241E+01},
            { 0.0000349172750296, 0.28289867162553E+01, 0.29885749150000E-04},
            {-0.0000325709094646, 0.53721409780230E+01, 0.12495233774000E-03},
            { 0.0000205826473860, 0.15258464215508E+01, 0.29001315522200E-02},
            {-0.0000192706087556, 0.29783311531879E+01, 0.44555739317536E+01},
            { 0.0000168028316254, 0.24879414119403E+01, 0.25119609725650E-01},
            {-0.0000141628733606, 0.29183576504413E+01, 0.64930403718000E-03},
            { 0.0000140713155600, 0.19872319369353E+00, 0.24675315510310E-01},
            { 0.0000131946915760, 0.99584744364935E+00, 0.71289183312483E+01},
            { 0.0000106598617620, 0.53356907396678E+01, 0.30233219231900E-02},
            {-0.0000104011727738, 0.62608296198866E+01, 0.27856729211550E+01},
            { 0.0000100746080234, 0.44288900030073E+01, 0.55297871931000E-03},
            { 0.0000097414019416, 0.27312462188296E+01, 0.89111510230745E+01},
            {-0.0000094651366640, 0.25010358163865E+01, 0.93478322470000E-04},
            { 0.0000091108073324, 0.15765182522628E+01, 0.25244441682120E-01},
            {-0.0000087720567668, 0.15376962386886E+01, 0.15676393315070E-01},
            {-0.0000078429703340, 0.58128473756772E+01, 0.25973069246350E-01},
            {-0.0000075566039418, 0.30586251688920E+01, 0.43252872559000E-03},
            {-0.0000066580990752, 0.19591270593390E+01, 0.22928567412490E-01},
            {-0.0000065854142774, 0.18617673337640E+01, 0.26093058384670E-01},
            {-0.0000058131135230, 0.16563893807978E+01, 0.62378035398422E+01},
            { 0.0000055720865276, 0.39565695752204E+01, 0.25481216339900E-02},
            {-0.0000048198508906, 0.62720230965345E+01, 0.13928364605775E+01},
            { 0.0000042728431266, 0.46220912946918E+01, 0.10693377982182E+02},
            { 0.0000042175545304, 0.13509343368359E+01, 0.30164435787800E-02},
            { 0.0000037707624520, 0.51034507119889E+01, 0.25219658202250E-01},
        },
        .z_len = 41,
        .z = {
            {-0.0093589104136341, 0.40899396509039E+01,-0.12906864146660E-01},
            { 0.0002988994545555, 0.59097265185595E+01, 0.17693227079462E+01},
            { 0.0002139036390350, 0.21256289300016E+01, 0.12727418407000E-03},
            { 0.0001980963564781, 0.27435168292650E+01, 0.67797343009000E-03},
            { 0.0001210388158965, 0.55839943711203E+01, 0.32056614900000E-04},
            { 0.0000837042048393, 0.16094538368039E+01,-0.90402165808846E+00},
            { 0.0000823525166369, 0.14461887708689E+01, 0.35515522949802E+01},
            {-0.0000315906532820, 0.28751224400811E+00, 0.87820791951527E+00},
            {-0.0000294503681314, 0.45078002968967E+00,-0.35773660260022E+01},
            {-0.0000278946698536, 0.22704374310903E+01,-0.17951364497113E+01},
            { 0.0000144958688621, 0.29313956641719E+01,-0.26862512422390E+01},
            { 0.0000139052321679, 0.60542576187622E+01,-0.25941002404500E-01},
            { 0.0000108374431350, 0.59320761116863E+01,-0.10163502160128E+01},
            {-0.0000082175838585, 0.49144730088838E+01,-0.53595956184347E+01},
            { 0.0000073925894084, 0.25962855881215E+01,-0.25845792927870E-01},
            { 0.0000062618381566, 0.62252936384007E+01,-0.71418248393794E+01},
            {-0.0000051968296512, 0.54353355159239E+01,-0.26491696725040E-01},
            {-0.0000043507065743, 0.51150292346242E+01, 0.37654221060604E+00},
            { 0.0000042081682285, 0.31202613836361E+01, 0.44427230757158E+01},
            { 0.0000041298266970, 0.42533371370636E+01,-0.44684808200578E+01},
            {-0.0000036991221930, 0.52487564172390E+01, 0.26604375002057E+01},
            {-0.0000027357551003, 0.12734806685602E+01,-0.89240546532297E+01},
            {-0.0000026854901206, 0.75596663258784E+00, 0.15806953180460E-01},
            { 0.0000023074479953, 0.19438998534712E+01, 0.71160114825227E+01},
            { 0.0000020163445050, 0.58484195254467E+01,-0.24091843401454E+01},
            {-0.0000018506530067, 0.26838225102582E+01, 0.68236609075000E-03},
            { 0.0000018159137522, 0.26048690461733E+01, 0.62248966818788E+01},
            {-0.0000017894118824, 0.57385537790777E+01,-0.10706284328884E+02},
            { 0.0000016518864520, 0.32658492478888E+01, 0.53337818460633E+01},
            {-0.0000015660692561, 0.61789350505156E+01,-0.11453588847960E-01},
            { 0.0000014426949422, 0.60014075911383E+01,-0.17664769149811E+01},
            { 0.0000013196935928, 0.55753025652974E+01,-0.62507103665413E+01},
            {-0.0000011726743714, 0.50242932747650E+01,-0.14351210704140E-01},
            {-0.0000009550285338, 0.28409403047363E+01, 0.14257595044900E-02},
            {-0.0000007569857746, 0.38098760906143E+01,-0.27271627793800E-02},
            {-0.0000007495662748, 0.29896372346394E+01, 0.20097553243900E-02},
            { 0.0000007091149133, 0.27139331814919E+01,-0.19924932783300E-02},
            { 0.0000005646670312, 0.21683602575236E+01,-0.12871803232940E-01},
            {-0.0000002004455524, 0.12893849410519E+01,-0.32724923477800E-02},
            {-0.0000001623489363, 0.24189454629613E+00, 0.44609337678800E-02},
            { 0.0000001058862562, 0.45356953407129E+01, 0.39269908172100E-02},
        },
        .zeta_len = 25,
        .zeta = {
            { 0.0040404917832303, 0.10477063169425E+01,-0.56920640540000E-03},
            { 0.0002200421034564, 0.33368857864364E+01,-0.12491307307000E-03},
            { 0.0001662544744719, 0.24134862374711E+01, 0.00000000000000E+00},
            { 0.0000590282470983, 0.59719930968366E+01,-0.30561602250000E-04},
            {-0.0000105030331400, 0.27964978379152E+01,-0.23150966123800E-02},
            {-0.0000102943248250, 0.84898796322150E+00,-0.25244521901650E-01},
            { 0.0000072600013020, 0.55603730312676E+01, 0.29003676713100E-02},
            { 0.0000018391258758, 0.28480515491153E+00,-0.14500579196900E-02},
            { 0.0000014880605763, 0.48429974929766E+01,-0.25688815138710E-01},
            {-0.0000008828196274, 0.65011185407635E+00, 0.34696170683100E-02},
            { 0.0000008714042768, 0.17639430319108E+01, 0.14501352157600E-02},
            { 0.0000008536188044, 0.45568506666427E+01, 0.43504641410100E-02},
            { 0.0000006846214331, 0.57542117253981E+01,-0.25813768702630E-01},
            { 0.0000004471826348, 0.53834694321520E+01,-0.23498632366370E-01},
            { 0.0000003034392168, 0.22078201315180E+01,-0.25783170906020E-01},
            { 0.0000001799083735, 0.31858868501531E+01, 0.88086056517000E-03},
            {-0.0000001792048645, 0.51949494917342E+01,-0.20193236931900E-02},
            {-0.0000001098546626, 0.59286821904995E+01, 0.49197316579700E-02},
            {-0.0000001083128732, 0.45808061408794E+01,-0.59959459406000E-03},
            { 0.0000001062153531, 0.38387102863271E+01,-0.53795085847000E-03},
            { 0.0000000768496749, 0.35553768729770E+01, 0.58005587812700E-02},
            {-0.0000000692273841, 0.46440611341931E+01, 0.30253219029200E-02},
            { 0.0000000676969224, 0.13621319661456E+00,-0.44430413602000E-03},
            {-0.0000000621559952, 0.30093497179950E+01,-0.13603287200690E-01},
            { 0.0000000000608298, 0.40529569532600E+01,-0.32510869900940E-01},
        },
    },
    // sat 3 (Ganymede)
    {
        .mu = 0.2824981841847230E-06,
        .al = {0.2874089391143348E+00,  0.8782079235893280E+00},
        .a_len = 38,
        .a = {
            { 0.0071566594572575, 0.00000000000000E+00, 0.00000000000000E+00},
            { 0.0000013930299110, 0.11586745884981E+01, 0.26733443704266E+01},
            { 0.0000006449829346, 0.56222145702102E+01, 0.89111478887838E+00},
            { 0.0000002298059520, 0.12995924044108E+01, 0.10034433456729E+01},
            {-0.0000001221434370, 0.49612436330515E+01, 0.17822295777568E+01},
            { 0.0000001095798176, 0.19486708778350E+01, 0.15051650461529E+01},
            { 0.0000000701435616, 0.64978508114196E+00, 0.50172166963138E+00},
            { 0.0000000547868566, 0.25992050672074E+01, 0.20068866945508E+01},
            {-0.0000000394635858, 0.23173535605652E+01, 0.53466887181044E+01},
            {-0.0000000363221428, 0.36393008632056E+01, 0.35644591656241E+01},
            { 0.0000000290949364, 0.20123392230605E+01, 0.17535157350384E+01},
            { 0.0000000281244968, 0.32490010762048E+01, 0.25086083721948E+01},
            {-0.0000000207924698, 0.29783308899923E+01, 0.44555739317536E+01},
            { 0.0000000146896774, 0.38988244013504E+01, 0.30103300418262E+01},
            {-0.0000000119930042, 0.16563968316083E+01, 0.62378035398422E+01},
            { 0.0000000112067460, 0.43188665692819E+01, 0.13034138285340E-01},
            {-0.0000000109535132, 0.49372826282154E+01, 0.13584834937940E-01},
            { 0.0000000099867772, 0.96700720263958E+00, 0.62699633776977E+00},
            { 0.0000000077668260, 0.45486373016444E+01, 0.35120517182683E+01},
            { 0.0000000074143972, 0.16140449852661E+00, 0.12531361661566E+00},
            { 0.0000000066346638, 0.33441073536010E+00, 0.80200331112799E+01},
            { 0.0000000057842684, 0.14936630646671E+01, 0.12938928799370E-01},
            {-0.0000000055768352, 0.44651777597613E+01, 0.11287386144710E+01},
            {-0.0000000049395106, 0.61563894598809E+01, 0.17520662093793E+01},
            { 0.0000000041439704, 0.51984558307998E+01, 0.40137734147421E+01},
            {-0.0000000040765630, 0.99543742426922E+00, 0.71289183312483E+01},
            {-0.0000000036862062, 0.46386836178626E+01, 0.10693377254218E+02},
            { 0.0000000033617538, 0.37493658441448E+01, 0.87808669180168E+00},
            { 0.0000000033348284, 0.22668196818990E+01, 0.16304394485673E+01},
            {-0.0000000025754698, 0.33293196902303E+00, 0.17952648120307E+01},
            { 0.0000000024363084, 0.19604838407749E+01, 0.11232854513197E+00},
            { 0.0000000022265432, 0.58482745704418E+01, 0.45154950951905E+01},
            { 0.0000000020032676, 0.29166648062069E+01, 0.21321610765333E+01},
            {-0.0000000018115706, 0.99782757414001E+00, 0.90414978368384E+00},
            { 0.0000000014535006, 0.18748212041600E+01, 0.89112137093506E+01},
            {-0.0000000006819260, 0.19871670124324E+00, 0.24675315493830E-01},
            { 0.0000000004433776, 0.24880003003965E+01, 0.25119610196650E-01},
            {-0.0000000002836658, 0.58126277034761E+01, 0.25973068607520E-01},
        },
        .l_len = 31,
        .l = {
            { 0.0002310797886226, 0.21402987195942E+01, 0.14500978438400E-02},
            {-0.0001828635964118, 0.43188672736968E+01, 0.13034138282630E-01},
            { 0.0001512378778204, 0.49373102372298E+01, 0.13584834812520E-01},
            {-0.0001163720969778, 0.43002659861490E+01, 0.26733443704266E+01},
            {-0.0000955478069846, 0.14936612842567E+01, 0.12938928798570E-01},
            { 0.0000815246854464, 0.56222137132535E+01, 0.89111478887838E+00},
            {-0.0000801219679602, 0.12995922951532E+01, 0.10034433456729E+01},
            {-0.0000607017260182, 0.64978769669238E+00, 0.50172167043264E+00},
            { 0.0000543922473002, 0.27927547440639E+01, 0.29880873700000E-04},
            {-0.0000489253646474, 0.53711728089803E+01, 0.12495278292000E-03},
            {-0.0000427574981536, 0.18196513407448E+01, 0.17822295777568E+01},
            {-0.0000307360417826, 0.19498372703786E+01, 0.15051650064903E+01},
            {-0.0000169767346458, 0.19078637281659E+01, 0.30507678226700E-02},
            { 0.0000154725890508, 0.56912713028984E+01, 0.65164073556000E-03},
            {-0.0000145268863648, 0.18863875475387E+00, 0.12530827181195E+00},
            {-0.0000135654458738, 0.27930238268852E+01, 0.55663681407000E-03},
            {-0.0000134648621904, 0.25991972928128E+01, 0.20068866945508E+01},
            { 0.0000095524017320, 0.23173520454449E+01, 0.53466887181044E+01},
            { 0.0000087955125170, 0.36393024031096E+01, 0.35644591656241E+01},
            { 0.0000075462003630, 0.53560617584395E+01, 0.92426977490000E-04},
            {-0.0000071146195958, 0.20120561622463E+01, 0.17535157644008E+01},
            { 0.0000064153141218, 0.15526366820734E+01, 0.29001309732400E-02},
            {-0.0000063221625942, 0.32490122452649E+01, 0.25086083721948E+01},
            {-0.0000056564973024, 0.24862139082596E+01, 0.44834622386000E-03},
            { 0.0000052570245720, 0.19871532348033E+00, 0.24675315501580E-01},
            { 0.0000047020767994, 0.29783317790630E+01, 0.44555739317536E+01},
            {-0.0000047004229470, 0.96617050453708E+00, 0.62699712737505E+00},
            {-0.0000046565198820, 0.36125113449716E+01, 0.43633231340000E-03},
            {-0.0000042349322008, 0.19604744669606E+01, 0.11232854282257E+00},
            {-0.0000038755741918, 0.22619624763183E+01, 0.25146663939730E-01},
            {-0.0000032577733688, 0.56861827246039E+01, 0.17074576501600E-02},
        },
        .z_len = 50,
        .z = {
            { 0.0014289811307319, 0.21256295942739E+01, 0.12727413029000E-03},
            { 0.0007710931226760, 0.55836330003496E+01, 0.32064341100000E-04},
            { 0.0005925911780766, 0.40899396636448E+01,-0.12906864146660E-01},
            { 0.0002045597496146, 0.52713683670372E+01,-0.12523544076106E+00},
            { 0.0001785118648258, 0.28743156721063E+00, 0.87820792442520E+00},
            { 0.0001131999784893, 0.14462127277818E+01, 0.35515522949802E+01},
            {-0.0000658778169210, 0.22702423990985E+01,-0.17951364394537E+01},
            { 0.0000497058888328, 0.59096792204858E+01, 0.17693227129285E+01},
            {-0.0000316384926978, 0.16093054939404E+01,-0.90402165028424E+00},
            { 0.0000287801237327, 0.46217321268757E+01,-0.62695712341840E+00},
            {-0.0000181744317896, 0.59210641379360E+01, 0.37648623991673E+00},
            { 0.0000105558175161, 0.39720191398746E+01,-0.11286788041058E+01},
            {-0.0000070808673396, 0.60542548894164E+01,-0.25941002415210E-01},
            {-0.0000070804404020, 0.27978433776854E+01, 0.67774258703000E-03},
            {-0.0000061046181888, 0.14151685760988E+01,-0.87530769416913E+00},
            {-0.0000057610853129, 0.42530537622646E+01,-0.44684807882788E+01},
            {-0.0000057310334964, 0.29311803223072E+01,-0.26862512192699E+01},
            { 0.0000048299146941, 0.27138294508149E+01, 0.27731329671900E-02},
            { 0.0000046610005483, 0.33222980229554E+01,-0.16304004832039E+01},
            {-0.0000038142769361, 0.25962943627643E+01,-0.25845792955510E-01},
            { 0.0000034982417330, 0.15866568011217E+01, 0.18816512920593E+01},
            {-0.0000030091617315, 0.35921173988567E+01,-0.35773660056343E+01},
            {-0.0000024732926446, 0.53461730094807E+01, 0.25122576835111E+00},
            { 0.0000024416432533, 0.47049477027963E+01,-0.25049613834712E+00},
            { 0.0000024171568015, 0.34508032389167E+01, 0.00000000000000E+00},
            { 0.0000023143850535, 0.55385759257808E+01, 0.28683339028800E-02},
            { 0.0000022651772374, 0.55608006706192E+01, 0.14501892967800E-02},
            { 0.0000022247695560, 0.26725424635341E+01,-0.21321221654766E+01},
            { 0.0000020947921969, 0.22350374116258E+01, 0.23833730192673E+01},
            {-0.0000014042712722, 0.93718044411525E+00, 0.13799296041822E+01},
            { 0.0000011932531874, 0.28861941414418E+01, 0.28850946416201E+01},
            {-0.0000011180389240, 0.49139919849718E+01,-0.53595955727170E+01},
            { 0.0000011076384510, 0.20227538540345E+01,-0.26338438454332E+01},
            {-0.0000010371714944, 0.40722739402948E+00,-0.87385759089274E+00},
            {-0.0000008993128501, 0.30942691883530E+01,-0.71418251640916E+01},
            { 0.0000007268381420, 0.54334774230433E+01,-0.26491687896550E-01},
            {-0.0000007178049665, 0.52487423493616E+01, 0.26604375002057E+01},
            { 0.0000006908412319, 0.40596134184175E+01,-0.75221793556997E+00},
            {-0.0000006784151570, 0.38846818226669E+01, 0.42496535534400E-02},
            { 0.0000006772314920, 0.23013479896873E+01, 0.26317235358158E+01},
            { 0.0000006659820028, 0.35359530295550E+01, 0.33868163258510E+01},
            {-0.0000006339665249, 0.39268665697903E+01, 0.44426670658559E+01},
            {-0.0000006286307601, 0.19440608894162E+01, 0.71160114019304E+01},
            {-0.0000006128705113, 0.25027415074658E+01, 0.62249001971556E+01},
            { 0.0000005660807396, 0.13729316457251E+01,-0.31355655165873E+01},
            {-0.0000005206551413, 0.55749300982469E+01,-0.62507103665413E+01},
            {-0.0000004718481418, 0.45366605084874E+01, 0.16786677353000E-03},
            {-0.0000004583970422, 0.19351070248496E+01,-0.98151695574500E+01},
            {-0.0000004577854173, 0.62350780976534E+01, 0.17563373058434E+01},
            { 0.0000003466029660, 0.75412427489767E+00, 0.15807097495700E-01},
        },
        .zeta_len = 18,
        .zeta = {
            { 0.0015932721570848, 0.33368862796665E+01,-0.12491307058000E-03},
            { 0.0008533093128905, 0.24133881688166E+01, 0.00000000000000E+00},
            { 0.0003513347911037, 0.59720789850127E+01,-0.30561017710000E-04},
            {-0.0001441929255483, 0.10477061764435E+01,-0.56920632124000E-03},
            { 0.0000157303527750, 0.55604041197704E+01, 0.29003665011200E-02},
            { 0.0000025161319881, 0.28477653709685E+00,-0.14500554486800E-02},
            { 0.0000020438305183, 0.17652628559998E+01, 0.14501383926500E-02},
            { 0.0000017939612784, 0.45568977341583E+01, 0.43504621590400E-02},
            { 0.0000013614276895, 0.84898872627945E+00,-0.25244521900630E-01},
            {-0.0000008996109017, 0.46441156003340E+01, 0.30253214588300E-02},
            {-0.0000008702078430, 0.27972000093551E+01,-0.23150965645100E-02},
            {-0.0000004371144064, 0.48429530385679E+01,-0.25688816011500E-01},
            {-0.0000002174259374, 0.57543785603741E+01,-0.25813642993310E-01},
            {-0.0000001926397869, 0.20118539705648E+01, 0.29330596864500E-02},
            { 0.0000001589279656, 0.35554727018503E+01, 0.58005577768400E-02},
            {-0.0000001432228753, 0.11966574544002E+01,-0.15750124983800E-02},
            {-0.0000000926213408, 0.22052538606469E+01,-0.25782797426020E-01},
            { 0.0000000000106902, 0.45764213311755E+01,-0.32611614716800E-01},
        },
    },
    // Sat 4 (Callisto)
    {
        .mu = 0.2824921448899090E-06,
        .al = {-0.3620341291375704E+00,  0.3764862334338280E+00},

        .a_len = 22,
        .a = {
            { 0.0125879701715314, 0.00000000000000E+00, 0.00000000000000E+00},
            { 0.0000035952049470, 0.64965776007116E+00, 0.50172168165034E+00},
            { 0.0000027580210652, 0.18084235781510E+01, 0.31750660413359E+01},
            { 0.0000012874896172, 0.62718908285025E+01, 0.13928364698403E+01},
            {-0.0000004173729106, 0.12990650292663E+01, 0.10034433697108E+01},
            { 0.0000002790757718, 0.71428870045577E+00, 0.75007225869130E+00},
            {-0.0000001998252258, 0.19489881012004E+01, 0.15051650461529E+01},
            {-0.0000001001149838, 0.25987168731338E+01, 0.20068867266014E+01},
            {-0.0000000513967092, 0.32484798706247E+01, 0.25086084022422E+01},
            {-0.0000000475687992, 0.48635521917696E+01, 0.74862216593606E+00},
            { 0.0000000348242240, 0.15082713497295E+00, 0.37645917070525E+00},
            { 0.0000000283840630, 0.51672973364888E+01, 0.12530678073049E+00},
            {-0.0000000263234638, 0.33499822210495E+01, 0.30103491232578E+01},
            { 0.0000000239106346, 0.43573519442736E+01, 0.62698238798737E+00},
            { 0.0000000219977422, 0.15075404808879E+01, 0.27986109086768E+01},
            {-0.0000000171144478, 0.62607361864777E+01, 0.27856729335053E+01},
            {-0.0000000141956834, 0.45481077718910E+01, 0.35120517575303E+01},
            {-0.0000000120003630, 0.18583887479127E+01, 0.11287042579152E+01},
            { 0.0000000108418904, 0.54873138800427E+01, 0.67395238593127E+01},
            { 0.0000000108218254, 0.59772630516669E+01, 0.10163811590412E+01},
            { 0.0000000002477642, 0.56894071957878E+01, 0.65165021654000E-03},
            {-0.0000000001874576, 0.28598333265121E+01, 0.55639542661000E-03},
        },
        .l_len = 19,
        .l = {
            { 0.0005586040123824, 0.21404207189815E+01, 0.14500979323100E-02},
            {-0.0003805813868176, 0.27358844897853E+01, 0.29729650620000E-04},
            { 0.0002205152863262, 0.64979652596400E+00, 0.50172167243580E+00},
            { 0.0001877895151158, 0.18084787604005E+01, 0.31750660413359E+01},
            { 0.0000766916975242, 0.62720114319755E+01, 0.13928364636651E+01},
            { 0.0000747056855106, 0.12995916202344E+01, 0.10034433456729E+01},
            {-0.0000388323297366, 0.71289234751879E+00, 0.75007236972328E+00},
            { 0.0000335036484314, 0.53712641184981E+01, 0.12494011725000E-03},
            { 0.0000293032677938, 0.19493939340593E+01, 0.15051650209131E+01},
            { 0.0000185940935472, 0.14630998372377E+01, 0.29001339405200E-02},
            {-0.0000170438022886, 0.56893382353856E+01, 0.65165044781000E-03},
            { 0.0000151393833114, 0.28749516044614E+01, 0.55646069067000E-03},
            {-0.0000148825637256, 0.33321074618840E+01, 0.12530790075011E+00},
            { 0.0000129927896682, 0.25991973549465E+01, 0.20068866945508E+01},
            {-0.0000116117398772, 0.56192268627131E+01, 0.93166256720000E-04},
            { 0.0000066211702894, 0.48564958193206E+01, 0.74862286166569E+00},
            { 0.0000065387442486, 0.35580120361824E+01, 0.16550513741900E-02},
            { 0.0000061580798140, 0.32490037889701E+01, 0.25086083721948E+01},
            { 0.0000046797140778, 0.96612169919707E+00, 0.62699716616712E+00},
        },
        .z_len = 46,
        .z = {
            { 0.0073755808467977, 0.55836071576084E+01, 0.32065099140000E-04},
            { 0.0002065924169942, 0.59209831565786E+01, 0.37648624194703E+00},
            { 0.0001589869764021, 0.28744006242623E+00, 0.87820792442520E+00},
            {-0.0001561131605348, 0.21257397865089E+01, 0.12727441285000E-03},
            { 0.0001486043380971, 0.14462134301023E+01, 0.35515522949802E+01},
            { 0.0000635073108731, 0.59096803285954E+01, 0.17693227129285E+01},
            { 0.0000599351698525, 0.41125517584798E+01,-0.27985797954589E+01},
            { 0.0000540660842731, 0.55390350845569E+01, 0.28683408228300E-02},
            {-0.0000489596900866, 0.46218149483338E+01,-0.62695712529519E+00},
            { 0.0000333682283528, 0.52066975238880E+01,-0.37358601734497E+00},
            { 0.0000295832427279, 0.59322697896516E+01,-0.10163502275209E+01},
            { 0.0000292325461337, 0.52707623402008E+01,-0.12523542448602E+00},
            { 0.0000197588369441, 0.33317768022759E+01, 0.00000000000000E+00},
            {-0.0000183551029746, 0.39720443249757E+01,-0.11286788041058E+01},
            { 0.0000090411191759, 0.55606719963947E+01, 0.14501837490800E-02},
            {-0.0000081987970452, 0.33223313720086E+01,-0.16304004832039E+01},
            {-0.0000060406575087, 0.13970265485562E+01, 0.43191832032300E-02},
            { 0.0000056895636122, 0.41990956668120E+01,-0.37213592656720E+00},
            {-0.0000040434854859, 0.47008406172134E+01,-0.25049602889288E+00},
            {-0.0000039403527376, 0.26725832255243E+01,-0.21321221654766E+01},
            { 0.0000036901291978, 0.35207772267753E+00, 0.11265585018525E+01},
            {-0.0000028551622596, 0.55601265129356E+01,-0.31584886140000E-04},
            {-0.0000026588026505, 0.25969882784477E+00, 0.14182025553300E-02},
            {-0.0000019711212463, 0.20228019680496E+01,-0.26338438454332E+01},
            { 0.0000019322089806, 0.51418595457408E+01, 0.25123117908847E+00},
            {-0.0000018673159813, 0.93674892088247E+00, 0.13799296163047E+01},
            { 0.0000016838424078, 0.60796033426941E+01, 0.75294520843775E+00},
            {-0.0000016695689644, 0.15867810488422E+01, 0.18816512864243E+01},
            { 0.0000016317841395, 0.45789534393209E+01, 0.14822429153000E-02},
            {-0.0000016159095087, 0.30157253757329E+00,-0.14180284447900E-02},
            {-0.0000014034621874, 0.59433512039442E+01,-0.24091866865037E+01},
            {-0.0000012029942283, 0.27137754880270E+01, 0.27731373092900E-02},
            {-0.0000011758260607, 0.40581098970285E+01,-0.75221789504525E+00},
            {-0.0000010798624964, 0.22364861319452E+01, 0.23833729650229E+01},
            {-0.0000010108880552, 0.13729872033949E+01,-0.31355655165873E+01},
            {-0.0000008876681807, 0.50534107615010E+01,-0.50169281575682E+00},
            { 0.0000008869382117, 0.50147420853991E+01, 0.68353864231000E-03},
            {-0.0000008194699011, 0.62190878357566E+01,-0.37503615934219E+00},
            { 0.0000007093782158, 0.44118312641559E+01,-0.24221246131672E+01},
            { 0.0000006728737059, 0.31910016062920E+01,-0.37068584881726E+00},
            { 0.0000006297345982, 0.13595719733984E+01, 0.11251084135564E+01},
            { 0.0000006128899757, 0.51402161299290E+01, 0.71160095483087E+01},
            {-0.0000005580987049, 0.34117733109010E+01,-0.12539396666771E+01},
            { 0.0000005321318002, 0.35377046967957E+01, 0.57685340271300E-02},
            {-0.0000004739086661, 0.21645217929478E+01, 0.58845474482000E-03},
            { 0.0000004518928658, 0.44963664372727E+01, 0.29023325111200E-02},
        },
        .zeta_len = 18,
        .zeta = {
            { 0.0038422977898495, 0.24133922085557E+01, 0.00000000000000E+00},
            { 0.0022453891791894, 0.59721736773277E+01,-0.30561255250000E-04},
            {-0.0002604479450559, 0.33368746306409E+01,-0.12491309972000E-03},
            { 0.0000332112143230, 0.55604137742337E+01, 0.29003768850700E-02},
            { 0.0000049727136261, 0.28488229706820E+00,-0.14500571761900E-02},
            {-0.0000049416729114, 0.10476908456459E+01,-0.56920298857000E-03},
            { 0.0000043945193428, 0.17684273746003E+01, 0.14501344524700E-02},
            { 0.0000037630501589, 0.45567680530533E+01, 0.43504645407000E-02},
            {-0.0000030823418750, 0.20094360655956E+01, 0.29313051376700E-02},
            { 0.0000004719790711, 0.18055417618741E+01, 0.14195445432000E-02},
            {-0.0000004637177865, 0.38277528822158E+01,-0.14808731001600E-02},
            { 0.0000003497224175, 0.46444360330108E+01, 0.30253130162300E-02},
            {-0.0000003467132626, 0.10120757927163E+01, 0.43816126822900E-02},
            { 0.0000003324412570, 0.35549391686606E+01, 0.58005379032100E-02},
            { 0.0000001945374351, 0.61251687150860E+01, 0.28808264872800E-02},
            { 0.0000001727743329, 0.11900773236610E+01,-0.29001068524700E-02},
            {-0.0000001485176585, 0.62335834706368E+01, 0.14807679092700E-02},
            { 0.0000000000666922, 0.40616225761771E+01,-0.32724923474890E-01},
        },
    },
};
