<?php
/**
 * Forwards_Driver_sql:: implements the Forwards_Driver API for SQL servers.
 *
 * $Horde: forwards/lib/Driver/customsql.php,v 1.13.2.5 2009/01/24 15:26:34 chuck Exp $
 *
 * Copyright 2004-2007 Kendrick Vargas <ken@hudat.com>
 *
 * See the enclosed file LICENSE for license information (BSDL). If you
 * did not receive this file, see http://www.horde.org/licenses/bsdl.php.
 *
 * @author  Kendrick Vargas <ken@hudat.com>
 * @author  Jan Schneider <jan@horde.org>
 * @since   Forwards 3.0
 * @package Forwards
 */
class Forwards_Driver_customsql extends Forwards_Driver {

    /**
     * SQL connection object.
     *
     * @var DB
     */
    var $_db;

    /**
     * State of SQL connection.
     *
     * @var boolean
     */
    var $_connected = false;

    /**
     * List of required parameters.
     *
     * @var array
     */
    var $_requiredParams = array('phptype', 'query_select', 'query_set',
                                 'query_disable', 'column_target',
                                 'column_keeplocal');

    /**
     * Constructor.
     *
     * @param string $user   A user name.
     * @param string $realm  The realm of the user.
     * @param array $params  A hash containing connection parameters.
     */
    function Forwards_Driver_customsql($user, $realm, $params = null)
    {
        if (is_null($params)) {
            $params = Horde::getDriverConfig('server', 'sql');
        }
        parent::Forwards_Driver($user, $realm, $params);
    }

    /**
     * Begins forwarding of mail for a user.
     *
     * @param string $password    The password of the user.
     * @param string $target      The email address that mail should be
     *                            forwarded to.
     * @param boolean $keeplocal  Keep a copy of forwarded mail in the local
     *                            mailbox.
     */
    function enableForwarding($password, $target, $keeplocal)
    {
        return $this->_enableForwarding($this->_params['query_set'], $password,
                                        $target, $keeplocal);
    }

    /**
     * Execute the query that begins forwarding of mail for a user.
     *
     * @param string $query       The query to execute, including placeholders.
     * @param string $password    The password of the user.
     * @param string $target      The email address that mail should be
     *                            forwarded to.
     * @param boolean $keeplocal  Keep a copy of forwarded mail in the local
     *                            mailbox.
     */
    function _enableForwarding($query, $password, $target, $keeplocal = null)
    {
        // Connect to database.
        if (is_a($connected = $this->_connect(), 'PEAR_Error')) {
            return $connected;
        }

        // Build the SQL query.
        $query = str_replace(
            array('\U', '\T', '\L', '\P'),
            array($this->_db->quote($this->_buildUsername()),
                  $this->_db->quote($target),
                  $this->_db->quote($keeplocal == 'on' ? 'Y' : 'N'),
                  $this->_db->quote($password)),
            $query);

        // Execute the query.
        $result = $this->_db->query($query);
        if (is_a($result, 'PEAR_Error')) {
            return $result;
        }
        if ($result !== DB_OK) {
            return PEAR::raiseError(_("An unknown error occured while enabling forwarding."));
        }
    }

    /**
     * Stops forwarding of mail for a user.
     *
     * @param string $password  The password of the user.
     */
    function disableForwarding($password)
    {
        return $this->_disableForwarding($this->_params['query_disable'],
                                         $password);
    }

    /**
     * Execute the query that stops forwarding of mail for a user.
     *
     * @param string $query     The query to execute, including placeholders.
     * @param string $password  The password of the user.
     */
    function _disableForwarding($query, $password)
    {
        // Connect to database.
        if (is_a($connected = $this->_connect(), 'PEAR_Error')) {
            return $connected;
        }

        // Build username.
        $myuser = $this->_buildUsername();

        // Build the SQL query.
        $query = str_replace(array('\U', '\P'),
                             array($this->_db->quote($this->_buildUsername()),
                                   $this->_db->quote($password)),
                             $query);

        // Execute the query.
        $result = $this->_db->query($query);
        if (is_a($result, 'PEAR_Error')) {
            return $result;
        }
        if ($this->_db->affectedRows() == 0) {
            return PEAR::raiseError(_("The forwarding cannot be disabled. Check the password."));
        }
        if ($result !== DB_OK) {
            return PEAR::raiseError(_("An unknown error occured while disabling forwarding."));
        }
    }

    /**
     * Retrieves current state of mail redirection for a user.
     *
     * @param string $password  The password of the user.
     *
     * @return mixed  Returns 'Y' if forwarding is enabled for the user, 'N' if
     *                forwarding is currently disabled, false if the status
     *                cannot be determined, and PEAR_Error on error.
     */
    function isEnabledForwarding($password)
    {
        // Build username.
        $myuser = $this->_buildUsername();

        // Get current details.
        $current_details = $this->_getUserDetails($password);
        if ($current_details === false ||
            is_a($current_details, 'PEAR_Error')) {
            return $current_details;
        }

        return 'Y';
    }

    /**
     * Checks if user is keeping a local copy of forwarded mail.
     *
     * @param string $password  The password of the user.
     *
     * @return boolean  True if user is keeping a local copy of mail,
     *                  otherwise false.
     */
    function isKeepLocal($password)
    {
        // Get current details.
        $current_details = $this->_getUserDetails($password);
        if ($current_details === false ||
            is_a($current_details, 'PEAR_Error')) {
            return $current_details;
        }

        // Check retain copy flag.
        return substr($current_details[$this->_params['column_keeplocal']], 0, 1) == 'Y';
    }

    /**
     * Retrieves current target of mail redirection for a user.
     *
     * @param string $password  The password of the user.
     *
     * @return string  The current forwarding mail address, false if no
     *                 forwarding is set, or PEAR_Error on error.
     */
    function currentTarget($password)
    {
        $current_details = $this->_getUserDetails($password);
        if ($current_details === false ||
            is_a($current_details, 'PEAR_Error')) {
            return $current_details;
        }

        // Check current forwarding mail address.
        return $current_details[$this->_params['column_target']];
    }

    /**
     * Retrieves user details from the backend.
     *
     * @param string $password  The password of the user.
     *
     * @return array  Hash with user details, or PEAR_Error.
     */
    function _getUserDetails($password)
    {
        static $row;

        // If we already have the details, return now.
        if (isset($row)) {
            return $row;
        }

        // Connect to database.
        if (is_a($connected = $this->_connect(), 'PEAR_Error')) {
            return $connected;
        }

        // Build the SQL query.
        $query = str_replace(array('\U', '\P'),
                             array($this->_db->quote($this->_buildUsername()),
                                   $this->_db->quote($password)),
                             $this->_params['query_select']);

        // Execute the query.
        $result = $this->_db->query($query);

        if (is_a($result, 'PEAR_Error')) {
            return $result;
        }

        $row = $result->fetchRow(DB_FETCHMODE_ASSOC);
        if (is_array($row)) {
            return $row;
        }

        return false;
    }

    /**
     * Builds a username based on presence of realm.
     *
     * @return string  Fully qualified username.
     */
    function _buildUsername()
    {
        if ($this->_realm === 'default' ||
            $this->_realm === '') {
            return $this->_user;
        } else {
            return $this->_user . '@' . $this->_realm;
        }
    }

    /**
     * Does an SQL connect and logs in as user with privilege to change
     * vacation.
     *
     * @return boolean  True or PEAR_Error based on success of connect.
     */
    function _connect()
    {
        if ($this->_connected) {
            return;
        }

        // Build the params array to pass to DB
        $args = array_merge($this->_params, isset($this->_params[$this->_realm]) ? $this->_params[$this->_realm] : array());
        Horde::assertDriverConfig($args, 'server', $this->_requiredParams, 'Forwards SQL');

        if (!isset($args['database'])) {
            $args['database'] = '';
        }
        if (!isset($args['username'])) {
            $args['username'] = '';
        }
        if (!isset($args['hostspec'])) {
            $args['hostspec'] = '';
        }

        // Connect to the SQL server using the supplied parameters.
        require_once 'DB.php';
        $this->_db = &DB::connect($args, array('persistent' => !empty($args['persistent'])));
        if (is_a($this->_db, 'PEAR_Error')) {
            return $this->_db;
        }

        // Set DB portability options.
        switch ($this->_db->phptype) {
        case 'mssql':
            $this->_db->setOption(
                'portability',
                DB_PORTABILITY_LOWERCASE | DB_PORTABILITY_ERRORS | DB_PORTABILITY_RTRIM);
            break;

        default:
            $this->_db->setOption(
                'portability',
                DB_PORTABILITY_LOWERCASE | DB_PORTABILITY_ERRORS);
        }

        $this->_connected = true;
    }

}
