/*
 * This file is part of the Score-P software (http://www.score-p.org)
 *
 * Copyright (c) 2009-2013,
 *    RWTH Aachen University, Germany
 *    Gesellschaft fuer numerische Simulation mbH Braunschweig, Germany
 *    Technische Universitaet Dresden, Germany
 *    University of Oregon, Eugene, USA
 *    Forschungszentrum Juelich GmbH, Germany
 *    German Research School for Simulation Sciences GmbH, Juelich/Aachen, Germany
 *    Technische Universitaet Muenchen, Germany
 *
 * See the COPYING file in the package base directory for details.
 *
 */

/**
 * @file otf2_config_library_dependecies.hpp
 * @status alpha
 * @maintainer Daniel Lorenz <d.lorenz@fz-juelich.de>
 *
 * Declares classes for the representation and analysis of library dependencies
 */

#ifndef OTF2_CONFIG_LIBRARY_DEPENDENCY_HPP
#define OTF2_CONFIG_LIBRARY_DEPENDENCY_HPP

#include <map>
#include <deque>
#include <string>


/**
 * A class to store and analyze the dependencies from libraries. To generate
 * the dependecies, the .la files are parsed and initialization code is
 * generated by vendor/common/build-config/generate-library-dependecies.sh
 */
class OTF2_Config_LibraryDependencies
{
    // ------------------------------------- Protected types
protected:

    /**
     * Internal class to represent teh content of one .la file.
     */
    class la_object
    {
public:
        /**
         * Empty contructor. Needed to allow copies of STL containers containing this
         * class
         */
        la_object( void );

        /**
         * Copy constructor.
         */
        la_object( const la_object& source );

        /**
         * Regular constructor.
         */
        la_object( const std::string&             lib_name,
                   const std::string&             build_dir,
                   const std::string&             install_dir,
                   const std::deque<std::string>& libs,
                   const std::deque<std::string>& ldflags,
                   const std::deque<std::string>& rpaths,
                   const std::deque<std::string>& dependency_las );

        /**
         * Destructor.
         */
        virtual
        ~la_object();
public:
        std::string             m_lib_name;
        std::string             m_build_dir;
        std::string             m_install_dir;
        std::deque<std::string> m_libs;
        std::deque<std::string> m_ldflags;
        std::deque<std::string> m_rpath;
        std::deque<std::string> m_dependency_las;
    };


    // ------------------------------------- Public functions
public:
    /**
     * Constructs the library dependecies
     */
    OTF2_Config_LibraryDependencies( void );

    /**
     * Destructor
     */
    virtual
    ~OTF2_Config_LibraryDependencies();

    /**
     * Returns the string containing the @a input_libs and its dependencies.
     * @param input_libs  A list of libraries, that should be linked.
     * @param frontend    Specifies whether the data for the frontend or
     *                    for the backend is requested. Pass true, if you
     *                    need frontend data. Pass false is you want
     *                    backend data.
     */
    std::string
    GetLibraries( const std::deque<std::string>& input_libs,
                  bool                           frontend );

    /**
     * Returns the string containing the library path flags for the @a input_libs and
     * its dependencies.
     * @param libs     A list of library names.
     * @param install  If true the install paths are used. If false the
     *                 build path are used.
     * @param frontend Specifies whether the data for the frontend or
     *                 for the backend is requested. Pass true, if you
     *                 need frontend data. Pass false is you want
     *                 backend data.
     */
    std::string
    GetLDFlags( const std::deque<std::string>& libs,
                bool                           install,
                bool                           frontend  );

    /**
     * Returns the string containing the rpath flags for the @a input_libs and
     * its dependencies. The string that is constructed can be customized for
     * the use on different systems. E.g. specify an empty string as head and
     * " -Wl,-rpath," as delimiter on most Linuy machines. On AIX, specify
     * " -Wl,-lbibpath" and head and ":" as delimiter.
     * @param libs      A list of library names.
     * @param install   If true the install paths are used. If false the
     *                  build path are used.
     * @param frontend  Specifies whether the data for the frontend or
     *                  for the backend is requested. Pass true, if you
     *                  need frontend data. Pass false is you want
     *                  backend data.
     * @param head      A string that appears once at the beginning
     * @param delimiter A string that separates the pathes.
     * @param tail      A string that is appended after the last flag.
     */
    std::string
    GetRpathFlags( const std::deque<std::string>& libs,
                   bool                           install,
                   bool                           frontend,
                   const std::string&             head,
                   const std::string&             delimiter,
                   const std::string&             tail );

    // ------------------------------------- Protected functions
protected:
    /**
     * Calculates the dependencies for a given set of libraries.
     * @param libs      A list of library names.
     * @param frontend  Specifies whether the data for the frontend or
     *                  for the backend is requested. Pass true, if you
     *                  need frontend data. Pass false is you want
     *                  backend data.
     */
    std::deque<std::string>
    get_dependencies( const std::deque<std::string>& libs,
                      bool                           frontend );

    // ------------------------------------- Public members
private:
    std::map< std::string, la_object> m_backend_objects;
    std::map< std::string, la_object> m_frontend_objects;
};

#endif
