/****************************************************************************
**  CUBE        http://www.score-p.org/                                    **
**  SCALASCA    http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2013                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  Copyright (c) 2009-2013                                                **
**  German Research School for Simulation Sciences GmbH,                   **
**  Laboratory for Parallel Programming                                    **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/
/**
 * \file types.h
   \brief Provides needed typedefs for cube writer.
 */
#ifndef TYPES_H
#define TYPES_H

//#include <config.h>
#include <stdint.h>
#include <sys/types.h>
#include <limits.h>
#include <float.h>
#ifndef DBL_MAX
#define DBL_MAX 1.7976931348623158e+308
#endif



#ifdef __cplusplus
extern "C" {
#endif

/* CUBE TYPES  */

/**
 * General flat array of pointers on general data (void*) with dynamical memory allocation
 */
typedef struct dyn_array
{
    int    size;
    int    capacity;
    void** data;
} dyn_array;

/**
 * Char mapping. Is used to save string parameters of cnodes.
 */
typedef struct cmap
{
    char* key;
    char* value;
} cmap;

/**
 * Numerical mapping . Is used to save numerical parameters of cnodes.
 */
typedef struct nmap
{
    char*  key;
    double value;
} nmap;

/**
   A dynamic array with numerical parametes for cnodes.
 */
typedef struct nparam_parray
{
    int    size;
    int    capacity;
    nmap** data;
} nparam_parray;

/**
   A dynamic array with string parameters for cnodes.
 */
typedef struct cparam_parray
{
    int    size;
    int    capacity;
    cmap** data;
} cparam_parray;





/**
 * General flat array of pointers on 'char mapping' (cmap*)
 */

typedef struct att_array
{
    int    size;
    int    capacity;
    cmap** data;
} att_array;


/*  METRIC TYPES */
typedef struct marray
{
    int                  size;
    int                  capacity;
    struct cube_metric** data;
} marray;  /**< Dynamical array containing metrics */


/*   CNODE TYPES */
typedef struct carray
{
    int                 size;
    int                 capacity;
    struct cube_cnode** data;
} carray; /**<  Dynamic array of cnodes. */


/* REGION TYPES */

typedef struct rarray
{
    int                 size;
    int                 capacity;
    struct cube_cnode** data;
} rarray;                          /**< A dynamic array of regions. */
typedef struct rarray cube_rarray; /**< A synonym of the arrays containing only regions. */






/*  types for the new flexible system tree */


/* forwart declarations*/
struct cube_system_tree_node;
struct location_group;
struct location;

typedef struct cube_stnarray
{
    int                            size;
    int                            capacity;
    struct cube_system_tree_node** data;
} cube_stnarray;

typedef struct cube_lgarray
{
    int                          size;
    int                          capacity;
    struct cube_location_group** data;
} cube_lgarray;

typedef struct cube_larray
{
    int                    size;
    int                    capacity;
    struct cube_location** data;
} cube_larray;




/**
 * Structure collection name and description  of a system tree node.
 */
typedef struct cube_system_tree_node
{
    cube_stnarray*                child;
    cube_lgarray*                 group;
    struct cube_system_tree_node* parent;
    int                           id;
    char*                         name;
    char*                         desc;
    char*                         stn_class;
} cube_system_tree_node;

/**
 * Structure collection name, ID and rank of a location group.
 */

typedef enum cube_location_group_type { CUBE_LOCATION_GROUP_TYPE_PROCESS = 0, CUBE_LOCATION_GROUP_TYPE_METRICS = 1 } cube_location_group_type;

typedef struct cube_location_group
{
    cube_larray*                  child;
    struct cube_system_tree_node* parent;
    char*                         name;
    int                           rank;
    int                           id;
    cube_location_group_type      type;
} cube_location_group;

/**
 * Structure collection name, ID and rank of a location group.
 */
typedef enum cube_location_type { CUBE_LOCATION_TYPE_CPU_THREAD = 0, CUBE_LOCATION_TYPE_GPU = 1, CUBE_LOCATION_TYPE_METRIC = 2 } cube_location_type;

typedef struct cube_location
{
    struct cube_location_group* parent;
    char*                       name;
    int                         rank;
    int                         id;
    cube_location_type          type;
} cube_location;

/**
   Typedef for old types
 */

typedef cube_system_tree_node cube_node;
typedef cube_system_tree_node cube_machine;
typedef cube_location_group   cube_process;
typedef cube_location         cube_thread;

typedef cube_larray           cube_tarray;
typedef cube_lgarray          cube_parray;
typedef cube_stnarray         cube_narray;




enum CubeFlavours_t { CUBE_MASTER = 0, CUBE_WRITER = 1, CUBE_SLAVE = 2 };

/*  Code is not clear for reader. Why False or FALSE? Why not CUBE_FALSE and CUBE_TRUE. */
enum bool_t { CUBE_FALSE = 0, CUBE_TRUE = 1 };

/* Value compatibility status*/
enum value_compat_t { CUBE_VALUE_SAME = 0, CUBE_VALUE_COMPATIBLE = 1, CUBE_VALUE_INCOMPATIBLE = 2, CUBE_VALUE_PRECISION_LOST = 3, CUBE_VALUE_MEANING_LOST = 4, CUBE_VALUE_RANGE_LOST = 5, CUBE_VALUE_UNKNOWN_RELATION = 127 };

/* Value initialization */
enum value_init_t { CUBE_UNDEFINED_INIT = 0, CUBE_ZERO_INIT = 1, CUBE_VALUE_INIT = 2 };

typedef struct dyn_array cube_dyn_array;
typedef struct cmap      cube_map;
typedef struct att_array cube_att_array;
typedef struct carray    cube_carray;



/*  Set of cannonical definitions of various types supported by cube:
    It is always a packed structure.
 */
/**
 * Cannonical type : uint8 (unsigned char (0-255)).
 */

typedef struct cube_type_uint8
{
    uint8_t value;
} cube_type_uint8;
static const cube_type_uint8 cube_undefined_uint8 = { UINT8_C( 0xff ) };
static const cube_type_uint8 cube_zero_uint8 =     { UINT8_C( 0x00 ) };

/**
 * Cannonical type : int8 (signed char (-127 - 127)).
 */

typedef struct cube_type_int8
{
    int8_t value;
} cube_type_int8;
static const cube_type_int8 cube_undefined_int8 = { ( int8_t )INT8_C( 0xff ) };
static const cube_type_int8 cube_zero_int8 =     { INT8_C( 0x00 ) };
/**
 * Cannonical type : uint16 (unsigned short ).
 */
typedef struct cube_type_uint16
{
    uint16_t value;
} cube_type_uint16;
static const cube_type_uint16 cube_undefined_uint16 = { UINT16_C( 0xffff ) };
static const cube_type_uint16 cube_zero_uint16 =     { UINT64_C( 0x0000 ) };
/**
 * Cannonical type : int16 (signed short ).
 */

typedef struct cube_type_int16
{
    int16_t value;
} cube_type_int16;
static const cube_type_int16 cube_undefined_int16 = { ( int16_t )INT16_C( 0xffff ) };
static const cube_type_int16 cube_zero_int16 =     { INT16_C( 0x0000 ) };

/**
 * Cannonical type : uint32 (unsigned int ).
 */
typedef struct cube_type_uint32
{
    uint32_t value;
} cube_type_uint32;
static const cube_type_uint32 cube_undefined_uint32 = { UINT32_C( 0xffffffff ) };
static const cube_type_uint32 cube_zero_uint32 =     { UINT32_C( 0x00000000 ) };



/**
 * Cannonical type : int32 (signed int ).
 */
typedef struct cube_type_int32
{
    int32_t value;
} cube_type_int32;
static const cube_type_int32 cube_undefined_int32 = { ( int32_t )INT32_C( 0xffffffff ) };
static const cube_type_int32 cube_zero_int32 =     { UINT32_C( 0x00000000 ) };


/**
 * Cannonical type : uint64 (unsigned long int  ).
 */
typedef struct cube_type_uint64
{
    uint64_t value;
} cube_type_uint64;
static const cube_type_uint64 cube_undefined_uint64 = { UINT64_C( 0xffffffffffffffff ) };
static const cube_type_uint64 cube_zero_uint64 =     { UINT64_C( 0x0000000000000000 ) };

/**
 * Cannonical type : int64 (signed long int  ).
 */
typedef struct cube_type_int64
{
    int64_t value;
} cube_type_int64;
static const cube_type_int64 cube_undefined_int64 =  { ( int64_t )INT64_C( 0xffffffffffffffff ) };
static const cube_type_int64 cube_zero_int64 =      { INT64_C( 0x0000000000000000 ) };

/**
 * Cannonical type : double
 */
typedef struct cube_type_double
{
    double value;
} cube_type_double;
static const cube_type_double cube_undefined_double =  { UINT64_C( 0xffffffffffffffff ) };
static const cube_type_double cube_zero_double =      { UINT64_C( 0x0000000000000000 ) };




/**
 * Cannonical type : tau atomic
 */
typedef struct cube_type_tau_atomic cube_type_tau_atomic;
struct cube_type_tau_atomic
{
    uint32_t N;
    double   Min;
    double   Max;
    double   Sum;
    double   Sum2;
}  __attribute__ ( ( __packed__ ) );

static const cube_type_tau_atomic cube_undefined_tau_atomic =  {
    UINT32_C( 0xffffffff ),
    UINT64_C( 0xffffffffffffffff ),
    UINT64_C( 0xffffffffffffffff ),
    UINT64_C( 0xffffffffffffffff ),
    UINT64_C( 0xffffffffffffffff )
};
static const cube_type_tau_atomic cube_zero_tau_atomic =  {
    UINT32_C( 0x00000000 ),
    DBL_MAX,
    -DBL_MAX,
    UINT64_C( 0x0000000000000000 ),
    UINT64_C( 0x0000000000000000 )
};


/**
 * Cannonical type : double
 */
typedef struct cube_type_scale_func cube_type_scale_func;
struct cube_type_scale_func
{
    double value[ 27 ]; /* there are 27 possible combinations of powers*/
}  __attribute__ ( ( __packed__ ) );
static const cube_type_scale_func cube_undefined_scale_func =  {
    { UINT64_C( 0xffffffffffffffff ),
      UINT64_C( 0xffffffffffffffff ),
      UINT64_C( 0xffffffffffffffff ),
      UINT64_C( 0xffffffffffffffff ),
      UINT64_C( 0xffffffffffffffff ),
      UINT64_C( 0xffffffffffffffff ),
      UINT64_C( 0xffffffffffffffff ),
      UINT64_C( 0xffffffffffffffff ),
      UINT64_C( 0xffffffffffffffff ),
      UINT64_C( 0xffffffffffffffff ),
      UINT64_C( 0xffffffffffffffff ),
      UINT64_C( 0xffffffffffffffff ),
      UINT64_C( 0xffffffffffffffff ),
      UINT64_C( 0xffffffffffffffff ),
      UINT64_C( 0xffffffffffffffff ),
      UINT64_C( 0xffffffffffffffff ),
      UINT64_C( 0xffffffffffffffff ),
      UINT64_C( 0xffffffffffffffff ),
      UINT64_C( 0xffffffffffffffff ),
      UINT64_C( 0xffffffffffffffff ),
      UINT64_C( 0xffffffffffffffff ),
      UINT64_C( 0xffffffffffffffff ),
      UINT64_C( 0xffffffffffffffff ),
      UINT64_C( 0xffffffffffffffff ),
      UINT64_C( 0xffffffffffffffff ),
      UINT64_C( 0xffffffffffffffff ),
      UINT64_C( 0xffffffffffffffff ) }
};
static const cube_type_scale_func cube_zero_scale_func =      {
    {
        UINT64_C( 0x0000000000000000 ),
        UINT64_C( 0x0000000000000000 ),
        UINT64_C( 0x0000000000000000 ),
        UINT64_C( 0x0000000000000000 ),
        UINT64_C( 0x0000000000000000 ),
        UINT64_C( 0x0000000000000000 ),
        UINT64_C( 0x0000000000000000 ),
        UINT64_C( 0x0000000000000000 ),
        UINT64_C( 0x0000000000000000 ),
        UINT64_C( 0x0000000000000000 ),
        UINT64_C( 0x0000000000000000 ),
        UINT64_C( 0x0000000000000000 ),
        UINT64_C( 0x0000000000000000 ),
        UINT64_C( 0x0000000000000000 ),
        UINT64_C( 0x0000000000000000 ),
        UINT64_C( 0x0000000000000000 ),
        UINT64_C( 0x0000000000000000 ),
        UINT64_C( 0x0000000000000000 ),
        UINT64_C( 0x0000000000000000 ),
        UINT64_C( 0x0000000000000000 ),
        UINT64_C( 0x0000000000000000 ),
        UINT64_C( 0x0000000000000000 ),
        UINT64_C( 0x0000000000000000 ),
        UINT64_C( 0x0000000000000000 ),
        UINT64_C( 0x0000000000000000 ),
        UINT64_C( 0x0000000000000000 ),
        UINT64_C( 0x0000000000000000 )
    }
};



/**
 * Cannonical type : rate
 */
typedef struct cube_type_rate cube_type_rate;
struct cube_type_rate
{
    double numerator;
    double denominator;
}  __attribute__ ( ( __packed__ ) );

static const cube_type_rate cube_undefined_rate =  {
    UINT64_C( 0xffffffffffffffff ),
    UINT64_C( 0xffffffffffffffff )
};
static const cube_type_rate cube_zero_rate =  {
    UINT64_C( 0x0000000000000000 ),
    UINT64_C( 0x0000000000000000 )
};



/**
 * Cannonical type : complex
 */
typedef struct cube_type_complex cube_type_complex;
struct cube_type_complex
{
    double re_value;
    double im_value;
}  __attribute__ ( ( __packed__ ) );

static const cube_type_complex cube_undefined_complex =  {
    UINT64_C( 0xffffffffffffffff ),
    UINT64_C( 0xffffffffffffffff )
};
static const cube_type_complex cube_zero_complex =  {
    UINT64_C( 0x0000000000000000 ),
    UINT64_C( 0x0000000000000000 )
};



#ifdef __cplusplus
}
#endif

#endif
