/****************************************************************************
**  SCALASCA    http://www.scalasca.org/                                   **
**  KOJAK       http://www.fz-juelich.de/jsc/kojak/                        **
*****************************************************************************
**  Copyright (c) 1998-2010                                                **
**  Forschungszentrum Juelich, Juelich Supercomputing Centre               **
**                                                                         **
**  Copyright (c) 2003-2008                                                **
**  University of Tennessee, Innovative Computing Laboratory               **
**                                                                         **
**  See the file COPYRIGHT in the package base directory for details       **
****************************************************************************/
/**
 * \file location_group.c
   \brief Defines types and functions to deal with location group of running application.
 */
#include <config.h>
#include <stdlib.h>
#include <string.h>
#include "cubew_system_tree_node.h"
#include "cubew_vector.h"
#include "cubew_services.h"

/**
 * Allocates memory for a location_group structure.
 *
 */
cube_location_group*
cube_location_group_create( cube_location_group* this )
{
    if ( this == NULL )
    {
        ALLOC( this, 1, cube_location_group );
    }
    if ( this != NULL )
    {
        cube_location_group_construct_child( this );
    }
    return this;
}

/**
 * Fills the structure with name and rank of location_group.
 *
 */
void
cube_location_group_init( cube_location_group*     this,
                          const char*              name,
                          int                      rank,
                          cube_location_group_type type,
                          cube_system_tree_node*   parent )
{
    this->name   = cubew_strdup( name );
    this->rank   = rank;
    this->parent = parent;
    this->type   = type;
    if ( parent != NULL )
    {
        cube_location_group_add_child( this->parent, this );
    }
}

/**
 * Creates a child of this location_group.
 *
 */
void
cube_location_group_construct_child( cube_location_group* this )
{
    XALLOC( this->child, 1, cube_larray );
    this->child->size     = 0;
    this->child->capacity = 0;
}

/**
 * Ads a location_group "proc" to the location_group "this".
 *
 */
void
cube_location_group_add_child( cube_system_tree_node* parent,
                               cube_location_group*   lg )
{
    ADD_NEXT( parent->group, lg, cube_location_group* );
}

/**
 * Releases memory for a location_group structure.
 *
 */
void
cube_location_group_free( cube_location_group* this )
{
    if ( this != NULL )
    {
        if ( this->name != 0 )
        {
            free( this->name );
        }
        if ( this->child != NULL )
        {
            free( this->child->data );
        }
        free( this->child );
        free( this );
    }
}

cube_location*
cube_location_group_get_child( cube_location_group* this,
                               int                  i )
{
    if ( i < 0 || i >= this->child->size )
    {
        printf( "cube_location_group_get_child: out of range\n" );
    }
    return this->child->data[ i ];
}

cube_system_tree_node*
cube_location_group_get_parent( cube_location_group* this )
{
    return this->parent;
}

int
cube_location_group_get_rank( cube_location_group* this )
{
    return this->rank;
}

char*
cube_location_group_get_name( cube_location_group* this )
{
    return this->name;
}

cube_location_group_type
cube_location_group_get_type( cube_location_group* this )
{
    return this->type;
}

int
cube_location_group_num_children( cube_location_group* this )
{
    return this->child->size;
}

/**
 * Gets a level of the location_group.
 * Does it recursiv with "deep search" algorithm
 */
int
cube_location_group_get_level( cube_location_group* this )
{
    if ( cube_location_group_get_parent( this ) == NULL )
    {
        return 0;
    }
    else
    {
        return cube_system_tree_node_get_level( cube_location_group_get_parent( this ) ) + 1;
    }
}

/**
 * Writes XML output for location_group in to .cube file.
 * Does it recursiv with "deep search" algorithm
 */
void
cube_location_group_writeXML( cube_location_group* this,
                              FILE*                fp )
{
    char* _name = ( char* )services_escape_to_xml( cube_location_group_get_name( this ) );

    int   i = 0, num = cube_location_group_get_level( this );
    char  ind[ 80 ];

    strcpy( ind, "" );
    for ( i = 0; i < 2 * num; i++ )
    {
        strcat( ind, " " );
    }
    cube_location_group_type __type = cube_location_group_get_type( this );
    char*                    _type  = ( __type == CUBE_LOCATION_GROUP_TYPE_PROCESS ) ? "process" :
                                      ( __type == CUBE_LOCATION_GROUP_TYPE_METRICS ) ? "metrics" :
                                      "not supported";
    fprintf( fp, "%s    <locationgroup Id=\"%d\">\n", ind, cube_location_group_get_id( this ) );
    fprintf( fp, "%s      <name>%s</name>\n", ind, _name ? _name : cube_location_group_get_name( this ) );
    fprintf( fp, "%s      <rank>%d</rank>\n", ind, cube_location_group_get_rank( this ) );
    fprintf( fp, "%s      <type>%s</type>\n", ind, _type );
    for ( i = 0; i < cube_location_group_num_children( this ); i++ )
    {
        cube_location* thrd = cube_location_group_get_child( this, i );
        cube_location_writeXML( thrd, fp );
    }
    fprintf( fp, "%s    </locationgroup>\n", ind );

    if ( _name )
    {
        free( _name );
    }
}

void
cube_location_group_set_id( cube_location_group* this,
                            int                  new_id )
{
    this->id = new_id;
}

int
cube_location_group_get_id( cube_location_group* this )
{
    return this->id;
}

/**
 * Compares equality of two location_groupes.
 */
int
cube_location_group_equal( cube_location_group* a,
                           cube_location_group* b )
{
    unsigned _a = cube_location_group_get_rank( a );
    unsigned _b = cube_location_group_get_rank( b );
    if ( _a == _b  )
    {
        return 1;
    }
    return 0;
}
