/****************************************************************************
**  SCALASCA    http://www.scalasca.org/                                   **
**  KOJAK       http://www.fz-juelich.de/jsc/kojak/                        **
*****************************************************************************
**  Copyright (c) 1998-2010                                                **
**  Forschungszentrum Juelich, Juelich Supercomputing Centre               **
**                                                                         **
**  Copyright (c) 2003-2008                                                **
**  University of Tennessee, Innovative Computing Laboratory               **
**                                                                         **
**  See the file COPYRIGHT in the package base directory for details       **
****************************************************************************/
/**
 * \file location.c
   \brief Defines types and functions to deal with locations of running application.
 */
#include <config.h>
#include <stdlib.h>
#include <string.h>
#include "cubew_location.h"
#include "cubew_location_group.h"
#include "cubew_vector.h"
#include "cubew_services.h"

/**
 * Allocates memory for the structure "location"
 */
cube_location*
cube_location_create( cube_location* this )
{
    if ( this == NULL )
    {
        ALLOC( this, 1, cube_location );
    }
    return this;
}


/**
 * Fills the location with the information (rank, namem, id).
 *
 */
void
cube_location_init( cube_location*       this,
                    const char*          name,
                    int                  rank,
                    cube_location_type   type,
                    cube_location_group* parent )
{
    this->name   = cubew_strdup( name );
    this->rank   = rank;
    this->type   = type;
    this->parent = parent;
    if ( parent != NULL )
    {
        cube_location_add_child( this->parent, this );
    }
}


/**
 * Adds a child of this location.
 *
 */
void
cube_location_add_child( cube_location_group* parent,
                         cube_location*       thrd )
{
    ADD_NEXT( parent->child, thrd, cube_location* );
}

/**
 * Releases memory of the structure "location"
 */
void
cube_location_free( cube_location* this )
{
    if ( this != NULL )
    {
        if ( this->name != 0 )
        {
            free( this->name );
        }
        free( this );
    }
}



cube_location_group*
cube_location_get_parent( cube_location* this )
{
    return this->parent;
}

char*
cube_location_get_name( cube_location* this )
{
    return this->name;
}

cube_location_type
cube_location_get_type( cube_location* this )
{
    return this->type;
}

int
cube_location_get_rank( cube_location* this )
{
    return this->rank;
}

int
cube_location_get_level( cube_location* this )
{
    if ( cube_location_get_parent( this ) == NULL )
    {
        return 0;
    }
    else
    {
        return cube_location_group_get_level( cube_location_get_parent( this ) ) + 1;
    }
}

/**
 * Writes XML output for location in to .cube file.
 * No recursiv. Plain one after another.
 *
 */
void
cube_location_writeXML( cube_location* this,
                        FILE*          fp )
{
    char* _name = ( char* )services_escape_to_xml( cube_location_get_name( this ) );

    char  ind[ 80 ];
    int   i = 0, num = cube_location_get_level( this );

    strcpy( ind, "" );
    for ( i = 0; i < 2 * num; i++ )
    {
        strcat( ind, " " );
    }
    cube_location_type __type = cube_location_get_type( this );
    char*              _type  =  ( __type == CUBE_LOCATION_TYPE_CPU_THREAD ) ? "thread" :
                                ( __type == CUBE_LOCATION_TYPE_GPU ) ? "gpu" :
                                ( __type == CUBE_LOCATION_TYPE_METRIC ) ? "metric" : "not supported";

    fprintf( fp, "%s    <location Id=\"%d\">\n", ind, cube_location_get_id( this ) );
    fprintf( fp, "%s      <name>%s</name>\n", ind, _name ? _name : cube_location_get_name( this ) );
    fprintf( fp, "%s      <rank>%d</rank>\n", ind, cube_location_get_rank( this ) );
    fprintf( fp, "%s      <type>%s</type>\n", ind, _type );
    fprintf( fp, "%s    </location>\n", ind );

    if ( _name )
    {
        free( _name );
    }
}

void
cube_location_set_id( cube_location* this,
                      int            new_id )
{
    this->id = new_id;
}

int
cube_location_get_id( cube_location* this )
{
    return this->id;
}

/**
 * Compares equality of two locations.
 */
int
cube_location_equal( cube_location* a,
                     cube_location* b )
{
    int _a = cube_location_get_rank( a );
    int _b = cube_location_get_rank( b );
    if ( _a == _b )
    {
        int _p_a = cube_location_group_get_rank( cube_location_get_parent( a ) );
        int _p_b = cube_location_group_get_rank( cube_location_get_parent( b ) );
        if (  _p_a == _p_b )
        {
            return 1;
        }
    }
    return 0;
}
