/****************************************************************************
**  SCALASCA    http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2013                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  Copyright (c) 2009-2013                                                **
**  German Research School for Simulation Sciences GmbH,                   **
**  Laboratory for Parallel Programming                                    **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include <config.h>
#include "SynchronizeTask.h"

#include <cstddef>

#include <pearl/Location.h>

#include <Synchronizer.h>

#include "TaskData.h"

using namespace std;
using namespace pearl;
using namespace scout;


//---------------------------------------------------------------------------
//
//  class SynchronizeTask
//
//---------------------------------------------------------------------------

//--- Constructors & destructor ---------------------------------------------

/**
 *  @brief Constructor. Creates a new synchronizer task.
 *
 *  This constructor creates a new timestamp synchronization task working on
 *  the given trace data object @p trace.
 *
 *  @param  privateData  Private task data object
 **/
SynchronizeTask::SynchronizeTask(const TaskDataPrivate& privateData)
  : Task(),
    mPrivateData(privateData),
    mSynchronizer(NULL)
{
}


/**
 *  @brief Destructor.
 *
 *  Destructor. Destroys the instance and releases all occupied resources.
 **/
SynchronizeTask::~SynchronizeTask()
{
  delete mSynchronizer;
}


//--- Execution control -----------------------------------------------------

/**
 *  @brief Executes the task.
 *
 *  Performs the timestamp synchronization based on the controlled logical
 *  clock algorithm.
 *
 *  @return Always @em true
 **/
bool SynchronizeTask::execute()
{
  const LocalTrace& trace = *mPrivateData.mTrace;

  int rank = trace.get_location().getRank();
  mSynchronizer = new Synchronizer(rank, trace);
  mSynchronizer->synchronize();

  return true;
}


//--- Printing statistics ---------------------------------------------------

/**
 *  @brief Prints some synchronizer-related statistics.
 *
 *  This method prints some additional statistics about the timestamp
 *  synchronization process. In case the synchronizer task has not been
 *  called, it defaults to a no-op.
 **/
void SynchronizeTask::print_statistics() const
{
  if (mSynchronizer)
    mSynchronizer->print_statistics();
}
