/****************************************************************************
**  SCALASCA    http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2013                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  Copyright (c) 2009-2013                                                **
**  German Research School for Simulation Sciences GmbH,                   **
**  Laboratory for Parallel Programming                                    **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#ifndef PEARL_READCB_H
#define PEARL_READCB_H


#include <map>
#include <stack>
#include <string>
#include <vector>

#include <pearl/Location.h>
#include <pearl/pearl_types.h>


/*-------------------------------------------------------------------------*/
/**
 *  @file    readcb.h
 *  @ingroup PEARL_base
 *  @brief   Definition of helper classes which are passed to the EPILOG
 *           read callbacks.
 *
 *  This header file provides the definition of the two helper classes
 *  DefsCbData and TraceCbData which are passed as user data to the
 *  EPILOG file read callbacks.
 **/
/*-------------------------------------------------------------------------*/


namespace pearl {

//--- Forward declarations --------------------------------------------------

class EpikArchive;
class GlobalDefs;
class LocalTrace;
class MappingTable;


/*-------------------------------------------------------------------------*/
/**
 *  @struct  DefsCbData
 *  @ingroup PEARL_base
 *  @brief   Callback data object used while reading definition records.
 **/
/*-------------------------------------------------------------------------*/

struct DefsCbData
{
  /// @name Constructors & destructor
  /// @{

  DefsCbData(EpikArchive& archive,
             GlobalDefs&  defs);

  /// @}
  /// @name Mapping functions
  /// @{

  std::string get_string(ident_t id) const;
  ident_t     get_file  (ident_t id) const;

  /// @}


  /// EPIK archive abstraction
  EpikArchive& m_archive;

  /// %GlobalDefs object storing the definitions read
  GlobalDefs& m_defs;

  /// Number of "open" string continuation records
  uint8_t m_strcnt;

  /// Current string ID, used for string continuation records
  ident_t m_strid;

  /// Size of MPI_COMM_WORLD (always 1 in case of serial/OpenMP experiments)
  uint32_t m_worldsize;

  /// Mapping table string ID |-@> string
  std::vector<std::string> m_strmap;

  /// Mapping table file ID |-@> string ID
  std::vector<ident_t> m_filemap;

  /// Mapping table machine ID |-@> system node ID
  std::map<ident_t, ident_t> m_machinemap;

  /// Mapping table node ID |-@> system node ID
  std::map<ident_t, ident_t> m_nodemap;

  /// Mapping table process ID |-@> string ID
  std::map<ident_t, ident_t> m_processmap;

  /// Mapping table (process ID, thread ID) |-@> string ID
  std::map<std::pair<ident_t, ident_t>, ident_t> m_threadmap;

  /// Stores the what() string of exceptions thrown in callbacks,
  /// empty if everything was ok
  std::string m_message;
};


/*-------------------------------------------------------------------------*/
/**
 *  @struct  TraceCbData
 *  @ingroup PEARL_base
 *  @brief   Callback data object used while reading event trace records.
 **/
/*-------------------------------------------------------------------------*/

struct TraceCbData
{
  /// @name Constructors & destructor
  /// @{

  TraceCbData(const EpikArchive& archive,
              const GlobalDefs&  defs,
              LocalTrace&        trace,
              MappingTable&      table,
              uint32_t           version,
              Location::IdType   locId,
              uint32_t           tid);

  /// @}


  /// EPIK archive abstraction
  const EpikArchive& m_archive;

  /// %GlobalDefs object providing the global definitions
  const GlobalDefs& m_defs;

  /// %LocalTrace object storing the event representations read
  LocalTrace& m_trace;

  /// %MappingTable object used for converting local into global identifiers
  MappingTable& m_table;

  /// EPILOG file format version
  uint32_t m_version;

  /// %Location identifier for the processed trace file
  Location::IdType m_location;

  /// %Thread identifier in case of OpenMP, always 0 for MPI
  uint32_t m_tid;

  /// %Last request identifier
  uint64_t m_requestId;

  /// Callstack in terms of region IDs
  std::stack<uint32_t> m_callstack;

  /// Stores the what() string of exceptions thrown in callbacks,
  /// empty if everything was ok
  std::string m_message;
};


}   // namespace pearl


#endif   // !PEARL_READCB_H
