/****************************************************************************
**  SCALASCA    http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2013                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  Copyright (c) 2009-2013                                                **
**  German Research School for Simulation Sciences GmbH,                   **
**  Laboratory for Parallel Programming                                    **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include <config.h>
#include <pearl/Region.h>

#include <algorithm>
#include <cassert>
#include <cctype>
#include <functional>
#include <iostream>

#include <pearl/String.h>

#include "pearl_iomanip.h"

using namespace std;
using namespace pearl;


/*-------------------------------------------------------------------------*/
/**
 *  @file    Region.cpp
 *  @ingroup PEARL_base
 *  @brief   Implementation of the class Region.
 *
 *  This file provides the implementation of the class Region and related
 *  functions.
 *
 *  @todo
 *    - Revise classification functions to use region roles/paradigms.
 *    - Move lowercase() to some other, more central location
 **/
/*-------------------------------------------------------------------------*/


//--- Local helper functions ------------------------------------------------

namespace
{
    // *INDENT-OFF*
    const string roleToString(const Region::Role role);
    const string paradigmToString(const Region::Paradigm paradigm);
    string lowercase(const string& str);
    uint32_t classify(const string& name, const string& file, Region::Role role, Region::Paradigm paradigm);
    uint32_t classify_omp(const string& name);
    uint32_t classify_mpi(const string& name);
    // *INDENT-ON*
}   // unnamed namespace


//--- Constructors & destructor ---------------------------------------------

Region::Region(const IdType   id,
               const String&  canonicalName,
               const String&  displayName,
               const String&  description,
               const Role     role,
               const Paradigm paradigm,
               const String&  filename,
               const uint32_t startLine,
               const uint32_t endLine)
    : mIdentifier(id),
      mCanonicalName(canonicalName),
      mDisplayName(displayName),
      mDescription(description),
      mRole(role),
      mParadigm(paradigm),
      mFilename(filename),
      mStartLine(startLine),
      mEndLine(endLine)
{
    // Arbitrary int's can be assigned to an enum, i.e., we need a range check
    assert((role >= Region::ROLE_UNKNOWN)
           && (role < Region::NUMBER_OF_ROLES));
    assert((paradigm >= Region::PARADIGM_UNKNOWN)
           && (paradigm < Region::NUMBER_OF_PARADIGMS));

    mClass = classify(canonicalName.getString(), filename.getString(),
                      role, paradigm);
}


//--- Access definition data ------------------------------------------------

Region::IdType
Region::getId() const
{
    return mIdentifier;
}


const String&
Region::getCanonicalName() const
{
    return mCanonicalName;
}


const String&
Region::getDisplayName() const
{
    return mDisplayName;
}


const String&
Region::getDescription() const
{
    return mDescription;
}


Region::Role
Region::getRole() const
{
    return mRole;
}


Region::Paradigm
Region::getParadigm() const
{
    return mParadigm;
}


const String&
Region::getFilename() const
{
    return mFilename;
}


uint32_t
Region::getStartLine() const
{
    return mStartLine;
}


uint32_t
Region::getEndLine() const
{
    return mEndLine;
}


//--- Related functions -----------------------------------------------------

namespace pearl
{
//--- Stream I/O operators ---------------------------------

ostream&
operator<<(ostream&      stream,
           const Region& item)
{
    // Special case: undefined region
    if (Region::UNDEFINED == item) {
        return stream << "<undefined>";
    }

    // Adjust indentation
    int indent = getIndent(stream);
    setIndent(stream, indent + 16);

    // Print data
    stream << "Region {" << iendl(indent)
           << "  id          = " << item.getId() << iendl(indent)
           << "  name        = " << item.getCanonicalName();
    if (item.getCanonicalName() != item.getDisplayName()) {
        stream << iendl(indent + 16)
               << item.getDisplayName();
    }
    stream << iendl(indent)
           << "  description = " << item.getDescription() << iendl(indent)
           << "  role        = " << roleToString(item.getRole()) << iendl(indent)
           << "  paradigm    = " << paradigmToString(item.getParadigm()) << iendl(indent)
           << "  filename    = " << item.getFilename() << iendl(indent)
           << "  startLine   = " << item.getStartLine() << iendl(indent)
           << "  endLine     = " << item.getEndLine() << iendl(indent)
           << "}";

    // Reset indentation
    return setIndent(stream, indent);
}


//--- Comparison operators ---------------------------------

bool
operator==(const Region& lhs,
           const Region& rhs)
{
    return ((lhs.getId() == rhs.getId())
            && (lhs.getCanonicalName() == rhs.getCanonicalName())
            && (lhs.getDisplayName() == rhs.getDisplayName())
            && (lhs.getDescription() == rhs.getDescription())
            && (lhs.getRole() == rhs.getRole())
            && (lhs.getParadigm() == rhs.getParadigm())
            && (lhs.getFilename() == rhs.getFilename())
            && (lhs.getStartLine() == rhs.getStartLine())
            && (lhs.getEndLine() == rhs.getEndLine()));
}


bool
operator!=(const Region& lhs,
           const Region& rhs)
{
    return !(lhs == rhs);
}
}   // namespace pearl


//--- Local helper functions ------------------------------------------------

namespace
{
/// @brief Get string representation of a region role.
///
/// Returns a human-readable string representation of the given region
/// @a role, used by the stream I/O functionality.
///
/// @param  role  %Region role
///
/// @return Corresponding string representation
///
const string
roleToString(const Region::Role role)
{
    switch (role) {
        case Region::ROLE_UNKNOWN:
            return "UNKNOWN";

        case Region::ROLE_FUNCTION:
            return "FUNCTION";

        case Region::ROLE_WRAPPER:
            return "WRAPPER";

        case Region::ROLE_LOOP:
            return "LOOP";

        case Region::ROLE_CODE:
            return "CODE";

        case Region::ROLE_PARALLEL:
            return "PARALLEL";

        case Region::ROLE_SECTIONS:
            return "SECTIONS";

        case Region::ROLE_SECTION:
            return "SECTION";

        case Region::ROLE_SINGLE:
            return "SINGLE";

        case Region::ROLE_SINGLE_SBLOCK:
            return "SINGLE_SBLOCK";

        case Region::ROLE_WORKSHARE:
            return "WORKSHARE";

        case Region::ROLE_MASTER:
            return "MASTER";

        case Region::ROLE_CRITICAL:
            return "CRITICAL";

        case Region::ROLE_CRITICAL_SBLOCK:
            return "CRITICAL_SBLOCK";

        case Region::ROLE_BARRIER:
            return "BARRIER";

        case Region::ROLE_IMPLICIT_BARRIER:
            return "IMPLICIT_BARRIER";

        case Region::ROLE_ATOMIC:
            return "ATOMIC";

        case Region::ROLE_FLUSH:
            return "FLUSH";

        case Region::ROLE_ORDERED:
            return "ORDERED";

        case Region::ROLE_ORDERED_SBLOCK:
            return "ORDERED_SBLOCK";

        case Region::ROLE_TASK:
            return "TASK";

        case Region::ROLE_TASK_CREATE:
            return "TASK_CREATE";

        case Region::ROLE_TASK_WAIT:
            return "TASK_WAIT";

        case Region::ROLE_COLL_ONE2ALL:
            return "COLL_ONE2ALL";

        case Region::ROLE_COLL_ALL2ONE:
            return "COLL_ALL2ONE";

        case Region::ROLE_COLL_ALL2ALL:
            return "COLL_ALLTOALL";

        case Region::ROLE_COLL_OTHER:
            return "COLL_OTHER";

        case Region::ROLE_FILE_IO:
            return "FILE_IO";

        case Region::ROLE_POINT2POINT:
            return "POINT2POINT";

        case Region::ROLE_RMA:
            return "RMA";

        case Region::ROLE_DATA_TRANSFER:
            return "DATA_TRANSFER";

        case Region::ROLE_ARTIFICIAL:
            return "ARTIFICIAL";

        // For "NUMBER_OF_ROLES" -- to make the compiler happy...
        default:
            break;
    }

    // Since all possible region roles should be handled in the switch
    // statement above, something went wrong if we reach this point...
    assert(false);
    return "";
}


/// @brief Get string representation of a region paradigm.
///
/// Returns a human-readable string representation of the given region
/// @a paradigm, used by the stream I/O functionality.
///
/// @param  paradigm  %Region paradigm
///
/// @return Corresponding string representation
///
const string
paradigmToString(const Region::Paradigm paradigm)
{
    switch (paradigm) {
        case Region::PARADIGM_UNKNOWN:
            return "UNKNOWN";

        case Region::PARADIGM_USER:
            return "USER";

        case Region::PARADIGM_COMPILER:
            return "COMPILER";

        case Region::PARADIGM_OPENMP:
            return "OPENMP";

        case Region::PARADIGM_MPI:
            return "MPI";

        case Region::PARADIGM_CUDA:
            return "CUDA";

        case Region::PARADIGM_MEASUREMENT_SYSTEM:
            return "MEASUREMENT SYSTEM";

        // For "NUMBER_OF_PARADIGMS" -- to make the compiler happy...
        default:
            break;
    }

    // Since all possible region paradigms should be handled in the switch
    // statement above, something went wrong if we reach this point...
    assert(false);
    return "";
}


struct fo_tolower
    : public std:: unary_function<int, int>
{
    int
    operator()(int x) const
    {
        return std::tolower(x);
    }
};


string
lowercase(const string& str)
{
    string result(str);

    transform(str.begin(), str.end(), result.begin(), fo_tolower());

    return result;
}


uint32_t
classify(const string&    name,
         const string&    file,
         Region::Role     role,
         Region::Paradigm paradigm)
{
    // Measurement-related regions
    if (paradigm == Region::PARADIGM_MEASUREMENT_SYSTEM || file == "EPIK") {
        return Region::cCLASS_INTERNAL;
    }

    // MPI regions
    if (file == "MPI") {
        return classify_mpi(lowercase(name.substr(4)));
    }

    // OpenMP regions
    if ((name.compare(0, 5, "!$omp") == 0) || (file == "OMP")) {
        return classify_omp(lowercase(name));
    }

    // User regions
    return Region::cCLASS_USER;
}


uint32_t
classify_omp(const string& name)
{
    string ompName(name);

    string::size_type pos = name.find('@');
    if (pos != string::npos) {
        ompName.erase(pos - 1);
    }

    if (ompName == "!$omp parallel") {
        return Region::cCLASS_OMP | Region::cCAT_OMP_PARALLEL;
    }
    if (ompName == "!$omp barrier") {
        return Region::cCLASS_OMP | Region::cCAT_OMP_SYNC
               | Region::cTYPE_OMP_BARRIER | Region::cMODE_OMP_EXPLICIT;
    }
    if ((ompName == "!$omp ibarrier") || (ompName == "!$omp implicit barrier")) {
        return Region::cCLASS_OMP | Region::cCAT_OMP_SYNC
               | Region::cTYPE_OMP_BARRIER | Region::cMODE_OMP_IMPLICIT;
    }
    return Region::cCLASS_OMP;
}


uint32_t
classify_mpi(const string& name)
{
    // MPI setup
    if ((name == "init") || (name == "init_thread")) {
        return Region::cCLASS_MPI | Region::cCAT_MPI_SETUP | Region::cTYPE_MPI_INIT;
    }
    if (name == "finalize") {
        return Region::cCLASS_MPI | Region::cCAT_MPI_SETUP | Region::cTYPE_MPI_FINALIZE;
    }

    // MPI collective
    if (name == "barrier") {
        return Region::cCLASS_MPI | Region::cCAT_MPI_COLLECTIVE | Region::cTYPE_MPI_BARRIER;
    }
    if ((name == "bcast") || (name == "scatter") || (name == "scatterv")) {
        return Region::cCLASS_MPI | Region::cCAT_MPI_COLLECTIVE | Region::cTYPE_MPI_ONE_TO_N;
    }
    if ((name == "reduce") || (name == "gather") || (name == "gatherv")) {
        return Region::cCLASS_MPI | Region::cCAT_MPI_COLLECTIVE | Region::cTYPE_MPI_N_TO_ONE;
    }
    if ((name == "allgather") || (name == "allgatherv") || (name == "allreduce")
        || (name == "alltoall") || (name == "alltoallv") || (name == "alltoallw")
        || (name == "reduce_scatter") || (name == "reduce_scatter_block")) {
        return Region::cCLASS_MPI | Region::cCAT_MPI_COLLECTIVE | Region::cTYPE_MPI_N_TO_N;
    }
    if ((name == "scan") || (name == "exscan")) {
        return Region::cCLASS_MPI | Region::cCAT_MPI_COLLECTIVE | Region::cTYPE_MPI_PARTIAL;
    }

    // MPI point-to-point
    if (name.compare(0, 8, "sendrecv") == 0) {
        return Region::cCLASS_MPI | Region::cCAT_MPI_P2P | Region::cTYPE_MPI_SENDRECV | Region::cMODE_MPI_NONBLOCKING;
    }
    if (name.find("send") != string::npos) {
        uint32_t result = Region::cCLASS_MPI | Region::cCAT_MPI_P2P | Region::cTYPE_MPI_SEND;

        if (name[0] == 'i') {
            result |= Region::cMODE_MPI_NONBLOCKING;
        } else if (name.find("init") != string::npos) {
            result |= Region::cMODE_MPI_PERSISTENT;
        }
        if (name.find("ssend") != string::npos) {
            result |= Region::cMODE_MPI_SYNCHRONOUS;
        } else if (name.find("bsend") != string::npos) {
            result |= Region::cMODE_MPI_BUFFERED;
        } else if (name.find("rsend") != string::npos) {
            result |= Region::cMODE_MPI_READY;
        } else {
            result |= Region::cMODE_MPI_STANDARD;
        }
        return result;
    }

    if (name == "recv") {
        return Region::cCLASS_MPI | Region::cCAT_MPI_P2P | Region::cTYPE_MPI_RECV;
    }
    if (name == "irecv") {
        return Region::cCLASS_MPI | Region::cCAT_MPI_P2P | Region::cTYPE_MPI_RECV | Region::cMODE_MPI_NONBLOCKING;
    }
    if (name == "recv_init") {
        return Region::cCLASS_MPI | Region::cCAT_MPI_P2P | Region::cTYPE_MPI_RECV | Region::cMODE_MPI_PERSISTENT;
    }
    if ((name == "test") || (name == "testany")) {
        return Region::cCLASS_MPI | Region::cCAT_MPI_P2P | Region::cTYPE_MPI_TEST | Region::cMODE_MPI_SINGLE;
    }
    if ((name == "testall") || (name == "testsome")) {
        return Region::cCLASS_MPI | Region::cCAT_MPI_P2P | Region::cTYPE_MPI_TEST | Region::cMODE_MPI_MANY;
    }
    if ((name == "wait") || (name == "waitany")) {
        return Region::cCLASS_MPI | Region::cCAT_MPI_P2P | Region::cTYPE_MPI_WAIT | Region::cMODE_MPI_SINGLE;
    }
    if ((name == "waitall") || (name == "waitsome")) {
        return Region::cCLASS_MPI | Region::cCAT_MPI_P2P | Region::cTYPE_MPI_WAIT | Region::cMODE_MPI_MANY;
    }
    if ((name.compare(0, 6, "buffer") == 0)
        || (name == "cancel")
        || (name == "get_count")
        || (name.find("probe") != string::npos)
        || (name.compare(0, 7, "request") == 0)
        || (name.compare(0, 5, "start") == 0)) {
        return Region::cCLASS_MPI | Region::cCAT_MPI_P2P;
    }

    // MPI I/O
    if (name.compare(0, 4, "file") == 0) {
        return Region::cCLASS_MPI | Region::cCAT_MPI_IO;
    }

    // MPI RMA
    if ((name == "accumulate") || (name == "put")) {
        return Region::cCLASS_MPI | Region::cCAT_MPI_RMA | Region::cTYPE_MPI_RMA_COMM | Region::cMODE_RMA_PUT;
    }
    if (name == "get") {
        return Region::cCLASS_MPI | Region::cCAT_MPI_RMA | Region::cTYPE_MPI_RMA_COMM | Region::cMODE_RMA_GET;
    }
    if (name.compare(0, 3, "win") == 0) {
        uint32_t result = Region::cCLASS_MPI | Region::cCAT_MPI_RMA;
        string   mode   = name.substr(4);

        if (mode == "fence") {
            return result | Region::cTYPE_MPI_RMA_COLL | Region::cMODE_RMA_FENCE;
        }
        if (mode == "create") {
            return result | Region::cTYPE_MPI_RMA_COLL | Region::cMODE_RMA_WIN_CREATE;
        }
        if (mode == "free") {
            return result | Region::cTYPE_MPI_RMA_COLL | Region::cMODE_RMA_WIN_FREE;
        }
        if (mode == "start") {
            return result | Region::cTYPE_MPI_RMA_GATS | Region::cMODE_RMA_START;
        }
        if (mode == "complete") {
            return result | Region::cTYPE_MPI_RMA_GATS | Region::cMODE_RMA_COMPLETE;
        }
        if (mode == "post") {
            return result | Region::cTYPE_MPI_RMA_GATS | Region::cMODE_RMA_POST;
        }
        if (mode == "wait") {
            return result | Region::cTYPE_MPI_RMA_GATS | Region::cMODE_RMA_WAIT;
        }
        if (mode == "test") {
            return result | Region::cTYPE_MPI_RMA_GATS | Region::cMODE_RMA_TEST;
        }
        if (mode == "lock") {
            return result | Region::cTYPE_MPI_RMA_PASSIVE | Region::cMODE_RMA_LOCK;
        }
        if (mode == "unlock") {
            return result | Region::cTYPE_MPI_RMA_PASSIVE | Region::cMODE_RMA_UNLOCK;
        }
    }

    return Region::cCLASS_MPI;
}
}   // unnamed namespace
