/****************************************************************************
**  SCALASCA    http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2013                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  Copyright (c) 2009-2013                                                **
**  German Research School for Simulation Sciences GmbH,                   **
**  Laboratory for Parallel Programming                                    **
**                                                                         **
**  Copyright (c) 2006                                                     **
**  TU Dresden, Zentrum fuer Informationsdienste und Hochleistungsrechnen  **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include <config.h>
#include <pearl/LocalTrace.h>

#include <algorithm>
#include <cassert>
#include <cstdlib>
#include <cstring>
#include <iomanip>
#include <map>
#include <sstream>
#include <stack>

#include <pearl/Callpath.h>
#include <pearl/Enter_rep.h>
#include <pearl/Error.h>
#include <pearl/GlobalDefs.h>
#include <pearl/Leave_rep.h>
#include <pearl/LocalTrace.h>
#include <pearl/Region.h>

#include "Calltree.h"

using namespace std;
using namespace pearl;


//---------------------------------------------------------------------------
//
//  class LocalTrace
//
//---------------------------------------------------------------------------

//--- Constructors & destructor ---------------------------------------------

LocalTrace::LocalTrace(const GlobalDefs& defs,
                       const Location&   location)
  : m_defs(defs),
    m_omp_parallel(0),
    m_omp_tasks(0),
    m_mpi_regions(0),
    m_location(const_cast<Location*>(&location))
{
}


LocalTrace::~LocalTrace()
{
  /* Delete events */
  container_type::reverse_iterator it = m_events.rbegin();
  while (it != m_events.rend()) {
    delete *it;
    ++it;
  }
}


//--- Iterator handling -----------------------------------------------------

LocalTrace::iterator LocalTrace::begin() const
{
  return Event(this);
}

LocalTrace::iterator LocalTrace::end() const
{
  return Event(this, true);
}


LocalTrace::reverse_iterator LocalTrace::rbegin() const
{
  return reverse_iterator(end());
}


LocalTrace::reverse_iterator LocalTrace::rend() const
{
  return reverse_iterator(begin());
}



//--- Get trace information -------------------------------------------------

uint32_t LocalTrace::size() const
{
  return m_events.size();
}


uint32_t LocalTrace::num_events() const
{
  return m_events.size();
}


uint32_t LocalTrace::num_omp_parallel() const
{
  return m_omp_parallel;
}


uint32_t LocalTrace::num_omp_tasks() const
{
  return m_omp_tasks;
}


uint32_t LocalTrace::num_mpi_regions() const
{
  return m_mpi_regions;
}


const Location& LocalTrace::get_location() const
{
  return *m_location;
}


const GlobalDefs& LocalTrace::get_definitions() const
{
  return m_defs;
}


//--- Store new events ------------------------------------------------------

void LocalTrace::add_event(Event_rep* event)
{
  m_events.push_back(event);
}


//--- Inserting & removing elements -----------------------------------------

LocalTrace::iterator LocalTrace::remove_if(bool (*predicate)(const Event_rep* event))
{
  uint32_t index = std::remove_if(m_events.begin(),
                                  m_events.end(),
                                  predicate) - m_events.begin();

  return Event(this, index);
}


LocalTrace::iterator LocalTrace::insert(LocalTrace::iterator pos,
                                        Event_rep*           event)
{
  m_events.insert(m_events.begin() + pos.m_index, event);
  m_defs.get_calltree()->setModified();

  return pos;
}


LocalTrace::iterator LocalTrace::erase(LocalTrace::iterator pos)
{
  assert(pos.m_index < size());

  m_events.erase(m_events.begin() + pos.m_index);
  m_defs.get_calltree()->setModified();

  return pos;
}


LocalTrace::iterator LocalTrace::erase(LocalTrace::iterator begin,
                                       LocalTrace::iterator end)
{
  assert(begin.m_index < size());

  m_events.erase(m_events.begin() + begin.m_index, 
                 m_events.begin() + end.m_index);
  m_defs.get_calltree()->setModified();

  return begin;
}


LocalTrace::iterator LocalTrace::replace(LocalTrace::iterator pos, Event_rep* event)
{
  m_events[pos.m_index] = event;
  m_defs.get_calltree()->setModified();

  return pos;
}


LocalTrace::iterator LocalTrace::swap(const LocalTrace::iterator& a, const LocalTrace::iterator& b)
{
  std::iter_swap(m_events.begin()+a.m_index, m_events.begin()+b.m_index);
  m_defs.get_calltree()->setModified();

  return a;
}


//--- Private methods -------------------------------------------------------

Event_rep* LocalTrace::operator[](uint32_t index) const
{
  return m_events[index];
}


void LocalTrace::preprocess()
{
  uint32_t numEvents = m_events.size();

  // Rectify consistency of buffer flush events
  for (uint32_t remaining = numEvents; remaining > 0; --remaining) {
    uint32_t   current  = remaining - 1;
    Event_rep* eventRep = m_events[current];
    bool       isFlush  = false;

    // Determine whether <current> points to a FLUSHING event (ENTER or LEAVE)
    if (ENTER == eventRep->getType()) {
      Enter_rep*    enterRep = static_cast<Enter_rep*>(eventRep);
      const Region& region   = enterRep->getRegionEntered();

      if (m_defs.getFlushingRegion() == region)
        isFlush = true;
    }
    else if (LEAVE == eventRep->getType()) {
      Leave_rep*    leaveRep = static_cast<Leave_rep*>(eventRep);
      const Region& region   = leaveRep->getRegionLeft();

      if (m_defs.getFlushingRegion() == region)
        isFlush = true;
    }

    // "Bubble" the FLUSHING event to the correct position in the event stream
    if (isFlush) {
      timestamp_t eventTime = eventRep->getTimestamp();
      uint32_t    next      = current + 1;
      while (next < numEvents && m_events[next]->getTimestamp() <= eventTime) {
        std::swap(m_events[current], m_events[next]);
        ++current;
        ++next;
      }
    }
  }

  std::map<uint64_t, uint32_t> requestmap;
  for (uint32_t index = 0; index < numEvents; ++index) {
    Event_rep* eventRep = m_events[index];

    // When entering an OpenMP parallel or MPI region, increment counter
    if (eventRep->isOfType(GROUP_ENTER)) {
      Enter_rep*    enter  = static_cast<Enter_rep*>(eventRep);
      const Region& region = enter->getRegionEntered();

      if (is_omp_parallel(region))
        m_omp_parallel++;

      if (is_mpi_api(region))
        m_mpi_regions++;
    }

    // When an OpenMP task is created, increment counter
    if (eventRep->isOfType(OMP_TASK_CREATE))
      m_omp_tasks++;

    // Precompute next/previous request offset
    event_t type = eventRep->getType();
    if ((type == MPI_SEND_REQUEST) ||
        (type == MPI_SEND_COMPLETE) ||
        (type == MPI_RECV_REQUEST) ||
        (type == MPI_RECV_COMPLETE) ||
        (type == MPI_REQUEST_TESTED) ||
        (type == MPI_CANCELLED)) {
      uint64_t requestId = eventRep->getRequestId();

      std::map<uint64_t, uint32_t>::iterator rmapit = requestmap.find(requestId);
      if (rmapit != requestmap.end()) {
        m_events[rmapit->second]->set_next_reqoffs(index - rmapit->second);
        eventRep                ->set_prev_reqoffs(index - rmapit->second);

        if (eventRep->isOfType(MPI_REQUEST_TESTED))
          rmapit->second = index;
        else
          requestmap.erase(rmapit);
      } else {
        requestmap.insert(std::make_pair(requestId, index));
      }
    }
  }
}
