/****************************************************************************
**  SCALASCA    http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2013                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  Copyright (c) 2009-2013                                                **
**  German Research School for Simulation Sciences GmbH,                   **
**  Laboratory for Parallel Programming                                    **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include <config.h>
#include <pearl/GlobalDefs.h>

#include <cfloat>
#include <cstddef>
#include <cstdlib>
#include <algorithm>
#include <functional>

#include <pearl/Cartesian.h>
#include <pearl/Error.h>
#include <pearl/Metric.h>
#include <pearl/RmaWindow.h>

#include "Calltree.h"
#include "SystemTree.h"

using namespace std;
using namespace pearl;


//--- Utility functions -----------------------------------------------------

namespace
{

template<class containerT>
inline void delete_seq_container(const containerT& container)
{
  typename containerT::const_iterator it = container.begin();
  while (it != container.end()) {
    delete *it;
    ++it;
  }
}


template<class containerT>
inline typename containerT::value_type get_entry(const containerT& container,
                                                 ident_t           id,
                                                 const string&     desc)
{
  if (id >= container.size()){
    throw RuntimeError("GlobalDefs::" + desc + " -- ID out of bounds.");
  }

  return container[id];
}


template<class containerT>
inline void add_entry(containerT&                     container,
                      typename containerT::value_type entry,
                      const string&                   desc)
{
  if (entry->get_id() == container.size())
    container.push_back(entry);
  else   // This should never happen!
    throw RuntimeError("GlobalDefs::" + desc + " -- Invalid ID.");
}


template<class containerT>
inline void addEntry(containerT&                     container,
                     typename containerT::value_type entry,
                     const string&                   desc)
{
  if (entry->getId() == container.size())
    container.push_back(entry);
  else   // This should never happen!
    throw RuntimeError("GlobalDefs::" + desc + " -- Invalid ID.");
}


inline bool locCmp(Location* item, Location::IdType id)
{
    return item->getId() < id;
}


inline bool commCmp(Communicator* item, Communicator::IdType id)
{
    return item->getId() < id;
}


inline bool procGrpCmp(ProcessGroup* item, ProcessGroup::IdType id)
{
    return item->getId() < id;
}


}   // unnamed namespace


//---------------------------------------------------------------------------
//
//  class GlobalDefs
//
//---------------------------------------------------------------------------

//--- Constructors & destructor ---------------------------------------------

GlobalDefs::GlobalDefs()
  : mGlobalOffset(DBL_MAX),
    mPausingRegion(NULL),
    mFlushingRegion(NULL)
{
  // Create call tree & system tree objects
  mCalltree   = new Calltree();
  mSystemTree = new SystemTree();
}


GlobalDefs::~GlobalDefs()
{
  // Delete entity containers
  // NOTE: The "MEASUREMENT OFF" and "TRACE BUFFER FLUSH" regions will be
  //       implicitly deleted.
  delete_seq_container(mCallsites);
  delete_seq_container(mCommunicators);
  delete_seq_container(mLocations);
  delete_seq_container(mLocationGroups);
  delete_seq_container(mProcessGroups);
  delete_seq_container(mRegions);
  delete_seq_container(mStrings);

  delete_seq_container(m_metrics);
  delete_seq_container(m_cartesians);
  delete_seq_container(m_windows);

  // Delete call tree & system tree objects
  delete mCalltree;
  delete mSystemTree;
}


//--- Get number of stored definitions --------------------------------------

uint32_t GlobalDefs::numCallpaths() const
{
  return mCalltree->numCallpaths();
}


uint32_t GlobalDefs::numCallsites() const
{
  return mCallsites.size();
}


uint32_t GlobalDefs::numCommunicators() const
{
  return mCommunicators.size();
}


uint32_t GlobalDefs::numLocations() const
{
  return mLocations.size();
}


uint32_t GlobalDefs::numLocationGroups() const
{
  return mLocationGroups.size();
}


uint32_t GlobalDefs::numProcessGroups() const
{
  return mProcessGroups.size();
}


uint32_t GlobalDefs::numRegions() const
{
  return mRegions.size();
}


uint32_t GlobalDefs::numStrings() const
{
  return mStrings.size();
}


uint32_t GlobalDefs::numSystemNodes() const
{
  return mSystemTree->numSystemNodes();
}


//--- Get stored definitions by ID ------------------------------------------

const Callpath& GlobalDefs::getCallpath(Callpath::IdType id) const
{
  if (Callpath::NO_ID == id)
    return Callpath::UNDEFINED;

  Callpath* entry = mCalltree->getCallpath(id);
  return *entry;
}


const Callsite& GlobalDefs::getCallsite(Callsite::IdType id) const
{
  if (Callsite::NO_ID == id)
    return Callsite::UNDEFINED;

  Callsite* entry = get_entry(mCallsites, id, "getCallsite(Callsite::IdType)");
  return *entry;
}


const Communicator& GlobalDefs::getCommunicator(Communicator::IdType id) const
{
    communicator_container::const_iterator it = lower_bound(mCommunicators.begin(),
                                                            mCommunicators.end(),
                                                            id,
                                                            ptr_fun(commCmp));
    if (mCommunicators.end() == it || (*it)->getId() != id)
        throw RuntimeError("GlobalDefs::getCommunicator(Communicator::IdType) -- Invalid ID.");

    return *(*it);
}


const Location& GlobalDefs::getLocation(Location::IdType id) const
{
    location_container::const_iterator it = lower_bound(mLocations.begin(),
                                                        mLocations.end(),
                                                        id,
                                                        ptr_fun(locCmp));
    if (mLocations.end() == it || (*it)->getId() != id)
        throw RuntimeError("GlobalDefs::getLocation(Location::IdType) -- Invalid ID.");

    return *(*it);
}


const LocationGroup& GlobalDefs::getLocationGroup(LocationGroup::IdType id) const
{
    LocationGroup* entry = get_entry(mLocationGroups, id,
                                     "getLocationGroup(LocationGroup::IdType)");
    return *entry;
}


const ProcessGroup& GlobalDefs::getProcessGroup(ProcessGroup::IdType id) const
{
    pgroup_container::const_iterator it = lower_bound(mProcessGroups.begin(),
                                                      mProcessGroups.end(),
                                                      id,
                                                      ptr_fun(procGrpCmp));
    if (mProcessGroups.end() == it || (*it)->getId() != id)
        throw RuntimeError("GlobalDefs::getProcessGroup(ProcessGroup::IdType) -- Invalid ID.");

    return *(*it);
}


const Region& GlobalDefs::getRegion(Region::IdType id) const
{
  if (Region::NO_ID == id)
    return Region::UNDEFINED;

  Region* entry = get_entry(mRegions, id, "getRegion(Region::IdType)");
  return *entry;
}


const String& GlobalDefs::getString(String::IdType id) const
{
  switch (id) {
    case String::NODE_ID:
      return String::NODE;

    case String::MACHINE_ID:
      return String::MACHINE;

    case String::NO_ID:
      return String::UNDEFINED;

    default:
      break;
  }

  String* entry = get_entry(mStrings, id, "getString(String::IdType)");
  return *entry;
}


const SystemNode& GlobalDefs::getSystemNode(SystemNode::IdType id) const
{
  if (SystemNode::NO_ID == id)
    return SystemNode::UNDEFINED;

  SystemNode* entry = mSystemTree->getSystemNode(id);
  return *entry;
}


const Location& GlobalDefs::getLocationByIndex(uint32_t index) const
{
  Location* entry = get_entry(mLocations, index, "getLocationByIndex(uint32_t)");
  return *entry;
}


Callpath* GlobalDefs::get_cnode(ident_t id) const
{
  if (id == PEARL_NO_ID)
    return NULL;

  Callpath* result = mCalltree->getCallpath(id);
  if (!result)
    throw RuntimeError("GlobalDefs::get_cnode(ident_t) -- ID out of bounds.");

  return result;
}


Metric* GlobalDefs::get_metric(ident_t id) const
{
  return get_entry(m_metrics, id, "get_metric(ident_t)");
}


ProcessGroup* GlobalDefs::get_group(ident_t id) const
{
  return get_entry(mProcessGroups, id, "get_group(ident_t)");
}


Communicator* GlobalDefs::get_comm(ident_t id) const
{
  if (Communicator::NO_ID == id)
    return NULL;

  communicator_container::const_iterator it = lower_bound(mCommunicators.begin(),
                                                          mCommunicators.end(),
                                                          id,
                                                          ptr_fun(commCmp));
  if (mCommunicators.end() == it || (*it)->getId() != id)
      throw RuntimeError("GlobalDefs::get_comm(ident_t) -- Invalid ID.");

  return (*it);
}


Cartesian* GlobalDefs::get_cartesian(ident_t id) const
{
  return get_entry(m_cartesians, id, "get_cartesian(ident_t)");
}


RmaWindow* GlobalDefs::get_window(ident_t id) const
{
  return get_entry(m_windows, id, "get_window(ident_t)");
}


SystemNode* GlobalDefs::get_systemnode(ident_t id) const
{
  return mSystemTree->getSystemNode(id);
}


LocationGroup* GlobalDefs::get_lgroup(ident_t id) const
{
  return get_entry(mLocationGroups, id, "get_lgroup(ident_t)");
}


//--- Get internal regions --------------------------------------------------

const Region& GlobalDefs::getPausingRegion() const
{
  return *mPausingRegion;
}


const Region& GlobalDefs::getFlushingRegion() const
{
  return *mFlushingRegion;
}


//--- Call tree handling ----------------------------------------------------

Calltree* GlobalDefs::get_calltree() const
{
  return mCalltree;
}


void GlobalDefs::set_calltree(Calltree* calltree)
{
  if (mCalltree == calltree)
    return;

  delete mCalltree;
  mCalltree = calltree;
}


//--- System tree handling --------------------------------------------------

SystemTree* GlobalDefs::get_systree() const
{
  return mSystemTree;
}


//--- Global time offset handling -------------------------------------------

timestamp_t GlobalDefs::getGlobalOffset() const
{
  return mGlobalOffset;
}


void GlobalDefs::setGlobalOffset(timestamp_t offset)
{
  mGlobalOffset = offset;
}


//--- Internal methods ------------------------------------------------------

/// @todo Check whether UNKNOWN callpath ist still necessary
void GlobalDefs::setup()
{
  // Create special "MEASUREMENT OFF" region if necessary
  if (!mPausingRegion) {
    // Create "MEASUREMENT OFF" string
    String::IdType stringId = numStrings();
    String*        pauseStr = new String(stringId, "MEASUREMENT OFF");
    addString(pauseStr);

    // Create "MEASUREMENT OFF" region
    Region::IdType regionId = numRegions();
    mPausingRegion = new Region(regionId,
                                *pauseStr,
                                *pauseStr,
                                String::UNDEFINED,
                                Region::ROLE_ARTIFICIAL,
                                Region::PARADIGM_USER,
                                String::UNDEFINED,
                                PEARL_NO_NUM,
                                PEARL_NO_NUM);
    addRegion(mPausingRegion);
  }

  // Create special "TRACE BUFFER FLUSH" region if necessary
  if (!mFlushingRegion) {
    // Create "TRACE BUFFER FLUSH" string
    String::IdType stringId = numStrings();
    String*        flushStr = new String(stringId, "TRACE BUFFER FLUSH");
    addString(flushStr);

    // Create "TRACE BUFFER FLUSH" region
    Region::IdType regionId = numRegions();
    mFlushingRegion = new Region(regionId,
                                 *flushStr,
                                 *flushStr,
                                 String::UNDEFINED,
                                 Region::ROLE_ARTIFICIAL,
                                 Region::PARADIGM_MEASUREMENT_SYSTEM,
                                 String::UNDEFINED,
                                 PEARL_NO_NUM,
                                 PEARL_NO_NUM);
    addRegion(mFlushingRegion);
  }

/*
  // Create "UNKNOWN" call path
  if (mCalltree->numCallpaths() == 0)
    mCalltree->getCallpath(Region::UNDEFINED, Callsite::UNDEFINED, NULL);
*/

  // Reset call tree modification flag
  mCalltree->setModified(false);
}


//--- Add new entities ------------------------------------------------------

void GlobalDefs::addCallpath(Callpath* cnode)
{
  mCalltree->addCallpath(cnode);
}


void GlobalDefs::addCallsite(Callsite* callsite)
{
  addEntry(mCallsites, callsite, "addCallsite(Callsite*)");
}


void GlobalDefs::addCommunicator(Communicator* comm)
{
  // Simple case: first location or increasing location identifiers
  if (mCommunicators.empty()
      || mCommunicators.back()->getId() < comm->getId()) {
    mCommunicators.push_back(comm);
  }

  // Complex case: arbitrary location identifiers
  else {
    communicator_container::iterator it = lower_bound(mCommunicators.begin(),
                                                      mCommunicators.end(),
                                                      comm->getId(),
                                                      ptr_fun(commCmp));
    if (mCommunicators.end() == it || (*it)->getId() == comm->getId())
        throw RuntimeError("GlobalDefs::addCommunicator(Communicator*) -- Invalid ID.");

    mCommunicators.insert(it, comm);
  }
}


void GlobalDefs::addLocation(Location* location)
{
  // Simple case: first location or increasing location identifiers
  if (mLocations.empty()
      || mLocations.back()->getId() < location->getId()) {
    mLocations.push_back(location);
  }

  // Complex case: arbitrary location identifiers
  else {
    location_container::iterator it = lower_bound(mLocations.begin(),
                                                  mLocations.end(),
                                                  location->getId(),
                                                  ptr_fun(locCmp));
    if (mLocations.end() == it || (*it)->getId() == location->getId())
        throw RuntimeError("GlobalDefs::addLocation(Location*) -- Invalid ID.");

    mLocations.insert(it, location);
  }
}


void GlobalDefs::addLocationGroup(LocationGroup* group)
{
  addEntry(mLocationGroups, group, "addLocationGroup(LocationGroup*)");
}


void GlobalDefs::addProcessGroup(ProcessGroup* group)
{
  // Simple case: first location or increasing location identifiers
  if (mProcessGroups.empty()
      || mProcessGroups.back()->getId() < group->getId()) {
    mProcessGroups.push_back(group);
  }

  // Complex case: arbitrary location identifiers
  else {
    pgroup_container::iterator it = lower_bound(mProcessGroups.begin(),
                                                mProcessGroups.end(),
                                                group->getId(),
                                                ptr_fun(procGrpCmp));
    if (mProcessGroups.end() == it || (*it)->getId() == group->getId())
        throw RuntimeError("GlobalDefs::addProcessGroup(ProcessGroup*) -- Invalid ID.");

    mProcessGroups.insert(it, group);
  }
}


void GlobalDefs::addRegion(Region* region)
{
  addEntry(mRegions, region, "addRegion(Region*)");
}


void GlobalDefs::addString(String* str)
{
  addEntry(mStrings, str, "addString(String*)");
}


void GlobalDefs::addSystemNode(SystemNode* node)
{
  mSystemTree->addSystemNode(node);
}


void GlobalDefs::add_metric(Metric* metric)
{
  add_entry(m_metrics, metric, "add_metric(Metric*)");
}


void GlobalDefs::add_cartesian(Cartesian* cart)
{
  add_entry(m_cartesians, cart, "add_cartesian(Cartesian*)");
}


void GlobalDefs::add_window(RmaWindow* window)
{
  add_entry(m_windows, window, "add_window(RmaWindow*)");
}


//--- Set internal regions --------------------------------------------------

void GlobalDefs::setPausingRegion(Region* const region)
{
  if (mPausingRegion)
    throw FatalError("Duplicate definition of PAUSING region!");

  mPausingRegion = region;
}


void GlobalDefs::setFlushingRegion(Region* const region)
{
  if (mFlushingRegion)
    throw FatalError("Duplicate definition of FLUSHING region!");

  mFlushingRegion = region;
}
