/****************************************************************************
**  SCALASCA    http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2013                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  Copyright (c) 2009-2013                                                **
**  German Research School for Simulation Sciences GmbH,                   **
**  Laboratory for Parallel Programming                                    **
**                                                                         **
**  Copyright (c) 2006                                                     **
**  TU Dresden, Zentrum fuer Informationsdienste und Hochleistungsrechnen  **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include <config.h>
#include <pearl/Event.h>

#include <iostream>

#include <pearl/Buffer.h>
#include <pearl/Callpath.h>
#include <pearl/Enter_rep.h>
#include <pearl/Error.h>
#include <pearl/Event_rep.h>
#include <pearl/GlobalDefs.h>
#include <pearl/Leave_rep.h>
#include <pearl/LocalTrace.h>
#include <pearl/Location.h>

#include "pearl_iomanip.h"

using namespace std;
using namespace pearl;


//---------------------------------------------------------------------------
//
//  class Event
//
//---------------------------------------------------------------------------

#define ITEM   (*m_trace)[m_index]

//--- Default constructor ---------------------------------------------------

Event::Event()
  : m_trace(0), m_index(0)
{
}

//--- Comparison operators --------------------------------------------------

bool Event::operator==(const Event& rhs) const
{
  return m_trace == rhs.m_trace &&
         m_index == rhs.m_index;
}


bool Event::operator!=(const Event& rhs) const
{
  return m_trace != rhs.m_trace ||
         m_index != rhs.m_index;
}


bool Event::operator<(const Event& rhs) const
{
  // Compare timestamps
  return ITEM->getTimestamp() < (*rhs.m_trace)[rhs.m_index]->getTimestamp();
}


bool Event::operator>(const Event& rhs) const
{
  // Compare timestamps
  return rhs < *this;
}


//--- Forward iterator interface --------------------------------------------

Event& Event::operator++()
{
  ++m_index;
  return *this;
}


Event Event::operator++(int)
{
  Event result(*this);
  ++m_index;
  return result;
}


//--- Backward iterator interface -------------------------------------------

Event& Event::operator--()
{
  --m_index;
  return *this;
}


Event Event::operator--(int)
{
  Event result(*this);
  --m_index;
  return result;
}


//--- Check iterator validity -----------------------------------------------

bool Event::is_valid() const
{
  return m_index < m_trace->size();
}


//--- Access related events -------------------------------------------------

Event Event::enterptr() const
{
  // Search for associated ENTER event while keeping track of nested
  // ENTERs/LEAVEs (necessary because of possible buffer flushing)
  int depth = 0;
  Event result(*this);
  --result;
  while (!(result->isOfType(GROUP_ENTER) && 0 == depth)) {
    if (result->isOfType(LEAVE))
      ++depth;
    if (result->isOfType(GROUP_ENTER))
      --depth;

    --result;
  }

  return result;
}


Event Event::leaveptr() const
{
  // Search for associated LEAVE event while keeping track of nested
  // ENTERs/LEAVEs (necessary because of possible buffer flushing)
  int depth = 0;
  Event result(*this);
  ++result;
  while (!(result->isOfType(LEAVE) && 0 == depth)) {
    if (result->isOfType(GROUP_ENTER))
      ++depth;
    if (result->isOfType(LEAVE))
      --depth;

    ++result;
  }

  return result;
}


Event Event::beginptr() const
{
  // Sanity check
  if (!ITEM->isOfType(GROUP_END))
    throw FatalError("Event::beginptr() -- "
                     "Method only allowed for END-type events!");

  // Search associated BEGIN-type event
  // NOTE: This assumes that no other BEGIN/END pair can be in between
  Event result(*this);
  --result;
  while (!result->isOfType(GROUP_BEGIN)) {
    --result;
  }

  return result;
}


Event Event::endptr() const
{
  // Sanity check
  if (!ITEM->isOfType(GROUP_BEGIN))
    throw FatalError("Event::endptr() -- "
                     "Method only allowed for BEGIN-type events!");

  // Search associated END-type event
  // NOTE: This assumes that no other BEGIN/END pair can be in between
  Event result(*this);
  ++result;
  while (!result->isOfType(GROUP_END)) {
    ++result;
  }

  return result;
}


Event Event::request() const
{
  // Search for a corresponding request event.
  // For blocking events, reqoffs is 0, hence *this is returned.

  uint32_t i = m_index;

  for (uint32_t offs = (*m_trace)[i]->get_prev_reqoffs(); offs; i -= offs)
    offs = (*m_trace)[i]->get_prev_reqoffs();

  return Event(m_trace, i);
}


Event Event::completion() const
{
  // Search for a corresponding completion event

  uint32_t i = m_index;

  for (uint32_t offs = (*m_trace)[i]->get_next_reqoffs(); offs; i += offs)
    offs = (*m_trace)[i]->get_next_reqoffs();
  
  return Event(m_trace, i);
}


Event Event::next_request_event() const
{
  // Search for next corresponding test or completion event
  return Event(m_trace, m_index + ITEM->get_next_reqoffs());
}


Event Event::prev_request_event() const
{
  // Search for previous corresponding test or request event
  return Event(m_trace, m_index - ITEM->get_prev_reqoffs());
}


//--- Access local event information ----------------------------------------

Callpath* Event::get_cnode() const
{
  if (const Enter_rep* event = dynamic_cast<const Enter_rep*>(ITEM))
    return event->getCallpath();

  if (const Leave_rep* event = dynamic_cast<const Leave_rep*>(ITEM))
    return event->getCallpath();

  return enterptr().get_cnode();
}


const Location& Event::get_location() const
{
  return m_trace->get_location();
}


//--- Access event representation -------------------------------------------

Event_rep& Event::operator*() const
{
  return *ITEM;
}


Event_rep* Event::operator->() const
{
  return ITEM;
}


//--- Private methods -------------------------------------------------------

Event::Event(const LocalTrace* trace, bool isEnd)
  : m_trace(trace),
    m_index(isEnd ? trace->size() : 0)
{
}


Event::Event(const LocalTrace* trace, uint32_t index)
  : m_trace(trace),
    m_index(index)
{
}


void Event::pack(Buffer& buffer) const
{
  // Store type & location
  buffer.put_uint32(static_cast<uint32_t>(ITEM->getType()));
  buffer.put_uint64(m_trace->get_location().getId());
  buffer.put_id(get_cnode()->getId());

  // Pack event data
  ITEM->pack(buffer);
}


ostream& Event::output(ostream& stream) const
{
  // Print data
  int indent = getIndent(stream);
  setIndent(stream, indent + 11);
  stream << "EVENT {" << iendl(indent)
         << "  type   = " << event_typestr(ITEM->getType()) << iendl(indent)
         << "  loc    = " << m_trace->get_location() << iendl(indent);
  ITEM->output(stream);
  stream << "}";

  return setIndent(stream, indent);
}


//---------------------------------------------------------------------------
//
//  Related functions
//
//---------------------------------------------------------------------------

ostream& pearl::operator<<(ostream& stream, const Event& event)
{
  return event.output(stream);
}


string pearl::event_typestr(event_t type)
{
  static const char* const type_strings[] = {
                             "ENTER",
                             "ENTER_CS",
                             "LEAVE",
                             "MPI_COLLECTIVE_BEGIN",
                             "MPI_COLLECTIVE_END",
                             "MPI_SEND",
                             "MPI_SEND_REQUEST",
                             "MPI_SEND_COMPLETE",
                             "MPI_RECV",
                             "MPI_RECV_REQUEST",
                             "MPI_RECV_COMPLETE",
                             "MPI_REQUEST_TESTED",
                             "MPI_CANCELLED",
                             "OMP_FORK",
                             "OMP_JOIN",
                             "OMP_ACQUIRE_LOCK",
                             "OMP_RELEASE_LOCK",
                             "OMP_TASK_CREATE",
                             "OMP_TASK_COMPLETE",
                             "OMP_TASK_SWITCH",
                             "RMA_PUT_START",
                             "RMA_PUT_END",
                             "RMA_GET_START",
                             "RMA_GET_END",
                             "MPI_RMA_PUT_START",
                             "MPI_RMA_PUT_END",
                             "MPI_RMA_GET_START",
                             "MPI_RMA_GET_END",
                             "MPI_RMA_GATS",
                             "MPI_RMA_COLLECTIVE_BEGIN",
                             "MPI_RMA_COLLECTIVE_END",
                             "MPI_RMA_LOCK",
                             "MPI_RMA_UNLOCK"
                             "GROUP_ALL",
                             "GROUP_ENTER",
                             "GROUP_SEND",
                             "GROUP_RECV",
                             "GROUP_BEGIN",
                             "GROUP_END"
                           };

  return type_strings[type];
}
