/****************************************************************************
**  SCALASCA    http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2013                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  Copyright (c) 2009-2013                                                **
**  German Research School for Simulation Sciences GmbH,                   **
**  Laboratory for Parallel Programming                                    **
**                                                                         **
**  This software may be modified and distributed under the terms of       **
**  a BSD-style license.  See the COPYING file in the package base         **
**  directory for details.                                                 **
****************************************************************************/


#include <config.h>
#include "OmpEventFactory.h"

#include <pearl/Error.h>
#include <pearl/GlobalDefs.h>

#include <pearl/OmpAcquireLock_rep.h>
#include <pearl/OmpFork_rep.h>
#include <pearl/OmpJoin_rep.h>
#include <pearl/OmpReleaseLock_rep.h>
#include <pearl/OmpTaskCreate_rep.h>
#include <pearl/OmpTaskComplete_rep.h>
#include <pearl/OmpTaskSwitch_rep.h>

using namespace pearl;


//---------------------------------------------------------------------------
//
//  class OmpEventFactory
//
//---------------------------------------------------------------------------

//--- OpenMP specific factory methods ----------------------------------------

Event_rep* OmpEventFactory::createOmpFork(const GlobalDefs& defs,
                                          timestamp_t       time,
                                          uint32_t          teamSize) const 
{
  return new OmpFork_rep(time, teamSize);
}


Event_rep* OmpEventFactory::createOmpJoin(const GlobalDefs& defs,
                                          timestamp_t       time) const 
{
  return new OmpJoin_rep(time);
}


Event_rep* OmpEventFactory::createOmpAcquireLock(const GlobalDefs& defs,
                                                 timestamp_t       time,
                                                 uint32_t          lockId,
                                                 uint32_t          order) const
{
  return new OmpAcquireLock_rep(time, lockId, order);
}


Event_rep* OmpEventFactory::createOmpReleaseLock(const GlobalDefs& defs,
                                                 timestamp_t       time,
                                                 uint32_t          lockId,
                                                 uint32_t          order) const 
{
  return new OmpReleaseLock_rep(time, lockId, order);
}


Event_rep* OmpEventFactory::createOmpTaskCreate(const GlobalDefs& defs,
                                                timestamp_t       time,
                                                uint64_t          taskId) const
{
  return new OmpTaskCreate_rep(time, taskId);
}


Event_rep* OmpEventFactory::createOmpTaskComplete(const GlobalDefs& defs,
                                                  timestamp_t       time,
                                                  uint64_t          taskId) const
{
  return new OmpTaskComplete_rep(time, taskId);
}


Event_rep* OmpEventFactory::createOmpTaskSwitch(const GlobalDefs& defs,
                                                timestamp_t       time,
                                                uint64_t          taskId) const
{
  return new OmpTaskSwitch_rep(time, taskId);
}


//--- Buffer-based factory methods ------------------------------------------

Event_rep* OmpEventFactory::createEvent(event_t           type,
                                        const GlobalDefs& defs,
                                        Buffer&           buffer) const
{
  Event_rep* result = NULL;

  switch (type) {
    case OMP_FORK:
      result = new OmpFork_rep(defs, buffer);
      break;

    case OMP_JOIN:
      result = new OmpJoin_rep(defs, buffer);
      break;

    case OMP_ACQUIRE_LOCK:
      result = new OmpAcquireLock_rep(defs, buffer);
      break;

    case OMP_RELEASE_LOCK:
      result = new OmpReleaseLock_rep(defs, buffer);
      break;

    case OMP_TASK_CREATE:
      result = new OmpTaskCreate_rep(defs, buffer);
      break;

    case OMP_TASK_COMPLETE:
      result = new OmpTaskComplete_rep(defs, buffer);
      break;

    case OMP_TASK_SWITCH:
      result = new OmpTaskSwitch_rep(defs, buffer);
      break;

    default:
      result = EventFactory::createEvent(type, defs, buffer);
      break;
  }

  return result;
}
