//! Serde `Deserializer` module

use crate::{
    errors::Error,
    escape::unescape,
    events::{
        attributes::{Attribute, Attributes},
        BytesStart, BytesText, Event,
    },
    reader::Decoder,
    Reader,
};
use serde::de::value::BorrowedBytesDeserializer;
use serde::de::{self, DeserializeOwned, DeserializeSeed, SeqAccess, MapAccess, Visitor};
use serde::{self, forward_to_deserialize_any, Deserialize};
use std::borrow::Cow;
use std::io::BufRead;

/// Deserialization error
#[derive(Debug)]
pub enum DeError {
    /// Custom error
    Custom(String),
    /// Cannot parse to integer
    Int(std::num::ParseIntError),
    /// Cannot parse to float
    Float(std::num::ParseFloatError),
    /// Xml parsing error
    Xml(Error),
    /// Unexpected end of attributes
    EndOfAttributes,
    /// Unexpected `Event::Start`
    StartEvent(String),
    /// Cannot deserialize owned event
    OwnedEvent,
    /// Unexpected of of file
    Eof,
    /// Cannot peek event
    Peek,
    /// Unexpected Start event name
    NameMismatch { 
        /// Expected name
        expected: &'static str, 
        /// Event name found
        found: String 
    },
    /// Expecting Text event
    Text,
    /// Expecting Start event
    Start,
    /// Expecting End event
    End,
}

mod error {
    use super::DeError;
    use crate::errors::Error;
    use std::fmt;

    impl fmt::Display for DeError {
        fn fmt(&self, f: &mut fmt::Formatter) -> Result<(), fmt::Error> {
            match self {
                DeError::Custom(s) => write!(f, "{}", s),
                DeError::Xml(e) => write!(f, "{}", e),
                DeError::Int(e) => write!(f, "{}", e),
                DeError::Float(e) => write!(f, "{}", e),
                DeError::EndOfAttributes => write!(f, "Unexpected end of attributes"),
                DeError::StartEvent(n) => write!(f, "Unexpected `Event::Start`: {}", n),
                DeError::OwnedEvent => write!(f, "Cannot deserialize owned event"),
                DeError::Eof => write!(f, "Unexpected `Event::Eof`"),
                DeError::Peek => write!(f, "Cannot peek event"),
                DeError::NameMismatch { expected, found } => {
                    write!(f, "Start event name mismatch:\nExpecting: {}\nFound: {}", expected, found)
                }
                DeError::Text => write!(f, "Expecting Text event"),
                DeError::Start => write!(f, "Expecting Start event"),
                DeError::End => write!(f, "Expecting End event"),
            }
        }
    }

    impl ::std::error::Error for DeError {
        fn description(&self) -> &str {
            "xml deserialize error"
        }
        fn cause(&self) -> Option<&::std::error::Error> {
            None
        }
    }

    impl serde::de::Error for DeError {
        fn custom<T: fmt::Display>(msg: T) -> Self {
            DeError::Custom(msg.to_string())
        }
    }

    impl From<Error> for DeError {
        fn from(e: Error) -> Self {
            DeError::Xml(e)
        }
    }

    impl From<std::num::ParseIntError> for DeError {
        fn from(e: std::num::ParseIntError) -> Self {
            DeError::Int(e)
        }
    }

    impl From<std::num::ParseFloatError> for DeError {
        fn from(e: std::num::ParseFloatError) -> Self {
            DeError::Float(e)
        }
    }
}

/// A deserializer for a xml escaped and encoded value
///
/// # Note
///
/// Escaping the value is actually not always necessary, for instance
/// when converting to float, we don't expect any escapable character
/// anyway
struct EscapedDeserializer<'de> {
    decoder: Decoder,
    escaped_value: &'de [u8],
}

impl<'de> EscapedDeserializer<'de> {
    fn unescaped(&self) -> Result<Cow<'de, [u8]>, DeError> {
        unescape(self.escaped_value).map_err(|e| DeError::Xml(Error::EscapeError(e)))
    }
}

macro_rules! deserialize_num {
    ($method:ident, $visit:ident) => {
        fn $method<V>(self, visitor: V) -> Result<V::Value, Self::Error>
        where
            V: Visitor<'de>,
        {
            let v = self.decoder.decode(self.escaped_value)?;
            visitor.$visit(v.parse()?)
        }
    }
}

impl<'a, 'de> serde::Deserializer<'de> for &'a mut EscapedDeserializer<'de> {
    type Error = DeError;

    fn deserialize_any<V>(self, visitor: V) -> Result<V::Value, Self::Error>
    where
        V: Visitor<'de>,
    {
        self.deserialize_str(visitor)
    }

    fn deserialize_str<V>(self, visitor: V) -> Result<V::Value, Self::Error>
    where
        V: Visitor<'de>,
    {
        let v = self.unescaped()?;
        visitor.visit_str(self.decoder.decode(&v)?)
    }

    fn deserialize_bytes<V>(self, visitor: V) -> Result<V::Value, Self::Error>
    where
        V: Visitor<'de>,
    {
        let v = self.unescaped()?;
        visitor.visit_bytes(&v)
    }

    fn deserialize_byte_buf<V>(self, visitor: V) -> Result<V::Value, Self::Error>
    where
        V: Visitor<'de>,
    {
        self.deserialize_bytes(visitor)
    }

    fn deserialize_string<V>(self, visitor: V) -> Result<V::Value, Self::Error>
    where
        V: Visitor<'de>,
    {
        self.deserialize_str(visitor)
    }

    fn deserialize_bool<V>(self, visitor: V) -> Result<V::Value, Self::Error>
    where
        V: Visitor<'de>,
    {
        match &*self.decoder.decode(&self.escaped_value)? {
            "TRUE" | "true" | "True" | "1" => visitor.visit_bool(true),
            _ => visitor.visit_bool(false),
        }
    }

    fn deserialize_char<V>(self, visitor: V) -> Result<V::Value, Self::Error>
    where
        V: Visitor<'de>,
    {
        let s = self.decoder.decode(&self.escaped_value)?;
        visitor.visit_char(s.chars().next().expect("s not empty"))
    }

    fn deserialize_unit<V>(self, visitor: V) -> Result<V::Value, Self::Error>
    where
        V: Visitor<'de>,
    {
        if self.escaped_value.is_empty() {
            visitor.visit_unit()
        } else {
            Err(DeError::Custom(
                "Expecting unit, got non empty attribute".into(),
            ))
        }
    }

    fn deserialize_option<V>(self, visitor: V) -> Result<V::Value, Self::Error>
    where
        V: Visitor<'de>,
    {
        if self.escaped_value.is_empty() {
            visitor.visit_none()
        } else {
            visitor.visit_some(self)
        }
    }

    fn deserialize_newtype_struct<V>(
        self,
        _name: &'static str,
        visitor: V,
    ) -> Result<V::Value, Self::Error>
    where
        V: Visitor<'de>,
    {
        visitor.visit_newtype_struct(self)
    }

    deserialize_num!(deserialize_i64, visit_i64);
    deserialize_num!(deserialize_i32, visit_i32);
    deserialize_num!(deserialize_i16, visit_i16);
    deserialize_num!(deserialize_i8, visit_i8);
    deserialize_num!(deserialize_u64, visit_u64);
    deserialize_num!(deserialize_u32, visit_u32);
    deserialize_num!(deserialize_u16, visit_u16);
    deserialize_num!(deserialize_u8, visit_u8);
    deserialize_num!(deserialize_f64, visit_f64);
    deserialize_num!(deserialize_f32, visit_f32);

    forward_to_deserialize_any! {
        unit_struct seq tuple tuple_struct map struct enum identifier ignored_any
    }
}

/// A deserializer for `Attributes`
struct AttributesDeserializer<'de> {
    decoder: Decoder,
    attributes: Attributes<'de>,
    de: Option<EscapedDeserializer<'de>>,
}

impl<'a, 'de> MapAccess<'de> for &'a mut AttributesDeserializer<'de> {
    type Error = DeError;

    fn next_key_seed<K: DeserializeSeed<'de>>(
        &mut self,
        seed: K,
    ) -> Result<Option<K::Value>, Self::Error> {
        self.attributes
            .next()
            .transpose()?
            .map(|a| {
                let de = BorrowedBytesDeserializer::<Self::Error>::new(a.key);
                let key = seed.deserialize(de)?;
                let escaped_value = match a.value {
                    Cow::Borrowed(v) => v,
                    _ => unreachable!("we deserialize borrowed data"),
                };
                self.de = Some(EscapedDeserializer {
                    decoder: self.decoder,
                    escaped_value,
                });
                Ok(key)
            })
            .transpose()
    }

    fn next_value_seed<K: DeserializeSeed<'de>>(
        &mut self,
        seed: K,
    ) -> Result<K::Value, Self::Error> {
        let mut de = self.de.take().ok_or(DeError::EndOfAttributes)?;
        seed.deserialize(&mut de)
    }
}

/// A `Reader` deserializer
pub struct Deserializer<'de, R: BufRead> {
    /// a buffer containing the deserialized data
    buf: &'de mut Vec<u8>,
    /// a buffer which can be deleted safely
    tmp_buf: Vec<u8>,
    /// the reader
    reader: Reader<R>,
    peek: Option<Event<'static>>, // FIXME: no need for static
}

impl<'de, 'a> Deserializer<'de, &'a [u8]> {
    /// Create a deserializer from a `&str`
    pub fn from_str(s: &'a str, buf: &'de mut Vec<u8>) -> Self {
        Self::from_reader(s.as_bytes(), buf)
    }
}

impl<'de, R: BufRead> Deserializer<'de, R> {
    /// Create a deserializer from a `reader`
    pub fn from_reader(reader: R, buf: &'de mut Vec<u8>) -> Self {
        let mut reader = Reader::from_reader(reader);
        reader.expand_empty_elements(true).trim_text(true);
        Deserializer {
            reader,
            buf,
            tmp_buf: Vec::new(),
            peek: None,
        }
    }

    fn next<'b>(&mut self, buf: &'b mut Vec<u8>) -> Result<Event<'b>, DeError> {
        let len = buf.len();
        loop {
            let event = self.reader.read_event(buf)?; 
            match event {
                Event::Start(_) | Event::End(_) | Event::Text(_) => return Ok(event),
                Event::Eof => return Err(DeError::Peek),
                _ => buf.truncate(len),
            }
        }
    }

    fn peek(&mut self) -> Result<(), DeError> {
        if self.peek.is_some() {
            return Ok(());
        }
        let event = loop {
            self.tmp_buf.clear();
            let event = self.reader.read_event(&mut self.tmp_buf)?; 
            match event {
                Event::Start(_) | Event::End(_) | Event::Text(_)  => break event,
                Event::Eof => return Err(DeError::Peek),
                _ => (),
            }
        };
        self.peek = Some(event.into_owned());
        Ok(())
    }

    /// Read next event and return a EscapedDeserializer whose value is either Text or Empty
    ///
    /// As for attribute values, text data is not unescaped or decode. It is up to the actual
    /// deserializer to decide what is the best strategy
    fn text_deserializer(&mut self) -> Result<EscapedDeserializer<'de>, DeError> {
        let len = self.buf.len();
        let txt = loop {
            match self.reader.read_event(&mut self.tmp_buf)? {
                Event::Text(txt) => break txt,
                Event::End(_) => break BytesText::from_escaped("".as_bytes()),
                Event::Eof => return Err(DeError::Peek),
                Event::Start(_) => return Err(DeError::Text),
                _ => self.buf.truncate(len),
            }
        };

        let txt = unsafe {
            // Safety note:
            //
            // We need to extend the lifetime of `buf` because it needs to be alive for 'de.
            // In practice we only append data to buf or truncate it so that previous data is never
            // deleted.
            std::slice::from_raw_parts(txt.as_ptr(), txt.len())
        };

        Ok(EscapedDeserializer {
            decoder: self.reader.decoder(),
            escaped_value: txt,
        })
    }

}

macro_rules! text_deserialize {
    ($($method:ident),*) => {
        $(
        fn $method<V: Visitor<'de>>(self, visitor: V) -> Result<V::Value, Self::Error> {
            let mut de = self.text_deserializer()?;
            de.$method(visitor)
        }
        )*
    }
}

impl<'a, 'de, R: BufRead> serde::Deserializer<'de> for &'a mut Deserializer<'de, R> {
    type Error = DeError;

    fn deserialize_any<V>(self, visitor: V) -> Result<V::Value, Self::Error>
    where
        V: Visitor<'de>,
    {
        unimplemented!()
    }

    fn deserialize_struct<V: Visitor<'de>>(
        self,
        name: &'static str,
        _fields: &'static [&'static str],
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let event = if let Some(Event::Start(event)) = self.peek.take() {
            event
        } else {
            return Err(DeError::Start);
        };

        if event.local_name() != name.as_bytes() {
            return Err(DeError::NameMismatch { expected: name, found: std::str::from_utf8(event.local_name()).unwrap_or("").into() });
        }

        let name = event.name().to_owned();
        let attributes = event.attributes();
        let attributes = unsafe {
            // Safety note:
            //
            // We need to extend the lifetime of `buf` because it needs to be alive for 'de.
            // In practice we only append data to buf or truncate it so that previous data is never
            // deleted.
            attributes.into_extended_lifetime()
        };
        let decoder = self.reader.decoder();
        let mut de = AttributesDeserializer {
            decoder,
            attributes,
            de: None,
        };
        let res = visitor.visit_map(&mut de)?;
        self.reader.read_to_end(&name, &mut self.tmp_buf)?;
        Ok(res)
    }

    fn deserialize_seq<V: Visitor<'de>>(mut self, visitor: V) -> Result<V::Value, Self::Error> {
        let name = match self.peek.take() {
            Some(Event::Start(event)) => event.name().to_vec(),
            _ => loop {
                self.tmp_buf.clear();
                match self.reader.read_event(&mut self.tmp_buf)? {
                    Event::Start(s) => break s.name().to_vec(),
                    Event::Eof => return Err(DeError::Eof),
                    _ => (),
                }
            },
        };

        let seq = PeekReader {
            de: &mut self,
            name,
        };
        visitor.visit_seq(seq)
    }

    //fn deserialize_map<V: Visitor<'de>>(mut self, visitor: V) -> Result<V::Value, Self::Error> {
    //    let name = match self.peek.take() {
    //        Some(event) => event.name().to_vec(),
    //        None => loop {
    //            self.tmp_buf.clear();
    //            match self.reader.read_event(&mut self.tmp_buf)? {
    //                Event::Start(s) => break s.name().to_vec(),
    //                Event::Eof => return Err(DeError::Eof),
    //                _ => (),
    //            }
    //        },
    //    };

    //    let map = PeekReader {
    //        de: &mut self,
    //        name,
    //    };
    //    visitor.visit_map(map)
    //}

    text_deserialize!(
        deserialize_bool, deserialize_u8, deserialize_i8, deserialize_i16, deserialize_i32, deserialize_i64, 
        deserialize_u16, deserialize_u32, deserialize_u64, deserialize_f32, deserialize_f64, deserialize_char, 
        deserialize_str, deserialize_string, deserialize_bytes, deserialize_byte_buf, deserialize_option
    );

    forward_to_deserialize_any! {
        unit unit_struct tuple tuple_struct enum identifier ignored_any newtype_struct map
    }
}

struct PeekReader<'a, 'de, R: BufRead> {
    de: &'a mut Deserializer<'de, R>,
    name: Vec<u8>,
}

impl<'a, 'de, R: BufRead> SeqAccess<'de> for PeekReader<'a, 'de, R> {
    type Error = DeError;

    fn next_element_seed<D>(&mut self, seed: D) -> Result<Option<D::Value>, Self::Error>
    where
        D: DeserializeSeed<'de>,
    {
        let value = loop {
            self.de.peek()?;
            match self.de.peek.as_ref() {
                Some(Event::End(e)) if e.name() == &*self.name => break Ok(None),
                Some(Event::Text(_)) => self.de.peek.take(),
                _ => break seed.deserialize(&mut *self.de).map(Some),
            };
        };
        value
    }
}

//impl<'a, 'de, R: BufRead> MapAccess<'de> for PeekReader<'a, 'de, R> {
//    type Error = DeError;
//
//    fn next_key_seed<K: DeserializeSeed<'de>>(
//        &mut self,
//        seed: K,
//    ) -> Result<Option<K::Value>, Self::Error> {
//        if self.de.peek_or_end(&self.name)? {
//            Ok(None)
//        } else {
//            let name = self.de.peek.as_ref().ok_or(DeError::Peek)?.local_name();
//            let de = BorrowedBytesDeserializer::<Self::Error>::new(name);
//            seed.deserialize(de).map(Some)
//        }
//    }
//
//    fn next_value_seed<K: DeserializeSeed<'de>>(
//        &mut self,
//        seed: K,
//    ) -> Result<K::Value, Self::Error> {
//        seed.deserialize(&mut *self.de)
//    }
//}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::events::Event;
    use serde::Deserialize;

    #[test]
    fn deserialize_struct() -> Result<(), DeError> {
        let xml = "<A x='1' y='test' z='true' />";
        #[derive(Debug, Deserialize, PartialEq)]
        struct A {
            x: u8,
            y: String,
            z: bool,
        }
        let mut buf = Vec::new();
        let mut de = Deserializer::from_str(xml, &mut buf);
        let a: A = Deserialize::deserialize(&mut de)?;
        assert_eq!(
            a,
            A {
                x: 1,
                y: "test".into(),
                z: true
            }
        );
        Ok(())
    }

    #[test]
    fn deserialize_vec_struct() -> Result<(), DeError> {
        let xml = "
            <Container>
                <A x='1' y='test' z='true' />
                <A x='2' y='test' z='true' />
            </Container>
        ";
        #[derive(Debug, Deserialize, PartialEq)]
        struct A {
            x: u8,
            y: String,
            z: bool,
        }
        let mut buf = Vec::new();
        let mut de = Deserializer::from_str(xml, &mut buf);
        let a: Vec<A> = Deserialize::deserialize(&mut de)?;
        assert_eq!(
            a[0],
            A {
                x: 1,
                y: "test".into(),
                z: true
            }
        );
        Ok(())
    }

    #[test]
    fn deserialize_struct_with_vec() -> Result<(), DeError> {
        let xml = "
            <Container>
                <A x='1' y='test' z='true' />
                <A x='2' y='test' z='true' />
            </Container>
        ";
        #[derive(Debug, Deserialize, PartialEq)]
        struct A {
            x: u8,
            y: String,
            z: bool,
        }
        #[derive(Debug, Deserialize, PartialEq)]
        struct Container {
            #[serde(rename = "Container")]
            container: Vec<A>,
        }

        let mut buf = Vec::new();
        let mut de = Deserializer::from_str(xml, &mut buf);
        let a: Container = Deserialize::deserialize(&mut de)?;
        assert_eq!(
            a.container[0],
            A {
                x: 1,
                y: "test".into(),
                z: true
            }
        );
        Ok(())
    }
}
