#ifndef _RHEOLEF_SOLVER_OPTION_H
#define _RHEOLEF_SOLVER_OPTION_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "rheolef/diststream.h"

namespace rheolef {

/*Class:solver
NAME:   @code{solver_option} - options for direct or interative solvers
@clindex solver
DESCRIPTION:
  @noindent
  This class implements a set of options for direct or iterative solvers,
  as implemented by the @ref{solver class}. An instance of this class
  can be supplied to the @code{solver} constructor.
OPTIONS:
@table @code
  @item iterative
  	Indicate if we want to use an iterative solver or a a direct one.
	Default is direct for 1D or 2D problems and iterative for 3D ones.
	Direct solver is @code{mumps} when available, otherwise @code{umfpack}
	or builtin solver when none of these solvers are available at Rheolef configure time.
@pindex configure
@cindex configure
	Iterative solver is @code{cg} for symmetric positive definite systems and @code{gmres} otherwise.
@end table
OPTIONS FOR ITERATIVE SOLVERS:
@table @code
  @item	tol
	Tolerance for the stopping criterion.
	Default is the machine epsilon for the default @code{Float} type.
	The default @code{Float} type is @code{double} defined
	at Rheolef configure time and is @code{double} by default,
	when no special configure option is used.
  @item max_iter
	Maximum number of iteration when using iterative method.
  @item absolute_stopping
	Absolute or relative stopping criterion.	
	With the absolute criterion, the algorithm stops when
	@code{norm(A*x-b) < tol}, otherwise it stops when
	@code{norm(A*x-b) < tol*norm(b)}.
	Default is to use an absolute stopping criterion.
  @item	residue	
	On return, gives the obtained residue, optionally divided
	by the initial one when using a relatiive stopping.
	It is less or equal to @code{tol} when the iterative algorithm stops with succes.
  @item	n_iter	
	On return, gives the number of iterations performed.
	It is always less or equal to @code{max_iter} when the iterative algorithm stops with succes.
  @item p_err
	A pointer to the @code{odiststeam} standard error where
	residues are printed during iterations.
	When this pointer is zero, no errors are printed.
	Default is to print to @code{derr}.
  @item label
	When printing errors, each line is prefixed by @code{[@var{label}]}.
	When the label is empty, each iterative algorithm uses
	its default label, e.g. @code{"cg"} for the conjugate gradient @code{cg}.
	By default the label is empty and this option is used to superset
	the algorithm default.
  @item krylov_dimension
	The dimension of the Krylov space used by the @code{gmres} algorithm.
	Default is @code{krylov_dimension=6}.
@end table
OPTIONS FOR DIRECT SOLVERS:
@table @code
  @item n_refinement
	Number of iterative refinement, when using direct method (@code{umfpack} only support it).
	This option is only active when using iterative solvers.
  @item compute_determinant
	Compute also the determinant of the matrix.
	This option is only active when using direct solvers.
	Requires @code{mumps} or @code{umfpack}.
  @item prefered_library
	When both @code{mumps} and @code{umfpack} are available, then @code{mumps} is the default.
	This option allows one to force to choose @code{umfpack}, by setting the string to @code{"umfpack"}.
	This option is only active when using direct solvers.
@end table
OPTIONS FOR DEVELOPPERS:
@table @code
  @item verbose_level
	Can be set to 0, 1, 2 or 3. The default is 0.
  @item level_of_fill
        Built an incomplete factorization with the prescribed
	level of fill [1:5].
  @item do_check
	Performs extra checks for debug.
  @item force_seq
	In distributed mode, restrict the linear system resolution
	to diagonal blocs per process.
	This option is only active when using the @code{mumps} direct solver.
  @item ooc
	Out-of-core limit (Mo/percent depending on compilation options).
	In development.
  @item amalgamation
	Level of amalgamation [10:70] for Kass.
	In development.
@end table
AUTHOR: Pierre.Saramito@imag.fr
DATE:   4 march 2011
end:
*/
//<verbatim:
class solver_option {
public:
  typedef std::size_t size_type;
  static const long int  decide = -1;
  mutable long int   iterative; 
  Float 	     tol;
  size_type	     max_iter;
  bool		     absolute_stopping;
  mutable Float      residue;
  mutable size_type  n_iter;
  odiststream*	     p_err;
  mutable std::string label;   
  size_type	     krylov_dimension;
  size_type	     n_refinement;
  bool               compute_determinant; 
  std::string        prefered_library;   
  size_type          verbose_level;     
  bool               do_check;
  bool               force_seq;	
  size_type          level_of_fill;
  size_type          amalgamation; 
  size_type          ooc;         

// allocator and default values:

  solver_option()
   : iterative     (decide),
#if defined(_RHEOLEF_HAVE_FLOAT128)
     tol           (1e6*std::numeric_limits<Float>::epsilon()),
#else
     tol           (std::numeric_limits<Float>::epsilon()),
#endif
     max_iter      (100000),
     absolute_stopping (true),
     residue       (0),
     n_iter        (0),
     p_err         (&derr),
     label         (),
     krylov_dimension (6),
     n_refinement  (2),
     compute_determinant(false),
     prefered_library(),
     verbose_level (0),
     do_check      (false),
     force_seq     (false),
     level_of_fill (1),
     amalgamation  (10),
     ooc           (20000)
  {
  }
  solver_option (const solver_option&);
  solver_option& operator= (const solver_option&);
};
//>verbatim:

// for backward compatibility:
using solver_option_type = solver_option;

// ---------------------------------------------------------------
// inlined
// ---------------------------------------------------------------
inline
solver_option::solver_option (const solver_option& x)
   : iterative     (x.iterative),
     tol           (x.tol),
     max_iter      (x.max_iter),
     absolute_stopping (x.absolute_stopping),
     residue       (x.residue),
     n_iter        (x.n_iter),
     p_err         (x.p_err),
     label         (x.label),
     krylov_dimension (x.krylov_dimension),
     n_refinement  (x.n_refinement),
     compute_determinant(x.compute_determinant),
     verbose_level (x.verbose_level),
     do_check      (x.do_check),
     force_seq     (x.force_seq),
     level_of_fill (x.level_of_fill),
     amalgamation  (x.amalgamation),
     ooc           (x.ooc)
{
}
inline
solver_option& 
solver_option::operator= (const solver_option& x)
{
     iterative     = x.iterative;
     tol           = x.tol;
     max_iter      = x.max_iter;
     absolute_stopping = x.absolute_stopping;
     residue       = x.residue;
     n_iter        = x.n_iter;
     p_err         = x.p_err;
     label         = x.label;
     krylov_dimension = x.krylov_dimension;
     n_refinement  = x.n_refinement;
     compute_determinant = x.compute_determinant;
     verbose_level = x.verbose_level;
     do_check      = x.do_check;
     force_seq     = x.force_seq;
     level_of_fill = x.level_of_fill;
     amalgamation  = x.amalgamation;
     ooc           = x.ooc;
     return *this;
}

} // namespace rheolef
#endif // _RHEOLEF_SOLVER_OPTION_H
