#ifndef _RHEO_DIS_CPU_TIME_H
#define _RHEO_DIS_CPU_TIME_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

#include "rheolef/distributed.h"
namespace rheolef {

/*D:dis_cpu_time
NAME: @code{dis_cpu_time}, @code{dis_wall_time}, @code{seq_cpu_time}, @code{seq_walltime} -- Time in seconds since an arbitrary time in the past.
SYNOPSIS:
  @noindent
  @example
    double dis_cpu_time();
    double dis_wall_time();
    double seq_cpu_time();
    double seq_wall_time();
  @end example

EXAMPLE:
  @noindent
  @example
    double t_start = dis_wall_time();
    //....  stuff to be timed  ...
    double t_end   = dis_wall_time();
    derr << "That took " << t_end - t_start << " seconds" << endl

    double cpu_start = dis_cpu_time();
    //....  stuff to be timed  ...
    double cpu_end   = dis_cpu_time();
    derr << "That needs " << cpu_end - cpu_start << " CPU seconds" << endl
  @end example

DESCRIPTION:       
  @noindent
  @code{dis_wall_time} returns a floating-point number of seconds, 
  representing elapsed wall-clock time since some time in the past.
  The @emph{time in the past} is guaranteed not to change during the life of the process.
  The user is responsible for converting large numbers of seconds to other units if they are preferred.
  This function is portable (it returns seconds, not @emph{ticks}), it allows high resolution,
  and carries no unnecessary baggage.
  In a distributed environment,
  @code{dis_wall_time} clocks are synchronized: different nodes return the same time
  value at the same instant.
  @code{seq_wall_time} is similar but the time returned is local to the node that called them
  and clocks are not synchronized: in a distributed environment,
  different nodes can return different local times,
  at different instant when the call to @code{seq_wall_time} is reached.

  @noindent
  @code{seq_cpu_time} is similar but returns the computing time per user for
  the current process.
  @code{dis_cpu_time} returns the accumulated CPU
  for all processed linked together via the default communicator.

AUTHOR: 
    Pierre Saramito
    Pierre.Saramito@imag.fr
    LJK-IMAG, 38041 Grenoble cedex 9, France
DATE: 
    5 november 2013
End:
*/

double seq_wall_time();
double seq_cpu_time();

# ifndef _RHEOLEF_HAVE_MPI
inline double dis_wall_time() { return seq_wall_time(); }
inline double dis_cpu_time()  { return seq_cpu_time(); }
# else  // _RHEOLEF_HAVE_MPI
inline double dis_wall_time() {
  if (! (mpi::environment::initialized() && !mpi::environment::finalized())) {
    return seq_wall_time();
  }
  double t = 0;
  communicator comm;
  comm.barrier();
  typedef std::size_t size_type;
  size_type io_proc = idiststream::io_proc();
  size_type my_proc = comm.rank();
  if (my_proc == io_proc) {
    t = seq_wall_time();
  }
  mpi::broadcast (comm, t, io_proc);
  return t;
}
inline double dis_cpu_time() {
  double cpu = seq_cpu_time();
  if (! (mpi::environment::initialized() && !mpi::environment::finalized())) {
    return cpu;
  }
  return rheolef::mpi::all_reduce (rheolef::communicator(), cpu, std::plus<double>());
}
# endif // _RHEOLEF_HAVE_MPI

} // namespace rheolef
#endif // _RHEO_DIS_CPU_TIME_H
