/* Table_extensions.cpp
	 *
 * Copyright (C) 1997-2019 David Weenink, Paul Boersma 2017
 *
 * This code is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This code is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this work. If not, see <http://www.gnu.org/licenses/>.
 */

/*
  djmw 20020619 GPL header
  djmw 20040113 Added comment header for Peterson& Barney data.
  djmw 20040512 Corrected Peterson& Barney ARPABET-labeling.
  djmw 20041213 Added Table_createFromWeeninkData.
  djmw 20080125 Corrected mislabeling of vowels in the Peterson& Barney dataset according to Watrous
  djmw 20080508 Labeling back to original PB article.
  djmw 20110329 Table_get(Numeric|String)Value is now Table_get(Numeric|String)Value_Assert
  djmw 20131219 Improved Table_scatterPlotWithConfidenceIntervals
*/
/*	speaker type (m|w|c), sex(m|f), id, vowel_number, vowel_label
	F0, F1, F2, F3
*/

#include "Discriminant.h"
#include "Formula.h"
#include "GraphicsP.h"
#include "Graphics_extensions.h"
#include "Index.h"
#include "Matrix_extensions.h"
#include "NUM2.h"
#include <ctype.h>
#include "Strings_extensions.h"
#include "SSCP.h"
#include "Table_extensions.h"

static bool Table_selectedColumnPartIsNumeric (Table me, integer column, constINTVEC selectedRows) {
	if (column < 1 || column > my numberOfColumns)
		return false;
	for (integer irow = 1; irow <= selectedRows.size; irow ++)
		if (! Table_isCellNumeric_ErrorFalse (me, selectedRows [irow], column))
			return false;
	return true;
}

// column and selectedRows must be valid
static void Table_columnExtremaFromSelectedRows (Table me, integer column, constINTVEC selectedRows, double *out_min, double *out_max) {
	MelderExtremaWithInit extrema;
	for (integer irow = 1; irow <= selectedRows.size; irow ++) {
		const double val = Table_getNumericValue_Assert (me, selectedRows [irow], column);
		extrema.update (val);
	}
	if (out_min)
		*out_min = extrema.min;
	if (out_max)
		*out_max = extrema.max;
}

/*
	The Peterson & Barney data were once (1991) obtained by me (djmw) as a compressed tar-file
	by anonymous ftp from ftp://linc.cis.upenn.edu/pub,
	However, this site appears no longer to be an anonymous ftp site.
	The compressed tar file contained two files: a header file 'pb.header'
	and a data file 'verified_pb.data'.
	The header file reads:

	"This file contains the vowel formant data reported by Gordon E.
	Peterson and Harold L. Barney in their classic paper, "Control methods
	used in a study of the vowels", JASA 24(2) 175-184, 1952. This data
	was supplied in printed form by Ignatius Mattingly, April, 1990.

	The data consists of the formant values F0, F1, F2, and F3 for each of
	two repetitions of ten vowels by 76 speakers (1520 utterances). The
	vowels were pronounced in isolated words consisting of hVd. Of the
	speakers, 33 were men, 28 were women and 15 were children. Dr.
	Mattingly reported that he obtained from G. Peterson the information
	that children speakers 62, 63, 65, 66, 67, 68, 73 and 76 were female.

	The data are organized by speaker type, speaker, and vowel into 1520
	lines of 8 fields. The fields are: Speaker Type, Speaker Number,
	Phoneme Number, Phoneme Label, F0, F1, F2 and F3. The speaker types
	are type 1 (men), type 2 (women) and type 3 (children)."
*/

autoTable Table_create_petersonBarney1952 () {
	const integer nrows = 1520, ncols = 9;
	const conststring32 columnLabels [9] = {U"Type", U"Sex", U"Speaker", U"Vowel", U"IPA", U"F0", U"F1", U"F2", U"F3"};
	const conststring32 type [3] = {U"m", U"w", U"c"};
	/*
		Wrong order before 20080125
		char32 *vowel [10] = {U"iy", U"ih", U"eh", U"ae", U"aa", U"ao", U"uh", U"uw", U"ah", U"er"};
		char32 *ipa [10] = {U"i", U"\\ic", U"\\ep", U"\\ae", U"\\as", U"\\ct", U"\\hs", U"u",
			U"\\vt", U"\\er\\hr"};
	*/
	const conststring32 vowel [10] = {U"iy", U"ih", U"eh", U"ae", U"ah", U"aa", U"ao", U"uh", U"uw", U"er"};
	/*
		Watrous IPA symbols
		char32 *ipa [10] = {U"i", U"\\ic", U"e", U"\\ae", U"\\vt", U"\\as", U"o", U"\\hs", U"u", U"\\er"};
	*/
	// P & B IPA symbols
	const conststring32 ipa [10] = {U"i", U"\\ic", U"\\ef", U"\\ae", U"\\vt", U"\\as", U"\\ct", U"\\hs", U"u", U"\\er\\hr"};
	const conststring32 sex [2] = {U"m", U"f"};
	const struct pbdatum {
		short star; // was there a * in front of the vowel-type?
		short f [4];	// f0, f1, f2, f3
	} pbdata [] = {
		{0, {160, 240, 2280, 2850}},
		{0, {186, 280, 2400, 2790}},
		{0, {203, 390, 2030, 2640}},
		{0, {192, 310, 1980, 2550}},
		{0, {161, 490, 1870, 2420}},
		{1, {155, 570, 1700, 2600}},
		{1, {140, 560, 1820, 2660}},
		{0, {180, 630, 1700, 2550}},
		{0, {144, 590, 1250, 2620}},
		{0, {148, 620, 1300, 2530}},
		{0, {148, 740, 1070, 2490}},
		{0, {170, 800, 1060, 2640}},
		{1, {161, 600,  970, 2280}},
		{1, {158, 660,  980, 2220}},
		{0, {163, 440, 1120, 2210}},
		{0, {190, 400, 1070, 2280}},
		{0, {160, 240, 1040, 2150}},
		{0, {157, 270,  930, 2280}},
		{0, {177, 370, 1520, 1670}},
		{0, {164, 460, 1330, 1590}},
		{0, {147, 220, 2220, 2910}},
		{0, {148, 210, 2360, 3250}},
		{0, {141, 410, 1890, 2680}},
		{0, {139, 420, 1850, 2500}},
		{0, {136, 500, 1760, 2590}},
		{1, {135, 510, 1710, 2380}},
		{0, {128, 690, 1610, 2560}},
		{0, {131, 700, 1690, 2580}},
		{1, {140, 650, 1080, 2420}},
		{1, {125, 625, 1060, 2490}},
		{0, {140, 650, 1040, 2450}},
		{0, {136, 670, 1100, 2430}},
		{0, {149, 580,  580, 2470}},
		{0, {140, 560,  560, 2410}},
		{0, {145, 450,  940, 1910}},
		{0, {141, 410,  830, 2240}},
		{0, {140, 280,  650, 3300}},
		{0, {137, 260,  660, 3300}},
		{0, {145, 510, 1210, 1570}},
		{0, {145, 510, 1130, 1510}},
		{0, {105, 250, 2180, 2680}},
		{0, {111, 244, 2300, 2780}},
		{0, {100, 400, 1930, 2610}},
		{0, {104, 400, 1990, 2700}},
		{0, {100, 550, 1810, 2500}},
		{0,  {95, 540, 1810, 2480}},
		{0,  {93, 630, 1710, 2400}},
		{0,  {94, 658, 1755, 2305}},
		{1, {100, 600, 1200, 2320}},
		{0, {105, 612, 1160, 2350}},
		{0,  {91, 640, 1080, 2140}},
		{0,  {94, 720, 1090, 2230}},
		{0,  {92, 550,  870, 2300}},
		{0, {120, 540,  840, 2280}},
		{0, {114, 460, 1150, 2290}},
		{0, {114, 456, 1030, 2300}},
		{0, {112, 340,  950, 2240}},
		{1, {112, 326,  900, 2190}},
		{0, {100, 500, 1370, 1780}},
		{0, {106, 530, 1330, 1800}},
		{0, {150, 300, 2240, 3400}},
		{0, {156, 280, 2450, 3200}},
		{1, {156, 450, 1960, 2400}},
		{0, {146, 440, 2050, 2360}},
		{0, {130, 570, 1780, 2410}},
		{0, {150, 555, 1890, 2440}},
		{0, {125, 750, 1610, 2340}},
		{0, {136, 770, 1580, 2350}},
		{0, {132, 660, 1200, 2330}},
		{1, {150, 675, 1140, 2380}},
		{0, {125, 750, 1100, 2550}},
		{0, {138, 800, 1120, 2500}},
		{0, {143, 540,  850, 2320}},
		{0, {150, 555,  890, 2370}},
		{0, {136, 460,  960, 2210}},
		{0, {156, 460, 1000, 2350}},
		{0, {140, 380,  950, 2050}},
		{0, {148, 385,  850, 2330}},
		{0, {150, 590, 1400, 1840}},
		{0, {145, 555, 1430, 1730}},
		{0, {140, 310, 2310, 2820}},
		{0, {131, 260, 2250, 2850}},
		{0, {137, 440, 2060, 2640}},
		{0, {134, 430, 1880, 2450}},
		{0, {140, 580, 1910, 2500}},
		{0, {137, 550, 1770, 2400}},
		{0, {143, 830, 1720, 2180}},
		{0, {135, 750, 1690, 2320}},
		{0, {136, 630, 1300, 1950}},
		{0, {130, 650, 1170, 2000}},
		{0, {131, 760, 1220, 2140}},
		{1, {126, 720, 1260, 2020}},
		{0, {136, 540,  970, 1980}},
		{0, {124, 550,  880, 1950}},
		{0, {133, 470, 1040, 1990}},
		{0, {132, 490,  990, 1920}},
		{0, {141, 380,  950, 2140}},
		{0, {133, 330,  800, 2130}},
		{0, {143, 560, 1510, 1800}},
		{0, {136, 510, 1460, 1700}},
		{0, {125, 312, 2350, 2800}},
		{0, {119, 330, 2430, 2870}},
		{0, {133, 420, 2000, 2660}},
		{0, {125, 313, 2000, 2750}},
		{0, {120, 600, 1860, 2500}},
		{0, {114, 570, 1830, 2570}},
		{0, {119, 676, 1670, 2540}},
		{0, {125, 725, 1687, 2500}},
		{1, {118, 680, 1150, 2560}},
		{0, {125, 726, 1270, 2560}},
		{0, {125, 740, 1100, 2680}},
		{1, {113, 670,  960, 2650}},
		{1, {120, 660, 1030, 2690}},
		{1, {125, 720,  960, 2700}},
		{0, {120, 456, 1080, 2520}},
		{0, {120, 450, 1140, 2600}},
		{0, {125, 313,  838, 2340}},
		{0, {125, 288,  938, 2450}},
		{0, {120, 503, 1305, 1775}},
		{0, {120, 505, 1320, 1750}},
		{0, {186, 320, 2320, 3120}},
		{0, {172, 310, 2280, 3020}},
		{1, {167, 470, 2000, 2660}},
		{0, {170, 410, 2040, 2715}},
		{1, {167, 630, 1900, 2860}},
		{0, {146, 614, 1840, 2770}},
		{0, {143, 740, 1800, 2450}},
		{0, {162, 775, 1810, 2200}},
		{1, {167, 620, 1240, 2410}},
		{0, {160, 640, 1250, 2400}},
		{1, {162, 650,  970, 2580}},
		{0, {163, 650,  980, 2350}},
		{0, {145, 430,  720, 2450}},
		{0, {171, 510,  800, 2500}},
		{1, {170, 460, 1120, 2150}},
		{1, {170, 493, 1120, 2300}},
		{0, {175, 380, 1040, 2260}},
		{1, {200, 400, 1000, 2350}},
		{0, {167, 570, 1300, 1750}},
		{0, {157, 565, 1370, 1710}},
		{1, {105, 230, 2480, 3200}},
		{0, {109, 218, 2380, 3100}},
		{1, {110, 320, 2200, 2680}},
		{0, {103, 206, 2130, 2570}},
		{1, {107, 430, 2100, 2630}},
		{1, {105, 515, 1760, 2470}},
		{1, {107, 514, 2060, 2600}},
		{1, {106, 552, 1820, 2500}},
		{0, {108, 640, 1300, 2300}},
		{0, {104, 624, 1350, 2410}},
		{1, {111, 714, 1170, 2420}},
		{1,  {97, 650, 1150, 2350}},
		{1, {107, 590,  965, 2500}},
		{0, {109, 578,  970, 2460}},
		{0, {111, 467, 1110, 2400}},
		{0, {105, 475, 1220, 2310}},
		{0, {107, 270,  910, 2200}},
		{0, {108, 260,  975, 2320}},
		{0, {107, 460, 1400, 1790}},
		{0, {103, 425, 1410, 1760}},
		{0, {175, 316, 2200, 2800}},
		{0, {175, 280, 2275, 2775}},
		{0, {167, 450, 1820, 2475}},
		{0, {167, 434, 1850, 2425}},
		{0, {157, 582, 1725, 2375}},
		{0, {158, 586, 1800, 2425}},
		{0, {150, 600, 1750, 2375}},
		{1, {145, 582, 1775, 2375}},
		{1, {145, 626, 1125, 2200}},
		{0, {160, 641, 1120, 2225}},
		{0, {144, 708, 1054, 2420}},
		{1, {150, 705, 1050, 2375}},
		{0, {146, 614,  848, 2200}},
		{0, {143, 600,  860, 2175}},
		{0, {167, 500, 1000, 2325}},
		{0, {167, 500, 1000, 2325}},
		{0, {167, 334, 1150, 2200}},
		{0, {183, 312, 1020, 2300}},
		{0, {157, 518, 1305, 1570}},
		{0, {157, 504, 1210, 1510}},
		{0, {129, 260, 2260, 2820}},
		{0, {125, 250, 2200, 2825}},
		{0, {146, 400, 2040, 2500}},
		{0, {144, 389, 2000, 2425}},
		{0, {126, 500, 1870, 2500}},
		{0, {125, 500, 1775, 2450}},
		{0, {110, 660, 1650, 2500}},
		{0, {120, 624, 1700, 2475}},
		{0, {122, 650, 1220, 2550}},
		{0, {120, 672, 1260, 2500}},
		{0, {114, 750, 1080, 2680}},
		{0, {114, 777, 1026, 2625}},
		{0, {115, 580,  800, 2650}},
		{0, {117, 585,  819, 2625}},
		{1, {140, 480,  950, 2500}},
		{0, {127, 461,  993, 2350}},
		{0, {140, 280,  950, 2300}},
		{0, {133, 266,  920, 2300}},
		{0, {128, 500, 1340, 1700}},
		{0, {133, 532, 1275, 1600}},
		{0, {146, 248, 2225, 3100}},
		{0, {140, 238, 2175, 3075}},
		{0, {150, 405, 1925, 2550}},
		{0, {138, 416, 1940, 2600}},
		{1, {147, 588, 1790, 2500}},
		{1, {133, 586, 1725, 2650}},
		{0, {145, 725, 1700, 2425}},
		{0, {127, 710, 1650, 2220}},
		{1, {136, 586, 1078, 2300}},
		{0, {136, 627, 1038, 2360}},
		{1, {145, 725, 1046, 2325}},
		{1, {131, 746, 1018, 2300}},
		{0, {140, 560,  840, 2500}},
		{1, {140, 560,  924, 2350}},
		{0, {150, 495, 1080, 2275}},
		{0, {143, 430, 1030, 2275}},
		{0, {162, 290,  760, 2300}},
		{0, {157, 315,  850, 2025}},
		{0, {150, 511, 1561, 1876}},
		{0, {138, 530, 1450, 1887}},
		{0, {110, 220, 2410, 3000}},
		{0, {125, 240, 2440, 3280}},
		{0, {120, 450, 1880, 2450}},
		{0, {118, 380, 1930, 2420}},
		{0, {115, 560, 1650, 2300}},
		{0, {123, 560, 1720, 2300}},
		{0, {110, 680, 1720, 2330}},
		{0, {133, 630, 1680, 2280}},
		{1, {110, 560, 1430, 2250}},
		{1, {120, 560, 1390, 2240}},
		{1, {108, 800, 1330, 2260}},
		{0, {110, 740, 1240, 2280}},
		{1, {120, 600,  920, 2080}},
		{1, {133, 580,  910, 2000}},
		{0, {130, 400, 1200, 2210}},
		{0, {110, 420, 1230, 2230}},
		{0, {122, 300,  900, 2130}},
		{0, {123, 260, 1010, 2240}},
		{0, {125, 400, 1450, 1650}},
		{0, {128, 360, 1410, 1640}},
		{0, {142, 290, 2290, 2600}},
		{0, {135, 260, 2290, 2700}},
		{0, {132, 390, 1950, 2550}},
		{1, {135, 400, 1900, 2450}},
		{1, {124, 490, 1740, 2500}},
		{1, {125, 500, 1780, 2430}},
		{0, {125, 660, 1630, 2500}},
		{0, {132, 670, 1630, 2380}},
		{0, {140, 600, 1220, 2530}},
		{0, {125, 600, 1210, 2430}},
		{1, {125, 680, 1120, 2630}},
		{0, {128, 670, 1100, 2700}},
		{0, {127, 510,  720, 2450}},
		{0, {120, 480,  710, 2540}},
		{1, {133, 380,  910, 2350}},
		{0, {140, 440, 1030, 2400}},
		{0, {127, 350,  720, 2750}},
		{0, {140, 380,  740, 2880}},
		{0, {128, 430, 1370, 1610}},
		{0, {135, 440, 1360, 1600}},
		{0, {114, 228, 2350, 2860}},
		{0, {118, 220, 2350, 2920}},
		{1, {110, 407, 2070, 2500}},
		{1, {112, 420, 1900, 2450}},
		{0, {106, 445, 2020, 2420}},
		{0, {115, 470, 2020, 2500}},
		{0, {103, 721, 1680, 2400}},
		{0, {109, 750, 1710, 2440}},
		{1, {104, 552, 1122, 2500}},
		{1, {115, 580, 1150, 2600}},
		{0,  {98, 686, 1078, 2570}},
		{1, {103, 700, 1050, 2680}},
		{0, {102, 560,  665, 2620}},
		{0, {106, 550,  650, 2700}},
		{0, {112, 448,  980, 2370}},
		{0, {104, 410,  940, 2370}},
		{0, {116, 232,  696, 2200}},
		{0, {117, 222,  665, 2080}},
		{0, {120, 432, 1300, 1400}},
		{0, {111, 420, 1300, 1570}},
		{0, {121, 230, 2100, 2850}},
		{0, {118, 240, 2000, 2980}},
		{0, {130, 365, 1900, 2340}},
		{0, {119, 300, 2040, 2560}},
		{1, {112, 440, 1980, 2310}},
		{1, {120, 410, 2050, 2500}},
		{1, {133, 620, 1710, 2110}},
		{0, {124, 660, 1800, 2150}},
		{0, {120, 660, 1000, 2380}},
		{0, {110, 660,  960, 2450}},
		{1, {122, 600,  830, 2250}},
		{1, {119, 620,  820, 2400}},
		{0, {117, 500,  620, 2250}},
		{0, {106, 550,  700, 2550}},
		{0, {140, 390,  730, 2180}},
		{0, {130, 360,  740, 2200}},
		{0, {131, 260,  720, 2100}},
		{0, {132, 260,  740, 2040}},
		{0, {125, 450, 1230, 1600}},
		{0, {127, 460, 1300, 1650}},
		{0, {150, 300, 2355, 3250}},
		{0, {150, 300, 2460, 3280}},
		{0, {160, 385, 2242, 2805}},
		{0, {150, 407, 2250, 2780}},
		{0, {140, 504, 2090, 2720}},
		{0, {146, 543, 1980, 2640}},
		{0, {133, 680, 1958, 2542}},
		{0, {141, 708, 1840, 2535}},
		{0, {150, 675, 1320, 2550}},
		{0, {150, 704, 1393, 2550}},
		{0, {137, 825, 1168, 2750}},
		{0, {135, 840, 1210, 2680}},
		{1, {143, 671, 1000, 2670}},
		{1, {147, 690,  968, 2660}},
		{0, {143, 443, 1273, 2430}},
		{0, {153, 459, 1286, 2410}},
		{0, {146, 395, 1300, 2160}},
		{0, {153, 400, 1320, 2150}},
		{0, {140, 532, 1500, 1890}},
		{0, {146, 538, 1460, 1818}},
		{0, {120, 264, 2290, 2700}},
		{0, {128, 256, 2305, 2635}},
		{0, {112, 380, 1880, 2440}},
		{0, {115, 346, 1930, 2390}},
		{0, {100, 510, 1780, 2300}},
		{0, {108, 520, 1730, 2275}},
		{0, {100, 630, 1770, 2350}},
		{0, {105, 630, 1642, 2170}},
		{0, {103, 601, 1273, 2130}},
		{1, {105, 590, 1283, 2150}},
		{1, {100, 750, 1150, 2440}},
		{0,  {95, 703, 1092, 2320}},
		{0,  {97, 565,  780, 2350}},
		{0, {106, 584,  849, 2460}},
		{0, {105, 420, 1100, 2140}},
		{0, {111, 422, 1200, 2175}},
		{0, {117, 315, 1080, 2260}},
		{0, {125, 326, 1125, 2210}},
		{0, {111, 444, 1300, 1625}},
		{0, {109, 469, 1288, 1600}},
		{0, {124, 210, 2100, 3090}},
		{0, {130, 220, 2080, 3180}},
		{0, {128, 280, 2000, 2710}},
		{0, {130, 310, 1950, 2670}},
		{0, {121, 470, 1910, 2580}},
		{1, {129, 490, 1930, 2650}},
		{0, {116, 640, 1620, 2200}},
		{0, {118, 650, 1580, 2360}},
		{0, {121, 610, 1100, 2230}},
		{0, {126, 620, 1120, 2330}},
		{1, {118, 700, 1100, 2240}},
		{1, {120, 670, 1100, 2220}},
		{0, {122, 460,  720, 2180}},
		{0, {118, 470,  690, 2200}},
		{0, {129, 320,  770, 1860}},
		{1, {130, 310,  790, 1920}},
		{0, {140, 210,  670, 1900}},
		{0, {148, 240,  730, 1850}},
		{0, {128, 390, 1320, 1550}},
		{0, {124, 420, 1240, 1510}},
		{0, {129, 190, 2650, 3280}},
		{0, {135, 190, 2700, 3170}},
		{0, {132, 370, 1750, 2700}},
		{0, {130, 370, 1800, 2750}},
		{1, {122, 370, 1680, 2560}},
		{1, {125, 375, 1700, 2500}},
		{1, {121, 550, 1570, 2600}},
		{1, {120, 530, 1610, 2650}},
		{1, {118, 570, 1050, 2500}},
		{1, {125, 590, 1100, 2480}},
		{1, {112, 640,  970, 2870}},
		{1, {122, 670,  980, 2900}},
		{1, {113, 560,  860, 2900}},
		{0, {121, 570,  820, 2820}},
		{1, {125, 350, 1000, 2500}},
		{1, {130, 380,  920, 2370}},
		{0, {130, 250, 1000, 2100}},
		{0, {140, 210,  960, 1940}},
		{0, {130, 360, 1300, 1920}},
		{0, {133, 370, 1300, 1760}},
		{0, {127, 250, 2180, 2660}},
		{0, {131, 260, 2210, 2780}},
		{0, {121, 400, 1900, 2440}},
		{0, {122, 350, 1980, 2480}},
		{1, {116, 560, 1670, 2310}},
		{1, {124, 530, 1700, 2380}},
		{0, {120, 680, 1470, 2280}},
		{0, {119, 620, 1580, 2320}},
		{0, {120, 620, 1100, 2390}},
		{1, {125, 640, 1110, 2370}},
		{1, {115, 630,  980, 2330}},
		{1, {121, 670,  940, 2380}},
		{0, {112, 560,  790, 2480}},
		{0, {120, 610,  840, 2420}},
		{0, {121, 360,  860, 2200}},
		{0, {120, 400,  840, 2200}},
		{0, {140, 280,  670, 2140}},
		{0, {126, 250,  720, 2190}},
		{0, {120, 480, 1410, 1760}},
		{0, {121, 470, 1330, 1700}},
		{0, {155, 280, 2400, 2910}},
		{0, {150, 300, 2320, 2960}},
		{1, {142, 410, 2060, 2680}},
		{0, {150, 450, 2050, 2670}},
		{0, {135, 540, 1900, 2530}},
		{0, {135, 540, 1920, 2520}},
		{0, {138, 620, 1800, 2440}},
		{0, {140, 690, 1820, 2480}},
		{1, {150, 630, 1200, 2600}},
		{0, {140, 680, 1290, 2600}},
		{0, {145, 740, 1110, 2500}},
		{1, {143, 700, 1060, 2720}},
		{1, {146, 600,  970, 2570}},
		{0, {138, 650,  880, 2660}},
		{0, {142, 430, 1130, 2440}},
		{0, {143, 430, 1150, 2420}},
		{0, {142, 280,  990, 2330}},
		{0, {145, 290, 1000, 2300}},
		{0, {150, 420, 1350, 1600}},
		{0, {150, 450, 1350, 1600}},
		{0, {135, 300, 2300, 2800}},
		{0, {135, 350, 2240, 2760}},
		{1, {136, 410, 2200, 2680}},
		{1, {138, 440, 2080, 2520}},
		{1, {133, 580, 1870, 2320}},
		{1, {127, 520, 1900, 2400}},
		{0, {130, 760, 1920, 2480}},
		{0, {132, 670, 1850, 2560}},
		{1, {139, 810, 1110, 2100}},
		{1, {131, 770, 1150, 2100}},
		{1, {141, 700, 1040, 2120}},
		{0, {125, 750, 1160, 2080}},
		{1, {133, 670,  920, 2240}},
		{1, {142, 570,  850, 2250}},
		{0, {140, 550,  970, 2200}},
		{1, {141, 490,  870, 2240}},
		{0, {150, 300,  600, 2300}},
		{0, {148, 230,  570, 2100}},
		{0, {140, 560, 1520, 2100}},
		{0, {140, 540, 1570, 2050}},
		{0, {125, 240, 2100, 2900}},
		{0, {119, 240, 2150, 2860}},
		{0, {130, 380, 1870, 2450}},
		{0, {120, 430, 1710, 2350}},
		{1, {119, 580, 1770, 2500}},
		{1, {117, 570, 1750, 2400}},
		{1, {115, 760, 1580, 2440}},
		{0, {110, 715, 1500, 2300}},
		{1, {124, 620,  880, 2500}},
		{1, {124, 650, 1000, 2520}},
		{0, {119, 710,  950, 2520}},
		{1, {120, 690,  960, 2520}},
		{0, {125, 460,  610, 2500}},
		{0, {120, 470,  710, 2500}},
		{0, {125, 390,  900, 2100}},
		{0, {125, 460,  920, 2140}},
		{0, {125, 250,  690, 2080}},
		{0, {130, 270,  650, 2050}},
		{0, {122, 540, 1280, 1720}},
		{0, {118, 510, 1280, 1650}},
		{0, {148, 280, 2450, 2700}},
		{0, {160, 288, 2500, 2880}},
		{0, {160, 400, 2080, 2530}},
		{0, {153, 384, 2110, 2500}},
		{1, {138, 590, 1900, 2200}},
		{1, {153, 583, 1840, 2250}},
		{0, {145, 680, 1850, 2400}},
		{0, {140, 685, 1780, 2160}},
		{0, {143, 660, 1370, 2110}},
		{1, {145, 680, 1300, 2100}},
		{1, {140, 760, 1260, 2120}},
		{0, {135, 770, 1140, 2020}},
		{0, {145, 500,  800, 1850}},
		{0, {132, 600, 1000, 2000}},
		{0, {157, 380, 1060, 1950}},
		{0, {150, 470, 1220, 2150}},
		{0, {162, 324,  800, 2220}},
		{0, {139, 290,  800, 2150}},
		{0, {150, 560, 1350, 1780}},
		{0, {150, 600, 1470, 1820}},
		{0, {110, 250, 2190, 3000}},
		{0, {106, 254, 2085, 2890}},
		{0, {111, 330, 1967, 2670}},
		{0, {108, 430, 1940, 2590}},
		{0, {116, 464, 2100, 2700}},
		{0, {105, 504, 1995, 2780}},
		{0,  {94, 595, 1900, 2700}},
		{0, {100, 670, 1860, 2500}},
		{0,  {96, 620, 1200, 2420}},
		{0, {105, 630, 1127, 2420}},
		{0, {100, 750, 1160, 2360}},
		{0,  {96, 740, 1155, 2330}},
		{0, {101, 460,  740, 2300}},
		{1, {105, 494,  789, 2420}},
		{0, {113, 400, 1020, 2200}},
		{0, {128, 450, 1028, 2160}},
		{0, {140, 392, 1000, 2120}},
		{0, {116, 350,  898, 2140}},
		{0, {117, 547, 1340, 1688}},
		{0, {128, 512, 1280, 1570}},
		{0, {123, 246, 2185, 2730}},
		{0, {133, 267, 2280, 2800}},
		{0, {140, 420, 2300, 2800}},
		{0, {120, 384, 2110, 2620}},
		{0, {120, 480, 1920, 2540}},
		{1, {112, 551, 1788, 2450}},
		{0, {114, 628, 1837, 2570}},
		{1, {111, 622, 1890, 2560}},
		{0, {114, 628, 1254, 2470}},
		{0, {114, 617, 1255, 2480}},
		{1, {117, 690, 1072, 2660}},
		{1, {103, 630, 1000, 2530}},
		{0, {117, 510,  700, 2650}},
		{0, {120, 504,  756, 2540}},
		{0, {122, 465,  990, 2440}},
		{0, {125, 462,  976, 2450}},
		{0, {120, 324,  708, 2440}},
		{0, {157, 387,  786, 2518}},
		{0, {122, 488, 1468, 1712}},
		{0, {118, 472, 1465, 1725}},
		{0, {138, 275, 2060, 2800}},
		{0, {136, 270, 2020, 2790}},
		{0, {133, 349, 2030, 2760}},
		{0, {136, 340, 1940, 2560}},
		{1, {120, 444, 1800, 2500}},
		{1, {127, 380, 1800, 2440}},
		{0, {125, 688, 1600, 2300}},
		{0, {122, 660, 1570, 2380}},
		{1, {128, 565, 1157, 2310}},
		{0, {130, 550, 1150, 2250}},
		{0, {125, 712, 1024, 2250}},
		{0, {125, 670, 1080, 2300}},
		{0, {125, 550,  913, 2360}},
		{0, {126, 550,  890, 2280}},
		{0, {128, 360, 1028, 2160}},
		{1, {140, 390, 1060, 2150}},
		{0, {133, 294,  930, 2050}},
		{0, {140, 280, 1000, 2160}},
		{0, {125, 440, 1250, 1625}},
		{0, {130, 480, 1160, 1520}},
		{0, {125, 320, 2160, 2900}},
		{0, {133, 267, 2230, 3000}},
		{0, {115, 440, 1750, 2400}},
		{0, {116, 390, 1780, 2450}},
		{0, {117, 525, 1800, 2480}},
		{0, {110, 520, 1750, 2390}},
		{0, {111, 660, 1600, 2400}},
		{0, {120, 720, 1680, 2430}},
		{0, {117, 600, 1250, 2300}},
		{1, {125, 575, 1170, 2240}},
		{1, {111, 730, 1160, 2340}},
		{0, {117, 860, 1280, 2470}},
		{0, {114, 560,  810, 2290}},
		{0, {116, 584,  840, 2280}},
		{0, {130, 455,  970, 2140}},
		{0, {120, 456, 1040, 2038}},
		{0, {125, 350,  820, 2130}},
		{0, {128, 366,  772, 2058}},
		{1, {111, 450, 1420, 1870}},
		{0, {118, 472, 1430, 1840}},
		{0, {133, 333, 2305, 3200}},
		{0, {131, 326, 2260, 3030}},
		{1, {125, 375, 2188, 2750}},
		{1, {133, 400, 2150, 2680}},
		{1, {125, 500, 1980, 2480}},
		{1, {150, 480, 1950, 2340}},
		{0, {116, 640, 1710, 2450}},
		{0, {123, 615, 1720, 2220}},
		{0, {116, 583, 1110, 2360}},
		{1, {117, 608, 1120, 2700}},
		{0, {111, 777, 1170, 2600}},
		{0, {114, 750, 1175, 2820}},
		{1, {105, 630,  891, 2519}},
		{1, {114, 572,  924, 2660}},
		{0, {125, 438,  975, 2300}},
		{0, {140, 420,  938, 2300}},
		{0, {133, 333,  800, 2130}},
		{0, {140, 320,  840, 2150}},
		{0, {120, 480, 1320, 1870}},
		{0, {127, 483, 1335, 1844}},
		{0, {166, 267, 2300, 2940}},
		{0, {156, 220, 2300, 2900}},
		{0, {154, 431, 2040, 2460}},
		{0, {155, 360, 2010, 2400}},
		{1, {150, 565, 1950, 2500}},
		{0, {180, 540, 2000, 2450}},
		{1, {143, 600, 2000, 2570}},
		{0, {138, 590, 1950, 2460}},
		{1, {157, 630, 1140, 2200}},
		{1, {186, 630, 1170, 2280}},
		{1, {146, 730, 1048, 2450}},
		{0, {155, 730, 1130, 2320}},
		{1, {150, 600,  900, 2400}},
		{1, {178, 640,  890, 2280}},
		{1, {160, 448,  960, 2200}},
		{0, {196, 450, 1000, 2180}},
		{0, {167, 333,  835, 2170}},
		{0, {198, 280,  750, 2170}},
		{0, {163, 488, 1300, 1600}},
		{0, {163, 490, 1380, 1620}},
		{0, {120, 312, 2380, 2900}},
		{0, {120, 300, 2350, 3000}},
		{0, {140, 490, 2000, 2620}},
		{1, {140, 490, 1960, 2600}},
		{0, {125, 640, 2000, 2620}},
		{0, {111, 555, 1870, 2540}},
		{0, {112, 697, 1610, 2540}},
		{0, {114, 684, 1634, 2510}},
		{0, {115, 633, 1260, 2530}},
		{1, {120, 660, 1213, 2460}},
		{0, {112, 730, 1203, 2700}},
		{0, {107, 752, 1125, 2620}},
		{0, {108, 507,  755, 2420}},
		{0, {116, 538,  816, 2450}},
		{0, {114, 456, 1040, 2300}},
		{0, {120, 480, 1120, 2160}},
		{1, {123, 344,  960, 2150}},
		{0, {125, 350, 1000, 2250}},
		{0, {112, 539, 1370, 1800}},
		{0, {117, 549, 1353, 1728}},
		{0, {146, 292, 2500, 3150}},
		{0, {133, 266, 2370, 3100}},
		{0, {143, 372, 2220, 2640}},
		{0, {131, 350, 2130, 2610}},
		{1, {133, 574, 1840, 2260}},
		{1, {133, 563, 1960, 2450}},
		{0, {125, 650, 1738, 2400}},
		{0, {130, 663, 1820, 2400}},
		{0, {137, 600, 1370, 2180}},
		{1, {125, 625, 1312, 2250}},
		{0, {133, 735, 1070, 2100}},
		{0, {117, 713, 1180, 2200}},
		{0, {125, 625,  875, 2180}},
		{1, {115, 700, 1000, 2250}},
		{0, {150, 420, 1100, 2000}},
		{0, {140, 420, 1120, 2100}},
		{0, {125, 350,  980, 2200}},
		{0, {133, 320,  918, 2100}},
		{0, {143, 554, 1480, 1800}},
		{0, {128, 484, 1505, 1890}},
		{0, {143, 286, 2415, 2860}},
		{0, {150, 300, 2415, 2860}},
		{1, {140, 400, 1980, 2500}},
		{1, {145, 407, 2095, 2620}},
		{1, {125, 525, 1988, 2610}},
		{0, {144, 553, 1935, 2530}},
		{0, {133, 640, 1773, 2490}},
		{0, {133, 640, 1840, 2560}},
		{0, {143, 672, 1272, 2640}},
		{0, {146, 658, 1241, 2560}},
		{0, {130, 780, 1170, 2640}},
		{0, {131, 788, 1115, 2645}},
		{1, {138, 633,  891, 2500}},
		{1, {150, 600,  935, 2550}},
		{0, {175, 490, 1102, 2420}},
		{0, {154, 492, 1077, 2306}},
		{1, {160, 320,  960, 2240}},
		{0, {160, 320,  960, 2290}},
		{0, {143, 543, 1310, 1643}},
		{0, {145, 508, 1309, 1600}},
		{1, {230, 370, 2670, 3100}},
		{0, {234, 390, 2760, 3060}},
		{0, {234, 468, 2330, 2930}},
		{0, {205, 410, 2380, 2950}},
		{0, {190, 550, 2200, 2880}},
		{0, {191, 570, 2100, 3040}},
		{0, {200, 800, 1980, 2810}},
		{0, {192, 860, 1920, 2850}},
		{0, {227, 635, 1200, 3250}},
		{0, {200, 700, 1200, 3100}},
		{1, {210, 880, 1240, 2870}},
		{0, {188, 830, 1200, 2880}},
		{0, {207, 570,  830, 3300}},
		{1, {200, 700, 1000, 3130}},
		{0, {240, 410,  940, 3040}},
		{0, {225, 450,  970, 3190}},
		{0, {238, 480,  955, 2960}},
		{1, {208, 395,  810, 2900}},
		{0, {200, 500, 1850, 2100}},
		{0, {200, 560, 1750, 2100}},
		{0, {225, 270, 2760, 3550}},
		{0, {240, 290, 2700, 3350}},
		{0, {245, 460, 2500, 3220}},
		{0, {220, 410, 2400, 3240}},
		{0, {220, 620, 2300, 3200}},
		{0, {210, 630, 2300, 3170}},
		{0, {220, 820, 2180, 2850}},
		{0, {195, 740, 2120, 3070}},
		{0, {240, 800, 1300, 2900}},
		{0, {225, 760, 1400, 2830}},
		{0, {214, 850, 1120, 2620}},
		{0, {190, 880, 1220, 2850}},
		{0, {228, 460,  900, 2830}},
		{1, {222, 440,  880, 2850}},
		{0, {250, 500, 1040, 2750}},
		{0, {245, 490, 1000, 2720}},
		{0, {250, 400,  940, 2720}},
		{0, {245, 410,  860, 2700}},
		{0, {225, 440, 1560, 1750}},
		{0, {210, 420, 1600, 1750}},
		{0, {210, 290, 2700, 3020}},
		{0, {215, 280, 2630, 3240}},
		{1, {211, 420, 2300, 2950}},
		{0, {211, 420, 2220, 2980}},
		{1, {207, 640, 2120, 2900}},
		{1, {221, 700, 2000, 2900}},
		{0, {212, 1000, 1830, 2820}},
		{0, {204, 980, 1800, 2820}},
		{1, {205, 780, 1410, 2720}},
		{0, {208, 710, 1450, 2750}},
		{0, {205, 950, 1280, 2600}},
		{0, {210, 870, 1260, 2740}},
		{0, {203, 610,  900, 2710}},
		{0, {210, 630,  840, 2700}},
		{0, {211, 440, 1050, 2780}},
		{0, {210, 420, 1050, 2740}},
		{0, {222, 380,  860, 2500}},
		{0, {208, 330,  750, 2740}},
		{0, {208, 580, 1450, 1720}},
		{0, {212, 540, 1560, 1900}},
		{0, {210, 294, 2800, 3100}},
		{0, {222, 270, 2880, 3160}},
		{0, {202, 420, 2430, 3030}},
		{0, {212, 420, 2370, 2930}},
		{1, {200, 580, 2180, 2770}},
		{1, {217, 540, 2160, 2770}},
		{0, {200, 820, 1970, 2620}},
		{0, {210, 840, 2000, 2700}},
		{1, {208, 690, 1200, 2900}},
		{0, {201, 666, 1206, 2900}},
		{1, {200, 800, 1200, 2920}},
		{1, {190, 760, 1140, 2850}},
		{0, {200, 560,  760, 2800}},
		{0, {207, 560,  770, 3000}},
		{1, {215, 430, 1075, 2580}},
		{0, {213, 430, 1000, 2700}},
		{0, {220, 330,  840, 2550}},
		{0, {213, 280,  850, 2500}},
		{0, {205, 430, 1800, 1930}},
		{0, {200, 420, 1740, 1960}},
		{0, {175, 350, 2800, 3160}},
		{0, {187, 338, 2870, 3300}},
		{0, {200, 400, 2540, 3200}},
		{0, {210, 420, 2680, 3000}},
		{0, {180, 518, 2470, 3200}},
		{0, {200, 600, 2400, 3150}},
		{0, {171, 773, 2000, 2870}},
		{0, {175, 875, 2100, 2970}},
		{1, {183, 733, 1468, 2700}},
		{0, {200, 740, 1280, 2900}},
		{1, {178, 730, 1210, 2740}},
		{1, {175, 735, 1220, 2850}},
		{0, {160, 560,  960, 2850}},
		{0, {192, 536,  850, 2850}},
		{0, {212, 424, 1040, 2780}},
		{0, {200, 520, 1060, 2670}},
		{0, {190, 380,  770, 2900}},
		{0, {187, 340,  750, 2780}},
		{1, {177, 490, 2120, 2480}},
		{0, {197, 493, 1930, 2300}},
		{0, {250, 325, 2700, 3100}},
		{0, {225, 310, 2750, 3225}},
		{0, {214, 350, 2580, 3000}},
		{0, {267, 390, 2700, 3200}},
		{0, {233, 560, 2330, 2800}},
		{0, {200, 520, 2500, 3000}},
		{0, {171, 806, 1970, 2600}},
		{0, {150, 825, 1860, 2550}},
		{0, {186, 708, 1485, 2760}},
		{0, {188, 676, 1500, 2590}},
		{0, {200, 800, 1200, 2800}},
		{1, {205, 714, 1154, 2850}},
		{0, {267, 530,  800, 2780}},
		{1, {180, 485,  810, 2750}},
		{0, {214, 450, 1460, 2550}},
		{0, {233, 467, 1400, 2450}},
		{0, {225, 450, 1080, 2350}},
		{0, {200, 400, 1000, 2400}},
		{0, {193, 524, 1700, 2130}},
		{0, {180, 507, 1800, 2380}},
		{0, {200, 300, 3100, 3400}},
		{1, {216, 300, 3100, 3500}},
		{0, {214, 428, 2570, 3000}},
		{0, {220, 440, 2640, 3080}},
		{0, {210, 528, 2540, 3170}},
		{1, {210, 504, 2520, 3200}},
		{0, {187, 940, 2250, 2760}},
		{0, {200, 820, 2200, 2920}},
		{0, {204, 816, 1450, 2700}},
		{1, {214, 858, 1500, 2700}},
		{0, {200, 960, 1280, 3000}},
		{1, {180, 1040, 1300, 3000}},
		{0, {220, 520,  880, 2500}},
		{0, {217, 574,  890, 2510}},
		{1, {233, 466, 1330, 2750}},
		{1, {233, 466, 1165, 2800}},
		{0, {180, 300,  850, 2800}},
		{1, {175, 350,  840, 2750}},
		{0, {216, 432, 1790, 2060}},
		{0, {219, 360, 1900, 2320}},
		{0, {225, 337, 2700, 3300}},
		{0, {233, 340, 2720, 3200}},
		{0, {237, 474, 2370, 3095}},
		{0, {237, 475, 2400, 3090}},
		{0, {229, 526, 2360, 3090}},
		{0, {233, 580, 2360, 3150}},
		{1, {230, 690, 2185, 2990}},
		{1, {220, 660, 2200, 3020}},
		{1, {225, 675, 1551, 2923}},
		{0, {233, 690, 1630, 2900}},
		{0, {222, 845, 1334, 2890}},
		{0, {233, 888, 1290, 2800}},
		{0, {225, 631,  923, 2250}},
		{0, {233, 543,  980, 2300}},
		{0, {233, 537, 1360, 2920}},
		{0, {240, 480, 1345, 2680}},
		{0, {235, 400, 1180, 2760}},
		{0, {233, 396, 1120, 2560}},
		{0, {225, 450, 1640, 2250}},
		{0, {233, 489, 1630, 2090}},
		{0, {225, 225, 2760, 3900}},
		{0, {230, 230, 2850, 3800}},
		{0, {238, 429, 2560, 3200}},
		{1, {230, 430, 2575, 3100}},
		{0, {214, 579, 2570, 3300}},
		{0, {214, 536, 2570, 3100}},
		{0, {205, 823, 2220, 2870}},
		{0, {200, 800, 2100, 2900}},
		{0, {250, 750, 1500, 2750}},
		{0, {217, 738, 1300, 2820}},
		{0, {200, 840, 1300, 3100}},
		{1, {206, 990, 1340, 3100}},
		{0, {214, 579,  856, 2790}},
		{0, {205, 545,  905, 2750}},
		{1, {233, 490, 1220, 2610}},
		{1, {250, 513, 1500, 2650}},
		{0, {250, 400, 1250, 2500}},
		{0, {225, 405, 1080, 2500}},
		{0, {233, 466, 1860, 2260}},
		{0, {225, 540, 1780, 2220}},
		{0, {240, 290, 3000, 3840}},
		{0, {250, 325, 2900, 3500}},
		{0, {250, 500, 2370, 3120}},
		{1, {238, 476, 2380, 3090}},
		{0, {238, 760, 2380, 3205}},
		{0, {233, 746, 2290, 3030}},
		{0, {206, 1008, 1990, 2870}},
		{0, {200, 1040, 2000, 2800}},
		{0, {220, 830, 1540, 2860}},
		{0, {237, 900, 1510, 2840}},
		{0, {206, 970, 1343, 3018}},
		{1, {236, 592, 1230, 2600}},
		{0, {233, 650,  900, 2920}},
		{0, {229, 687, 1060, 2780}},
		{1, {233, 512, 1211, 2630}},
		{0, {233, 467, 1167, 2595}},
		{0, {250, 450,  875, 2750}},
		{0, {233, 420,  935, 2710}},
		{0, {230, 622, 1750, 2070}},
		{0, {225, 652, 1710, 2043}},
		{0, {255, 275, 2800, 3310}},
		{0, {245, 245, 2800, 3300}},
		{0, {267, 534, 2500, 3250}},
		{0, {264, 528, 2640, 3370}},
		{0, {238, 700, 2380, 3250}},
		{0, {250, 750, 2480, 3000}},
		{0, {237, 1020, 1900, 2960}},
		{0, {233, 1005, 2050, 2870}},
		{0, {263, 750, 1500, 2850}},
		{0, {250, 850, 1400, 2750}},
		{0, {258, 978, 1290, 2840}},
		{0, {246, 935, 1230, 2730}},
		{0, {250, 500,  750, 2750}},
		{0, {243, 632,  850, 2850}},
		{0, {250, 350, 1170, 2750}},
		{0, {266, 450, 1000, 2800}},
		{0, {256, 358,  640, 2560}},
		{0, {250, 300,  750, 2500}},
		{0, {260, 520, 1560, 1820}},
		{0, {250, 500, 1500, 1750}},
		{0, {236, 236, 2790, 3760}},
		{0, {242, 242, 2770, 3800}},
		{0, {222, 444, 2555, 3110}},
		{0, {242, 420, 2700, 3120}},
		{0, {226, 634, 2325, 2940}},
		{0, {225, 608, 2475, 3100}},
		{0, {210, 1010, 2060, 2900}},
		{0, {200, 980, 2160, 2920}},
		{0, {217, 818, 1450, 2500}},
		{0, {200, 750, 1280, 2650}},
		{0, {220, 820, 1200, 2640}},
		{0, {210, 900, 1120, 2900}},
		{0, {220, 440,  749, 2640}},
		{0, {210, 567,  752, 2600}},
		{1, {204, 460, 1045, 2504}},
		{0, {240, 480, 1105, 2400}},
		{0, {250, 420, 1000, 2500}},
		{0, {275, 350, 1100, 2400}},
		{0, {217, 487, 1500, 1780}},
		{0, {206, 467, 1420, 1640}},
		{0, {225, 360, 2920, 3400}},
		{0, {233, 340, 2840, 3300}},
		{0, {257, 514, 2570, 3070}},
		{0, {238, 500, 2680, 3260}},
		{0, {238, 650, 2495, 3090}},
		{1, {216, 650, 2380, 3030}},
		{0, {225, 1020, 2030, 2700}},
		{0, {225, 1000, 2200, 2770}},
		{0, {225, 788, 1462, 2920}},
		{0, {217, 736, 1500, 2900}},
		{1, {214, 987, 1330, 2830}},
		{0, {214, 1009, 1415, 3080}},
		{1, {226, 672, 1084, 2495}},
		{0, {209, 627, 1045, 2504}},
		{0, {250, 500, 1200, 2450}},
		{0, {230, 460, 1150, 2880}},
		{0, {267, 420,  990, 2860}},
		{0, {190, 380,  893, 2920}},
		{0, {246, 610, 1630, 2020}},
		{0, {225, 585, 1700, 1850}},
		{0, {285, 285, 2900, 3500}},
		{0, {286, 310, 2900, 3400}},
		{0, {297, 480, 2670, 3260}},
		{1, {220, 440, 2620, 3380}},
		{0, {173, 550, 2370, 3140}},
		{0, {260, 520, 2340, 3040}},
		{0, {167, 790, 2180, 3020}},
		{0, {280, 840, 2160, 3020}},
		{0, {280, 840, 1400, 2750}},
		{0, {270, 760, 1330, 2950}},
		{0, {252, 900, 1290, 2750}},
		{1, {260, 900, 1240, 3110}},
		{0, {175, 700, 1050, 2750}},
		{0, {190, 720, 1080, 3030}},
		{0, {286, 540, 1200, 2860}},
		{0, {205, 570, 1200, 2970}},
		{0, {328, 400,  980, 2630}},
		{0, {290, 440,  990, 2900}},
		{0, {286, 570, 2000, 2480}},
		{0, {260, 510, 1850, 2350}},
		{0, {170, 340, 2750, 3120}},
		{0, {238, 360, 2760, 3120}},
		{0, {167, 480, 2390, 2950}},
		{1, {194, 520, 2450, 3000}},
		{0, {220, 620, 2520, 2920}},
		{0, {222, 620, 2440, 2880}},
		{0, {222, 1110, 2160, 2700}},
		{0, {214, 1070, 1960, 2750}},
		{1, {217, 820, 1240, 2600}},
		{1, {216, 860, 1300, 2670}},
		{0, {150, 840, 1110, 2930}},
		{0, {170, 850, 1120, 2850}},
		{0, {200, 500,  700, 2930}},
		{0, {212, 380,  720, 2700}},
		{0, {235, 400,  940, 2820}},
		{0, {214, 380,  860, 2680}},
		{0, {196, 330,  760, 2870}},
		{0, {188, 350,  710, 2760}},
		{0, {182, 550, 1780, 2080}},
		{0, {201, 600, 1750, 2000}},
		{0, {200, 320, 2360, 2980}},
		{0, {203, 304, 2380, 3050}},
		{1, {211, 444, 2220, 2740}},
		{1, {210, 420, 2090, 2780}},
		{0, {200, 500, 2350, 2830}},
		{0, {200, 600, 2200, 2700}},
		{1, {192, 845, 1700, 2300}},
		{0, {187, 860, 1724, 2530}},
		{0, {200, 720, 1440, 2380}},
		{0, {191, 707, 1470, 2440}},
		{0, {200, 700, 1080, 2420}},
		{1, {192, 767, 1150, 2590}},
		{0, {200, 600,  860, 2410}},
		{0, {200, 600,  900, 2400}},
		{0, {210, 546, 1090, 2400}},
		{0, {210, 462, 1240, 2310}},
		{0, {257, 360,  930, 2260}},
		{0, {220, 440, 1100, 2300}},
		{0, {200, 540, 1400, 1800}},
		{0, {204, 460, 1350, 1560}},
		{0, {203, 406, 2600, 2945}},
		{1, {200, 400, 2600, 3100}},
		{0, {200, 460, 2300, 2800}},
		{0, {210, 420, 2305, 2835}},
		{0, {190, 570, 2100, 2720}},
		{0, {207, 538, 2175, 2880}},
		{0, {189, 850, 1853, 2685}},
		{0, {193, 830, 1800, 2620}},
		{0, {200, 720, 1500, 2560}},
		{0, {200, 800, 1400, 2420}},
		{0, {194, 915, 1280, 2530}},
		{0, {206, 723, 1196, 2600}},
		{0, {192, 575, 1073, 2490}},
		{0, {200, 600, 1100, 2600}},
		{0, {202, 520, 1210, 2420}},
		{0, {212, 468, 1275, 2550}},
		{0, {207, 370, 1000, 2470}},
		{0, {205, 330,  970, 2460}},
		{0, {200, 560, 1600, 1900}},
		{0, {206, 514, 1540, 1955}},
		{0, {240, 380, 2880, 3360}},
		{0, {250, 380, 2820, 3300}},
		{1, {233, 514, 2600, 2930}},
		{0, {237, 473, 2660, 2970}},
		{0, {223, 567, 2460, 3122}},
		{0, {224, 521, 2460, 2920}},
		{0, {218, 808, 2070, 2880}},
		{1, {203, 678, 2420, 3080}},
		{0, {200, 800, 1340, 2700}},
		{1, {214, 772, 1280, 2660}},
		{0, {183, 843, 1190, 2860}},
		{0, {205, 740, 1160, 2780}},
		{1, {222, 623, 1022, 2700}},
		{0, {220, 594,  990, 2640}},
		{1, {240, 480,  960, 2820}},
		{0, {242, 484,  900, 2640}},
		{0, {233, 370,  933, 2520}},
		{0, {250, 325,  750, 2500}},
		{0, {225, 450, 1680, 2050}},
		{1, {233, 466, 1630, 1865}},
		{0, {200, 320, 2750, 3100}},
		{0, {178, 356, 2755, 3200}},
		{0, {194, 388, 2622, 3050}},
		{0, {194, 426, 2460, 3040}},
		{0, {187, 592, 2242, 2765}},
		{1, {191, 535, 2290, 2870}},
		{1, {188, 750, 2060, 2770}},
		{1, {162, 650, 2110, 2618}},
		{0, {187, 618, 1518, 2700}},
		{0, {183, 624, 1430, 2660}},
		{0, {163, 766, 1180, 2340}},
		{0, {167, 750, 1065, 2640}},
		{1, {170, 595,  918, 2600}},
		{1, {176, 630,  985, 2630}},
		{0, {200, 420, 1200, 2600}},
		{0, {200, 460, 1260, 2640}},
		{0, {187, 375, 1124, 2685}},
		{1, {188, 375, 1143, 2700}},
		{0, {180, 504, 1565, 1835}},
		{0, {183, 513, 1578, 1830}},
		{0, {280, 357, 2800, 3360}},
		{0, {275, 340, 2860, 3350}},
		{0, {290, 480, 2600, 3060}},
		{0, {292, 465, 2598, 3060}},
		{1, {250, 700, 2350, 2980}},
		{0, {240, 737, 2325, 3100}},
		{0, {200, 960, 2100, 3000}},
		{0, {217, 1030, 2200, 3260}},
		{0, {275, 920, 1512, 2950}},
		{0, {260, 910, 1688, 3050}},
		{0, {275, 990, 1237, 2360}},
		{0, {267, 987, 1172, 3180}},
		{0, {267, 587, 1068, 3270}},
		{0, {293, 560,  990, 3150}},
		{1, {275, 520, 1350, 3190}},
		{1, {280, 510, 1415, 3130}},
		{0, {300, 420, 1045, 3060}},
		{0, {300, 390,  960, 3030}},
		{0, {230, 460, 1860, 2250}},
		{0, {214, 504, 1820, 2290}},
		{0, {200, 240, 2760, 3700}},
		{0, {220, 220, 2850, 3800}},
		{0, {228, 319, 2500, 3020}},
		{0, {216, 324, 2500, 3010}},
		{0, {220, 616, 2380, 2900}},
		{0, {212, 615, 2300, 2800}},
		{1, {212, 710, 2120, 2600}},
		{1, {210, 690, 2250, 2680}},
		{0, {221, 800, 1520, 2380}},
		{0, {210, 780, 1470, 2400}},
		{0, {199, 995, 1392, 2290}},
		{0, {200, 1000, 1400, 2440}},
		{0, {205, 656,  944, 2250}},
		{0, {200, 720,  960, 2380}},
		{0, {223, 335, 1049, 2470}},
		{0, {210, 420, 1009, 2300}},
		{0, {219, 329,  877, 2550}},
		{0, {230, 340,  900, 2530}},
		{0, {206, 400, 1380, 1560}},
		{0, {201, 400, 1240, 1480}},
		{0, {220, 286, 2800, 3550}},
		{0, {241, 289, 2800, 3400}},
		{0, {225, 383, 2420, 3080}},
		{0, {240, 384, 2400, 3050}},
		{1, {209, 418, 2430, 3110}},
		{0, {230, 460, 2300, 3050}},
		{0, {187, 861, 2100, 2800}},
		{0, {224, 896, 2040, 3000}},
		{0, {218, 654, 1160, 2800}},
		{0, {230, 690, 1195, 2770}},
		{0, {208, 860, 1103, 2700}},
		{0, {212, 806, 1060, 2850}},
		{0, {202, 606,  910, 2900}},
		{0, {201, 583,  860, 2840}},
		{0, {225, 340,  900, 2650}},
		{0, {235, 470, 1100, 2560}},
		{0, {205, 308, 1025, 2650}},
		{0, {235, 329, 1151, 2560}},
		{0, {213, 533, 1425, 1830}},
		{0, {214, 535, 1412, 1800}},
		{0, {236, 307, 2670, 3150}},
		{0, {245, 340, 2700, 3250}},
		{0, {231, 417, 2300, 3000}},
		{1, {239, 410, 2200, 2910}},
		{1, {222, 644, 2250, 3000}},
		{0, {224, 670, 2300, 2880}},
		{1, {224, 784, 1800, 2750}},
		{1, {234, 820, 1750, 2890}},
		{0, {225, 765, 1300, 2700}},
		{0, {221, 730, 1390, 2790}},
		{1, {225, 834, 1282, 2800}},
		{0, {212, 850, 1270, 2760}},
		{1, {229, 688, 1029, 2750}},
		{0, {222, 670, 1040, 2640}},
		{0, {251, 427, 1506, 2640}},
		{0, {240, 460, 1370, 2610}},
		{0, {236, 378, 1416, 2580}},
		{0, {239, 380, 1430, 2610}},
		{0, {230, 460, 1200, 1909}},
		{0, {225, 410, 1580, 1800}},
		{0, {256, 384, 2860, 3210}},
		{0, {250, 375, 3000, 3400}},
		{0, {230, 460, 2665, 3140}},
		{0, {233, 467, 2680, 3150}},
		{0, {229, 640, 2400, 2860}},
		{0, {233, 630, 2530, 3030}},
		{1, {233, 700, 2560, 3150}},
		{1, {225, 675, 2510, 3145}},
		{1, {240, 768, 1440, 2855}},
		{0, {234, 794, 1447, 2920}},
		{0, {227, 978, 1362, 2724}},
		{0, {233, 933, 1350, 2610}},
		{0, {240, 700, 1080, 2810}},
		{0, {240, 720, 1090, 2840}},
		{0, {243, 500, 1215, 2870}},
		{0, {239, 500, 1240, 2860}},
		{0, {263, 470, 1000, 2820}},
		{0, {272, 378,  950, 2990}},
		{0, {243, 480, 1410, 1700}},
		{0, {243, 493, 1580, 1775}},
		{0, {268, 320, 2900, 3200}},
		{0, {263, 290, 2750, 3050}},
		{1, {258, 460, 2380, 2940}},
		{1, {251, 480, 2260, 2980}},
		{1, {246, 640, 2220, 2900}},
		{1, {250, 670, 2250, 2960}},
		{0, {243, 950, 1970, 2890}},
		{0, {244, 980, 1950, 2920}},
		{1, {251, 750, 1280, 2760}},
		{0, {258, 770, 1340, 2800}},
		{1, {250, 950, 1130, 3160}},
		{1, {256, 850, 1150, 2940}},
		{0, {242, 530,  870, 2680}},
		{0, {250, 600,  900, 2770}},
		{1, {250, 600, 1225, 2500}},
		{1, {264, 630, 1320, 2560}},
		{0, {258, 440, 1290, 2530}},
		{0, {269, 460, 1080, 2640}},
		{0, {250, 600, 1500, 2000}},
		{0, {254, 610, 1520, 1950}},
		{0, {234, 280, 2690, 3040}},
		{0, {261, 280, 2740, 2980}},
		{0, {260, 470, 2500, 3400}},
		{0, {262, 440, 2480, 3240}},
		{0, {242, 730, 2300, 3100}},
		{0, {260, 750, 2340, 3120}},
		{0, {233, 860, 2070, 2880}},
		{0, {240, 890, 1920, 2710}},
		{0, {257, 770, 1540, 2840}},
		{0, {257, 800, 1410, 2860}},
		{0, {240, 790, 1250, 3080}},
		{1, {241, 820, 1210, 2960}},
		{0, {234, 408,  695, 3040}},
		{0, {246, 420,  590, 3100}},
		{0, {251, 500, 1230, 2520}},
		{0, {256, 480, 1230, 2750}},
		{0, {263, 419, 1050, 2850}},
		{0, {278, 390, 1060, 2800}},
		{0, {220, 420, 1720, 1900}},
		{0, {255, 510, 1680, 1890}},
		{0, {208, 270, 2820, 3450}},
		{0, {225, 250, 2880, 3350}},
		{0, {220, 370, 2530, 3060}},
		{0, {250, 400, 2600, 3120}},
		{0, {214, 640, 2360, 3020}},
		{0, {219, 650, 2430, 3040}},
		{0, {205, 900, 2090, 3000}},
		{0, {200, 860, 2160, 2870}},
		{0, {214, 750, 1540, 2800}},
		{0, {214, 770, 1530, 2780}},
		{0, {195, 920, 1350, 2550}},
		{1, {210, 920, 1470, 2690}},
		{0, {194, 720, 1110, 2420}},
		{0, {200, 700, 1100, 2780}},
		{0, {222, 470, 1200, 2900}},
		{0, {237, 470, 1190, 2800}},
		{0, {240, 380,  980, 3100}},
		{0, {188, 340,  920, 3050}},
		{0, {222, 530, 1670, 2050}},
		{0, {200, 500, 1720, 1900}},
		{0, {258, 310, 2740, 3200}},
		{0, {262, 262, 2680, 3170}},
		{0, {262, 450, 2310, 3020}},
		{0, {263, 472, 2270, 2950}},
		{1, {245, 640, 1980, 2920}},
		{1, {235, 700, 2110, 2940}},
		{0, {194, 810, 1860, 2620}},
		{0, {234, 890, 1800, 2700}},
		{1, {230, 710, 1340, 2780}},
		{1, {245, 740, 1470, 2940}},
		{1, {225, 830, 1020, 2650}},
		{1, {219, 830, 1095, 2610}},
		{0, {240, 600,  850, 2760}},
		{0, {253, 455,  810, 2750}},
		{0, {282, 400, 1070, 2530}},
		{0, {250, 450, 1050, 2450}},
		{0, {260, 290,  670, 2380}},
		{0, {275, 330,  630, 2460}},
		{0, {240, 500, 1630, 2040}},
		{0, {243, 490, 1580, 2190}},
		{0, {228, 460, 3300, 3950}},
		{0, {200, 400, 3400, 3850}},
		{1, {205, 600, 2550, 4000}},
		{1, {205, 610, 2500, 4100}},
		{1, {225, 600, 2750, 3600}},
		{1, {210, 760, 2500, 3850}},
		{0, {200, 1000, 2300, 3900}},
		{1, {200, 800, 2500, 4050}},
		{1, {200, 1000, 1750, 3550}},
		{0, {223, 1110, 1690, 4040}},
		{1, {205, 1220, 1560, 3650}},
		{1, {200, 1300, 1800, 3450}},
		{0, {219, 660, 1100, 3850}},
		{1, {217, 690, 1090, 3900}},
		{1, {206, 620, 1420, 3700}},
		{0, {220, 620, 1410, 3520}},
		{0, {233, 440,  900, 3900}},
		{0, {200, 400,  650, 3800}},
		{0, {210, 610, 2300, 2900}},
		{0, {200, 450, 2150, 2550}},
		{0, {290, 320, 3500, 4260}},
		{0, {305, 350, 3400, 4100}},
		{0, {322, 640, 3200, 3660}},
		{0, {325, 650, 3000, 3800}},
		{0, {270, 850, 2900, 3680}},
		{1, {285, 700, 3120, 3750}},
		{0, {256, 1130, 2560, 3500}},
		{0, {285, 1140, 2000, 3560}},
		{0, {310, 1130, 1740, 3670}},
		{0, {300, 1000, 1800, 3450}},
		{1, {265, 1170, 1500, 3440}},
		{1, {283, 980, 1300, 3100}},
		{1, {265, 530, 1060, 3450}},
		{1, {272, 540, 1080, 3000}},
		{0, {285, 560, 1440, 3500}},
		{0, {294, 570, 1450, 3500}},
		{0, {333, 350, 1280, 3650}},
		{0, {290, 340, 1160, 2950}},
		{1, {275, 560, 1740, 2460}},
		{0, {302, 600, 1800, 2200}},
		{0, {240, 380, 3140, 3700}},
		{0, {258, 310, 3350, 3650}},
		{1, {290, 580, 2760, 3400}},
		{0, {250, 500, 2660, 3500}},
		{1, {250, 780, 2450, 3400}},
		{1, {240, 672, 2550, 3400}},
		{1, {240, 660, 2900, 3370}},
		{1, {215, 760, 2850, 3300}},
		{0, {250, 880, 1500, 3200}},
		{0, {243, 850, 1700, 3250}},
		{0, {250, 940, 1380, 2400}},
		{0, {276, 1200, 1500, 3160}},
		{0, {250, 750, 1250, 3450}},
		{0, {225, 675,  950, 3240}},
		{0, {300, 610, 1500, 3300}},
		{0, {275, 500, 1370, 3500}},
		{0, {256, 300, 1280, 3150}},
		{0, {250, 400, 1300, 3700}},
		{0, {250, 500, 1540, 1700}},
		{0, {242, 580, 1620, 1790}},
		{0, {291, 410, 3200, 3800}},
		{0, {264, 420, 3400, 3900}},
		{1, {291, 580, 2900, 3820}},
		{1, {280, 560, 2840, 3900}},
		{1, {292, 810, 2640, 4200}},
		{1, {270, 780, 2720, 4100}},
		{0, {270, 1080, 2480, 3950}},
		{0, {245, 1050, 2420, 4000}},
		{1, {286, 970, 1600, 3950}},
		{1, {250, 800, 1680, 3800}},
		{1, {275, 1040, 1350, 3850}},
		{0, {250, 1100, 1460, 4250}},
		{1, {286, 770, 1150, 3950}},
		{1, {273, 710, 1200, 3900}},
		{1, {285, 680, 1420, 3800}},
		{0, {278, 640, 1350, 3950}},
		{0, {300, 420, 1110, 3640}},
		{0, {280, 505, 1050, 3400}},
		{0, {320, 640, 1940, 2820}},
		{0, {265, 610, 2100, 2600}},
		{0, {330, 460, 2800, 3550}},
		{0, {333, 490, 2730, 3550}},
		{1, {310, 560, 2500, 3450}},
		{1, {310, 580, 2500, 3450}},
		{1, {286, 800, 2300, 3750}},
		{1, {310, 835, 2420, 3740}},
		{0, {282, 950, 2150, 3650}},
		{0, {310, 1000, 2150, 3700}},
		{0, {293, 880, 1700, 3750}},
		{0, {340, 900, 1600, 3650}},
		{1, {299, 990, 1410, 3750}},
		{0, {280, 1050, 1320, 3730}},
		{1, {285, 770,  940, 3750}},
		{1, {333, 680, 1020, 3700}},
		{0, {322, 550, 1195, 3750}},
		{0, {350, 550, 1340, 3500}},
		{1, {316, 600, 1200, 3600}},
		{0, {345, 550, 1100, 3470}},
		{0, {310, 805, 1705, 2420}},
		{0, {310, 710, 1700, 2400}},
		{0, {210, 340, 3400, 4320}},
		{0, {227, 590, 3610, 4220}},
		{0, {235, 680, 3250, 4380}},
		{1, {220, 440, 3000, 3790}},
		{0, {212, 660, 2900, 3610}},
		{1, {216, 610, 2760, 3650}},
		{0, {214, 1240, 2700, 3640}},
		{0, {215, 1050, 2550, 3550}},
		{0, {216, 820, 1470, 3500}},
		{1, {211, 970, 1410, 3200}},
		{0, {218, 1090, 1380, 3050}},
		{1, {212, 860, 1250, 2800}},
		{0, {211, 800, 1220, 3700}},
		{0, {214, 640, 1070, 3000}},
		{0, {219, 660, 1360, 3700}},
		{0, {214, 730, 1500, 3600}},
		{0, {220, 620, 1100, 3250}},
		{0, {216, 600, 1280, 3650}},
		{0, {222, 670, 2130, 2360}},
		{0, {205, 760, 2240, 2460}},
		{0, {253, 330, 3250, 3720}},
		{0, {262, 340, 3100, 3400}},
		{1, {250, 500, 2500, 3640}},
		{1, {278, 530, 2630, 3640}},
		{1, {255, 710, 2550, 3560}},
		{1, {250, 750, 2480, 3470}},
		{0, {233, 1140, 2260, 3640}},
		{0, {245, 1110, 2230, 3380}},
		{0, {256, 770, 1540, 3500}},
		{0, {257, 800, 1490, 3300}},
		{0, {240, 940, 1400, 3400}},
		{1, {245, 930, 1370, 3120}},
		{0, {240, 530,  860, 3400}},
		{0, {240, 520,  910, 3420}},
		{0, {255, 510, 1250, 3320}},
		{0, {260, 520, 1140, 3320}},
		{0, {274, 360,  660, 3050}},
		{0, {260, 310,  730, 3500}},
		{0, {250, 550, 1500, 1800}},
		{0, {239, 480, 1650, 1960}},
		{0, {250, 300, 2950, 3600}},
		{0, {270, 320, 3210, 3600}},
		{0, {290, 550, 2610, 3560}},
		{1, {286, 540, 2570, 3600}},
		{0, {280, 700, 2500, 3580}},
		{0, {263, 600, 2360, 3400}},
		{0, {260, 970, 2400, 3200}},
		{0, {250, 950, 2270, 3200}},
		{0, {270, 780, 1650, 3350}},
		{1, {250, 720, 1500, 3240}},
		{1, {278, 950, 1200, 2950}},
		{1, {250, 920, 1080, 2770}},
		{1, {262, 790, 1050, 2900}},
		{0, {250, 750, 1000, 2500}},
		{0, {275, 540, 1430, 3320}},
		{0, {263, 530, 1580, 3200}},
		{0, {295, 420, 1500, 3010}},
		{0, {260, 450, 1330, 2840}},
		{0, {272, 570, 1880, 2400}},
		{0, {255, 510, 1610, 1910}},
		{0, {235, 280, 2820, 3400}},
		{0, {244, 317, 3125, 3500}},
		{0, {230, 460, 2520, 3300}},
		{0, {212, 420, 2480, 3140}},
		{0, {235, 657, 2300, 3300}},
		{0, {232, 672, 2275, 3300}},
		{0, {231, 808, 1950, 3300}},
		{0, {225, 870, 2000, 3200}},
		{0, {236, 706, 1410, 3200}},
		{0, {211, 720, 1480, 2880}},
		{1, {250, 950, 1350, 3100}},
		{1, {227, 910, 1360, 2950}},
		{0, {203, 700, 1120, 3070}},
		{0, {230, 690,  920, 2760}},
		{0, {250, 475, 1250, 3150}},
		{0, {212, 460, 1210, 2750}},
		{0, {244, 403, 1100, 2950}},
		{0, {242, 363,  920, 2900}},
		{0, {226, 452, 1580, 1810}},
		{0, {232, 510, 1550, 1740}},
		{0, {230, 280, 3140, 3830}},
		{0, {250, 300, 3400, 3950}},
		{1, {225, 450, 2700, 3650}},
		{1, {250, 400, 2840, 3700}},
		{0, {215, 580, 2650, 3550}},
		{1, {220, 620, 2660, 3770}},
		{0, {240, 910, 2370, 3160}},
		{0, {233, 930, 2350, 3450}},
		{0, {250, 770, 1650, 3420}},
		{0, {230, 690, 1600, 3350}},
		{1, {242, 970, 1450, 3260}},
		{1, {225, 1010, 1650, 3150}},
		{0, {232, 670, 1160, 3550}},
		{0, {225, 720, 1260, 3400}},
		{0, {216, 500, 1640, 3580}},
		{0, {250, 450, 1440, 3500}},
		{0, {290, 350, 1160, 3260}},
		{0, {273, 330, 1090, 3350}},
		{0, {240, 430, 1800, 2400}},
		{0, {233, 470, 1840, 2400}},
		{0, {275, 330, 3050, 3800}},
		{0, {286, 340, 2860, 3610}},
		{0, {280, 500, 2720, 3360}},
		{0, {230, 600, 2750, 3550}},
		{0, {245, 735, 2450, 3300}},
		{0, {258, 780, 2560, 3300}},
		{0, {235, 940, 2020, 2580}},
		{0, {232, 1070, 2320, 2900}},
		{0, {268, 860, 1530, 3100}},
		{0, {256, 970, 1500, 3050}},
		{1, {245, 780, 1250, 3180}},
		{1, {236, 970,  970, 3120}},
		{1, {258, 825, 1210, 3100}},
		{1, {300, 930,  930, 2900}},
		{0, {260, 490, 1460, 2860}},
		{0, {286, 570, 1320, 2840}},
		{0, {275, 470, 1400, 2800}},
		{0, {286, 370, 1160, 2800}},
		{0, {268, 510, 1660, 2100}},
		{0, {250, 480, 1700, 1830}},
		{0, {295, 380, 3200, 4000}},
		{0, {267, 350, 3250, 3700}},
		{0, {294, 380, 2960, 3800}},
		{0, {300, 520, 2900, 3600}},
		{0, {280, 670, 2790, 3600}},
		{1, {275, 620, 2750, 3500}},
		{0, {262, 1070, 2380, 3100}},
		{0, {275, 1130, 2320, 3110}},
		{1, {290, 700, 1730, 2960}},
		{0, {270, 725, 1570, 2900}},
		{0, {278, 1110, 1630, 2780}},
		{0, {280, 1130, 1400, 3000}},
		{1, {292, 580,  930, 2950}},
		{0, {270, 540, 1070, 3000}},
		{0, {300, 450, 1350, 3000}},
		{0, {320, 520, 1600, 3150}},
		{0, {307, 460, 1460, 3070}},
		{0, {300, 400, 1700, 3000}},
		{0, {300, 540, 1770, 2040}},
		{0, {286, 540, 2050, 2300}},
		{0, {300, 300, 3250, 3850}},
		{0, {275, 275, 3280, 3800}},
		{0, {286, 570, 2850, 3400}},
		{0, {267, 485, 2630, 3450}},
		{0, {264, 650, 2880, 3500}},
		{0, {284, 570, 2900, 3600}},
		{0, {260, 1300, 2280, 3130}},
		{0, {260, 1300, 2160, 3300}},
		{0, {275, 850, 1540, 3020}},
		{0, {262, 840, 1580, 2880}},
		{0, {250, 1230, 1300, 3200}},
		{0, {286, 1090, 1230, 2980}},
		{0, {265, 850,  850, 2920}},
		{1, {285, 685, 1030, 2900}},
		{0, {283, 540, 1420, 3050}},
		{0, {300, 600, 1440, 2900}},
		{0, {280, 390, 1340, 2830}},
		{0, {284, 340, 1110, 3080}},
		{0, {280, 530, 1650, 1740}},
		{0, {286, 550, 1660, 1770}},
		{0, {265, 370, 2950, 3400}},
		{0, {290, 370, 2910, 3480}},
		{0, {271, 515, 2740, 3280}},
		{0, {290, 485, 2600, 3200}},
		{0, {262, 630, 2520, 3150}},
		{0, {272, 565, 2440, 3120}},
		{0, {262, 970, 2030, 2880}},
		{0, {275, 915, 2130, 2900}},
		{0, {270, 810, 1600, 3230}},
		{1, {280, 760, 1530, 3180}},
		{1, {270, 810, 1350, 2940}},
		{0, {275, 1000, 1360, 3000}},
		{1, {270, 535,  970, 2960}},
		{1, {275, 550, 1080, 2850}},
		{1, {275, 550, 1420, 3040}},
		{0, {295, 570, 1500, 3000}},
		{0, {283, 510, 1700, 3020}},
		{1, {278, 500, 1640, 3050}},
		{0, {261, 522, 1830, 2350}},
		{0, {282, 530, 1800, 2250}},
		{0, {320, 350, 3240, 3760}},
		{0, {344, 344, 3120, 3640}},
		{0, {308, 590, 2760, 3500}},
		{0, {320, 540, 2900, 3500}},
		{0, {307, 830, 2750, 3650}},
		{0, {308, 800, 2640, 3540}},
		{0, {294, 1140, 2450, 3230}},
		{0, {239, 1130, 2550, 3150}},
		{0, {310, 930, 1540, 3120}},
		{0, {315, 950, 1670, 3150}},
		{0, {350, 1190, 1470, 3150}},
		{1, {314, 1070, 1460, 2950}},
		{0, {300, 910, 1200, 3180}},
		{0, {330, 830, 1250, 3250}},
		{0, {327, 630, 1310, 3270}},
		{0, {322, 610, 1550, 3400}},
		{0, {345, 520, 1250, 3460}},
		{0, {334, 500, 1140, 3380}},
		{0, {308, 740, 1850, 2160}},
		{0, {328, 660, 1830, 2200}}
	};

	try {
		autoTable me = Table_create (nrows, ncols);

		for (integer i = 1; i <= nrows; i ++) {
			const TableRow row = my rows.at [i];
			const integer vowel_id = ( (i - 1) % 20) / 2 + 1;	/* 1 - 10 */
			const integer speaker_id = (i - 1) / 20 + 1;		/* 1 - 76 */
			integer speaker_type, speaker_sex;

			if (speaker_id <= 33) { /* 33 men */
				speaker_type = 0;
				speaker_sex = 0;
			} else if (speaker_id <= (33 + 28)) { /* 28 women */
				speaker_type = 1;
				speaker_sex = 1;
			} else { /*15  children */
				speaker_type = 2;
				speaker_sex = 0;
				if (speaker_id == 62 || speaker_id == 63 ||
					(speaker_id >= 65 && speaker_id <= 68) ||
					speaker_id == 73 || speaker_id == 76)
					speaker_sex = 1;
			}

			row -> cells [1]. string = Melder_dup (type [speaker_type]);
			row -> cells [2]. string = Melder_dup (sex [speaker_sex]);
			row -> cells [3]. string = Melder_dup (Melder_integer (speaker_id));
			row -> cells [4]. string = Melder_dup (vowel [vowel_id - 1]);
			row -> cells [5]. string = Melder_dup (ipa [vowel_id - 1]);
			for (integer j = 0; j <= 3; j ++)
				row -> cells [j + 6]. string = Melder_dup (Melder_integer (pbdata [i - 1].f [j]));
		}
		for (integer j = 1; j <= ncols; j ++) {
			Table_setColumnLabel (me.get(), j, columnLabels [j - 1]);
			my columnHeaders [j]. numericized = false;
		}
		return me;
	} catch (MelderError) {
		Melder_throw (U"Table not created from Peterson & Barney data.");
	}
}

autoTable Table_create_polsVanNierop1973 () {
	const integer nrows = 900, ncols = 10;
	const conststring32 columnLabels [10] = {U"Sex", U"Speaker", U"Vowel", U"IPA", U"F1", U"F2", U"F3", U"L1", U"L2", U"L3"};
	const conststring32 vowel [12] = {U"oe", U"aa", U"oo", U"a", U"eu", U"ie", U"uu", U"ee", U"u", U"e", U"o", U"i"};
	const conststring32 ipa [12] = {U"u", U"a", U"o", U"\\as", U"\\o/", U"i", U"y", U"e", U"\\yc", U"\\ep", U"\\ct", U"\\ic"};
	const conststring32 sex [2] = {U"m", U"f"};
	const struct polsdatum {
		short f [3]; // frequency F1, F2, F3
		short l [3];	// level f1, f2, f3
	} polsdata [] = {
		/* 50*12 males */
		/* male 1 */
		{{320,  630,  2560},  {6,  13,  48}}, 	// poet
		{{780, 1300,  2460},  {6,   8,  30}},	// paat
		{{500,  940,  2420},  {3,  12,  35}},	// poot
		{{720, 1060,  2420},  {3,   8,  27}},	// pat
		{{430, 1580,  2260},  {2,  24,  36}},	// peut
		{{280, 2300,  2780}, {14,  22,  27}},	// piet
		{{320, 1680,  2140},  {6,  23,  30}},	// puut
		{{420, 2000,  2620},  {5,  20,  23}},	// peet
		{{420, 1540,  2380},  {4,  19,  24}},	// put
		{{600, 1720,  2700},  {3,  17,  29}},	// pet
		{{520, 1000,  2520},  {4,  13,  31}},	// pot
		{{350, 2000,  2520},  {7,  19,  18}},	// pit
		/* male 2 */
		{{440,  780,  2600},  {7,  20,  35}},
		{{940, 1300,  2780},  {5,  13,  26}},
		{{500,  740,  2700},  {7,  11,  32}},
		{{800, 1000,  2480}, {11,  11,  31}},
		{{460, 1500,  2300}, {10,  20,  20}},
		{{320, 2400,  3040},  {9,  28,  27}},
		{{340, 1600,  2050},  {7,  30,  32}},
		{{420, 2200,  2650},  {4,  10,  12}},
		{{540, 1370,  2320},  {6,  23,  31}},
		{{760, 1660,  2600},  {8,  13,  18}},
		{{560,  800,  2800},  {5,  17,  37}},
		{{430, 2030,  2660},  {7,  18,  17}},
		/* male 3 */
		{{280,  740,  2160},  {8,  32,  46}},
		{{860, 1500,  2580},  {7,  13,  19}},
		{{480,  820,  2280},  {4,  11,  33}},
		{{680, 1020,  2460},  {9,  14,  25}},
		{{360, 1520,  2080},  {5,  16,  21}},
		{{270, 2040,  2860},  {5,  18,  18}},
		{{280, 1600,  1900},  {5,  16,  21}},
		{{380, 1940,  2580},  {5,  15,  17}},
		{{400, 1520,  2120},  {4,  21,  24}},
		{{560, 1840,  2520},  {7,  18,  22}},
		{{500,  820,  2520},  {3,  11,  30}},
		{{350, 2000,  2660},  {4,  15,  19}},
		/* male 4 */
		{{360,  820,  2220},  {6,  11,  30}},
		{{840, 1300,  2280},  {9,   7,  22}},
		{{500,  900,  2320},  {7,  13,  35}},
		{{680, 1000,  2480}, {12,  12,  27}},
		{{440, 1320,  2060},  {5,  12,  27}},
		{{240, 2060,  2580},  {5,  20,  23}},
		{{300, 1540,  2020},  {4,  18,  24}},
		{{460, 1920,  2460},  {7,  18,  17}},
		{{480, 1320,  2200},  {6,  15,  22}},
		{{660, 1660,  2340},  {7,  16,  21}},
		{{500,  800,  2520},  {7,  14,  34}},
		{{440, 1920,  2380},  {3,  19,  18}},
		/* male 5 */
		{{440,  880,  2300},  {6,  22,  41}},
		{{820, 1420,  2180},  {6,   7,  28}},
		{{540,  960,  2460},  {4,  10,  28}},
		{{780, 1040,  2620},  {8,  10,  32}},
		{{540, 1540,  2160},  {2,  17,  19}},
		{{300, 2300,  2900},  {5,  23,  30}},
		{{360, 1860,  2200},  {2,  15,  15}},
		{{520, 1960,  2400},  {4,  21,  21}},
		{{560, 1440,  2280},  {2,  17,  27}},
		{{700, 1720,  2300},  {7,  13,  22}},
		{{600,  880,  3000},  {5,   5,  25}},
		{{560, 1920,  2400},  {5,  22,  25}},
		/* male 6 */
		{{260,  700,  2550},  {3,  24,  45}},
		{{820, 1460,  2760}, {10,  15,  27}},
		{{450,  900,  2460}, {19,  20,  45}},
		{{700, 1080,  2660}, {13,  22,  32}},
		{{460, 1750,  2300},  {7,  32,  40}},
		{{240, 2500,  3000},  {3,  33,  37}},
		{{260, 2100,  2500}, {10,  37,  43}},
		{{300, 2320,  2860},  {8,  26,  26}},
		{{440, 1700,  2660},  {7,  27,  32}},
		{{560, 2080,  2840}, {12,  15,  16}},
		{{550,  900,  2740},  {7,  25,  30}},
		{{340, 2340,  3000},  {8,  31,  31}},
		/* male 7 */
		{{280,  860,  2340},  {5,  15,  33}},
		{{800, 1320,  2540},  {7,  14,  26}},
		{{520,  920,  2600},  {8,  15,  33}},
		{{600, 1000,  2760},  {7,   7,  25}},
		{{450, 1660,  2260},  {7,  20,  24}},
		{{260, 2340,  2640},  {3,  17,  20}},
		{{280, 1780,  2160},  {3,  25,  29}},
		{{400, 2040,  2400},  {9,  19,  21}},
		{{460, 1560,  2400},  {4,  19,  22}},
		{{620, 1760,  2560},  {6,  18,  25}},
		{{560,  960,  2760},  {9,  16,  33}},
		{{340, 2000,  2600},  {6,  14,  18}},
		/* male 8 */
		{{320,  880,  2200},  {6,  16,  40}},
		{{800, 1160,  2600},  {6,  13,  26}},
		{{560,  980,  2360},  {5,   8,  35}},
		{{700, 1080,  2540}, {10,  14,  34}},
		{{500, 1480,  2300},  {8,  17,  29}},
		{{280, 2080,  2620},  {3,  27,  27}},
		{{320, 1760,  2060},  {4,  25,  32}},
		{{400, 1940,  2540},  {9,  17,  22}},
		{{400, 1560,  2280},  {6,  18,  28}},
		{{540, 1860,  2540},  {4,  14,  19}},
		{{560,  920,  2320},  {8,  13,  32}},
		{{340, 1960,  2480},  {6,  13,  17}},
		/* male 9 */
		{{300,  680,  2400},  {3,  19,  32}},
		{{860, 1300,  2660}, {12,  18,  26}},
		{{500,  940,  2500},  {3,  13,  33}},
		{{700, 1120,  2620},  {9,  17,  24}},
		{{500, 1500,  2280},  {3,  17,  22}},
		{{300, 2380,  2960},  {2,  20,  22}},
		{{300, 1760,  2160},  {1,  19,  23}},
		{{480, 2100,  2580},  {4,  15,  15}},
		{{500, 1580,  2400},  {5,  12,  22}},
		{{640, 1700,  2620},  {3,  15,  19}},
		{{560,  900,  2940},  {4,   8,  31}},
		{{400, 2040,  2600},  {7,  17,  19}},
		/* male 10 */
		{{360,  900,  2220}, {11,  21,  46}},
		{{880, 1400,  2660},  {9,  17,  29}},
		{{460,  940,  2400},  {3,  13,  37}},
		{{660, 1040,  2660},  {5,   5,  31}},
		{{460, 1580,  2360},  {4,  22,  26}},
		{{340, 2200,  2920}, {14,  30,  26}},
		{{400, 1880,  2800}, {12,  19,  33}},
		{{460, 2080,  2800},  {4,  18,  17}},
		{{460, 1480,  2260},  {5,  27,  31}},
		{{600, 1860,  2640},  {7,  18,  23}},
		{{520,  860,  2880},  {2,  24,  32}},
		{{460, 2100,  2800},  {6,  19,  22}},
		/* male 11 */
		{{320,  830,  2060},  {5,  16,  45}},
		{{820, 1340,  2200},  {7,   9,  26}},
		{{520,  840,  2040},  {7,  13,  31}},
		{{660, 1060,  2300},  {8,   9,  28}},
		{{440, 1520,  2040},  {7,  16,  22}},
		{{300, 2100,  2600},  {5,  23,  30}},
		{{300, 1740,  2040},  {2,  17,  19}},
		{{340, 2040,  2460},  {5,  20,  24}},
		{{500, 1440,  2200},  {8,  17,  21}},
		{{600, 1760,  2380}, {10,  15,  19}},
		{{560,  900,  2300},  {9,  10,  28}},
		{{300, 1960,  2400},  {7,  22,  24}},
		/* male 12 */
		{{400,  860,  2700},  {7,  15,  46}},
		{{940, 1520,  3040}, {10,  19,  33}},
		{{580, 1040,  2960},  {5,  17,  42}},
		{{860, 1280,  3000},  {8,  19,  34}},
		{{480, 1600,  2620},  {7,  15,  32}},
		{{300, 2500,  2880},  {3,  29,  29}},
		{{300, 1670,  2350},  {3,  31,  39}},
		{{480, 2220,  2640},  {7,  30,  31}},
		{{380, 1600,  2540},  {4,  22,  33}},
		{{630, 2140,  2880},  {7,  27,  27}},
		{{640,  900,  3000},  {8,  13,  39}},
		{{360, 2220,  2780},  {5,  23,  31}},
		/* male 13 */
		{{260,  780,  2460},  {5,  20,  44}},
		{{900, 1560,  2860},  {9,  13,  26}},
		{{440,  850,  2600},  {3,  17,  42}},
		{{860, 1140,  2820}, {10,  12,  30}},
		{{460, 1580,  2400},  {3,  19,  27}},
		{{260, 2560,  3240},  {3,  28,  34}},
		{{300, 1960,  2500},  {2,  31,  29}},
		{{460, 2320,  2960},  {6,  25,  27}},
		{{460, 1600,  2460},  {6,  22,  30}},
		{{680, 2100,  2940}, {14,  19,  25}},
		{{540,  800,  2740},  {7,  13,  36}},
		{{380, 2500,  2980},  {7,  20,  23}},
		/* male 14 */
		{{340,  720,  2500},  {6,  16,  47}},
		{{900, 1500,  3020}, {10,  13,  33}},
		{{450,  900,  2700},  {4,  16,  42}},
		{{600, 1000,  2720}, {12,  10,  31}},
		{{420, 1740,  2560},  {8,  21,  25}},
		{{360, 2500,  3000},  {8,  10,  13}},
		{{360, 1900,  2420},  {6,  16,  12}},
		{{380, 1780,  2420},  {4,  36,  18}},
		{{500, 1640,  2620},  {4,  23,  29}},
		{{600, 1940,  2700},  {3,  16,  19}},
		{{500,  800,  2800},  {3,  10,  38}},
		{{400, 2360,  2740},  {7,  13,  27}},
		/* male 15 */
		{{360,  780,  2320},  {5,  23,  50}},
		{{860, 1420,  2420},  {8,  15,  35}},
		{{440,  840,  2480},  {9,  20,  38}},
		{{660,  980,  2500},  {6,  13,  33}},
		{{460, 1660,  2200},  {2,  21,  31}},
		{{300, 2360,  2960},  {7,  29,  27}},
		{{320, 1740,  2220},  {2,  28,  30}},
		{{400, 2240,  2560},  {5,  23,  24}},
		{{480, 1420,  2220},  {6,  23,  37}},
		{{680, 1640,  2340},  {9,  21,  28}},
		{{560,  860,  2780},  {4,  15,  44}},
		{{440, 2120,  2500},  {2,  22,  23}},
		/* male 16 */
		{{360,  760,  2300},  {7,  23,  52}},
		{{660, 1000,  2500},  {9,  15,  35}},
		{{500,  920,  2520},  {7,  15,  37}},
		{{780, 1060,  2380},  {5,   9,  38}},
		{{440, 1560,  2260},  {7,  25,  31}},
		{{280, 2200,  2880},  {7,  43,  38}},
		{{380, 1720,  2200},  {7,  29,  39}},
		{{360, 2140,  2620},  {3,  26,  28}},
		{{360, 1600,  2400},  {4,  26,  33}},
		{{520, 1800,  2480},  {8,  32,  35}},
		{{540,  920,  2640},  {6,  20,  44}},
		{{340, 2080,  2680},  {3,  28,  27}},
		/* male 17 */
		{{400,  820,  2200},  {5,  15,  48}},
		{{1100, 1480,  2260}, {10,  14,  30}},
		{{520,  940,  2560},  {5,  15,  43}},
		{{660,  940,  2820}, {11,  11,  35}},
		{{500, 1720,  2400},  {5,  19,  23}},
		{{360, 2300,  3260}, {11,  25,  17}},
		{{360, 2100,  2420}, {10,  19,  20}},
		{{440, 2360,  2860},  {6,  20,  25}},
		{{500, 1760,  2600},  {6,  15,  25}},
		{{660, 1840,  2620},  {7,  17,  24}},
		{{540,  860,  2860},  {3,   9,  41}},
		{{400, 2440,  3000},  {5,  28,  30}},
		/* male 18 */
		{{360,  860,  2520},  {6,  15,  35}},
		{{740, 1300,  2660},  {7,   9,  22}},
		{{460,  800,  2620},  {3,  20,  35}},
		{{740, 1040,  2800}, {15,  13,  30}},
		{{440, 1400,  2200},  {4,  21,  25}},
		{{340, 2040,  2500},  {6,  18,  20}},
		{{340, 1340,  2040},  {3,  15,  25}},
		{{420, 1760,  2420},  {6,  18,  21}},
		{{460, 1380,  2200},  {6,  20,  22}},
		{{560, 1640,  2400},  {9,  13,  19}},
		{{540,  920,  2520},  {9,  10,  23}},
		{{400, 1800,  2400},  {8,  19,  21}},
		/* male 19 */
		{{320,  840,  2360},  {5,  20,  45}},
		{{780, 1140,  2740},  {8,  15,  36}},
		{{460, 1020,  2700},  {6,  11,  43}},
		{{800, 1100,  2720}, {15,  17,  42}},
		{{440, 1500,  2500},  {5,  23,  37}},
		{{260, 2000,  2680},  {4,  39,  40}},
		{{300, 1540,  2100},  {5,  33,  42}},
		{{400, 1900,  2680},  {6,  25,  34}},
		{{440, 1500,  2400},  {4,  26,  42}},
		{{600, 1700,  2640}, {18,  33,  40}},
		{{500,  800,  3000},  {5,  20,  47}},
		{{400, 2000,  2500},  {5,  27,  37}},
		/* male 20 */
		{{400,  960,  2400}, {10,  16,  37}},
		{{800, 1220,  2380},  {7,  13,  25}},
		{{500, 1080,  2500}, {10,  14,  31}},
		{{780, 1100,  2600}, {10,  11,  30}},
		{{400, 1480,  2380},  {8,  20,  25}},
		{{280, 2380,  2720},  {6,  23,  24}},
		{{300, 1760,  2220},  {5,  14,  22}},
		{{400, 2000,  2600},  {4,  20,  24}},
		{{440, 1500,  2440},  {4,  15,  20}},
		{{440, 1800,  2620}, {12,  18,  22}},
		{{460,  860,  2600}, {12,  13,  33}},
		{{400, 2040,  2640},  {3,  18,  23}},
		/* male 21 */
		{{300,  700,  2100},  {3,  15,  37}},
		{{800, 1100,  2300}, {11,  15,  30}},
		{{420,  700,  2440},  {5,  12,  33}},
		{{660,  920,  2520},  {6,   9,  32}},
		{{400, 1300,  2000},  {2,  22,  30}},
		{{300, 1940,  2620},  {5,  25,  24}},
		{{260, 1920,  2900},  {4,  21,  31}},
		{{300, 1900,  2340},  {4,  20,  24}},
		{{400, 1200,  2000},  {4,  20,  29}},
		{{540, 1500,  2280},  {4,  15,  22}},
		{{400,  740,  2580},  {3,  11,  34}},
		{{300, 1900,  2380},  {2,  20,  20}},
		/* male 22 */
		{{400,  780,  2500},  {6,  14,  40}},
		{{760, 1260,  2620},  {6,  14,  30}},
		{{540,  860,  2600},  {4,   7,  34}},
		{{660, 1100,  2460},  {9,  12,  30}},
		{{540, 1460,  2260},  {5,  12,  20}},
		{{300, 2300,  2800},  {7,  20,  17}},
		{{300, 1980,  2900},  {6,  14,  32}},
		{{420, 2100,  2600},  {4,  17,  21}},
		{{500, 1440,  2300},  {4,  18,  26}},
		{{540, 1900,  2600}, {10,  20,  22}},
		{{550,  900,  3000},  {5,  11,  34}},
		{{300, 2200,  2700},  {6,  14,  20}},
		/* male 23 */
		{{360,  900,  2140},  {3,  14,  37}},
		{{800, 1250,  2650}, {12,  13,  26}},
		{{520,  960,  2200},  {5,   7,  28}},
		{{760, 1120,  2700}, {12,  10,  29}},
		{{400, 1500,  2160},  {7,  20,  30}},
		{{300, 2260,  3000},  {4,  18,  17}},
		{{320, 1800,  2500},  {6,  17,  32}},
		{{460, 2020,  2800},  {3,  10,  16}},
		{{500, 1500,  2300},  {7,  15,  23}},
		{{640, 1600,  2500},  {8,  22,  32}},
		{{550,  940,  2420},  {7,  12,  40}},
		{{460, 2100,  2880},  {2,  21,  24}},
		/* male 24 */
		{{360,  860,  2460},  {4,  15,  33}},
		{{840, 1400,  2500},  {5,   8,  20}},
		{{460,  900,  2520},  {1,   8,  31}},
		{{620, 1020,  2770},  {7,   6,  26}},
		{{410, 1460,  2360},  {2,  11,  18}},
		{{270, 2140,  2580},  {2,  15,  21}},
		{{300, 1870,  2300},  {1,  11,  16}},
		{{360, 2000,  2520},  {2,  17,  19}},
		{{400, 1520,  2400},  {2,  12,  17}},
		{{600, 1600,  2580},  {3,  12,  20}},
		{{500,  900,  2700},  {2,  11,  33}},
		{{360, 1940,  2550},  {1,  19,  21}},
		/* male 25 */
		{{360,  860,  2200},  {8,  22,  46}},
		{{880, 1240,  2400},  {7,  14,  25}},
		{{460,  920,  3300},  {2,  11,  40}},
		{{600, 1000,  2600},  {4,   7,  32}},
		{{400, 1440,  2160},  {2,  31,  37}},
		{{360, 2240,  2760},  {6,  23,  21}},
		{{380, 1660,  2000}, {13,  16,  26}},
		{{460, 2000,  2520},  {5,  23,  32}},
		{{460, 1500,  2300},  {7,  17,  25}},
		{{540, 1700,  2460}, {10,  22,  35}},
		{{540, 1000,  2600},  {8,  13,  40}},
		{{340, 2040,  2580},  {9,  27,  30}},
		/* male 26 */
		{{400,  800,  2500},  {6,  23,  42}},
		{{960, 1300,  2640},  {9,   8,  29}},
		{{460,  860,  2460},  {7,  16,  39}},
		{{740, 1140,  2400},  {9,  10,  30}},
		{{400, 1600,  2400},  {6,  21,  25}},
		{{360, 2500,  2840}, {10,  19,  20}},
		{{360, 1800,  2400},  {8,  21,  27}},
		{{360, 2080,  2680},  {5,  14,  16}},
		{{400, 1620,  2440},  {5,  15,  21}},
		{{600, 1940,  2600},  {5,  13,  23}},
		{{560,  980,  2900},  {5,  13,  32}},
		{{400, 2060,  2540},  {5,  21,  21}},
		/* male 27 */
		{{300,  900,  2300},  {3,  15,  38}},
		{{780, 1300,  2400},  {9,  18,  32}},
		{{550, 1000,  2480},  {5,  10,  35}},
		{{680, 1050,  2550},  {5,  10,  35}},
		{{520, 1480,  2400},  {5,  16,  27}},
		{{260, 2180,  2560},  {1,  30,  30}},
		{{250, 1720,  2220},  {1,  26,  32}},
		{{360, 2100,  2650},  {4,  31,  25}},
		{{440, 1440,  2440},  {4,  21,  26}},
		{{600, 1600,  2500},  {5,  15,  27}},
		{{560,  950,  2700},  {3,  10,  40}},
		{{360, 1900,  2600},  {3,  26,  31}},
		/* male 28 */
		{{280,  740,  2500},  {3,  20,  45}},
		{{780, 1300,  2840},  {4,  12,  25}},
		{{440,  860,  2860},  {5,  13,  41}},
		{{440,  700,  3040},  {9,  10,  37}},
		{{450, 1520,  2320},  {2,  25,  30}},
		{{220, 2340,  2960},  {1,  35,  36}},
		{{240, 1800,  2140},  {3,  36,  37}},
		{{300, 2200,  2600},  {4,  28,  33}},
		{{440, 1500,  2480},  {3,  26,  37}},
		{{500, 1660,  2620}, {10,  30,  38}},
		{{420,  700,  3000},  {4,  12,  35}},
		{{300, 2140,  2760},  {6,  32,  31}},
		/* male 29 */
		{{340,  660,  2320},  {4,  13,  37}},
		{{640, 1250,  2480}, {10,  16,  29}},
		{{560, 1000,  2480}, {10,  14,  31}},
		{{720, 1150,  2600}, {10,  13,  26}},
		{{480, 1400,  2160},  {9,  22,  28}},
		{{300, 2040,  2640},  {5,  20,  22}},
		{{280, 1540,  1960},  {3,  21,  23}},
		{{460, 1760,  2320},  {7,  19,  20}},
		{{440, 1550,  2200},  {8,  16,  23}},
		{{480, 1660,  1960}, {10,  16,  23}},
		{{480,  840,  2840},  {9,  12,  28}},
		{{400, 1780,  2360},  {7,  20,  23}},
		/* male 30 */
		{{360,  800,  2540},  {1,  11,  40}},
		{{600, 1300,  2600},  {6,   8,  27}},
		{{500,  860,  2440},  {2,   7,  36}},
		{{750, 1140,  2640},  {4,   9,  30}},
		{{460, 1400,  2340},  {1,  23,  28}},
		{{340, 2300,  2620},  {2,  26,  25}},
		{{300, 1540,  2300},  {5,  25,  35}},
		{{440, 2000,  2540},  {1,  14,  19}},
		{{440, 1360,  2360},  {1,  19,  26}},
		{{620, 1840,  2560},  {3,  18,  23}},
		{{520,  820,  2680},  {2,   7,  34}},
		{{420, 2000,  2640},  {1,  20,  25}}, /*L2 (10) corrected 20021211 */
		/* male 31 */
		{{340,  740,  2240},  {5,  15,  44}},
		{{820, 1200,  2250},  {5,  17,  40}},
		{{440,  820,  2540},  {3,  13,  37}},
		{{760, 1060,  2340},  {8,  15,  42}},
		{{460, 1540,  2380},  {3,  21,  25}},
		{{280, 2260,  2620},  {8,  31,  32}},
		{{300, 1800,  2220},  {6,  36,  37}},
		{{460, 1900,  2260},  {3,  34,  31}},
		{{380, 1540,  2400}, {15,  36,  40}},
		{{500, 1740,  2400},  {4,  26,  39}},
		{{460,  840,  2580},  {7,  28,  35}},
		{{320, 2100,  2460},  {7,  30,  27}},
		/* male 32 */
		{{360,  900,  2200},  {5,  18,  39}},
		{{640, 1280,  2340},  {7,  15,  26}},
		{{460,  920,  2360},  {7,  14,  33}},
		{{720, 1200,  2580}, {11,  14,  26}},
		{{420, 1520,  2260},  {6,  17,  23}},
		{{400, 2000,  2560},  {5,  15,  20}},
		{{380, 1700,  2100},  {6,  18,  21}},
		{{440, 1740,  2420},  {4,  14,  17}},
		{{500, 1520,  2440},  {5,  14,  21}},
		{{580, 1540,  2460},  {5,  12,  24}},
		{{580, 1020,  2700},  {9,  13,  30}},
		{{460, 1720,  2400},  {7,  22,  20}},
		/* male 33 */
		{{400,  700,  2600},  {4,  17,  45}},
		{{900, 1440,  2600}, {10,  17,  36}},
		{{460,  860,  2600},  {7,  18,  45}},
		{{680, 1000,  2200},  {7,  13,  39}},
		{{460, 1600,  2540},  {5,  28,  37}},
		{{300, 2260,  2880},  {7,  28,  26}},
		{{320, 1860,  2200},  {7,  22,  28}},
		{{440, 2180,  2660},  {3,  24,  28}},
		{{380, 1560,  2360},  {8,  26,  33}},
		{{620, 1720,  2060},  {6,  21,  28}},
		{{600,  860,  2900}, {12,  18,  37}},
		{{440, 2040,  2600},  {4,  26,  30}},
		/* male 34 */
		{{370,  900,  2230},  {3,  17,  35}},
		{{700, 1200,  2580}, {12,  17,  30}},
		{{500,  840,  2460},  {4,  13,  37}},
		{{720, 1080,  2640},  {7,  13,  37}},
		{{440, 1300,  2220},  {4,  20,  31}},
		{{300, 2040,  2580},  {7,  25,  25}},
		{{320, 1540,  2080},  {8,  21,  23}},
		{{380, 1860,  2450},  {7,  24,  33}},
		{{460, 1200,  2360},  {3,  20,  30}},
		{{580, 1500,  2380},  {9,  22,  25}},
		{{480,  820,  2580},  {8,  15,  32}},
		{{400, 1800,  2360},  {6,  23,  26}},
		/* male 35 */
		{{280, 1040,  2340},  {7,  26,  41}},
		{{820, 1300,  2760}, {10,  15,  32}},
		{{440, 1220,  2580},  {6,  18,  29}},
		{{600, 1040,  2540},  {8,  13,  27}},
		{{420, 1560,  2480},  {6,  22,  26}},
		{{300, 2160,  2700},  {5,  29,  30}},
		{{250, 1760,  2320},  {9,  30,  38}},
		{{440, 1940,  2550},  {5,  25,  28}},
		{{400, 1600,  2460},  {8,  26,  29}},
		{{580, 1820,  2460},  {5,  23,  30}},
		{{460,  860,  2660},  {5,  21,  37}},
		{{400, 2100,  2640},  {8,  27,  27}},
		/* male 36 */
		{{360,  740,  2160},  {2,  21,  40}},
		{{660, 1260,  2540}, {10,  18,  21}},
		{{500,  900,  2600},  {9,  20,  30}},
		{{640, 1000,  2880}, {11,  17,  29}},
		{{460, 1300,  2140},  {8,  19,  25}},
		{{300, 1900,  2580}, {11,  18,  22}},
		{{320, 1660,  2060},  {9,  17,  20}},
		{{400, 1780,  2320},  {8,  20,  21}},
		{{380, 1360,  2200},  {6,  20,  25}},
		{{540, 1600,  2260},  {8,  21,  22}},
		{{540,  860,  2720},  {7,  20,  32}},
		{{400, 1740,  2340},  {5,  22,  23}},
		/* male 37 */
		{{300,  900,  2140},  {5,  21,  39}},
		{{700, 1240,  2460}, {10,  18,  28}},
		{{480,  960,  2140},  {4,  12,  32}},
		{{640, 1120,  2480},  {9,  14,  32}},
		{{460, 1520,  2160},  {4,  18,  25}},
		{{320, 2120,  2600}, {10,  20,  26}},
		{{320, 1800,  2200},  {8,  25,  27}},
		{{320, 1920,  2460},  {8,  21,  27}},
		{{480, 1460,  2260},  {5,  22,  27}},
		{{600, 1600,  2480},  {7,  17,  24}},
		{{500,  950,  2450},  {4,  14,  38}},
		{{460, 1820,  2480},  {6,  18,  26}},
		/* male 38 */
		{{320,  760,  2080}, {11,  23,  41}},
		{{840, 1180,  2700}, {13,  20,  37}},
		{{500,  920,  2400}, {13,  17,  37}},
		{{660, 1060,  2700}, {13,  17,  36}},
		{{440, 1400,  2220},  {5,  32,  37}},
		{{280, 2240,  2700}, {14,  29,  35}},
		{{300, 1640,  2080}, {12,  31,  31}},
		{{440, 2040,  2600}, {11,  19,  24}},
		{{400, 1460,  2160},  {8,  32,  38}},
		{{580, 1700,  1900}, {13,  26,  26}},
		{{500,  840,  2920}, {10,  18,  40}},
		{{360, 2060,  2440},  {7,  21,  27}},
		/* male 39 */
		{{320,  760,  2480},  {9,  21,  46}},
		{{700, 1420,  2680},  {9,  16,  31}},
		{{500,  940,  2500},  {4,  16,  41}},
		{{700, 1060,  2720},  {8,  10,  38}},
		{{440, 1580,  2260}, {11,  34,  39}},
		{{260, 2200,  2700},  {5,  29,  32}},
		{{200, 1600,  2060},  {7,  33,  34}},
		{{400, 2200,  2600}, {13,  29,  31}},
		{{380, 1500,  2220},  {8,  25,  37}},
		{{540, 1750,  2420},  {8,  23,  32}},
		{{520,  820,  2560}, {10,  19,  43}},
		{{400, 1700,  2320},  {3,  38,  23}},
		/* male 40 */
		{{300,  680,  1920},  {7,  28,  48}},
		{{740, 1200,  2550},  {8,  10,  24}},
		{{420,  860,  2420},  {7,  17,  37}},
		{{640, 1120,  2500}, {12,  17,  37}},
		{{360, 1500,  2180},  {3,  27,  35}},
		{{280, 2160,  2920},  {4,  27,  31}},
		{{260, 1560,  2050},  {2,  26,  27}},
		{{360, 2020,  2500},  {4,  26,  28}},
		{{440, 1400,  2320},  {4,  21,  32}},
		{{460, 1660,  2460},  {5,  19,  27}},
		{{500,  840,  2580},  {6,  14,  35}},
		{{360, 1920,  2560},  {3,  31,  31}},
		/* male 41 */
		{{360,  880,  2320},  {2,  12,  43}},
		{{840, 1200,  2500}, {12,  17,  37}},
		{{580, 1060,  2300}, {11,  10,  32}},
		{{580, 1100,  2680},  {8,  12,  33}},
		{{560, 1600,  2200},  {9,  22,  27}},
		{{300, 2260,  2800},  {5,  25,  26}},
		{{320, 1760,  2100},  {5,  23,  25}},
		{{500, 2020,  2660},  {5,  17,  21}},
		{{420, 1520,  2320},  {1,  20,  28}},
		{{700, 1800,  2620},  {8,  17,  22}},
		{{540,  860,  2720},  {7,  13,  35}},
		{{420, 2080,  2600},  {4,  21,  25}},
		/* male 42 */
		{{420,  800,  2400},  {5,  15,  40}},
		{{800, 1400,  2900},  {9,  16,  41}},
		{{420,  820,  2480},  {5,   8,  40}},
		{{600, 1200,  2760},  {6,  12,  34}},
		{{400, 1560,  2120},  {2,  20,  31}},
		{{320, 2360,  2820},  {8,  25,  27}},
		{{340, 1680,  2240},  {9,  19,  35}},
		{{400, 2180,  2760},  {2,  19,  20}},
		{{400, 1440,  2360},  {3,  15,  26}},
		{{700, 1700,  2340}, {11,  18,  29}},
		{{500,  780,  2840},  {7,  14,  38}},
		{{380, 2120,  2720},  {2,  21,  25}},
		/* male 43 */
		{{300,  760,  2020},  {3,  16,  38}},
		{{740, 1200,  2360},  {8,  15,  29}},
		{{460,  860,  2200},  {8,  12,  39}},
		{{620,  900,  2500},  {8,  12,  27}},
		{{400, 1340,  2100},  {7,  20,  31}},
		{{240, 2000,  2340},  {2,  22,  28}},
		{{240, 1580,  1860},  {3,  16,  24}},
		{{360, 1640,  2080},  {5,  19,  26}},
		{{400, 1340,  2060},  {3,  16,  28}},
		{{580, 1400,  2120},  {6,  13,  24}},
		{{500,  800,  2460},  {6,   6,  31}},
		{{440, 1720,  2100},  {7,  19,  24}},
		/* male 44 */
		{{260,  800,  2400},  {3,  16,  48}},
		{{780, 1300,  2700},  {6,  14,  28}},
		{{480,  900,  2500},  {5,   8,  35}},
		{{620, 1000,  2820},  {5,   9,  28}},
		{{420, 1400,  2300},  {3,  18,  29}},
		{{240, 2040,  2680},  {1,  31,  28}},
		{{260, 1580,  2260},  {3,  31,  28}},
		{{380, 2000,  2600},  {5,  29,  26}},
		{{420, 1420,  2400},  {2,  21,  26}},
		{{540, 1640,  2440},  {5,  19,  26}},
		{{480,  840,  2800},  {6,  13,  32}},
		{{280, 1960,  2560},  {5,  27,  28}},
		/* male 45 */
		{{300,  840,  3060},  {3,  10,  38}},
		{{800, 1220,  2280},  {6,  10,  26}},
		{{500,  920,  2120},  {6,   8,  31}},
		{{700, 1020,  2600},  {3,  11,  28}},
		{{400, 1260,  2020},  {6,  17,  24}},
		{{260, 1960,  2440},  {1,  22,  22}},
		{{300, 1480,  1940},  {2,  18,  22}},
		{{440, 1880,  2380},  {6,  17,  17}},
		{{320, 1400,  2140},  {5,  18,  27}},
		{{500, 1560,  2300},  {7,  18,  22}},
		{{540,  780,  2400},  {8,  13,  34}},
		{{360, 1860,  2300},  {4,  20,  21}},
		/* male 46 */
		{{320,  860,  2380},  {3,  19,  41}},
		{{660, 1400,  2540}, {11,  20,  27}},
		{{520,  940,  2580},  {7,  11,  34}},
		{{700, 1040,  2720},  {4,   8,  23}},
		{{400, 1600,  2280},  {2,  27,  29}},
		{{320, 2340,  3140},  {3,  33,  29}},
		{{300, 1860,  2160},  {2,  25,  25}},
		{{420, 2200,  2760},  {1,  17,  23}},
		{{460, 2320,  3360},  {2,  33,  37}},
		{{500, 2100,  2760},  {6,  23,  38}},
		{{600,  920,  2700},  {8,  17,  29}},
		{{420, 2200,  2740},  {3,  30,  32}},
		/* male 47 */
		{{360,  800,  2120},  {3,  18,  33}},
		{{700, 1220,  2760},  {6,  13,  27}},
		{{540,  940,  2640},  {2,   8,  33}},
		{{620, 1080,  2800},  {4,  10,  33}},
		{{500, 1400,  2200},  {6,  18,  25}},
		{{320, 2240,  2940},  {2,  22,  27}},
		{{320, 1800,  2100},  {1,  26,  27}},
		{{420, 2040,  2400},  {3,  19,  24}},
		{{460, 1440,  2140},  {1,  13,  25}},
		{{600, 1600,  2520},  {6,  13,  27}},
		{{560,  700,  2780},  {5,  11,  31}},
		{{440, 1920,  2560},  {3,  22,  24}},
		/* male 48 */
		{{300,  760,  1900},  {3,  17,  42}},
		{{800, 1260,  2740},  {7,  10,  28}},
		{{460,  840,  1840},  {4,  13,  38}},
		{{540,  900,  2400}, {10,  14,  28}},
		{{420, 1380,  2100},  {2,  16,  29}},
		{{220, 2080,  2900},  {2,  28,  21}},
		{{220, 1760,  2120},  {1,  22,  25}},
		{{440, 2060,  2780},  {1,  19,  21}},
		{{440, 1440,  2560},  {3,  19,  31}},
		{{580, 1400,  2100},  {5,  15,  22}},
		{{520,  900,  2300},  {3,  10,  32}},
		{{420, 1720,  2720},  {6,  22,  21}},
		/* male 49 */
		{{320, 1000,  2220},  {3,  24,  43}},
		{{700, 1280,  2500},  {3,  13,  30}},
		{{460, 1060,  2380},  {4,  13,  31}},
		{{620, 1100,  2840}, {10,  18,  33}},
		{{340, 1440,  2260},  {4,  21,  30}},
		{{280, 2140,  2580},  {3,  31,  32}},
		{{280, 1820,  2220},  {2,  36,  35}},
		{{340, 2100,  2500},  {2,  29,  31}},
		{{380, 1460,  2400},  {2,  26,  38}},
		{{500, 1640,  2500},  {7,  27,  31}},
		{{500,  960,  2720},  {4,  17,  31}},
		{{420, 1960,  2700},  {1,  32,  32}},
		/* male 50 */
		{{340,  780,  2020}, {11,  22,  36}},
		{{660, 1220,  2500},  {9,  14,  22}},
		{{420,  760,  2440},  {2,  17,  33}},
		{{560, 1000,  2600},  {6,  13,  25}},
		{{400, 1320,  2120},  {8,  18,  21}},
		{{300, 1860,  2440},  {6,  22,  22}},
		{{280, 1600,  1900},  {6,  16,  19}},
		{{340, 1740,  2260},  {3,  12,  17}},
		{{400, 1360,  2160},  {4,  16,  22}},
		{{520, 1580,  2240},  {2,  12,  16}},
		{{380,  800,  2560},  {7,  11,  25}},
		{{360, 1740,  2260},  {5,  14,  17}},
		/* 25*12 females */
		{{250,  800, 2450},  {0,  8, 45}},	// poet
		{{950, 1500, 2650},  {5, 14, 30}},	// paat
		{{500, 1050, 2600},  {3,  5, 38}},	// poot
		{{720, 1100, 2950},  {8,  2, 24}},	// pat
		{{500, 1800, 2500},  {6, 14, 30}},	// peut
		{{280, 2500, 3100},  {0, 32, 26}},	// piet
		{{250, 1700, 2200},  {0, 18, 21}},	// puut
		{{500, 2350, 2750},  {2, 12, 12}},	// peet
		{{520, 1550, 2400},  {4, 15, 27}},	// put
		{{750, 2000, 2600},  {4, 20, 20}},	// pet
		{{550,  900, 2800},  {6,  3, 34}},	// pot
		{{480, 2150, 2650},  {5, 20, 22}},	// pit
		/* female 2 */
		{{300,  750, 2700},  {0, 10, 50}},
		{{1100, 1500, 3000},  {6,  9, 28}}, // djmw 20021212 L3 (was 20)
		{{520,  900, 2800},  {2,  8, 30}},
		{{800, 1150, 3000},  {2, 12, 34}}, // djmw 20021212 F3 (was 300 in Van Nierop data!)
		{{450, 1600, 2950},  {2, 22, 31}},
		{{250, 2700, 3300},  {0, 27, 31}},
		{{300, 1900, 2650},  {0, 23, 33}},
		{{550, 2300, 3000},  {0, 17, 22}},
		{{450, 1900, 2800},  {0, 20, 32}},
		{{700, 2150, 3100},  {0, 22, 30}},
		{{600,  950, 2800},  {0, 10, 40}},
		{{350, 2450, 2900},  {0, 27, 28}},
		/* female 3 */
		{{350,  800, 2700},  {0, 19, 50}},
		{{1000, 1450, 2800},  {7,  7, 34}},
		{{500, 1000, 2600},  {4, 11, 45}},
		{{650, 1000, 2700},  {7,  1, 29}},
		{{460, 1650, 2400},  {2, 21, 35}},
		{{250, 2450, 2900},  {0, 30, 26}},
		{{250, 1750, 2500},  {0, 22, 31}},
		{{480, 2600, 3600},  {4, 22, 33}},
		{{450, 1650, 2600},  {2, 16, 32}},
		{{650, 2450, 2800},  {6, 19, 20}},
		{{650, 1050, 2800},  {1, 10, 42}},
		{{350, 2600, 3200},  {0, 24, 35}},
		/* female 4 */
		{{350,  750, 2500},  {1, 10, 36}},
		{{1000, 1550, 2900},  {6, 14, 18}},
		{{450,  950, 2450},  {2,  6, 30}},
		{{800, 1100, 3200},  {6,  6, 26}},
		{{450, 1900, 2400},  {3, 14, 14}},
		{{250, 2450, 3050},  {0, 15, 22}},
		{{300, 1900, 2350},  {0, 12, 15}},
		{{450, 2250, 2850},  {3, 16, 12}},
		{{430, 1700, 2650},  {2, 12, 17}},
		{{600, 2250, 2950},  {4, 12, 15}},
		{{600,  850, 3200},  {4,  3, 30}},
		{{430, 2500, 3000},  {2, 16, 24}},
		/* female 5 */
		{{300,  900, 2700},  {0, 12, 42}},
		{{750, 1600, 2800},  {2, 15, 24}},
		{{480,  950, 2600},  {2, 13, 32}},
		{{800, 1200, 2900},  {3, 13, 30}},
		{{430, 1500, 2600},  {1, 20, 25}},
		{{250, 2800, 3150}, { -1, 22, 32}},
		{{300, 1400, 2500},  {0, 24, 28}},
		{{420, 2400, 2900},  {2, 27, 27}},
		{{500, 1650, 2850},  {1, 16, 25}},
		{{750, 1900, 3000},  {4, 25, 30}},
		{{600, 1200, 2850},  {3, 17, 23}},
		{{490, 1950, 2900},  {2, 29, 28}},
		/* female 6 */
		{{300,  750, 2350},  {2,  8, 37}}, // djmw 20021212 L2 (was  0)
		{{950, 1400, 2400},  {2, 12, 23}},
		{{650, 1100, 2200},  {4,  8, 34}}, // djmw 20021212 L2 (was  0)
		{{900, 1100, 2600},  {2,  6, 26}}, // djmw 20021212 L3 (was 20)
		{{490, 1700, 2400},  {1, 18, 24}}, // djmw 20021212 L2 (was 14)
		{{300, 2500, 2800},  {0, 16, 21}},
		{{300, 1800, 2400},  {0, 10, 14}},
		{{470, 2400, 2750},  {2, 17, 19}},
		{{570, 1750, 2550},  {0, 17, 18}}, // djmw 20021212 L3 (was 19)
		{{700, 1750, 2400},  {0, 10, 16}},
		{{600, 1200, 2500},  {0, 18, 28}}, // djmw 20021212 L2,3 (was 10,20)
		{{440, 2250, 2700},  {0, 17, 18}},
		/* female 7 */
		{{350,  850, 2600},  { -1, 16, 38}},
		{{950, 1200, 2900},  {7,  9, 40}},
		{{550, 1050, 2500},  {2,  6, 38}},
		{{850, 1200, 2800},  { -1, 12, 39}},
		{{440, 1800, 2500},  {0, 24, 30}},
		{{300, 2300, 3000},  { -1, 22, 30}},
		{{350, 1850, 2400},  { -1, 23, 27}},
		{{460, 2400, 2900},  {1, 28, 30}}, // djmw 20021212 L2 (was 20)
		{{490, 1650, 2700},  {1, 22, 28}},
		{{650, 1700, 2750},  {2, 28, 28}}, // djmw 20021212 L2 (was 20)
		{{450,  700, 3000},  {6,  2, 33}}, // djmw 20021212 L2 (was  8)
		{{440, 2550, 3000},  {1, 40, 41}},
		/* female 8 */
		{{350, 1000, 2500},  {2, 19, 40}},
		{{950, 1400, 2700},  {8, 14, 29}},
		{{500,  950, 2450},  {6, 10, 34}},
		{{850, 1230, 2950},  {8, 19, 42}},
		{{520, 1650, 2600},  {6, 22, 28}},
		{{250, 2500, 3000},  {0, 25, 30}},
		{{250, 1750, 2300},  {0, 23, 29}},
		{{510, 2250, 2850},  {4, 30, 24}},
		{{650, 1600, 2600},  {6, 18, 26}},
		{{720, 1750, 2650},  {4, 19, 27}},
		{{750, 1150, 2650},  {2, 15, 37}},
		{{510, 2050, 2950},  {2, 22, 23}},
		/* female 9 */
		{{300,  850, 2650},  {0, 14, 43}},
		{{950, 1350, 2700},  {4,  6, 25}},
		{{550,  900, 2700},  {5,  4, 37}},
		{{930, 1300, 2800},  {0, 15, 35}},
		{{550, 1500, 2400},  {0, 10, 22}},
		{{280, 2450, 3000},  {0, 13, 21}},
		{{300, 1650, 2400},  {0, 26, 32}},
		{{530, 2400, 3000},  {0,  4, 14}},
		{{600, 1600, 2000},  {0, 14, 16}},
		{{850, 1950, 2800},  {1, 12, 19}},
		{{750, 1050, 2900},  {2,  4, 32}},
		{{550, 2300, 2800},  {1, 14, 22}},
		/* female 10 */
		{{350,  800, 2500},  {0, 21, 44}},
		{{1100, 1450, 2900},  {5,  9, 31}},
		{{400,  850, 2600},  {0,  5, 46}},
		{{750, 1050, 2950},  {4,  9, 25}},
		{{420, 1650, 2550},  {0, 17, 38}},
		{{300, 2450, 3050},  { -1, 28, 35}},
		{{280, 1850, 2500},  { -1, 20, 30}},
		{{420, 2400, 2800},  {1, 25, 29}},
		{{420, 1500, 2350},  {0, 22, 34}},
		{{650, 1850, 2800},  {2, 28, 31}},
		{{550,  800, 2800},  {4,  7, 44}},
		{{420, 2350, 2750},  {2, 27, 28}},
		/* female 11 */
		{{300,  800, 2850},  { -1, 22, 54}},
		{{1150, 1550, 2850},  {6,  9, 26}},
		{{600, 1000, 2700},  {3,  9, 30}},
		{{750, 1100, 2700},  {2, 10, 34}},
		{{500, 1800, 2550},  {3, 17, 24}},
		{{250, 2400, 3100},  {0, 27, 35}},
		{{250, 2000, 3650},  {0, 30, 36}},
		{{420, 2400, 2900},  {5, 28, 31}}, // djmw 20021212 L2,3 (was 20,34)
		{{470, 1700, 2500},  {4, 26, 37}}, // djmw 20021212 L2 (was 20)
		{{700, 1880, 2650},  {2, 21, 26}},
		{{650,  900, 2650},  {5,  4, 33}},
		{{490, 2250, 2650},  {3, 22, 26}},
		/* female 12 */
		{{300,  900, 2500},  { -1, 20, 39}},
		{{950, 1500, 2700},  {3, 16, 20}},
		{{450, 1000, 2650},  {2,  8, 35}},
		{{600, 1050, 2800},  {4,  4, 33}},
		{{420, 1650, 2500},  {1, 14, 22}},
		{{250, 2300, 2850},  {0, 30, 24}},
		{{300, 1800, 2300},  { -1, 21, 28}},
		{{400, 2100, 2700},  {0, 29, 20}},
		{{420, 1650, 2450},  {0, 17, 28}},
		{{550, 1800, 2650},  {2, 19, 20}},
		{{450, 1050, 2800},  {2, 10, 26}},
		{{420, 1850, 2700},  {0, 27, 26}},
		/* female 13 */
		{{300,  850, 2800},  {0, 19, 47}},
		{{1100, 1400, 2800},  {6, 10, 25}},
		{{550,  900, 2900},  {2, 16, 44}},
		{{700, 1050, 2750},  {4,  6, 33}},
		{{490, 1700, 2500},  {0, 33, 35}},
		{{300, 2500, 2950},  {0, 40, 35}},
		{{300, 1900, 2450},  {0, 24, 36}},
		{{500, 2500, 3000},  {4, 31, 32}},
		{{460, 1450, 2500},  {6, 33, 38}},
		{{700, 1700, 2700},  {1, 18, 30}},
		{{500,  800, 2700},  {2,  6, 38}},
		{{550, 2250, 2750},  {3, 28, 26}},
		/* female 14 */
		{{350,  900, 3600},  {0, 16, 44}},
		{{950, 1420, 2950},  {5,  7, 26}},
		{{400, 1000, 2400},  {0, 12, 36}},
		{{600,  950, 2650},  {4,  7, 31}},
		{{420, 1400, 2500},  {1, 21, 24}},
		{{300, 2500, 2850},  {0, 26, 27}},
		{{250, 1450, 2200},  {0, 29, 36}},
		{{420, 1950, 2700},  {0, 20, 20}},
		{{410, 1550, 2400},  {0, 20, 28}},
		{{600, 1650, 2600},  {3,  9, 21}},
		{{430,  880, 2950},  {2,  6, 32}},
		{{410, 1950, 2250},  {0, 31, 30}},
		/* female 15 */
		{{350, 1000, 2350},  {0, 29, 41}},
		{{850, 1100, 2800},  {2,  5, 25}},
		{{600, 1000, 2800},  {2,  6, 30}},
		{{850, 1400, 2600},  {1, 20, 31}},
		{{470, 2000, 2550},  {2, 16, 18}},
		{{250, 2650, 3250},  {0, 31, 40}},
		{{300, 2000, 2400},  {0, 23, 24}},
		{{430, 2400, 2900},  {1, 22, 27}},
		{{460, 1900, 2500},  {2, 25, 28}},
		{{650, 1900, 2900},  {2, 19, 23}},
		{{600,  900, 2400},  {3,  7, 35}},
		{{460, 2400, 2900},  {1, 26, 27}},
		/* female 16 */
		{{300,  850, 3300},  {0, 20, 52}},
		{{1050, 1400, 2450},  {6,  8, 31}},
		{{500,  950, 2400},  {4, 10, 44}},
		{{850, 1050, 2850},  {6,  7, 37}},
		{{480, 1650, 2300},  {2, 14, 32}},
		{{300, 3000, 4200},  {0, 32, 36}},
		{{300, 1900, 2450},  {0, 27, 37}},
		{{460, 2450, 3000},  {1, 32, 30}},
		{{500, 1550, 3100},  {2, 18, 40}},
		{{650, 1800, 2700},  {2, 22, 32}},
		{{450,  750, 2850},  {4,  2, 38}},
		{{440, 2200, 2900},  {0, 30, 29}},
		/* female 17 */
		{{320,  850, 2500},  {0, 18, 50}}, // djmw 20021212 L2 (was 10)
		{{1100, 1350, 2900},  {7,  9, 25}},
		{{450,  900, 2600},  {2,  8, 39}}, // djmw 20021212 L3 (was 30)
		{{800, 1200, 2800},  {2, 14, 32}},
		{{480, 1800, 2500},  {2, 24, 29}},
		{{300, 2600, 3000},  {0, 32, 32}},
		{{300, 1900, 2400},  {0, 25, 28}},
		{{460, 2300, 2900},  {2, 26, 29}},
		{{480, 1800, 2650},  {2, 27, 28}}, // djmw 20021212 L3 (was 20)
		{{650, 1850, 3000},  {4, 22, 34}},
		{{550,  800, 2850},  {4,  4, 30}},
		{{470, 2200, 3000},  {2, 28, 33}}, // djmw 20021212 L2 (was 20)
		/* female 18 */
		{{350,  750, 2550},  {0, 14, 48}}, // djmw 20021212 L3 (was 40)
		{{1050, 1700, 2850},  {0, 14, 22}},
		{{550, 1000, 2600},  {3,  4, 32}},
		{{750, 1150, 2950},  {5,  3, 30}},
		{{550, 1750, 2600},  {2, 12, 19}},
		{{300, 2400, 3000},  {0, 21, 22}},
		{{300, 1950, 2500},  {0, 18, 22}},
		{{510, 2200, 3000},  {1, 24, 27}},
		{{490, 1750, 2450},  {2, 23, 29}},
		{{600, 2200, 3000},  {3, 14, 23}},
		{{700,  850, 2900},  {6,  6, 38}},
		{{540, 2200, 3000},  {9, 22, 22}},
		/* female 19 */
		{{350,  950, 2800},  {0, 24, 51}},
		{{1000, 1500, 2900},  {6, 10, 26}},
		{{450,  950, 2800},  {2, 16, 48}},
		{{700, 1050, 3000},  {5,  5, 35}},
		{{450, 1700, 2450},  {4, 23, 28}},
		{{250, 2600, 3000},  {0, 28, 34}},
		{{330, 1900, 2700},  {0, 30, 42}},
		{{520, 2500, 2900},  {2, 28, 30}},
		{{450, 1700, 2800},  {1, 24, 30}},
		{{700, 1850, 3000},  {2, 23, 24}},
		{{600,  900, 3150},  {2, 12, 38}},
		{{490, 2400, 3100},  {2, 35, 31}},
		/* female 20 */
		{{300,  950, 2650},  {1, 21, 29}},
		{{680, 1250, 2600},  {3, 23, 50}},
		{{400,  850, 2700},  {3, 16, 44}},
		{{500, 1050, 3000},  {4, 13, 42}},
		{{390, 1550, 2500},  {2, 28, 35}},
		{{250, 2300, 2900},  {0, 36, 38}},
		{{250, 1950, 3600},  {0, 34, 45}},
		{{420, 2100, 2600},  {2, 25, 34}},
		{{420, 2100, 2600},  {2, 26, 34}},
		{{600, 1500, 2800},  {5, 25, 36}},
		{{450,  750, 2800},  {2, 11, 48}},
		{{440, 2050, 2700},  {1, 27, 38}},
		/* female 21 */
		{{330,  750, 2900},  {1,  8, 44}},
		{{1150, 1500, 2750},  {3,  8, 35}},
		{{530,  950, 2850},  {0,  8, 40}},
		{{850, 1050, 3000},  {5,  5, 36}},
		{{500, 1700, 2700},  {0, 20, 31}},
		{{300, 2600, 3100},  { -1, 23, 28}},
		{{300, 1800, 2400},  {0, 22, 27}},
		{{520, 2550, 3050},  {0, 22, 26}},
		{{520, 1650, 2550},  {0, 20, 28}},
		{{700, 1850, 2900},  {0, 16, 24}},
		{{650, 1100, 2900},  { -1, 11, 36}},
		{{450, 2300, 2800},  {0, 16, 17}},
		/* female 22 */
		{{300,  850, 2800},  {0, 15, 42}},
		{{1080, 1400, 2750},  {2,  3, 21}},
		{{480,  950, 2300},  {0,  6, 30}},
		{{800, 1100, 2800},  {3,  3, 28}},
		{{470, 1800, 2500},  {0, 12, 23}},
		{{300, 2350, 2850},  { -1, 31, 31}},
		{{300, 2100, 2600},  { -1, 25, 31}},
		{{460, 2600, 2900},  {0, 20, 23}},
		{{470, 1800, 2500},  {0, 16, 29}},
		{{600, 2000, 2700},  {2, 13, 11}},
		{{500,  750, 2650},  {4,  2, 24}},
		{{500, 2300, 2750},  {0, 20, 22}},
		/* female 23 */
		{{350,  700, 2800},  {0, 10, 46}},
		{{950, 1500, 2900},  {0, 19, 23}},
		{{530,  980, 2850},  {0, 14, 34}},
		{{750, 1100, 3000},  {4,  1, 27}},
		{{520, 1800, 2450},  {2, 20, 25}},
		{{300, 2550, 2950},  {0, 22, 24}},
		{{300, 1950, 2450},  {0, 30, 31}},
		{{500, 2300, 2800},  {0, 21, 29}},
		{{550, 1700, 2750},  {0, 21, 24}},
		{{650, 1850, 2700},  { -1, 16, 20}},
		{{650,  980, 2650},  {1,  4, 34}},
		{{500, 2400, 2800},  {4, 16, 18}},
		/* female 24 */
		{{300,  900, 2900},  {0, 12, 45}},
		{{900, 1650, 2850},  {1,  6, 23}},
		{{500,  900, 2300},  {0,  4, 39}},
		{{750, 1150, 2750},  {2,  6, 27}},
		{{550, 1650, 2500},  {0, 16, 24}},
		{{300, 2500, 2850},  {0, 27, 34}},
		{{250, 1850, 2300},  {0, 30, 26}},
		{{490, 2400, 2850},  {2, 16, 18}},
		{{520, 1800, 2450},  {3, 22, 23}},
		{{650, 2100, 2800},  {2, 14, 13}},
		{{550, 1100, 3900},  {2, 11, 32}},
		{{510, 2350, 2950},  {1, 24, 23}}, // djmw 20021212 L3 (was 33)
		/* female 25 */
		{{300,  800, 3700},  {0, 13, 50}}, // djmw 20021212 L2 (was 17)
		{{1000, 1450, 2650},  {4,  7, 24}},
		{{550, 1000, 2850},  {8, 10, 43}},
		{{750, 1050, 3000},  {2,  6, 33}},
		{{500, 1550, 2400},  {4, 16, 25}},
		{{250, 2400, 2950},  {0, 26, 24}},
		{{300, 1800, 2400},  {0, 26, 26}},
		{{490, 2300, 2800},  {8, 15, 20}},
		{{550, 1550, 2500},  {1, 14, 25}},
		{{700, 1950, 2750},  {0,  4, 13}},
		{{630,  980, 2850},  {5,  7, 39}},
		{{490, 2300, 2900},  {4, 17, 22}}
	};

	try {
		autoTable me = Table_create (nrows, ncols);

		for (integer i = 1; i <= nrows; i ++) {
			TableRow row = my rows.at [i];
			const integer vowel_id = ( (i - 1) % 12) + 1;	/* 1 - 12 */
			const integer speaker_id = (i - 1) / 12 + 1;  /* 1 - 75 */
			const integer speaker_sex = ( speaker_id <= 50 ? 0 : 1 );

			row -> cells [1]. string = Melder_dup (sex [speaker_sex]);
			row -> cells [2]. string = Melder_dup (Melder_integer (speaker_id));
			row -> cells [3]. string = Melder_dup (vowel [vowel_id - 1]);
			row -> cells [4]. string = Melder_dup (ipa [vowel_id - 1]);
			for (integer j = 0; j <= 2; j ++) {
				row -> cells [j + 5]. string = Melder_dup (Melder_integer (polsdata [i - 1]. f [j]));
				row -> cells [j + 8]. string = Melder_dup (Melder_integer (polsdata [i - 1]. l [j]));
			}
		}
		for (integer j = 1; j <= ncols; j ++) {
			Table_setColumnLabel (me.get(), j, columnLabels [j - 1]);
			my columnHeaders [j]. numericized = false;
		}
		return me;
	} catch (MelderError) {
		Melder_throw (U"Table not created from Pols & van Nierop data.");
	}
}

autoTable Table_create_weenink1983 () {
	const integer nrows = 360, ncols = 9;
	const conststring32 columnLabels [9] = {U"Type", U"Sex", U"Speaker", U"Vowel", U"IPA", U"F0", U"F1", U"F2", U"F3"};
	const conststring32 type [3] = {U"m", U"w", U"c"};
	/*
		Our vowel order: "oe", "o", "oo", "a", "aa", "u", "eu", "uu", "ie", "i", "ee", "e".
		To Pols & van Nierop order
	*/
	const integer order [13] = { 0, 1, 5, 3, 4, 7, 9, 8, 11, 6, 12, 2, 10};
	const conststring32 vowel [13] = {U"", U"oe", U"aa", U"oo", U"a", U"eu", U"ie", U"uu", U"ee", U"u", U"e", U"o", U"i"};
	const conststring32 ipa [13] = {U"", U"u", U"a", U"o", U"\\as", U"\\o/", U"i", U"y", U"e", U"\\yc", U"\\ep", U"\\ct", U"\\ic"};
	const conststring32 sex [2] = {U"m", U"f"};
	const struct weeninkdatum {
		short f [4];	  // f0, f1, f2, f3
	} weeninkdata [] = {
		{{182, 335,  748, 2332}},
		{{170, 435,  669, 2886}},
		{{164, 488,  911, 2362}},
		{{170, 671, 1005, 2822}},
		{{160, 787, 1372, 3313}},
		{{165, 396, 1624, 2374}},
		{{165, 472, 1585, 2344}},
		{{180, 293, 1791, 2177}},
		{{176, 314, 2149, 3215}},
		{{164, 393, 2120, 2694}},
		{{161, 454, 1982, 2673}},
		{{165, 502, 1902, 2632}},
		/* Speaker 2 */
		{{161, 308,  709, 2222}},
		{{155, 461,  760, 2632}},
		{{148, 510,  907, 2532}},
		{{153, 645, 1004, 2604}},
		{{145, 758, 1244, 2658}},
		{{155, 449, 1489, 2279}},
		{{145, 454, 1388, 2308}},
		{{162, 314, 1620, 2111}},
		{{158, 303, 1984, 2835}},
		{{150, 429, 1888, 2502}},
		{{156, 460, 1748, 2442}},
		{{150, 580, 1679, 2383}},
		/* Speaker 3 */
		{{124, 343,  719, 2107}},
		{{124, 519,  818, 1975}},
		{{125, 532,  937, 2044}},
		{{126, 721, 1135, 2077}},
		{{125, 850, 1328, 2299}},
		{{131, 390, 1307, 2177}},
		{{129, 481, 1438, 2203}},
		{{127, 270, 1705, 2068}},
		{{121, 275, 2081, 2882}},
		{{129, 377, 2044, 2638}},
		{{131, 464, 1949, 2536}},
		{{124, 645, 1854, 2488}},
		/* Speaker 4 */
		{{119, 322,  590, 2146}},
		{{110, 438,  703, 2392}},
		{{106, 489,  845, 2157}},
		{{120, 708, 1126, 2302}},
		{{107, 818, 1333, 2312}},
		{{109, 406, 1505, 2133}},
		{{109, 422, 1518, 2116}},
		{{115, 298, 1581, 2162}},
		{{119, 268, 2189, 2950}},
		{{109, 400, 2151, 2870}},
		{{108, 420, 2075, 2495}},
		{{108, 550, 1905, 2383}},
		/* Speaker 5 */
		{{132, 282,  656, 2355}},
		{{118, 462,  662, 2852}},
		{{117, 482,  797, 2531}},
		{{115, 723, 1036, 2456}},
		{{117, 732, 1356, 2562}},
		{{118, 454, 1516, 2178}},
		{{119, 466, 1484, 2139}},
		{{129, 271, 1642, 2154}},
		{{122, 255, 2390, 2787}},
		{{120, 353, 2114, 2507}},
		{{114, 441, 2020, 2392}},
		{{122, 601, 1858, 2315}},
		/* Speaker 6 */
		{{155, 275,  560, 2397}},
		{{141, 395,  705, 2802}},
		{{149, 419,  831, 1824}},
		{{142, 674, 1113, 2455}},
		{{137, 696, 1262, 2778}},
		{{148, 419, 1493, 2242}},
		{{148, 422, 1599, 2319}},
		{{157, 291, 1585, 2160}},
		{{152, 281, 2324, 3124}},
		{{144, 358, 2269, 2936}},
		{{146, 426, 2174, 2786}},
		{{140, 560, 1941, 2722}},
		/* Speaker 7 */
		{{165, 315,  703, 2304}},
		{{138, 531,  837, 2338}},
		{{130, 471,  838, 2391}},
		{{135, 713, 1218, 2466}},
		{{140, 807, 1478, 2390}},
		{{126, 467, 1639, 2227}},
		{{135, 504, 1520, 2268}},
		{{151, 341, 1738, 2246}},
		{{145, 302, 2255, 2887}},
		{{140, 379, 2183, 2614}},
		{{133, 400, 2198, 2551}},
		{{137, 587, 2024, 2516}},
		/* Speaker 8 */
		{{179, 361,  730, 2256}},
		{{168, 500,  714, 2782}},
		{{165, 495,  961, 2405}},
		{{162, 642,  988, 2308}},
		{{156, 921, 1409, 2909}},
		{{166, 451, 1482, 2657}},
		{{175, 505, 1546, 2443}},
		{{177, 341, 1705, 2291}},
		{{176, 285, 2288, 2817}},
		{{173, 364, 2126, 2907}},
		{{153, 492, 2102, 2778}},
		{{158, 628, 1777, 2903}},
		/* Speaker 9 */
		{{180, 358,  700, 2672}},
		{{177, 524,  759, 2784}},
		{{169, 557,  856, 2727}},
		{{178, 685, 1075, 2677}},
		{{168, 855, 1311, 2701}},
		{{170, 499, 1496, 2725}},
		{{170, 502, 1385, 2112}},
		{{178, 348, 1504, 2129}},
		{{176, 318, 2252, 2846}},
		{{169, 422, 2186, 2816}},
		{{159, 509, 1941, 2732}},
		{{165, 582, 1920, 2773}},
		/* Speaker 10 */
		{{161, 277,  592, 2493}},
		{{141, 490,  718, 2757}},
		{{137, 392,  692, 2512}},
		{{142, 642,  991, 2707}},
		{{144, 753, 1306, 2617}},
		{{140, 388, 1570, 2175}},
		{{139, 357, 1675, 2105}},
		{{156, 248, 1846, 2117}},
		{{148, 258, 2280, 3350}},
		{{149, 367, 2129, 2731}},
		{{142, 357, 2163, 2624}},
		{{143, 581, 1865, 2572}},
		/* Speaker 11 */
		{{282, 305,  842, 2366}},
		{{273, 557,  986, 2677}},
		{{277, 606, 1090, 2422}},
		{{274, 819, 1327, 2819}},
		{{277, 938, 1580, 2953}},
		{{278, 503, 1826, 2518}},
		{{274, 579, 1630, 2501}},
		{{286, 312, 2065, 3737}},
		{{282, 291, 2540, 3225}},
		{{278, 506, 2270, 3040}},
		{{277, 559, 2245, 2846}},
		{{277, 770, 2137, 2919}},
		/* Speaker 12 */
		{{255, 307,  738, 2529}},
		{{265, 505,  938, 2641}},
		{{228, 524, 1067, 2510}},
		{{233, 902, 1183, 2603}},
		{{215, 841, 1504, 2712}},
		{{239, 464, 1655, 2556}},
		{{234, 443, 1754, 2607}},
		{{245, 303, 1762, 2460}},
		{{241, 277, 2439, 3407}},
		{{262, 490, 2371, 3198}},
		{{228, 484, 2044, 2593}},
		{{234, 495, 2296, 3204}},
		/* Speaker 13 */
		{{298, 309,  627, 3029}},
		{{267, 785,  918, 2956}},
		{{259, 600,  948, 2684}},
		{{257, 893, 1136, 2844}},
		{{245, 1028, 1498, 2715}},
		{{258, 501, 1746, 2529}},
		{{265, 536, 1799, 2601}},
		{{280, 282, 1391, 2520}},
		{{270, 275, 2664, 3174}},
		{{260, 511, 2320, 2882}},
		{{255, 506, 2482, 2846}},
		{{258, 783, 2027, 2606}},
		/* Speaker 14 */
		{{187, 327,  586, 2463}},
		{{177, 520,  690, 2828}},
		{{175, 540,  956, 2599}},
		{{178, 596, 1023, 2750}},
		{{184, 692, 1382, 2432}},
		{{186, 397, 1658, 2422}},
		{{175, 512, 1674, 2458}},
		{{186, 291, 1737, 2236}},
		{{203, 271, 2299, 3260}},
		{{187, 369, 2263, 2903}},
		{{182, 471, 2057, 2729}},
		{{196, 593, 2122, 2779}},
		/* Speaker 15 */
		{{263, 448,  830, 2687}},
		{{230, 570, 1010, 2614}},
		{{244, 595, 1109, 2686}},
		{{240, 806, 1386, 2591}},
		{{239, 890, 1574, 2965}},
		{{244, 482, 1900, 2710}},
		{{251, 502, 1529, 2716}},
		{{238, 436, 2134, 2626}},
		{{254, 374, 2580, 3181}},
		{{252, 494, 2323, 2989}},
		{{242, 484, 2323, 3025}},
		{{246, 595, 1897, 2865}},
		/* Speaker 16 */
		{{238, 325,  737, 2290}},
		{{224, 462,  764, 2600}},
		{{202, 577,  944, 2651}},
		{{216, 674, 1084, 2851}},
		{{210, 833, 1333, 2256}},
		{{214, 427, 1708, 2346}},
		{{202, 502, 1546, 2326}},
		{{242, 289, 1669, 2169}},
		{{232, 337, 2060, 2841}},
		{{224, 416, 2217, 2769}},
		{{214, 491, 1950, 2674}},
		{{188, 548, 2071, 2727}},
		/* Speaker 17 */
		{{277, 313,  578, 2457}},
		{{254, 549,  855, 2793}},
		{{250, 608,  904, 2724}},
		{{267, 890, 1294, 2732}},
		{{263, 1047, 1578, 2848}},
		{{289, 549, 1728, 2844}},
		{{258, 508, 1747, 2799}},
		{{284, 287, 1982, 2793}},
		{{268, 268, 2873, 4075}},
		{{258, 440, 2590, 3090}},
		{{248, 489, 2529, 3111}},
		{{261, 606, 2078, 2980}},
		/* Speaker 18 */
		{{330, 334,  729, 3332}},
		{{247, 629,  864, 3024}},
		{{234, 521,  935, 2760}},
		{{264, 964, 1246, 3112}},
		{{261, 1008, 1573, 2932}},
		{{278, 509, 1887, 2596}},
		{{243, 487, 1799, 2393}},
		{{303, 306, 1781, 2415}},
		{{282, 296, 2597, 3571}},
		{{217, 425, 2636, 3314}},
		{{250, 490, 2642, 3494}},
		{{275, 787, 2326, 2992}},
		/* Speaker 19 */
		{{228, 322,  661, 2721}},
		{{214, 589,  821, 2729}},
		{{219, 457,  857, 2582}},
		{{229, 801, 1260, 2911}},
		{{220, 898, 1628, 2770}},
		{{233, 463, 1630, 2306}},
		{{225, 456, 1671, 2381}},
		{{225, 283, 1779, 2211}},
		{{224, 276, 2495, 3372}},
		{{224, 440, 2385, 3007}},
		{{216, 435, 2363, 2866}},
		{{212, 637, 2130, 3029}},
		/* Speaker 20 */
		{{246, 324,  730, 2644}},
		{{230, 636,  923, 2757}},
		{{233, 534,  954, 2542}},
		{{234, 820, 1256, 3064}},
		{{248, 966, 1695, 2890}},
		{{237, 484, 1753, 2718}},
		{{240, 480, 1917, 2670}},
		{{247, 291, 1981, 2701}},
		{{243, 290, 2648, 3176}},
		{{256, 501, 2358, 3058}},
		{{238, 469, 2406, 3008}},
		{{236, 693, 2128, 2991}},
		/* Speaker 21 */
		{{342, 353,  758, 3269}},
		{{309, 627,  976, 3256}},
		{{334, 663, 1017, 3075}},
		{{321, 964, 1576, 3186}},
		{{316, 1129, 2249, 3404}},
		{{330, 650, 2336, 3284}},
		{{311, 636, 2083, 3328}},
		{{331, 357, 2389, 3185}},
		{{332, 337, 2960, 3595}},
		{{351, 586, 2833, 3595}},
		{{312, 621, 2839, 3614}},
		{{309, 765, 2755, 3636}},
		/* Speaker 22 */
		{{326, 359,  682, 2948}},
		{{318, 631,  795, 3537}},
		{{296, 688, 1135, 2930}},
		{{285, 746, 1135, 1926}},
		{{290, 1006, 2166, 3263}},
		{{326, 632, 1990, 3042}},
		{{285, 570, 2015, 3155}},
		{{321, 324, 2025, 3104}},
		{{338, 346, 3069, 3573}},
		{{320, 636, 3025, 3709}},
		{{306, 611, 2770, 3648}},
		{{331, 893, 2639, 3566}},
		/* Speaker 23 */
		{{312, 414,  828, 2908}},
		{{314, 609,  936, 3139}},
		{{305, 659, 1201, 3015}},
		{{298, 890, 1303, 3003}},
		{{300, 1150, 2116, 3205}},
		{{275, 537, 1968, 3032}},
		{{277, 534, 1997, 3009}},
		{{304, 398, 2298, 2940}},
		{{302, 330, 3010, 3746}},
		{{312, 547, 2965, 3673}},
		{{275, 551, 2786, 3442}},
		{{299, 641, 2662, 3410}},
		/* Speaker 24 */
		{{301, 315,  812, 3357}},
		{{316, 621,  960, 3701}},
		{{305, 604, 1179, 3659}},
		{{331, 981, 1701, 3374}},
		{{322, 954, 2010, 3291}},
		{{297, 581, 2085, 3288}},
		{{307, 613, 1991, 3392}},
		{{322, 331, 2099, 3096}},
		{{326, 326, 3235, 3935}},
		{{307, 547, 2747, 3618}},
		{{308, 613, 2763, 3710}},
		{{315, 622, 2459, 3616}},
		/* Speaker 25 */
		{{360, 417,  981, 3610}},
		{{354, 697, 1017, 3077}},
		{{363, 741, 1302, 3381}},
		{{340, 1028, 1611, 2973}},
		{{337, 1122, 1930, 3068}},
		{{310, 608, 2105, 3177}},
		{{323, 628, 1929, 3197}},
		{{378, 384, 2255, 2992}},
		{{363, 363, 3235, 4069}},
		{{338, 527, 2698, 3458}},
		{{308, 643, 2603, 3382}},
		{{326, 712, 2351, 3498}},
		/* Speaker 26 */
		{{375, 380,  798, 3316}},
		{{371, 456,  759, 3321}},
		{{324, 656, 1118, 3296}},
		{{340, 1031, 1690, 3163}},
		{{376, 1154, 1920, 3519}},
		{{362, 604, 2162, 3603}},
		{{342, 679, 2035, 3683}},
		{{371, 374, 2042, 3206}},
		{{349, 361, 3082, 3782}},
		{{357, 499, 2899, 3934}},
		{{324, 646, 2820, 3970}},
		{{353, 706, 2785, 3942}},
		/* Speaker 27 */
		{{312, 339,  850, 4356}},
		{{319, 619,  919, 3570}},
		{{288, 576, 1156, 3681}},
		{{301, 865, 1567, 3326}},
		{{296, 914, 2090, 3113}},
		{{297, 589, 2150, 3208}},
		{{287, 580, 2146, 3340}},
		{{303, 330, 2431, 3032}},
		{{291, 313, 3090, 4039}},
		{{305, 539, 3015, 3905}},
		{{291, 577, 2589, 2778}},
		{{307, 616, 2448, 3856}},
		/* Speaker 28 */
		{{353, 354, 1085, 3082}},
		{{343, 617,  990, 3027}},
		{{281, 822, 1180, 2931}},
		{{324, 993, 1424, 2941}},
		{{297, 1192, 1792, 2922}},
		{{314, 444, 2170, 3044}},
		{{275, 608, 1817, 2909}},
		{{355, 358, 2138, 3160}},
		{{321, 323, 3169, 3625}},
		{{315, 499, 2831, 3465}},
		{{303, 608, 2707, 3369}},
		{{313, 718, 2479, 3627}},
		/* Speaker 29 */
		{{361, 363, 1050, 3239}},
		{{365, 666, 1094, 3269}},
		{{327, 830, 1289, 3145}},
		{{325, 963, 1293, 3046}},
		{{356, 1163, 1707, 3188}},
		{{377, 580, 2191, 3234}},
		{{336, 673, 2339, 3167}},
		{{342, 345, 2344, 3103}},
		{{341, 346, 2729, 3370}},
		{{326, 541, 2568, 3297}},
		{{331, 677, 2561, 3320}},
		{{354, 897, 2463, 3338}},
		/* Speaker 30 */
		{{332, 352,  979, 2988}},
		{{286, 596,  973, 2690}},
		{{304, 753, 1174, 3005}},
		{{290, 884, 1432, 2806}},
		{{347, 1060, 2079, 2706}},
		{{303, 592, 2069, 2889}},
		{{299, 644, 1775, 3005}},
		{{296, 329, 2082, 2791}},
		{{315, 326, 2807, 3612}},
		{{298, 561, 2612, 3330}},
		{{281, 569, 2415, 3239}},
		{{280, 709, 2460, 3314}}
	};

	try {
		autoTable me = Table_create (nrows, ncols);

		for (integer i = 1; i <= nrows; i ++) {
			const TableRow row = my rows.at [i];
			const integer speaker_id = (i - 1) / 12 + 1;	// 1 - 30
			const integer vowel_id = (i - 1) % 12 + 1;	// 1 - 12
			const integer index_in_data = (speaker_id - 1) * 12 + order [vowel_id] - 1;
			integer speaker_type, speaker_sex;

			if (speaker_id <= 10) {   // 10 men
				speaker_type = 0;
				speaker_sex = 0;
			} else if (speaker_id <= 20) {   // 10 women
				speaker_type = 1;
				speaker_sex = 1;
			} else {   // 10 children
				speaker_type = 2;
				speaker_sex = 0;   // which children were m/f
			}

			row -> cells [1]. string = Melder_dup (type [speaker_type]);
			row -> cells [2]. string = Melder_dup (sex [speaker_sex]);
			row -> cells [3]. string = Melder_dup (Melder_integer (speaker_id));
			row -> cells [4]. string = Melder_dup (vowel [vowel_id]);
			row -> cells [5]. string = Melder_dup (ipa [vowel_id]);

			for (integer j = 0; j <= 3; j ++)
				row -> cells [j + 6]. string = Melder_dup (Melder_integer (weeninkdata [index_in_data]. f [j]));
		}
		for (integer j = 1; j <= ncols; j ++) {
			Table_setColumnLabel (me.get(), j, columnLabels [j - 1]);
			my columnHeaders [j]. numericized = false;
		}
		return me;
	} catch (MelderError) {
		Melder_throw (U"Table not created from Weenink data.");
	}
}

/*
	P.A. Keating & C. Esposito (2006): "Linguistic voice quality." UCLA Working Papers in Phonetics 105: 85-91.
*/
autoTable Table_create_esposito2006 () {
	try {
		autoTable me = Table_createWithColumnNames (10, U"Language Modal Breathy");
		Table_setStringValue (me.get(), 1, 1, U"Chong");
		Table_setNumericValue (me.get(), 1, 2, -1.5);
		Table_setNumericValue (me.get(), 1, 3, 5);
		Table_setStringValue (me.get(), 2, 1, U"Fuzhou");
		Table_setNumericValue (me.get(), 2, 2, -1.5);
		Table_setNumericValue (me.get(), 2, 3, 5);
		Table_setStringValue (me.get(), 3, 1, U"Green Hmong");
		Table_setNumericValue (me.get(), 3, 2, 3);
		Table_setNumericValue (me.get(), 3, 3, 12);
		Table_setStringValue (me.get(), 4, 1, U"White Hmong");
		Table_setNumericValue (me.get(), 4, 2, 2);
		Table_setNumericValue (me.get(), 4, 3, 11);
		Table_setStringValue (me.get(), 5, 1, U"Mon");
		Table_setNumericValue (me.get(), 5, 2, -1.5);
		Table_setNumericValue (me.get(), 5, 3, 0);
		Table_setStringValue (me.get(), 6, 1, U"SADV Zapotec");
		Table_setNumericValue (me.get(), 6, 2, -6);
		Table_setNumericValue (me.get(), 6, 3, -4);
		Table_setStringValue (me.get(), 7, 1, U"SLQ Zapotec");
		Table_setNumericValue (me.get(), 7, 2, 3.5);
		Table_setNumericValue (me.get(), 7, 3, 14);
		Table_setStringValue (me.get(), 8, 1, U"Tlacolula Zapotec");
		Table_setNumericValue (me.get(), 8, 2, 3);
		Table_setNumericValue (me.get(), 8, 3, 13);
		Table_setStringValue (me.get(), 9, 1, U"Tamang");
		Table_setNumericValue (me.get(), 9, 2, 1);
		Table_setNumericValue (me.get(), 9, 3, 1);
		Table_setStringValue (me.get(), 10, 1, U"!Xoo");
		Table_setNumericValue (me.get(), 10, 2, 1);
		Table_setNumericValue (me.get(), 10, 3, 14);
		return me;
	} catch (MelderError) {
		Melder_throw (U"Keating-Esposito table not created.");
	}
}

/*
	W.F. Ganong III (1980): "Phonetic categorization in auditory word perception." Journal of Experimental Psychology: 
		Human Perception and Performance 6: 110-125.
*/
autoTable Table_create_ganong1980 () {
	try {
		autoTable me = Table_createWithColumnNames (6, U"VOT dash-tash dask-task");
		Table_setNumericValue (me.get(), 1, 1, -17.5);
		Table_setNumericValue (me.get(), 1, 2, 0.98);
		Table_setNumericValue (me.get(), 1, 3, 0.92);
		Table_setNumericValue (me.get(), 2, 1, -7.5);
		Table_setNumericValue (me.get(), 2, 2, 0.95);
		Table_setNumericValue (me.get(), 2, 3, 0.83);
		Table_setNumericValue (me.get(), 3, 1, -2.5);
		Table_setNumericValue (me.get(), 3, 2, 0.71);
		Table_setNumericValue (me.get(), 3, 3, 0.33);
		Table_setNumericValue (me.get(), 4, 1, 2.5);
		Table_setNumericValue (me.get(), 4, 2, 0.29);
		Table_setNumericValue (me.get(), 4, 3, 0.10);
		Table_setNumericValue (me.get(), 5, 1, 7.5);
		Table_setNumericValue (me.get(), 5, 2, 0.12);
		Table_setNumericValue (me.get(), 5, 3, 0.02);
		Table_setNumericValue (me.get(), 6, 1, 17.5);
		Table_setNumericValue (me.get(), 6, 2, 0.10);
		Table_setNumericValue (me.get(), 6, 3, 0.02);
		return me;
	} catch (MelderError) {
		Melder_throw (U"Ganong table not created.");
	}
}

static bool intervalsIntersect (double x1, double x2, double xmin, double xmax, double *out_xc1, double *out_xc2) {
	if (x1 > x2)
		std::swap (x1, x2);
	if (xmin > xmax)
		std::swap (xmin, xmax);
	double xc1 = x1;
	double xc2 = x2;
	bool intersect = true;
	if (x2 <= xmin || x1 >= xmax)
		intersect = false;
	else {
		if (x1 < xmin)
			xc1 = xmin;
		if (x2 > xmax)
			xc2 = xmax;
	}
	if (out_xc1)
		*out_xc1 = xc1;
	if (out_xc2)
		*out_xc2 = xc2;
	return intersect;
}

void Table_horizontalErrorBarsPlotWhere (Table me, Graphics g, integer xcolumn, integer ycolumn, double xmin, double xmax,
	double ymin, double ymax, integer xci_min, integer xci_max, double bar_mm, bool garnish, conststring32 formula, Interpreter interpreter)
{
	try {
		const integer nrows = my rows.size;
		if (xcolumn < 1 || xcolumn > nrows || ycolumn < 1 || ycolumn > nrows ||
			(xci_min != 0 && xci_min > nrows) || (xci_max != 0 && xci_max > nrows))
				return;

		integer numberOfSelectedRows = 0;
		autoINTVEC selectedRows = Table_findRowsMatchingCriterion (me, formula, interpreter);
		if (ymin >= ymax) {
			Table_columnExtremaFromSelectedRows (me, ycolumn, selectedRows.get(), & ymin, & ymax);
			if (ymin >= ymax) {
				ymin -= 1.0;
				ymax += 1.0;
			}
		}
		double x1min, x1max;
		if (xmin >= xmax) {
			Table_columnExtremaFromSelectedRows (me, xcolumn, selectedRows.get(), & xmin, & xmax);
			if (xci_min > 0) {
				Table_columnExtremaFromSelectedRows (me, xci_min, selectedRows.get(), & x1min, & x1max);
				xmin -= x1max;
			}
			if (xci_max > 0) {
				Table_columnExtremaFromSelectedRows (me, xci_max, selectedRows.get(), & x1min, & x1max);
				xmax += x1max;
			}
			if (xmin >= xmax) {
				xmin -= 1.0;
				xmax += 1.0;
			}
		}
		Graphics_setWindow (g, xmin, xmax, ymin, ymax);
		Graphics_setInner (g);
		double dy = Graphics_dyMMtoWC (g, bar_mm);
		for (integer irow = 1; irow <= numberOfSelectedRows; irow ++) {
			const double x = Table_getNumericValue_Assert (me, selectedRows [irow], xcolumn);
			const double y = Table_getNumericValue_Assert (me, selectedRows [irow], ycolumn);
			const double dx1 =
				xci_min > 0 ? Table_getNumericValue_Assert (me, selectedRows [irow], xci_min) : 0.0;
			const double dx2 =
				xci_max > 0 ? Table_getNumericValue_Assert (me, selectedRows [irow], xci_max) : 0.0;
			const double x1 = x - dx1, x2 = x + dx2;

			if (x <= xmax && x >= xmin && y <= ymax && y >= ymin) {
				double xc1, yc1, xc2, yc2;
				if (intervalsIntersect (x1, x2, xmin, xmax, & xc1, & xc2)) {
					Graphics_line (g, xc1, y, xc2, y);
					if (dy > 0 && intervalsIntersect (y - dy / 2.0, y + dy / 2.0, ymin, ymax, & yc1, & yc2)) {
						if (xc1 >= xmin && dx1 > 0.0)
							Graphics_line (g, xc1, yc1, xc1, yc2);
						if (xc2 <= xmax && dx2 > 0.0)
							Graphics_line (g, xc2, yc1, xc2, yc2);
					}
				}
			}
		}
		Graphics_unsetInner (g);

		if (garnish) {
			Graphics_drawInnerBox (g);
			Graphics_marksLeft (g, 2, true, true, false);
			Graphics_marksBottom (g, 2, true, true, false);
		}
	} catch (MelderError) {
		Melder_clearError ();
	}
}

void Table_verticalErrorBarsPlotWhere (Table me, Graphics g,
	integer xcolumn, integer ycolumn, double xmin, double xmax,
	double ymin, double ymax, integer yci_min, integer yci_max,
	double bar_mm, bool garnish, conststring32 formula, Interpreter interpreter)
{
	try {
		const integer nrows = my rows.size;
		if (xcolumn < 1 || xcolumn > nrows || ycolumn < 1 || ycolumn > nrows ||
			(yci_min != 0 && yci_min > nrows) || (yci_max != 0 && yci_max > nrows))
				return;
		autoINTVEC selectedRows = Table_findRowsMatchingCriterion (me, formula, interpreter);
		if (xmin >= xmax) {
			Table_columnExtremaFromSelectedRows (me, ycolumn, selectedRows.get(), & ymin, & ymax);
			if (xmin >= xmax) {
				xmin -= 1.0;
				xmax += 1.0;
			}
		}
		if (ymin >= ymax) {
			double y1min, y1max;
			Table_columnExtremaFromSelectedRows (me, ycolumn, selectedRows.get(), & ymin, & ymax);
			if (yci_min > 0.0) {
				Table_columnExtremaFromSelectedRows (me, yci_min, selectedRows.get(), & y1min, & y1max);
				ymin -= y1max;
			}
			if (yci_max > 0.0) {
				Table_columnExtremaFromSelectedRows (me, yci_max, selectedRows.get(), & y1min, & y1max);
				ymax += y1max;
			}
			if (ymin >= ymax) {
				ymin -= 1.0;
				ymax += 1.0;
			}
		}
		Graphics_setWindow (g, xmin, xmax, ymin, ymax);
		Graphics_setInner (g);
		const double dx = Graphics_dxMMtoWC (g, bar_mm);
		for (integer irow = 1; irow <= selectedRows.size; irow ++) {
			const double x  = Table_getNumericValue_Assert (me, selectedRows [irow], xcolumn);
			const double y  = Table_getNumericValue_Assert (me, selectedRows [irow], ycolumn);
			const double dy1 =
				yci_min > 0 ? Table_getNumericValue_Assert (me, selectedRows [irow], yci_min) : 0.0;
			const double dy2 =
				yci_max > 0 ? Table_getNumericValue_Assert (me, selectedRows [irow], yci_max) : 0.0;
			const double y1 = y - dy1, y2 = y + dy2;

			if (x <= xmax && x >= xmin && y <= ymax && y >= ymin) {
				double xc1, yc1, xc2, yc2;
				if (intervalsIntersect (y1, y2, ymin, ymax, & yc1, & yc2)) {
					Graphics_line (g, x, yc1, x, yc2);
					if (dx > 0 && intervalsIntersect (x - dx / 2.0, x + dx / 2.0, xmin, xmax, & xc1, & xc2)) {
						if (yc1 >= ymin && dy1 > 0.0)
							Graphics_line (g, xc1, yc1, xc2, yc1);
						if (yc2 <= ymax && dy2 > 0.0)
							Graphics_line (g, xc1, yc2, xc2, yc2);
					}
				}
			}
		}
		Graphics_unsetInner (g);

		if (garnish) {
			Graphics_drawInnerBox (g);
			Graphics_marksLeft (g, 2, true, true, false);
			Graphics_marksBottom (g, 2, true, true, false);
		}
	} catch (MelderError) {
		Melder_clearError ();
	}
}

double Table_getMedianAbsoluteDeviation (Table me, integer columnNumber) {
	try {
		autoVEC data = Table_getColumnVector (me, columnNumber);
		double mad, location;
		NUMmad (data.get(), & location, true, & mad);
		return mad;
	} catch (MelderError) {
		Melder_throw (me, U": cannot compute median absolute deviation of column ", columnNumber, U".");
	}
}

autoVEC Table_getColumnVector (Table me, integer columnNumber) {
	try {
		Table_checkSpecifiedColumnNumberWithinRange (me, columnNumber);
		Table_numericize_Assert (me, columnNumber);
		Melder_require (my rows.size > 0,
			U"The table is empty.");
		autoVEC result = newVECraw (my rows.size);
		for (integer irow = 1; irow <= my rows.size; irow ++) {
			const TableRow row = my rows.at [irow];
			result [irow] = row -> cells [columnNumber].number;
			Melder_require (isdefined (result [irow]),
				U"The cell in row ", irow, U" of column ", Table_messageColumn (me, columnNumber), U" is undefined.");
		}
		return result;
	} catch (MelderError) {
		Melder_throw (me, U": cannot get column vector.");
	}
}

void Table_reportHuberMStatistics (Table me, integer columnNumber, double k_std, double tol, double *out_location, double *out_scale, integer maximumNumberOfIterations) {
	try {
		autoVEC data = Table_getColumnVector (me, columnNumber);
		double location, scale;
		NUMstatistics_huber (data.get(), & location, true, & scale, true, k_std, tol, maximumNumberOfIterations);
		if (out_location)
			*out_location = location;
		if (out_scale)
			*out_scale = scale;
	} catch (MelderError) {
		Melder_throw (me, U": cannot compute median absolute deviation of column ", columnNumber, U".");
	}
}

autoTable Table_getOneWayKruskalWallis (Table me, integer column, integer factorColumn, double *out_prob, double *out_kruskalWallis, double *out_df) {
	try {
		Melder_require (column > 0 && column <= my numberOfColumns,
			U"Invalid column number.");
		Melder_require (factorColumn > 0 && factorColumn <= my numberOfColumns && factorColumn != column,
			U"Invalid group column number.");

		const integer numberOfData = my rows.size;
		Table_numericize_Assert (me, column);
		autoVEC data = newVECraw (numberOfData);
		autoStringsIndex levels = Table_to_StringsIndex_column (me, factorColumn);
		const integer numberOfLevels = levels -> classes->size;
		
		Melder_require (numberOfLevels > 1, 
			U"There should be at least two levels.");

		for (integer irow = 1; irow <= numberOfData; irow ++)
			data [irow] = my rows.at [irow] -> cells [column]. number;

		NUMsortTogether <double, integer> (data.get(), levels -> classIndex.get());
		NUMrank (data.get());
		/*
			Get correctionfactor for ties (Hays pg. 831)
			W. Hays (1988), Statistics, Fourth edition, Sauders college publishing.
		*/
		longdouble c = 0.0;
		integer jt, j = 1;
		while (j < numberOfData) {
        	for (jt = j + 1; jt <= numberOfData && data [jt] == data [j]; jt ++);
        	const double multiplicity = jt - j;
			if (multiplicity > 1)
				c += multiplicity * (multiplicity *multiplicity - 1.0);
        	j = jt;
		}
		const double tiesCorrection = 1.0 - (double) c / (numberOfData * (numberOfData * numberOfData - 1.0));

		autoINTVEC factorLevelSizes = newINTVECzero (numberOfLevels);
		autoVEC factorLevelSums = newVECzero (numberOfLevels);
		autoINTVEC ties = newINTVECzero (numberOfLevels);
		for (integer i = 1; i <= numberOfData; i ++) {
			const integer index = levels -> classIndex [i];
			factorLevelSizes [index] ++;
			factorLevelSums [index] += data [i];
		}

		longdouble kruskalWallis = 0.0;
		for (integer level = 1; level <= numberOfLevels; level ++) {
			if (factorLevelSizes [level] < 2) {
				SimpleString ss = (SimpleString) levels -> classes->at [level];   // FIXME cast
				Melder_throw (U"Group ", ss -> string.get(), U" has fewer than two cases.");
			}
			/*
				factorLevelMeans * groupMean * factorLevelSizes
			*/
			kruskalWallis += factorLevelSums [level] * factorLevelSums [level] / factorLevelSizes [level];
		}
		kruskalWallis = (12.0 / (numberOfData * (numberOfData + 1.0))) * kruskalWallis - 3.0 * (numberOfData + 1);
		kruskalWallis /= tiesCorrection;
		const double df = numberOfLevels - 1.0;
		if (out_df)
			*out_df = df;
		if (out_kruskalWallis)
			*out_kruskalWallis = (double) kruskalWallis;
		if (out_prob)
			*out_prob = NUMchiSquareQ ((double) kruskalWallis, df);

		autoTable him = Table_createWithColumnNames (numberOfLevels, U"Group(R) Sums(R) Cases");
		for (integer irow = 1; irow <= numberOfLevels; irow ++) {
			const SimpleString ss = (SimpleString) levels -> classes->at [irow];
			Table_setStringValue  (him.get(), irow, 1, ss -> string.get());
			Table_setNumericValue (him.get(), irow, 2, factorLevelSums [irow]);
			Table_setNumericValue (him.get(), irow, 3, factorLevelSizes [irow]);
		}
		Table_numericize_Assert (him.get(), 2);
		Table_numericize_Assert (him.get(), 3);
		return him;
	} catch (MelderError) {
		Melder_throw (me, U": no one-way Kruskal-Wallis performed.");
	}
}

// Table with Group Means Cases
static void _Table_postHocTukeyHSD (Table me, double sumOfSquaresWithin, double degreesOfFreedomWithin, autoTable *out_meansDiff, autoTable *out_meansDiffProbabilities) {
	try {
		Table_numericize_Assert (me, 2);
		Table_numericize_Assert (me, 3);
		const integer numberOfMeans = my rows.size;
		autoVEC means = newVECraw (numberOfMeans);
		autoVEC cases = newVECraw (numberOfMeans);
		autoTable meansD = Table_create (numberOfMeans - 1, numberOfMeans);
		for (integer i = 1; i <= numberOfMeans; i ++) {
			const TableRow row = my rows.at [i];
			means [i] = row -> cells [2]. number;
			cases [i] = row -> cells [3]. number;
		}
		for (integer i = 1; i <= numberOfMeans - 1; i ++) {
			Table_setStringValue (meansD.get(), i, 1, my rows.at [i] -> cells [1]. string.get());
			Table_setColumnLabel (meansD.get(), i + 1, my rows.at [i + 1] -> cells [1]. string.get());
		}

		for (integer irow = 1; irow <= numberOfMeans - 1; irow ++) {
			for (integer icol = irow + 1; icol <= numberOfMeans; icol ++) {
				const double dif = fabs (means [irow] - means [icol]);
				Table_setNumericValue (meansD.get(), irow, icol, dif);
			}
		}
		autoTable meansP = Data_copy (meansD.get());
		for (integer irow = 1; irow <= numberOfMeans - 1; irow ++) {
			for (integer icol = irow + 1; icol <= numberOfMeans; icol ++) {
				/*
					Tukey-Kramer correction for unequal sample sizes
				*/
				const double oneOverNstar =  0.5 * (1.0 / cases [icol] + 1.0 / cases [irow]);
				const double s = sqrt (sumOfSquaresWithin * oneOverNstar);
				const double q = fabs (means [irow] - means [icol]) / s;
				const double p = NUMtukeyQ (q, numberOfMeans, degreesOfFreedomWithin, 1);
				Table_setNumericValue (meansP.get(), irow, icol, p);
			}
		}
		if (out_meansDiff)
			*out_meansDiff = meansD.move();
		if (out_meansDiffProbabilities)
			*out_meansDiffProbabilities = meansP.move();
	} catch (MelderError) {
		Melder_throw (me, U": no post-hoc performed.");
	}
}

// expect 6 columns, first text others numeric
void Table_printAsAnovaTable (Table me) {
	autoMelderString s;
	const integer width [7] = { 0, 25, 15, 10, 15, 10, 10 };
	if (my numberOfColumns < 6)
		return;
	MelderInfo_writeLine (
		Melder_pad (width [1], U"Source"), U"\t",
		Melder_pad (width [2], U"SS"), U"\t",
		Melder_pad (width [3], U"Df"), U"\t",
		Melder_pad (width [4], U"MS"), U"\t",
		Melder_pad (width [5], U"F"), U"\t",
		Melder_pad (width [6], U"P")
	);
	for (integer icol = 2; icol <= 6; icol ++)
		Table_numericize_Assert (me, icol);

	for (integer i = 1; i <= my rows.size; i ++) {
		const TableRow row = my rows.at [i];
		MelderString_copy (& s, Melder_padOrTruncate (width [1], row -> cells [1]. string.get()), U"\t");
		for (integer j = 2; j <= 6; j ++) {
			const double value = row -> cells [j]. number;
			if (isdefined (value))
				MelderString_append (& s, Melder_pad (width [j], Melder_single (value)), j == 6 ? U"" : U"\t");
			else
				MelderString_append (& s, Melder_pad (width [j], U""), j == 6 ? U"" : U"\t");
		}
		MelderInfo_writeLine (s.string);
	}
}

void Table_printAsMeansTable (Table me) {
	autoMelderString s;
	for (integer icol = 2; icol <= my numberOfColumns; icol ++)
		Table_numericize_Assert (me, icol);

	for (integer j = 1; j <= my numberOfColumns; j ++)
		MelderString_append (& s,
			Melder_padOrTruncate (10, ( my columnHeaders [j]. label ? my columnHeaders [j]. label.get() : U"" )),
			( j == my numberOfColumns ? U"" : U"\t" ));

	MelderInfo_writeLine (s.string);
	for (integer i = 1; i <= my rows.size; i ++) {
		const TableRow row = my rows.at [i];
		MelderString_copy (& s, Melder_padOrTruncate (10, row -> cells [1]. string.get()), U"\t");
		for (integer j = 2; j <= my numberOfColumns; j ++) {
			const double value = row -> cells [j].number;
			if (isdefined (value))
				MelderString_append (& s, Melder_pad (10, Melder_half (value)),
					( j == my numberOfColumns ? U"" : U"\t" ));
			else
				MelderString_append (& s, Melder_pad (10, U""), ( j == my numberOfColumns ? U"" : U"\t" ));
		}
		MelderInfo_writeLine (s.string);
	}
}

/*
	W. Hays (1988), Statistics, Fourth edition, Sauders college publishing.
	Chapter 10.14 outlines 12 steps to calculate the one-way analysis of variance.
*/
autoTable Table_getOneWayAnalysisOfVarianceF (Table me, integer column, integer factorColumn, autoTable *means, autoTable *meansDiff, autoTable *meansDiffProbabilities) {
	try {
		Melder_require (column > 0 && column <= my numberOfColumns,
			U"Invalid column number.");
		Melder_require (factorColumn > 0 && factorColumn <= my numberOfColumns && factorColumn != column,
			U"Invalid group column number.");
		const integer numberOfData = my rows.size;
		Table_numericize_Assert (me, column);
		autoStringsIndex levels = Table_to_StringsIndex_column (me, factorColumn);
		// copy data from Table
		autoVEC data = newVECraw (numberOfData);
		for (integer irow = 1; irow <= numberOfData; irow ++)
			data [irow] = my rows.at [irow] -> cells [column]. number;
		const integer numberOfLevels = levels -> classes->size;
		Melder_require (numberOfLevels > 1,
			U"There should be at least two levels.");
		autoINTVEC factorLevelSizes = newINTVECzero (numberOfLevels);
		autoVEC factorLevelMeans = newVECzero (numberOfLevels);

		const longdouble sumOfSquares = NUMsum2 (data.get());  // step 2
		const longdouble sumOfRawScores = NUMsum (data.get()); // step 3
		for (integer i = 1; i <= numberOfData; i ++) {
			const integer index = levels -> classIndex [i];
			factorLevelSizes [index] ++;
			factorLevelMeans [index] += data [i];
		}

		longdouble c = 0.0; // step 5.
		for (integer j = 1; j <= numberOfLevels; j ++) {
			if (factorLevelSizes [j] < 2) {
				const SimpleString ss = (SimpleString) levels -> classes->at [j];
				Melder_throw (U"Level \"", ss -> string.get(), U"\" has less then two members.");
			}
			c += factorLevelMeans [j] * factorLevelMeans [j] / factorLevelSizes [j]; // order of these two is important!
			factorLevelMeans [j] /= factorLevelSizes [j];
		}
		
		const double sumOfSquares_total = double (sumOfSquares - sumOfRawScores * sumOfRawScores / numberOfData); // step 6
		const double sumOfSquares_between = double (c - sumOfRawScores * sumOfRawScores / numberOfData); // step 7
		const double sumOfSquares_within = double (sumOfSquares - c); // step 8
		const double degreesOfFreedom_within = numberOfData - numberOfLevels;
		const double degreesOfFreedom_between = numberOfLevels - 1;

		autoTable anova = Table_createWithColumnNames (3, U"Source SS Df MS F P");
		const integer col_s = 1, col_ss = 2, col_df = 3, col_ms = 4, col_f = 5, col_p = 6;
		const integer row_b = 1, row_w = 2, row_t = 3;
		Table_setStringValue (anova.get(), row_b, col_s, U"Between");
		Table_setStringValue (anova.get(), row_w, col_s, U"Within");
		Table_setStringValue (anova.get(), row_t, col_s, U"Total");

		Table_setNumericValue (anova.get(), row_b, col_ss, sumOfSquares_between);
		Table_setNumericValue (anova.get(), row_b, col_df, degreesOfFreedom_between);
		const double ms_between = sumOfSquares_between / degreesOfFreedom_between; // step 9
		Table_setNumericValue (anova.get(), row_b, col_ms, ms_between);

		Table_setNumericValue (anova.get(), row_w, col_ss, sumOfSquares_within);
		Table_setNumericValue (anova.get(), row_w, col_df, degreesOfFreedom_within);
		const double ms_within = sumOfSquares_within / degreesOfFreedom_within; // step 10
		Table_setNumericValue (anova.get(), row_w, col_ms, ms_within);
		const double fisherF = ms_between / ms_within; // step 11
		const double probability = NUMfisherQ (fisherF, degreesOfFreedom_between, degreesOfFreedom_within); // step 12

		Table_setNumericValue (anova.get(), row_b, col_f, fisherF);
		Table_setNumericValue (anova.get(), row_b, col_p, probability);

		Table_setNumericValue (anova.get(), row_t, col_ss, sumOfSquares_total);
		Table_setNumericValue (anova.get(), row_t, col_df, degreesOfFreedom_within + degreesOfFreedom_between);

		autoTable ameans = Table_createWithColumnNames (numberOfLevels, U"Group Mean Cases");
		for (integer irow = 1; irow <= numberOfLevels; irow ++) {
			const SimpleString name = (SimpleString) levels -> classes->at [irow];
			Table_setStringValue (ameans.get(), irow, 1, name -> string.get());
			Table_setNumericValue (ameans.get(), irow, 2, factorLevelMeans [irow]);
			Table_setNumericValue (ameans.get(), irow, 3, factorLevelSizes [irow]);
		}
		const integer columns [1+1] { 0, 2 };   // sort by column 2
		Table_sortRows_Assert (ameans.get(), constINTVEC (columns, 1));
		_Table_postHocTukeyHSD (ameans.get(), ms_within, degreesOfFreedom_within, meansDiff, meansDiffProbabilities);
		if (means)
			*means = ameans.move();
		return anova;
	} catch (MelderError) {
		Melder_throw (me, U": no one-way anova performed.");
	}
}

autoTable Table_getTwoWayAnalysisOfVarianceF (Table me, integer column, integer factorColumnA, integer factorColumnB, autoTable *out_means, autoTable *out_levelSizes) {
	try {
		Melder_require (column > 0 && column <= my numberOfColumns,
			U"Invalid column number.");
		Melder_require (factorColumnA > 0 && factorColumnA <= my numberOfColumns && factorColumnA != column,
			U"Invalid A group column number.");
		Melder_require (factorColumnB > 0 && factorColumnB <= my numberOfColumns && factorColumnB != column && factorColumnA != factorColumnB,
			U"Invalid B group column number.");

		const char32 *label_A = my columnHeaders [factorColumnA]. label.get();
		const char32 *label_B = my columnHeaders [factorColumnB]. label.get();

		const integer numberOfData = my rows.size;
		Table_numericize_Assert (me, column);
		autoStringsIndex levelsA = Table_to_StringsIndex_column (me, factorColumnA);
		autoStringsIndex levelsB = Table_to_StringsIndex_column (me, factorColumnB);
		/*
			Copy data from Table
		*/
		autoVEC data = newVECraw (numberOfData);
		for (integer irow = 1; irow <= numberOfData; irow ++)
			data [irow] = my rows.at [irow] -> cells [column]. number;
		const integer numberOfLevelsA = levelsA -> classes -> size;
		const integer numberOfLevelsB = levelsB -> classes -> size;
		
		Melder_require (numberOfLevelsA > 1,
			U"There should be at least two levels in \"", label_A, U"\".");
		Melder_require (numberOfLevelsB > 1,
			U"There should be at least two levels in \"", label_B, U"\".");

		/*
			Formula's according to A. Khuri (1998), Unweighted sums of squares
			in unbalanced analysis of variance, Journal of Statistical Planning
			and Inference (74): 135--147.

			Model:

			y [i,j,k] = mu + alpha [i] + beta [j] + gamma [i,j] + eps [i,j,k]
			i=1..r, j = 1..s, k=1..n [i,j]
		
			ss(alpha) = nh * s * sum(i=1..r, (ystar [i.]-ystar [..])^2)
			ss(beta)  = nh * r * sum(j=1..s, (ystar [.j]-ystar [..])^2)
			ss(alpha,beta) = nh * sum (i=1..r, sum (j=1..s, (ymean [ij.]-ystar [i.] - ystar [.j] + ystar [..])^2)),

			where

			nh = r * s / sum (i=1..r, sum (j=1..s, 1/n [ij])),
			ymean [ij.] = sum (k=1..n [ij], y [ijk]/n [ij])
			ystar [i.] = sum (j=1..s, ymean [ij.]) / s,
			ystar [.j] = sum (i=1..r, ymean [ij.]) / r,
			ystar [..] = sum (i=1..r, sum (j=1..s, ymean [ij.])) / (r * s)
		 */

		autoMAT factorLevelSizes = newMATzero (numberOfLevelsA + 1, numberOfLevelsB + 1); // sum + weighted sum
		/*
			Extra column for ystar [i.], extra row for ystar [.j]
		*/
		autoMAT factorLevelMeans = newMATzero (numberOfLevelsA + 1, numberOfLevelsB + 1); // weighted mean + mean

		for (integer k = 1; k <= numberOfData; k ++) {
			const integer indexA = levelsA -> classIndex [k];
			const integer indexB = levelsB -> classIndex [k];
			factorLevelSizes [indexA] [indexB] ++;
			factorLevelMeans [indexA] [indexB] += data [k];
		}
		/*
			Check for unfilled cells and calculate cell means
		*/
		longdouble nh = 0.0;
		for (integer i = 1; i <= numberOfLevelsA; i ++) {
			for (integer j = 1; j <= numberOfLevelsB; j ++) {
				Melder_require (factorLevelSizes [i] [j] > 0,
					U"Level ", levelsA -> classes->at [i], U" of ", levelsA -> classes->at [j],
					U" has no data.");
				factorLevelMeans [i] [j] /= factorLevelSizes [i] [j];
				nh += 1.0 / factorLevelSizes [i] [j];
			}
		}
		nh = numberOfLevelsA * numberOfLevelsB / nh;
		/*
			Row marginals (ystar [i.])
		*/
		longdouble sum = 0.0;
		for (integer i = 1; i <= numberOfLevelsA; i ++) {
			for (integer j = 1; j <= numberOfLevelsB; j ++) {
				factorLevelMeans [i] [numberOfLevelsB + 1] += factorLevelMeans [i] [j];
				sum += factorLevelMeans [i] [j];
				factorLevelSizes [i] [numberOfLevelsB + 1] += factorLevelSizes [i] [j];
			}
			factorLevelMeans [i] [numberOfLevelsB + 1] /= numberOfLevelsB;
		}
		const double mean = double (sum) / (numberOfLevelsA * numberOfLevelsB);
		factorLevelMeans [numberOfLevelsA + 1] [numberOfLevelsB + 1] = mean;
		factorLevelSizes [numberOfLevelsA + 1] [numberOfLevelsB + 1] = numberOfData;
		/*
			Column marginals (ystar [.j])
		*/
		for (integer j = 1; j <= numberOfLevelsB; j ++) {
			for (integer i = 1; i <= numberOfLevelsA; i ++) {
				factorLevelMeans [numberOfLevelsA + 1] [j] += factorLevelMeans [i] [j];
				factorLevelSizes [numberOfLevelsA + 1] [j] += factorLevelSizes [i] [j];
			}
			factorLevelMeans [numberOfLevelsA + 1] [j] /= numberOfLevelsA;
		}
		/*
			The sums of squares
		*/
		longdouble ss_T = 0.0;
		for (integer k = 1; k <= numberOfData; k ++) {
			const double dif = data [k] - mean;
			ss_T += dif * dif;
		}

		longdouble ss_A = 0.0;
		for (integer i = 1; i <= numberOfLevelsA; i ++) {
			const double dif = factorLevelMeans [i] [numberOfLevelsB + 1] - mean;
			ss_A += dif * dif;
		}
		ss_A *= nh * numberOfLevelsB;

		longdouble ss_B = 0.0;
		for (integer j = 1; j <= numberOfLevelsB; j ++) {
			const double dif = factorLevelMeans [numberOfLevelsA + 1] [j] - mean;
			ss_B += dif * dif;
		}
		ss_B *= nh * numberOfLevelsA;

		longdouble ss_AB = 0.0;
		for (integer i = 1; i <= numberOfLevelsA; i ++) {
			for (integer j = 1; j <= numberOfLevelsB; j ++) {
				const double dif = factorLevelMeans [i] [j] - factorLevelMeans [i] [numberOfLevelsB + 1] - factorLevelMeans [numberOfLevelsA + 1] [j] + mean;
				ss_AB += dif * dif;
			}
		}
		ss_AB *= nh;

		double ss_E = double (ss_T - ss_A - ss_B - ss_AB);
		/*
			Are there any replications? If not then the error term is the AB interaction.
		*/
		bool replications = true;
		if (factorLevelSizes [numberOfLevelsA + 1] [1] == numberOfLevelsA)
			replications = false;
		/*
			Construct the means Table (numberOfLevelsA+1)x(numberOfLevelsB + 1 + 1)
		*/
		autoTable ameans = Table_createWithoutColumnNames (numberOfLevelsA + 1, numberOfLevelsB + 1 + 1);
		for (integer k = 2; k <= numberOfLevelsB + 1; k ++) {
			const SimpleString name = (SimpleString) levelsB -> classes->at [k - 1];
			Table_setColumnLabel (ameans.get(), k, name -> string.get());
		}
		Table_setColumnLabel (ameans.get(), numberOfLevelsB + 1 + 1, U"Mean");
		for (integer j = 1; j <= numberOfLevelsA; j ++) {
			const SimpleString name = (SimpleString) levelsA -> classes->at [j];
			Table_setStringValue (ameans.get(), j, 1, name -> string.get());
		}
		Table_setStringValue (ameans.get(), numberOfLevelsA + 1, 1, U"Mean");

		for (integer i = 1; i <= numberOfLevelsA + 1; i ++)
			for (integer j = 1; j <= numberOfLevelsB + 1; j ++)
				Table_setNumericValue (ameans.get(), i, j + 1, factorLevelMeans [i] [j]);


		if (out_levelSizes) {
			autoTable asizes = Data_copy (ameans.get());
			Table_setColumnLabel (asizes.get(), numberOfLevelsB + 1 + 1, U"Total");
			Table_setStringValue (asizes.get(), numberOfLevelsA + 1, 1, U"Total");
			for (integer i = 1; i <= numberOfLevelsA + 1; i ++)
				for (integer j = 1; j <= numberOfLevelsB + 1; j ++)
					Table_setNumericValue (asizes.get(), i, j + 1, factorLevelSizes [i] [j]);
			*out_levelSizes = asizes.move();
		}

		autoTable anova = Table_createWithColumnNames (replications ? 5 : 4, U"Source SS Df MS F P");
		const integer col_s = 1, col_ss = 2, col_df = 3, col_ms = 4, col_f = 5, col_p = 6;
		const integer row_A = 1, row_B = 2, row_AB = 3, row_E = replications ? 4 : 3, row_t = replications ? 5 : 4;
		Table_setStringValue (anova.get(), row_A, col_s, label_A);
		Table_setStringValue (anova.get(), row_B, col_s, label_B);
		Table_setStringValue (anova.get(), row_AB, col_s, Melder_cat (label_A, U" x ", label_B));
		if (replications)
			Table_setStringValue (anova.get(), row_E, col_s, U"Error");

		Table_setStringValue (anova.get(), row_t, col_s, U"Total");

		double dof_A = numberOfLevelsA - 1, ms_A = ss_A / dof_A;
		Table_setNumericValue (anova.get(), row_A, col_ss, ss_A);
		Table_setNumericValue (anova.get(), row_A, col_df, dof_A);
		Table_setNumericValue (anova.get(), row_A, col_ms, ms_A);

		double dof_B = numberOfLevelsB - 1, ms_B = ss_B / dof_B;
		Table_setNumericValue (anova.get(), row_B, col_ss, ss_B);
		Table_setNumericValue (anova.get(), row_B, col_df, dof_B);
		Table_setNumericValue (anova.get(), row_B, col_ms, ms_B);

		const double dof_AB = dof_A * dof_B;
		double dof_E, ms_E, ms_AB = 0.0;
		if (replications) {
			ms_AB = ss_AB / dof_AB;
			dof_E = numberOfData - dof_A - dof_B - dof_AB - 1;
			ms_E = ss_E / dof_E;
			Table_setNumericValue (anova.get(), row_AB, col_ss, ss_AB);
			Table_setNumericValue (anova.get(), row_AB, col_df, dof_AB);
			Table_setNumericValue (anova.get(), row_AB, col_ms, ms_AB);
		} else {
			ss_E = ss_AB;
			dof_E = numberOfData - dof_A - dof_B - 1;
			ms_E = ss_AB / dof_E;
		}
		Table_setNumericValue (anova.get(), row_E, col_ss, ss_E);
		Table_setNumericValue (anova.get(), row_E, col_df, dof_E);
		Table_setNumericValue (anova.get(), row_E, col_ms, ms_E);
		Table_setNumericValue (anova.get(), row_t, col_ss, ss_T);
		Table_setNumericValue (anova.get(), row_t, col_df, numberOfData - 1);
		/*
			Get f and p values wrt ms_E
		*/
		const double f_A = ms_A / ms_E;
		const double f_B = ms_B / ms_E;
		const double p_A = NUMfisherQ (f_A, dof_A, dof_E);
		const double p_B = NUMfisherQ (f_B, dof_B, dof_E);
		Table_setNumericValue (anova.get(), row_A, col_f, f_A);
		Table_setNumericValue (anova.get(), row_B, col_f, f_B);
		Table_setNumericValue (anova.get(), row_A, col_p, p_A);
		Table_setNumericValue (anova.get(), row_B, col_p, p_B);
		if (replications) {
			const double f_AB = ms_AB / ms_E;
			const double p_AB = NUMfisherQ (f_AB, dof_AB, dof_E);
			Table_setNumericValue (anova.get(), row_AB, col_f, f_AB);
			Table_setNumericValue (anova.get(), row_AB, col_p, p_AB);
		}
		if (out_means)
			*out_means = ameans.move();
		return anova;
	} catch (MelderError) {
		Melder_throw (me, U": two-way anova not created.");
	}
}

void Table_normalProbabilityPlot (Table me, Graphics g, integer column, integer numberOfQuantiles,
	double numberOfSigmas, double labelSize, conststring32 label, bool garnish)
{
	try {
		if (column < 1 || column > my numberOfColumns)
			return;
		Table_numericize_Assert (me, column);
		const integer numberOfData = my rows.size;
		autoVEC data = newVECraw (numberOfData);
		for (integer irow = 1; irow <= numberOfData; irow ++)
			data [irow] = my rows.at [irow] -> cells [column]. number;

		MelderGaussianStats stats = NUMmeanStdev (data.all());
		double xmin = 100.0, xmax = -xmin, ymin = 1e308, ymax = -ymin;
		if (numberOfSigmas != 0) {
			xmin = -numberOfSigmas;
			xmax =  numberOfSigmas;
			ymin = stats.mean - numberOfSigmas * stats.stdev;
			ymax = stats.mean + numberOfSigmas * stats.stdev;
		}
		VECsort_inplace (data.get());
		numberOfQuantiles = numberOfData < numberOfQuantiles ? numberOfData : numberOfQuantiles;
		autoTableOfReal thee = TableOfReal_create (numberOfQuantiles, 2);
		TableOfReal_setColumnLabel (thee.get(), 1, U"Normal distribution quantiles");
		TableOfReal_setColumnLabel (thee.get(), 2, my columnHeaders [column]. label.get());
		const double un = pow (0.5, 1.0 / numberOfQuantiles);
		for (integer irow = 1; irow <= numberOfQuantiles; irow ++) {
			const double ui = irow == 1 ? 1.0 - un : (irow == numberOfQuantiles ? un : (irow - 0.3175) / (numberOfQuantiles + 0.365));
			const double q = NUMquantile (numberOfData, data.at, ui);
			const double zq = - NUMinvGaussQ (ui);
			thy data [irow] [1] = zq; // along x
			thy data [irow] [2] = q;  // along y
			if (numberOfSigmas == 0.0) {
				xmin = zq < xmin ? zq : xmin;
				xmax = zq > xmax ? zq : xmax;
				ymin = q < ymin ? q : ymin;
				ymax = q > ymax ? q : ymax;
			}
		}

		TableOfReal_drawScatterPlot (thee.get(), g, 1, 2, 1, numberOfQuantiles, xmin, xmax, ymin, ymax, labelSize, false, label, garnish);

		Graphics_setInner (g);
		Graphics_setLineType (g, Graphics_DOTTED);
		Graphics_line (g, xmin, ymin, xmax, ymax);
		Graphics_setLineType (g, Graphics_DRAWN);
		Graphics_unsetInner (g);

	} catch (MelderError) {
		Melder_clearError ();   // drawing errors shall be ignored
	}
}

void Table_quantileQuantilePlot_betweenLevels (Table me, Graphics g,
	integer dataColumn, integer factorColumn, conststring32 xlevel, conststring32 ylevel, integer numberOfQuantiles,
	double xmin, double xmax, double ymin, double ymax, double labelSize, conststring32 plotLabel, bool garnish)
{
	try {
		if (dataColumn < 1 || dataColumn > my numberOfColumns || factorColumn < 1 || factorColumn > my numberOfColumns)
			return;
		Table_numericize_Assert (me, dataColumn);
		const integer numberOfData = my rows.size;
		autoVEC xdata = newVECraw (numberOfData);
		autoVEC ydata = newVECraw (numberOfData);
		integer xnumberOfData = 0, ynumberOfData = 0;
		for (integer irow = 1; irow <= numberOfData; irow ++) {
			char32 *label = my rows.at [irow] -> cells [factorColumn]. string.get();
			const double val = my rows.at [irow] -> cells [dataColumn]. number;
			if (Melder_equ (label, xlevel))
				xdata [ ++ xnumberOfData] = val;
			else if (Melder_equ (label, ylevel))
				ydata [ ++ ynumberOfData] = val;
		}
		if (xnumberOfData == 0 || ynumberOfData == 0)
			return;
		xdata.resize (xnumberOfData);
		ydata.resize (ynumberOfData);
		if (xmin == xmax) {
			NUMextrema (xdata.part (1, xnumberOfData), & xmin, & xmax);
			if (xmin == xmax) {
				xmin -= 1.0;
				xmax += 1.0;
			}
		}
		if (ymin == ymax) {
			NUMextrema (ydata.part (1, ynumberOfData), & ymin, & ymax);
			if (ymin == ymax) {
				ymin -= 1.0;
				ymax += 1.0;
			}
		}
		Graphics_setWindow (g, xmin, xmax, ymin, ymax);
		Graphics_setInner (g);
		Graphics_quantileQuantilePlot (g, numberOfQuantiles, xdata.get(), ydata.get(),
			xmin, xmax, ymin, ymax, labelSize, plotLabel);
		Graphics_unsetInner (g);
		if (garnish) {
			Graphics_drawInnerBox (g);

			Graphics_textBottom (g, true, Melder_cat (my columnHeaders [dataColumn]. label.get(), U" (", xlevel, U")"));
			Graphics_marksBottom (g, 2, true, true, false);

			Graphics_textLeft (g, true, Melder_cat (my columnHeaders [dataColumn]. label.get(), U" (", ylevel, U")"));
			Graphics_marksLeft (g, 2, true, true, false);
		}
	} catch (MelderError) {
		Melder_clearError ();   // drawing errors shall be ignored
	}
}

void Table_quantileQuantilePlot (Table me, Graphics g, integer xcolumn, integer ycolumn, integer numberOfQuantiles,
	double xmin, double xmax, double ymin, double ymax, double labelSize, conststring32 plotLabel, bool garnish)
{
	try {
		if (xcolumn < 1 || xcolumn > my numberOfColumns || ycolumn < 1 || ycolumn > my numberOfColumns)
			return;
		Table_numericize_Assert (me, xcolumn);
		Table_numericize_Assert (me, ycolumn);
		const integer numberOfData = my rows.size;
		autoVEC xdata = newVECraw (numberOfData);
		autoVEC ydata = newVECraw (numberOfData);
		for (integer irow = 1; irow <= numberOfData; irow ++) {
			xdata [irow] = my rows.at [irow] -> cells [xcolumn]. number;
			ydata [irow] = my rows.at [irow] -> cells [ycolumn]. number;
		}
		if (xmin == xmax) {
			NUMextrema (xdata.get(), & xmin, & xmax);
			if (xmin == xmax) {
				xmin -= 1.0;
				xmax += 1.0;
			}
		}
		if (ymin == ymax) {
			NUMextrema (ydata.get(), & ymin, & ymax);
			if (ymin == ymax) {
				ymin -= 1.0;
				ymax += 1.0;
			}
		}
		Graphics_setWindow (g, xmin, xmax, ymin, ymax);
		Graphics_setInner (g);
		Graphics_quantileQuantilePlot (g, numberOfQuantiles, xdata.get(), ydata.get(),
				xmin, xmax, ymin, ymax, labelSize, plotLabel);
		Graphics_unsetInner (g);
		if (garnish) {
			Graphics_drawInnerBox (g);
			if (my columnHeaders [xcolumn].label)
				Graphics_textBottom (g, true, my columnHeaders [xcolumn]. label.get());

			Graphics_marksBottom (g, 2, true, true, false);
			if (my columnHeaders [ycolumn].label)
				Graphics_textLeft (g, true, my columnHeaders [ycolumn]. label.get());

			Graphics_marksLeft (g, 2, true, true, false);
		}
	} catch (MelderError) {
		Melder_clearError ();   // drawing errors shall be ignored
	}
}

void Table_boxPlots (Table me, Graphics g, integer dataColumn, integer factorColumn, double ymin, double ymax, bool garnish) {
	try {
		if (dataColumn < 1 || dataColumn > my numberOfColumns || factorColumn < 1 || factorColumn > my numberOfColumns)
			return;
		Table_numericize_Assert (me, dataColumn);
		const integer numberOfData = my rows.size;
		autoStringsIndex si = Table_to_StringsIndex_column (me, factorColumn);
		const integer numberOfLevels = si -> classes->size;
		if (ymin == ymax) {
			ymax = Table_getMaximum (me, dataColumn);
			ymin = Table_getMinimum (me, dataColumn);
			if (ymax == ymin) {
				ymax += 1.0;
				ymin -= 1.0;
			}
		}
		Graphics_setWindow (g, 1.0 - 0.5, numberOfLevels + 0.5, ymin, ymax);
		Graphics_setInner (g);
		autoVEC data = newVECraw (numberOfData);
		for (integer ilevel = 1; ilevel <= numberOfLevels; ilevel ++) {
			integer numberOfDataInLevel = 0;
			for (integer k = 1; k <= numberOfData; k ++)
				if (si -> classIndex [k] == ilevel)
					data [ ++ numberOfDataInLevel] = Table_getNumericValue_Assert (me, k, dataColumn);
			Graphics_boxAndWhiskerPlot (g, data.part (1, numberOfDataInLevel), ilevel, 0.2, 0.35, ymin, ymax);
		}
		Graphics_unsetInner (g);
		if (garnish) {
			Graphics_drawInnerBox (g);
			for (integer ilevel = 1; ilevel <= numberOfLevels; ilevel ++) {
				const SimpleString ss = (SimpleString) si -> classes->at [ilevel];
				Graphics_markBottom (g, ilevel, false, true, false, ss -> string.get());
			}
			Graphics_marksLeft (g, 2, true, true, false);
		}
	} catch (MelderError) {
		Melder_clearError ();   // drawing errors shall be ignored
	}
}

void Table_boxPlotsWhere (Table me, Graphics g, conststring32 dataColumns_string, integer factorColumn, double ymin, double ymax,
	bool garnish, conststring32 formula, Interpreter interpreter)
{
	try {
		autoINTVEC dataColumns = Table_getColumnIndicesFromColumnLabelString (me, dataColumns_string);
		if (factorColumn < 1 || factorColumn > my numberOfColumns)
			return;
		const integer numberOfSelectedColumns = dataColumns.size;
		Formula_compile (interpreter, me, formula, kFormula_EXPRESSION_TYPE_NUMERIC, true);
		Formula_Result result;
		const integer numberOfData = my rows.size;
		autoStringsIndex si = Table_to_StringsIndex_column (me, factorColumn);
		const integer numberOfLevels = si -> classes->size;
		if (ymin == ymax) {
			ymin = 1e308, ymax = - ymin;
			for (integer icol = 1; icol <= numberOfSelectedColumns; icol ++) {
				const double ymaxi = Table_getMaximum (me, dataColumns [icol]);
				const double ymini = Table_getMinimum (me, dataColumns [icol]);
				if (ymaxi > ymax)
					ymax = ymaxi;
				if (ymini < ymin)
					ymin = ymini;
			}
			if (ymax == ymin) {
				ymax += 1.0;
				ymin -= 1.0;
			}
		}
		Graphics_setWindow (g, 1.0 - 0.5, numberOfLevels + 0.5, ymin, ymax);
		Graphics_setInner (g);
		const double boxWidth = 4.0, spaceBetweenBoxesInGroup = 1.0, barWidth = boxWidth / 3.0;
		const double spaceBetweenGroupsdiv2 = 3.0 / 2.0;
		const double widthUnit = 1.0 / (numberOfSelectedColumns * boxWidth + (numberOfSelectedColumns - 1) * spaceBetweenBoxesInGroup + spaceBetweenGroupsdiv2 + spaceBetweenGroupsdiv2);
		autoVEC data = newVECraw (numberOfData);
		for (integer ilevel = 1; ilevel <= numberOfLevels; ilevel ++) {
			const double xlevel = ilevel;
			for (integer icol = 1; icol <= numberOfSelectedColumns; icol ++) {
				integer numberOfDataInLevelColumn = 0;
				for (integer irow = 1; irow <= numberOfData; irow ++) {
					if (si -> classIndex [irow] == ilevel) {
						Formula_run (irow, dataColumns [icol], & result);
						if (result. numericResult != 0.0)
							data [++ numberOfDataInLevelColumn] = Table_getNumericValue_Assert (me, irow, dataColumns [icol]);
					}
				}
				if (numberOfDataInLevelColumn > 0) {
					/*
						Determine position
					*/
					const double xc = xlevel - 0.5 + (spaceBetweenGroupsdiv2 + (icol - 1) * (boxWidth + spaceBetweenBoxesInGroup) + boxWidth / 2) * widthUnit;
					Graphics_boxAndWhiskerPlot (g, data.part (1, numberOfDataInLevelColumn), xc, 0.5 * barWidth * widthUnit , 0.5 * boxWidth * widthUnit, ymin, ymax);
				}
			}
		}
		Graphics_unsetInner (g);
		if (garnish) {
			Graphics_drawInnerBox (g);
			for (integer ilevel = 1; ilevel <= numberOfLevels; ilevel ++) {
				const SimpleString ss = (SimpleString) si -> classes->at [ilevel];
				Graphics_markBottom (g, ilevel, false, true, false, ss -> string.get());
			}
			Graphics_marksLeft (g, 2, true, true, false);
		}
	} catch (MelderError) {
		Melder_clearError ();   // drawing errors shall be ignored
	}
}

void Table_distributionPlotWhere (Table me, Graphics g, integer dataColumn, double minimum, double maximum, integer nBins, double freqMin, double freqMax, bool garnish, conststring32 formula, Interpreter interpreter) {
	try {
		if (dataColumn < 1 || dataColumn > my numberOfColumns)
			return;
		Formula_compile (interpreter, me, formula, kFormula_EXPRESSION_TYPE_NUMERIC, true);
		Formula_Result result;

		Table_numericize_Assert (me, dataColumn);
		integer mrow = 0;
		autoMatrix thee = Matrix_create (1.0, 1.0, 1, 1.0, 1.0, 0.0, my rows.size + 1.0, my rows.size, 1.0, 1.0);
		for (integer irow = 1; irow <= my rows.size; irow ++) {
			Formula_run (irow, dataColumn, & result);
			if (result. numericResult != 0.0)
				thy z [1] [++ mrow] = Table_getNumericValue_Assert (me, irow, dataColumn);
		}
		Matrix_drawDistribution (thee.get(), g, 0, 1, 0.5, mrow + 0.5, minimum, maximum, nBins, freqMin, freqMax, false, false);
		if (garnish) {
			Graphics_drawInnerBox (g);
			Graphics_marksBottom (g, 2, true, true, false);
			if (my columnHeaders [dataColumn]. label)
				Graphics_textBottom (g, true, my columnHeaders [dataColumn]. label.get());
			Graphics_marksLeft (g, 2, true, true, false);
			Graphics_textLeft (g, true, U"Number / bin");
		}

	} catch (MelderError) {
		Melder_clearError ();   // drawing errors shall be ignored
	}
}

static autoStrings itemizeColourString (conststring32 colourString) {
	/*
		Remove all spaces within { } so each {1,2,3} can be itemized
	*/
	static const conststring32 searchRE = U"\\{\\s*( [0-9.]+)\\s*,\\s*( [0-9.]+)\\s*,\\s*( [0-9.]+)\\s*\\}";
	regexp *compiledRE = CompileRE_throwable (searchRE, 0);
	autostring32 colourStringWithoutSpaces = newSTRreplace_regex (colourString, compiledRE, U"{\\1,\\2,\\3}", 0);
	autoStrings thee = Strings_createAsTokens (colourStringWithoutSpaces.get(), U" ");
	return thee;
}

static MelderColour Strings_colourToValue  (Strings me, integer index) {
	if (index < 0 || index > my numberOfStrings)
		return Melder_GREY;
	MelderColour result = MelderColour_fromColourNameOrNumberStringOrRGBString (my strings [index].get());
	if (! result.valid())
		return Melder_GREY;
	return result;
}

integer Table_getNumberOfRowsWhere (Table me, conststring32 formula, Interpreter interpreter) {
	integer numberOfRows = 0;
	Formula_compile (interpreter, me, formula, kFormula_EXPRESSION_TYPE_NUMERIC, true);
	Formula_Result result;
	for (integer irow = 1; irow <= my rows.size; irow ++) {
		Formula_run (irow, 1, & result);
		if (result. numericResult != 0.0)
			numberOfRows ++;
	}
	return numberOfRows;
}

autoINTVEC Table_findRowsMatchingCriterion (Table me, conststring32 formula, Interpreter interpreter) {
	try {
		const integer numberOfMatches = Table_getNumberOfRowsWhere (me, formula, interpreter);
		Melder_require (numberOfMatches > 0,
			U"No rows selected.");
		Formula_compile (interpreter, me, formula, kFormula_EXPRESSION_TYPE_NUMERIC, true);
		Formula_Result result;
		autoINTVEC selectedRows = newINTVECzero (numberOfMatches);
		integer n = 0;
		for (integer irow = 1; irow <= my rows.size; irow ++) {
			Formula_run (irow, 1, & result);
			if (result. numericResult != 0.0)
				selectedRows [ ++ n] = irow;
		}
		Melder_assert (n == numberOfMatches);
		return selectedRows;
	} catch (MelderError) {
		Melder_throw (me, U": cannot find matches.");
	}
}

void Table_barPlotWhere (Table me, Graphics g,
	conststring32 columnLabels, double ymin, double ymax, conststring32 factorColumn,
	double xoffsetFraction, double interbarFraction, double interbarsFraction, conststring32 colours,
	double angle, bool garnish, conststring32 formula, Interpreter interpreter)
{
	try {
		autoINTVEC columnIndexes = Table_getColumnIndicesFromColumnLabelString (me, columnLabels);
		const integer labelIndex = Table_findColumnIndexFromColumnLabel (me, factorColumn);
		autoStrings colourText = itemizeColourString (colours);   // removes all spaces within { } so each {} can be parsed as 1 item
		
		autoINTVEC selectedRows = Table_findRowsMatchingCriterion (me, formula, interpreter);
		if (ymax <= ymin) {   // autoscaling
			ymin = 1e308;
			ymax = - ymin;
			for (integer icol = 1; icol <= columnIndexes.size; icol ++) {
				double cmin, cmax;
				Table_columnExtremaFromSelectedRows (me, columnIndexes [icol], selectedRows.get(), & cmin, & cmax);
				if (cmin < ymin)
					ymin = cmin;
				if (cmax > ymax)
					ymax = cmax;
			}
			ymin = std::min (0.0, ymin);
			ymax = std::max (0.0, ymax);
		}
		Graphics_setInner (g);
		Graphics_setWindow (g, 0, 1, ymin, ymax);

		const integer numberOfGroups = selectedRows.size;
		const integer groupSize = columnIndexes.size;
		const double bar_width = 1.0 / (numberOfGroups * groupSize + 2.0 * xoffsetFraction + (numberOfGroups - 1) * interbarsFraction + numberOfGroups * (groupSize - 1) * interbarFraction);
		const double dx = (interbarsFraction + groupSize + (groupSize - 1) * interbarFraction) * bar_width;

		for (integer icol = 1; icol <= groupSize; icol ++) {
			const double xb = xoffsetFraction * bar_width + (icol - 1) * (1 + interbarFraction) * bar_width;
			double x1 = xb;
			MelderColour colour = Strings_colourToValue (colourText.get(), icol);
			for (integer irow = 1; irow <= selectedRows.size; irow ++) {
				const double x2 = x1 + bar_width;
				double y2 = Table_getNumericValue_Assert (me, selectedRows [irow], columnIndexes [icol]);
				y2 = y2 > ymax ? ymax : y2 < ymin ? ymin : y2;
				const double y1 = std::max (0.0, ymin);
				
				Graphics_setColour (g, colour);
				Graphics_fillRectangle (g, x1, x2, y1, y2);
				Graphics_setGrey (g, 0.0);   // black
				Graphics_rectangle (g, x1, x2, y1, y2);

				x1 += dx;
			}
		}

		//Graphics_unsetInner (g);

		if (garnish) {
			if (labelIndex > 0) {
				double y = ymin, xb = (xoffsetFraction + 0.5 * (groupSize + (groupSize - 1) * interbarFraction)) * bar_width;
				const double lineSpacing = Graphics_dyMMtoWC (g, 1.5 * Graphics_inqFontSize (g) * 25.4 / 72.0);
				const double currentFontSize = Graphics_inqFontSize (g);
				Graphics_setTextRotation (g, angle);
				if (angle < 0.0) {
					y -= 0.3 * lineSpacing;
					xb -= 0.5 * bar_width;
					Graphics_setFontSize (g, currentFontSize - ( currentFontSize > 12.0 ? 2.0 : 1.0 ));
					Graphics_setTextAlignment (g, Graphics_LEFT, Graphics_TOP);
				} else if (angle > 0.0) {
					y -= 0.3*lineSpacing;
					xb += 0.5 * bar_width;
					Graphics_setFontSize (g, currentFontSize - ( currentFontSize > 12.0 ? 2.0 : 1.0 ));
					Graphics_setTextAlignment (g, Graphics_RIGHT, Graphics_TOP);
				} else {
					Graphics_setTextAlignment (g, Graphics_CENTRE, Graphics_TOP);
				}
				for (integer irow = 1; irow <= numberOfGroups; irow ++) {
					conststring32 label = Table_getStringValue_Assert (me, selectedRows [irow], labelIndex);
					if (label)
						Graphics_text (g, xb, ymin - g -> vertTick, label); // was y
					xb += dx;
				}
				Graphics_setFontSize (g, currentFontSize);
				Graphics_setTextRotation (g, 0);
			}
		}
		Graphics_unsetInner (g);
		if (garnish) {
			if (ymin * ymax < 0.0)
				Graphics_markLeft (g, 0.0, true, true, true, nullptr);
			Graphics_drawInnerBox (g);
			Graphics_marksLeft (g, 2, true, true, false);
		}
	} catch (MelderError) {
		Melder_clearError ();   // drawing errors shall be ignored
	}
}

static bool Graphics_getConnectingLine (Graphics g, conststring32 text1, double x1, double y1, conststring32 text2, double x2, double y2, double *x3, double *y3, double *x4, double *y4) {
	bool drawLine = false;
	const double width1 = Graphics_textWidth (g, text1);
	const double width2 = Graphics_textWidth (g, text2);
	const double h = Graphics_dyMMtoWC (g, 1.5 * Graphics_inqFontSize (g) * 25.4 / 72.0) / 1.5;
	const double xleft = x1 < x2 ? x1 : x2, xright = x2 > x1 ? x2 : x1;
	double xi [3], yi [3];
	int numberOfIntersections = NUMgetIntersectionsWithRectangle (x1, y1, x2, y2, xleft - width1 / 2.0, y1 - h/2, xleft + width1 / 2.0, y1 + h/2, xi, yi);
	if (numberOfIntersections == 1) {
		*x3 = xi [1];
		*y3 = yi [1];
		numberOfIntersections = NUMgetIntersectionsWithRectangle (x1, y1, x2, y2, xright - width2 / 2.0, y2 - h/2, xright + width2 / 2.0, y2 + h/2, xi, yi);
		if (numberOfIntersections == 1) {
			*x4 = xi [1];
			*y4 = yi [1];
			drawLine = true;
		}
	}
	return drawLine;
}

// take the xcolumn as labels if non-numeric column else as numbers and arrange distances accordingly.
void Table_lineGraphWhere (Table me, Graphics g, integer xcolumn, double xmin, double xmax, integer ycolumn, double ymin, double ymax, conststring32 symbol, double angle, bool garnish, conststring32 formula, Interpreter interpreter) {
	try {
		Melder_require (ycolumn >= 1 && ycolumn <= my numberOfColumns,
			U"The column for the vertical axis should exist.");
		Melder_require (xcolumn >= 1 && xcolumn <= my numberOfColumns,
			U"The column for the horizontal axis should exist.");
		
		autoINTVEC selectedRows = Table_findRowsMatchingCriterion (me, formula, interpreter);
		if (ymax <= ymin)
			Table_columnExtremaFromSelectedRows (me, ycolumn, selectedRows.get(), & ymin, & ymax);

		const bool xIsNumeric = Table_selectedColumnPartIsNumeric (me, xcolumn, selectedRows.get());
		if (xmin >= xmax) {
			if (xIsNumeric)
				Table_columnExtremaFromSelectedRows (me, xcolumn, selectedRows.get(), & xmin, & xmax);
			else {
				xmin = 0.0;
				xmax = selectedRows.size + 1;
			}
		}
		Graphics_setInner (g);
		Graphics_setWindow (g, xmin, xmax, ymin, ymax);
		Graphics_setTextAlignment (g, Graphics_CENTRE, Graphics_HALF);
		const double lineSpacing = Graphics_dyMMtoWC (g, 1.5 * Graphics_inqFontSize (g) * 25.4 / 72.0);
		//double symbolHeight = lineSpacing / 1.5;
		double x1, y1;
		for (integer i = 1; i <= selectedRows.size; i ++) {
			const double y2 = Table_getNumericValue_Assert (me, selectedRows [i], ycolumn);
			double x2 = xIsNumeric ? Table_getNumericValue_Assert (me, selectedRows [i], xcolumn) : i;
			//double symbolWidth = 0;
			if (x2 >= xmin && (x2 <= xmax || x1 < xmax)) {
				if (symbol && y2 >= ymin && y2 <= ymax && x2 <= xmax)
					Graphics_text (g, x2, y2, symbol);
				if (i > 1) {
					double x3, y3, x4, y4, xo1, yo1, xo2, yo2;
					if (Graphics_getConnectingLine (g, symbol, x1, y1, symbol, x2, y2, & x3, & y3, & x4, & y4) && 
							NUMclipLineWithinRectangle (x3, y3, x4, y4, xmin, ymin, xmax, ymax, & xo1, & yo1, & xo2, & yo2))
						Graphics_line (g, xo1, yo1, xo2, yo2);
				}
			} else {
				x2 = x2 < xmin ? xmin : xmax;
			}
			x1 = x2;
			y1 = y2;
		}
		
		if (garnish && ! xIsNumeric && xcolumn > 0) {
			const double currentFontSize = Graphics_inqFontSize (g);
			double y = ymin, dx = 0.0;
			Graphics_setTextRotation (g, angle);
			if (angle < 0.0) {
				y -= 0.3 * lineSpacing;
				dx = - 0.5;
				Graphics_setFontSize (g, currentFontSize - (currentFontSize > 12 ? 2 : 1));
				Graphics_setTextAlignment (g, Graphics_LEFT, Graphics_TOP);
			} else if (angle > 0.0) {
				y -= 0.3 * lineSpacing;
				dx = 0.5;
				Graphics_setFontSize (g, currentFontSize - (currentFontSize > 12 ? 2 : 1));
				Graphics_setTextAlignment (g, Graphics_RIGHT, Graphics_TOP);
			} else {
				Graphics_setTextAlignment (g, Graphics_CENTRE, Graphics_TOP);
			}
			for (integer i = 1; i <= selectedRows.size; i ++) {
				const double x2 = double (i);
				if (x2 >= xmin && x2 <= xmax) {
					conststring32 label = Table_getStringValue_Assert (me, selectedRows [i], xcolumn);
					if (label)
						Graphics_text (g, x2 + dx, ymin - g -> vertTick, label); // was y
				}
			}
			Graphics_setFontSize (g, currentFontSize);
			Graphics_setTextRotation (g, 0);
		}
		Graphics_unsetInner (g);

		if (garnish) {
			Graphics_drawInnerBox (g);
			Graphics_marksLeft (g, 2, true, true, false);
			if (xIsNumeric)
				Graphics_marksBottom (g, 2, true, true, false);
		}
	} catch (MelderError) {
		Melder_clearError ();   // drawing errors shall be ignored
	}
}

void Table_lagPlotWhere (Table me, Graphics g, integer column, integer lag, double xmin, double xmax,
	conststring32 symbol, double labelSize,
	bool garnish, conststring32 formula, Interpreter interpreter)
{
	try {
		if (column < 1 || column > my rows.size)
			return;
		autoINTVEC selectedRows = Table_findRowsMatchingCriterion (me, formula, interpreter);
		if (xmax <= xmin)   // autoscaling
			Table_columnExtremaFromSelectedRows (me, column, selectedRows.get(), & xmin, & xmax);
		autoVEC x = newVECraw (selectedRows.size);
		for (integer i = 1; i <= selectedRows.size; i ++)
			x [i] = Table_getNumericValue_Assert (me, selectedRows [i], column);
		Graphics_setInner (g);
		Graphics_setWindow (g, xmin, xmax, xmin, xmax);
		Graphics_lagPlot (g, x.get(), xmin, xmax, lag, labelSize, symbol);
		Graphics_unsetInner (g);
		if (garnish) {
			Graphics_drawInnerBox (g);
			Graphics_marksBottom (g, 2, true, true, false);
			Graphics_marksLeft (g, 2, true, true, false);
			if (my columnHeaders [column]. label) {
				Graphics_textLeft (g, true, my columnHeaders [column]. label.get());
				Graphics_textBottom (g, true, Melder_cat (my columnHeaders [column]. label.get(), U" (lag = ", lag, U")"));
			}
		}
	} catch (MelderError) {
		Melder_clearError ();   // drawing errors shall be ignored
	}
}

autoTable Table_extractRowsWhere (Table me, conststring32 formula, Interpreter interpreter) {
	try {
		Formula_compile (interpreter, me, formula, kFormula_EXPRESSION_TYPE_NUMERIC, true);
		Formula_Result result;
		autoTable thee = Table_create (0, my numberOfColumns);
		for (integer icol = 1; icol <= my numberOfColumns; icol ++)
			thy columnHeaders [icol]. label = Melder_dup (my columnHeaders [icol]. label.get());
		for (integer irow = 1; irow <= my rows.size; irow ++) {
			Formula_run (irow, 1, & result);
			if (result. numericResult != 0.0) {
				const TableRow row = my rows.at [irow];
				autoTableRow newRow = Data_copy (row);
				thy rows. addItem_move (newRow.move());
			}
		}
		if (thy rows.size == 0)
			Melder_warning (U"No row matches criterion.");
		return thee;
	} catch (MelderError) {
		Melder_throw (me, U": no Table could be extracted.");
	}
}

static autoTableOfReal Table_to_TableOfReal_where (Table me, conststring32 columnLabels, conststring32 factorColumn, conststring32 formula, Interpreter interpreter) {
	try {
		const integer factorColIndex = Table_findColumnIndexFromColumnLabel (me, factorColumn);
		autoINTVEC columnIndexes = Table_getColumnIndicesFromColumnLabelString (me, columnLabels);
		autoINTVEC selectedRows = Table_findRowsMatchingCriterion (me, formula, interpreter);
		autoTableOfReal thee = TableOfReal_create (selectedRows.size, columnIndexes.size);
		for (integer i = 1; i <= selectedRows.size; i ++) {
			for (integer icol = 1; icol <= columnIndexes.size; icol ++)
				thy data [i] [icol] = Table_getNumericValue_Assert (me, selectedRows [i], columnIndexes [icol]);
			if (factorColIndex > 0) { // if no factorColumn given labels may be empty
				const conststring32 label = Table_getStringValue_Assert (me, selectedRows [i], factorColIndex);
				TableOfReal_setRowLabel (thee.get(), i, label);
			}
		}
		for (integer icol = 1; icol <= columnIndexes.size; icol ++)
			TableOfReal_setColumnLabel (thee.get(), icol, my columnHeaders [columnIndexes [icol]]. label.get());
		return thee;
	} catch (MelderError) {
		Melder_throw (me, U"No TableOfReal created from Table.");
	}
}

static autoSSCPList Table_to_SSCPList_where (Table me,
	conststring32 columnLabels, conststring32 factorColumn, conststring32 formula, Interpreter interpreter) {
	try {
		autoTableOfReal thee = Table_to_TableOfReal_where (me, columnLabels, factorColumn, formula, interpreter);
		autoSSCPList him = TableOfReal_to_SSCPList_byLabel (thee.get());
		return him;
	} catch (MelderError) {
		Melder_throw (me, U"No Discriminant created from Table.");
	}
}

static integer SSCPList_findIndexOfGroupLabel (SSCPList me, conststring32 label) {
	for (integer i = 1; i <= my size; i ++)
		if (Melder_equ (Thing_getName (my at [i]), label))
			return i;
	return 0;
}

static autoTable Table_SSCPList_extractMahalanobisWhere (Table me, SSCPList thee, double numberOfSigmas, kMelder_number which, conststring32 factorColumn, conststring32 formula, Interpreter interpreter) {
	try {
		const integer numberOfGroups = thy size;
		Melder_assert (numberOfGroups > 0);

		const SSCP sscp = thy at [1];
		const integer numberOfColumns = sscp -> numberOfColumns;
		const integer factorColIndex = Table_findColumnIndexFromColumnLabel (me, factorColumn);   // can be absent
		autoINTVEC columnIndex = newINTVECraw (numberOfColumns);
		autoVEC vector = newVECraw (numberOfColumns);
		autoINTVEC selectedRows = Table_findRowsMatchingCriterion (me, formula, interpreter);
		for (integer icol = 1; icol <= numberOfColumns; icol ++)
			columnIndex [icol] = Table_getColumnIndexFromColumnLabel (me, sscp -> columnLabels [icol].get()); // throw if not present
		autoTable him = Table_create (0, my numberOfColumns);
		for (integer icol = 1; icol <= my numberOfColumns; icol ++)
			his columnHeaders [icol].label = Melder_dup (my columnHeaders [icol]. label.get());
		OrderedOf<structCovariance> covs;
		for (integer igroup = 1; igroup <= numberOfGroups; igroup ++) {
			autoCovariance cov = SSCP_to_Covariance (thy at [igroup], 1);
			SSCP_expandLowerCholeskyInverse (cov.get());
			covs. addItem_move (cov.move());
		}
		for (integer i = 1; i <= selectedRows.size; i ++) {
			const integer irow = selectedRows [i];
			integer igroup = 1;   // if factorColIndex == 0 we don't need labels
			if (factorColIndex > 0) {
				conststring32 label = Table_getStringValue_Assert (me, irow, factorColIndex);
				igroup = SSCPList_findIndexOfGroupLabel (thee, label);
				Melder_require (igroup > 0,
					U"The label \"", label, U"\" in row ", irow, U" is not valid in this context.");
			}
			const Covariance covi = covs.at [igroup];
			for (integer icol = 1; icol <= numberOfColumns; icol ++)
				vector [icol] = Table_getNumericValue_Assert (me, irow, columnIndex [icol]);
			const double dm2 = NUMmahalanobisDistanceSquared (covi -> lowerCholeskyInverse.get(), vector.get(), covi -> centroid.get());
			if (Melder_numberMatchesCriterion (sqrt (dm2), which, numberOfSigmas)) {
				const TableRow row = my rows.at [irow];
				autoTableRow newRow = Data_copy (row);
				his rows. addItem_move (newRow.move());
			}
		}
		return him;
	} catch (MelderError) {
		Melder_throw (me, U"Table (mahalanobis) not extracted.");
	}
}

autoTable Table_extractMahalanobisWhere (Table me,
	conststring32 columnLabels, conststring32 factorColumn, double numberOfSigmas,
	kMelder_number which, conststring32 formula, Interpreter interpreter) {
	try {
		autoSSCPList thee = Table_to_SSCPList_where (me, columnLabels, factorColumn, formula, interpreter);
		autoTable him = Table_SSCPList_extractMahalanobisWhere (me, thee.get(), numberOfSigmas, which, factorColumn, formula, interpreter);
		return him;
	} catch (MelderError) {
		Melder_throw (me, U"Table not extracted.");
	}
}

void Table_drawEllipsesWhere (Table me, Graphics g, integer xcolumn, integer ycolumn, integer factorColumn,
	double xmin, double xmax, double ymin, double ymax, double numberOfSigmas, double labelSize, bool garnish,
	conststring32 formula, Interpreter interpreter) {
	try {
		autoINTVEC selectedRows = Table_findRowsMatchingCriterion (me, formula, interpreter);
		autoTableOfReal thee = TableOfReal_create (selectedRows.size, 2);
		for (integer i = 1; i <= selectedRows.size; i ++) {
			const conststring32 label = Table_getStringValue_Assert (me, selectedRows [i], factorColumn);
			TableOfReal_setRowLabel (thee.get(), i, label);
			thy data [i] [1] = Table_getNumericValue_Assert (me, selectedRows [i], xcolumn);
			thy data [i] [2] = Table_getNumericValue_Assert (me, selectedRows [i], ycolumn);
		}
		autoSSCPList him = TableOfReal_to_SSCPList_byLabel (thee.get());
		constexpr bool confidence = false;
		if (ymax == ymin)   // autoscaling
			SSCPList_getEllipsesBoundingBoxCoordinates (him.get(), numberOfSigmas, confidence, & xmin, & xmax, & ymin, & ymax);
		Graphics_setWindow (g, xmin, xmax, ymin, ymax);
		Graphics_setInner (g);
		for (integer i = 1; i <= his size; i ++) {
			const SSCP sscpi = his at [i];
			const double scalei = SSCP_getEllipseScalefactor (sscpi, numberOfSigmas, confidence);
			if (scalei > 0.0)
				SSCP_drawTwoDimensionalEllipse_inside (sscpi, g, scalei, Thing_getName (sscpi), labelSize);
		}
		Graphics_unsetInner (g);

		if (garnish) {
			Graphics_drawInnerBox (g);
			Graphics_marksBottom (g, 2, true, true, false);
			Graphics_marksLeft (g, 2, true, true, false);
			if (my columnHeaders [xcolumn]. label)
				Graphics_textBottom (g, true, my columnHeaders [xcolumn]. label.get());
			if (my columnHeaders [ycolumn]. label)
				Graphics_textLeft (g, true, my columnHeaders [ycolumn]. label.get());
		}
	} catch (MelderError) {
		Melder_clearError ();   // drawing errors shall be ignored
	}
}

autoTable Table_extractColumnRanges (Table me, conststring32 ranges) {
	try {
		const integer numberOfRows = my rows.size;
		autoINTVEC columnRanges = NUMstring_getElementsOfRanges (ranges, my numberOfColumns, U"columnn number", true);
		autoTable thee = Table_createWithoutColumnNames (numberOfRows, columnRanges.size); 
		for (integer icol = 1; icol <= columnRanges.size; icol ++)
			Table_setColumnLabel (thee.get(), icol, my v_getColStr (columnRanges [icol]));
		for (integer irow = 1; irow <= numberOfRows; irow ++) {
			for (integer icol = 1; icol <= columnRanges.size; icol ++) {
				const conststring32 value = Table_getStringValue_Assert (me, irow, columnRanges [icol]);
				Table_setStringValue (thee.get(), irow, icol, value);
			}
		}
		return thee;
	} catch (MelderError) {
		Melder_throw (me, U": no column range extracted.");
	}
}

/* End of file Table_extensions.cpp */
