# Copyright (c) 2009, 2013, Oracle and/or its affiliates. All rights reserved.
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 2 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA 

# cmake -DWITH_EDITLINE=system|bundled
# bundled is the default

MACRO (MYSQL_CHECK_MULTIBYTE)
  CHECK_INCLUDE_FILE(wctype.h HAVE_WCTYPE_H)
  CHECK_INCLUDE_FILE(wchar.h HAVE_WCHAR_H)
  IF(HAVE_WCHAR_H)
    SET(CMAKE_EXTRA_INCLUDE_FILES wchar.h)
    CHECK_TYPE_SIZE(mbstate_t SIZEOF_MBSTATE_T)
    SET(CMAKE_EXTRA_INCLUDE_FILES)
    IF(SIZEOF_MBSTATE_T)
      SET(HAVE_MBSTATE_T 1)
    ENDIF()
  ENDIF()

  CHECK_C_SOURCE_COMPILES("
  #include <langinfo.h>
  int main(int ac, char **av)
  {
    char *cs = nl_langinfo(CODESET);
    return 0;
  }"
  HAVE_LANGINFO_CODESET)
  
  CHECK_FUNCTION_EXISTS(mbrlen HAVE_MBRLEN)
  CHECK_FUNCTION_EXISTS(mbscmp HAVE_MBSCMP)
  CHECK_FUNCTION_EXISTS(mbsrtowcs HAVE_MBSRTOWCS)
  CHECK_FUNCTION_EXISTS(wcrtomb HAVE_WCRTOMB)
  CHECK_FUNCTION_EXISTS(mbrtowc HAVE_MBRTOWC)
  CHECK_FUNCTION_EXISTS(wcscoll HAVE_WCSCOLL)
  CHECK_FUNCTION_EXISTS(wcsdup HAVE_WCSDUP)
  CHECK_FUNCTION_EXISTS(wcwidth HAVE_WCWIDTH)
  CHECK_FUNCTION_EXISTS(wctype HAVE_WCTYPE)
  CHECK_FUNCTION_EXISTS(iswlower HAVE_ISWLOWER)
  CHECK_FUNCTION_EXISTS(iswupper HAVE_ISWUPPER)
  CHECK_FUNCTION_EXISTS(towlower HAVE_TOWLOWER)
  CHECK_FUNCTION_EXISTS(towupper HAVE_TOWUPPER)
  CHECK_FUNCTION_EXISTS(iswctype HAVE_ISWCTYPE)

  SET(CMAKE_EXTRA_INCLUDE_FILES wchar.h)
  CHECK_TYPE_SIZE(wchar_t SIZEOF_WCHAR_T)
  IF(SIZEOF_WCHAR_T)
    SET(HAVE_WCHAR_T 1)
  ENDIF()

  SET(CMAKE_EXTRA_INCLUDE_FILES wctype.h)
  CHECK_TYPE_SIZE(wctype_t SIZEOF_WCTYPE_T)
  IF(SIZEOF_WCTYPE_T)
    SET(HAVE_WCTYPE_T 1)
  ENDIF()
  CHECK_TYPE_SIZE(wint_t SIZEOF_WINT_T)
  IF(SIZEOF_WINT_T)
    SET(HAVE_WINT_T 1)
  ENDIF()
  SET(CMAKE_EXTRA_INCLUDE_FILES)

ENDMACRO()

MACRO (FIND_CURSES)
 FIND_PACKAGE(Curses) 
 MARK_AS_ADVANCED(CURSES_CURSES_H_PATH CURSES_FORM_LIBRARY CURSES_HAVE_CURSES_H)
 IF(NOT CURSES_FOUND)
   SET(ERRORMSG "Curses library not found. Please install appropriate package,
    remove CMakeCache.txt and rerun cmake.")
   IF(CMAKE_SYSTEM_NAME MATCHES "Linux")
    SET(ERRORMSG ${ERRORMSG} 
    "On Debian/Ubuntu, package name is libncurses5-dev, on Redhat and derivates " 
    "it is ncurses-devel.")
   ENDIF()
   MESSAGE(FATAL_ERROR ${ERRORMSG})
 ENDIF()

 IF(CURSES_HAVE_CURSES_H)
   SET(HAVE_CURSES_H 1 CACHE INTERNAL "")
 ELSEIF(CURSES_HAVE_NCURSES_H)
   SET(HAVE_NCURSES_H 1 CACHE INTERNAL "")
 ENDIF()
 IF(CMAKE_SYSTEM_NAME MATCHES "HP")
   # CMake uses full path to library /lib/libcurses.sl 
   # On Itanium, it results into architecture mismatch+
   # the library is for  PA-RISC
   SET(CURSES_LIBRARY "curses" CACHE INTERNAL "" FORCE)
   SET(CURSES_CURSES_LIBRARY "curses" CACHE INTERNAL "" FORCE)
 ENDIF()
 IF(CMAKE_SYSTEM_NAME MATCHES "SunOS")
   # CMake generates /lib/64/libcurses.so -R/lib/64
   # The result is we cannot find
   # /opt/studio12u2/lib/stlport4/v9/libstlport.so.1
   # at runtime
   SET(CURSES_LIBRARY "curses" CACHE INTERNAL "" FORCE)
   SET(CURSES_CURSES_LIBRARY "curses" CACHE INTERNAL "" FORCE)
   MESSAGE(STATUS "CURSES_LIBRARY ${CURSES_LIBRARY}")
 ENDIF()

 IF(CMAKE_SYSTEM_NAME MATCHES "Linux")
   # -Wl,--as-needed breaks linking with -lcurses, e.g on Fedora 
   # Lower-level libcurses calls are exposed by libtinfo
   CHECK_LIBRARY_EXISTS(${CURSES_LIBRARY} tputs "" HAVE_TPUTS_IN_CURSES)
   IF(NOT HAVE_TPUTS_IN_CURSES)
     CHECK_LIBRARY_EXISTS(tinfo tputs "" HAVE_TPUTS_IN_TINFO)
     IF(HAVE_TPUTS_IN_TINFO)
       SET(CURSES_LIBRARY tinfo)
     ENDIF()
   ENDIF() 
 ENDIF()
ENDMACRO()

MACRO (MYSQL_USE_BUNDLED_EDITLINE)
  SET(USE_LIBEDIT_INTERFACE 1)
  SET(HAVE_HIST_ENTRY 1)
  SET(EDITLINE_INCLUDE_DIR ${CMAKE_SOURCE_DIR}/cmd-line-utils/libedit/editline)
  SET(EDITLINE_LIBRARY edit)
  FIND_CURSES()
  ADD_SUBDIRECTORY(${CMAKE_SOURCE_DIR}/cmd-line-utils/libedit)
ENDMACRO()

MACRO (FIND_SYSTEM_EDITLINE)
  FIND_PATH(FOUND_EDITLINE_READLINE
    NAMES editline/readline.h
  )
  IF(FOUND_EDITLINE_READLINE)
    SET(EDITLINE_INCLUDE_DIR "${FOUND_EDITLINE_READLINE}/editline")
  ELSE()
    # Different path on FreeBSD
    FIND_PATH(FOUND_EDIT_READLINE_READLINE
      NAMES edit/readline/readline.h
    )
    IF(FOUND_EDIT_READLINE_READLINE)
      SET(EDITLINE_INCLUDE_DIR "${FOUND_EDIT_READLINE_READLINE}/edit/readline")
    ENDIF()
  ENDIF()

  FIND_LIBRARY(EDITLINE_LIBRARY
    NAMES
    edit
  )
  MARK_AS_ADVANCED(EDITLINE_INCLUDE_DIR EDITLINE_LIBRARY)

  MESSAGE(STATUS "EDITLINE_INCLUDE_DIR ${EDITLINE_INCLUDE_DIR}")
  MESSAGE(STATUS "EDITLINE_LIBRARY ${EDITLINE_LIBRARY}")

  INCLUDE(CheckCXXSourceCompiles)
  IF(EDITLINE_LIBRARY AND EDITLINE_INCLUDE_DIR)
    SET(CMAKE_REQUIRED_INCLUDES ${EDITLINE_INCLUDE_DIR})
    SET(CMAKE_REQUIRED_LIBRARIES ${EDITLINE_LIBRARY})
    CHECK_CXX_SOURCE_COMPILES("
    #include <stdio.h>
    #include <readline.h>
    int main(int argc, char **argv)
    {
       HIST_ENTRY entry;
       return 0;
    }"
    EDITLINE_HAVE_HIST_ENTRY)

    CHECK_CXX_SOURCE_COMPILES("
    #include <stdio.h>
    #include <readline.h>
    int main(int argc, char **argv)
    {
      typedef int MYFunction(const char*, int);
      MYFunction* myf= rl_completion_entry_function;
      int res= (myf)(NULL, 0);
      completion_matches(0,0);
      return res;
    }"
    EDITLINE_HAVE_COMPLETION)

    IF(EDITLINE_HAVE_COMPLETION)
      SET(HAVE_HIST_ENTRY ${EDITLINE_HAVE_HIST_ENTRY})
      SET(USE_LIBEDIT_INTERFACE 1)
      SET(EDITLINE_FOUND 1)
    ENDIF()
  ENDIF()
ENDMACRO()


IF (NOT WITH_EDITLINE AND NOT WIN32)
  SET(WITH_EDITLINE "bundled" CACHE STRING "By default use bundled editline")
ENDIF()

MACRO (MYSQL_CHECK_EDITLINE)
  IF (NOT WIN32)
    MYSQL_CHECK_MULTIBYTE()

    IF(WITH_EDITLINE STREQUAL "bundled") 
      MYSQL_USE_BUNDLED_EDITLINE()
    ELSEIF(WITH_EDITLINE STREQUAL "system")
      FIND_SYSTEM_EDITLINE()
      IF(NOT EDITLINE_FOUND)
        MESSAGE(FATAL_ERROR "Cannot find system editline libraries.") 
      ENDIF()
    ELSE()
      MESSAGE(FATAL_ERROR "WITH_EDITLINE must be bundled or system")
    ENDIF()
  ENDIF(NOT WIN32)
ENDMACRO()
