/*=========================================================================

   Program: ParaView
   Module:    pqDataRepresentation.h

   Copyright (c) 2005-2008 Sandia Corporation, Kitware Inc.
   All rights reserved.

   ParaView is a free software; you can redistribute it and/or modify it
   under the terms of the ParaView license version 1.2.

   See License_v1.2.txt for the full ParaView license.
   A copy of this license can be obtained by contacting
   Kitware Inc.
   28 Corporate Drive
   Clifton Park, NY 12065
   USA

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/
#ifndef __pqDataRepresentation_h
#define __pqDataRepresentation_h

#include "pqRepresentation.h"

class pqDataRepresentationInternal;
class pqOutputPort;
class pqPipelineSource;
class pqScalarOpacityFunction;
class pqScalarsToColors;
class vtkPVArrayInformation;
class vtkPVDataInformation;
class vtkPVProminentValuesInformation;
class vtkPVTemporalDataInformation;

/// pqDataRepresentation is the superclass for a display for a pqPipelineSource
/// i.e. the input for this display proxy is a pqPiplineSource.
/// This class manages the linking between the pqPiplineSource
/// and pqDataRepresentation.
class PQCORE_EXPORT pqDataRepresentation : public pqRepresentation
{
  Q_OBJECT
  typedef pqRepresentation Superclass;
public:
  pqDataRepresentation(const QString& group, const QString& name,
    vtkSMProxy* display, pqServer* server,
    QObject* parent=0);
  virtual ~pqDataRepresentation();

  /// Get the source/filter of which this is a display.
  pqPipelineSource* getInput() const;

  /// Returns the input pqPipelineSource's output port to which this
  /// representation is connected.
  pqOutputPort* getOutputPortFromInput() const;

  /// Returns the data information for the data coming into the representation
  /// as input. 
  vtkPVDataInformation* getInputDataInformation() const;

  /// Returns the temporal data information for the input. This can be a very
  /// slow process. Use with extreme caution!!!
  vtkPVTemporalDataInformation* getInputTemporalDataInformation() const;

  /// Returns the represented data information. Depending on the representation
  /// this may differ from the input data information eg. if the representation
  /// shows an outline of the data, the this method will return the information
  /// about the polydata forming the outline not the input dataset.
  vtkPVDataInformation* getRepresentedDataInformation(bool update=true) const;

  /// Get the data bounds for the input of this display.
  /// Returns if the operation was successful.
  bool getDataBounds(double bounds[6]);

  /// Returns the lookuptable proxy, if any.
  /// Most consumer displays take a lookup table. This method
  /// provides access to the Lookup table, if one exists.
  virtual vtkSMProxy* getLookupTableProxy();

  /// Returns the pqScalarsToColors object for the lookup table
  /// proxy if any.
  /// Most consumer displays take a lookup table. This method
  /// provides access to the Lookup table, if one exists.
  virtual pqScalarsToColors* getLookupTable();

  /// Returns the proxy for the piecewise function used to
  /// map scalars to opacity.
  virtual vtkSMProxy* getScalarOpacityFunctionProxy() {return 0;}

  /// Returns the pqScalarOpacityFunction object for the piecewise
  /// function used to map scalars to opacity.
  virtual pqScalarOpacityFunction* getScalarOpacityFunction() {return 0;}

  /// Sets default values for the underlying proxy.
  /// This is during the initialization stage of the pqProxy
  /// for proxies created by the GUI itself i.e.
  /// for proxies loaded through state or created by python client
  /// this method won't be called.
  /// The default implementation iterates over all properties
  /// of the proxy and sets them to default values.
  virtual void setDefaultPropertyValues();

  /// Returns the data size for the full-res data.
  /// This may trigger a pipeline update to obtain correct data sizes.
  unsigned long getFullResMemorySize();

  /// This is convienience method to return first representation for the
  /// upstream stream filter/source in the same view as this representation.
  /// This is only applicable, if this representation is connected to a
  /// data-filter which has a valid input.
  pqDataRepresentation* getRepresentationForUpstreamSource() const;

  /// Convenience method to return the name of the active scalar array.
  virtual QString getProxyColorArrayName();
  /// Convenience method to return the active scalar array field association.
  virtual int getProxyScalarMode();
  /// Convenience method to return the array information associated with the active scalars.
  virtual vtkPVArrayInformation* getProxyColorArrayInfo();
  /// Convenience method to return prominent value list associated with the active scalars.
  virtual vtkPVProminentValuesInformation* getProxyColorProminentValuesInfo(
    double uncertainty = 1e-6, double minFrequency = 5e-4);

  virtual int getNumberOfComponents(const char* arrayname, int fieldType);
  virtual QString getComponentName( const char* arrayname, int fieldtype, int component);

signals:
  /// Fired when the representation proxy fires the vtkCommand::UpdateDataEvent.
  void dataUpdated();

protected slots:
  /// called when input property on display changes. We must detect if
  /// (and when) the display is connected to a new proxy.
  virtual void onInputChanged();

protected:
  /// Use this method to initialize the pqObject state using the
  /// underlying vtkSMProxy. This needs to be done only once,
  /// after the object has been created.
  virtual void initialize()
    {
    this->Superclass::initialize();
    this->onInputChanged();
    }

  virtual vtkPVArrayInformation* getArrayInformation( const char* arrayname, const int &fieldType );
  virtual vtkPVProminentValuesInformation* getProminentValuesInformation(
    const char* arrayname, const int &fieldType, double uncertainty, double minFrequency );

private:
  pqDataRepresentation(const pqDataRepresentation&); // Not implemented.
  void operator=(const pqDataRepresentation&); // Not implemented.

  pqDataRepresentationInternal* Internal;
};

#endif

