///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/data/units/ParameterUnit.h>
#include <core/scene/animation/controller/LookAtController.h>
#include <core/scene/SceneNode.h>

namespace Core {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(LookAtController, RotationController)
DEFINE_REFERENCE_FIELD(LookAtController, FloatController, "Roll", _rollCtrl)
DEFINE_FLAGS_REFERENCE_FIELD(LookAtController, SceneNode, "Target", PROPERTY_FIELD_NEVER_CLONE_TARGET, _targetNode)
SET_PROPERTY_FIELD_LABEL(LookAtController, _rollCtrl, "Roll")
SET_PROPERTY_FIELD_LABEL(LookAtController, _targetNode, "Target")
SET_PROPERTY_FIELD_UNITS(LookAtController, _rollCtrl, AngleParameterUnit)

/******************************************************************************
* Default constructor.
******************************************************************************/
LookAtController::LookAtController(bool isLoading) : RotationController(isLoading)
{
	INIT_PROPERTY_FIELD(LookAtController, _rollCtrl);
	INIT_PROPERTY_FIELD(LookAtController, _targetNode);
	if(!isLoading) {
		// Create sub-controller.
		_rollCtrl = CONTROLLER_MANAGER.createDefaultController<FloatController>();
	}
}

/******************************************************************************
* Queries the controller for its absolute value at a certain time.
******************************************************************************/
void LookAtController::getValue(TimeTicks time, Rotation& result, TimeInterval& validityInterval)
{
	// Get position of target node.
	Vector3 targetPos(NULL_VECTOR);
	if(targetNode()) {
		const AffineTransformation& targetTM = targetNode()->getWorldTransform(time, validityInterval);
		targetPos = targetTM.getTranslation();
	}
	
	if(!sourcePosValidity.isEmpty())
		validityInterval.intersect(sourcePosValidity);
	else
		validityInterval.intersect(TimeInterval(time));

	// Get rolling.
	FloatType rollAngle = 0.0;
	if(rollController())
		rollController()->getValue(time, rollAngle, validityInterval);

	if(targetPos == sourcePos) {
		result = NULL_ROTATION;
		return;
	}
	
	AffineTransformation tm = AffineTransformation::lookAt(ORIGIN + sourcePos, ORIGIN + targetPos, Vector3(0,0,1));
	tm.setTranslation(NULL_VECTOR);
	result = Rotation(tm);
	result.angle = -result.angle;

	if(rollAngle != 0.0)
		result = result * Rotation(Vector3(0,0,1), rollAngle);

	// Reset source validity.
	sourcePosValidity.setEmpty();
}

/******************************************************************************
* Sets the controller's value at the specified time.
******************************************************************************/
void LookAtController::setValue(TimeTicks time, const Rotation& newValue, bool isAbsoluteValue)
{
	// Cannot set value for this controller type.
}

/******************************************************************************
* Let the controller add its value at a certain time to the input value.
******************************************************************************/
void LookAtController::applyValue(TimeTicks time, AffineTransformation& result, TimeInterval& validityInterval)
{
	// Save source position for later use.
	sourcePos = result.getTranslation();
	sourcePosValidity = validityInterval;

	RotationController::applyValue(time, result, validityInterval);            			
}

/******************************************************************************
* Computes the largest time interval containing the given time during which the
* controller's value is constant.
******************************************************************************/
void LookAtController::validityInterval(TimeTicks time, TimeInterval& validityInterval)
{
	if(rollController()) rollController()->validityInterval(time, validityInterval);
	if(targetNode()) {		
		targetNode()->getWorldTransform(time, validityInterval);
	}
}

};
