#ifndef OSMIUM_INDEX_DETAIL_MMAP_VECTOR_BASE_HPP
#define OSMIUM_INDEX_DETAIL_MMAP_VECTOR_BASE_HPP

/*

This file is part of Osmium (http://osmcode.org/libosmium).

Copyright 2013-2015 Jochen Topf <jochen@topf.org> and others (see README).

Boost Software License - Version 1.0 - August 17th, 2003

Permission is hereby granted, free of charge, to any person or organization
obtaining a copy of the software and accompanying documentation covered by
this license (the "Software") to use, reproduce, display, distribute,
execute, and transmit the Software, and to prepare derivative works of the
Software, and to permit third-parties to whom the Software is furnished to
do so, all subject to the following:

The copyright notices in the Software and this entire statement, including
the above license grant, this restriction and the following disclaimer,
must be included in all copies of the Software, in whole or in part, and
all derivative works of the Software, unless such copies or derivative
works are solely in the form of machine-executable object code generated by
a source language processor.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.

*/

#include <cstddef>
#include <new>
#include <stdexcept>

#include <osmium/index/detail/typed_mmap.hpp>
#include <osmium/util/compatibility.hpp>

namespace osmium {

    namespace detail {

        constexpr size_t mmap_vector_size_increment = 1024 * 1024;

        /**
         * This is a base class for implementing classes that look like
         * STL vector but use mmap internally. This class can not be used
         * on it's own. Use the derived classes mmap_vector_anon or
         * mmap_vector_file.
         */
        template <typename T, template <typename> class TDerived>
        class mmap_vector_base {

        protected:

            int m_fd;
            size_t m_capacity;
            size_t m_size;
            T* m_data;

            explicit mmap_vector_base(int fd, size_t capacity, size_t size, T* data) noexcept :
                m_fd(fd),
                m_capacity(capacity),
                m_size(size),
                m_data(data) {
            }

            explicit mmap_vector_base(int fd, size_t capacity, size_t size) :
                m_fd(fd),
                m_capacity(capacity),
                m_size(size),
                m_data(osmium::detail::typed_mmap<T>::grow_and_map(capacity, m_fd)) {
            }

            void data(T* data) {
                m_data = data;
            }

        public:

            typedef T value_type;
            typedef T& reference;
            typedef const T& const_reference;
            typedef T* pointer;
            typedef const T* const_pointer;
            typedef T* iterator;
            typedef const T* const_iterator;

            ~mmap_vector_base() {
                osmium::detail::typed_mmap<T>::unmap(m_data, m_capacity);
            }

            size_t capacity() const noexcept {
                return m_capacity;
            }

            size_t size() const noexcept {
                return m_size;
            }

            bool empty() const noexcept {
                return m_size == 0;
            }

            const T* data() const noexcept {
                return m_data;
            }

            T* data() noexcept {
                return m_data;
            }

            T& operator[](size_t n) {
                return m_data[n];
            }

            T at(size_t n) const {
                if (n >= m_size) {
                    throw std::out_of_range("out of range");
                }
                return m_data[n];
            }

            void clear() noexcept {
                m_size = 0;
            }

            void shrink_to_fit() {
                // XXX do something here
            }

            void push_back(const T& value) {
                if (m_size >= m_capacity) {
                    resize(m_size+1);
                }
                m_data[m_size] = value;
                ++m_size;
            }

            void resize(size_t new_size) {
                if (new_size > capacity()) {
                    static_cast<TDerived<T>*>(this)->reserve(new_size + osmium::detail::mmap_vector_size_increment);
                }
                if (new_size > size()) {
                    new (data() + size()) T[new_size - size()];
                }
                m_size = new_size;
            }

            iterator begin() noexcept {
                return m_data;
            }

            iterator end() noexcept {
                return m_data + m_size;
            }

            const_iterator begin() const noexcept {
                return m_data;
            }

            const_iterator end() const noexcept {
                return m_data + m_size;
            }

            const_iterator cbegin() noexcept {
                return m_data;
            }

            const_iterator cend() noexcept {
                return m_data + m_size;
            }

        }; // class mmap_vector_base

    } // namespace detail

} // namespace osmium

#endif // OSMIUM_INDEX_DETAIL_MMAP_VECTOR_BASE_HPP
