/* pdu-ether.c

   PDU builder for Ethernet frames

   Copyright (C) 2007, 2008, 2009 Eloy Paris

   This is part of Network Expect.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
    
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
    
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "pbuild-priv.h"

static void
build(const GNode *pdu, void *dest)
{
    struct eth_hdr *ether;
    uint16_t eth_type, *proto;
    void *field;
    const GNode *next_pdu;

    ether = dest;

    /* Put in place Ethernet type */
    if (  !(field = _pb_pdata(pdu, "type") )
	&& (next_pdu = pb_nextpdu(pdu) ) ) {
	/*
	 * User didn't specify the ethertype so we try to guess the right
	 * value based on the next PDU.
	 */

	proto = g_hash_table_lookup(ethertypes, pb_getname(next_pdu) );
	eth_type = proto ? *proto : 0;
    } else
	/*
	 * Use for IP protocol whatever the user specified. Note that if
	 * there is no user-specified IP protocol *and* there is no
	 * next PDU then we end up here, but that's okay because
	 * num_next(NULL) is 0.
	 */
	eth_type = num_next(field);

    eth_type = htons(eth_type);
    SSVAL(ether, offsetof(struct eth_hdr, eth_type), eth_type);

    /* Put in place destination MAC address */
    if ( (field = _pb_pdata(pdu, "dst") ) )
	ether->eth_dst = *(eth_addr_t *) field;
    else
	memset(&ether->eth_dst, 0, ETH_ADDR_LEN);

    /* Put in place source MAC address */
    if ( (field = _pb_pdata(pdu, "src") ) )
	ether->eth_src = *(eth_addr_t *) field;
    else
	memset(&ether->eth_src, 0, ETH_ADDR_LEN);
}

#if 0
static void
pdu_etherhdr_dumper(pdu_t *p, const char *prefix)
{
    struct etherhdr_options *hdr_data;

    hdr_data = p->header_data;

    printf("%s  Parameters:\n", prefix);
    printf("%s    Source MAC address: %s\n", prefix,
	   pdu_mac_to_ascii(&hdr_data->ether_shost) );
    printf("%s    Destination MAC address: %s\n", prefix,
	   pdu_mac_to_ascii(&hdr_data->ether_dhost) );
    printf("%s    Ethernet protocol: %s\n", prefix,
	   num_info(hdr_data->ether_proto) );
}
#endif

static const pdu_t pdu_ether = {
    .name = "ether",
    .description = "Ethernet II frame",
    .documented_in = "IEEE Std 802.3",
    .fields = (field_t []) {
	{.name = "dst", .type = PDU_FTYPE_MACADDR},
	{.name = "src", .type = PDU_FTYPE_MACADDR},
	{.name = "type", .type = PDU_FTYPE_NUMTYPE},
	{.name = NULL}
    },
    .len = sizeof(struct eth_hdr),
    .build = &build
};

static const pdu_t pdu_802dot3 = {
    .name = "dot3",
    .description = "Ethernet 802.3 frame",
    .documented_in = "IEEE Std 802.3",
    .fields = (field_t []) {
	{
	    .name = "dst",
	    .type = PDU_FTYPE_MACADDR,
	    .offset = offsetof(struct eth_hdr, eth_dst)
	},
	{
	    .name = "src",
	    .type = PDU_FTYPE_MACADDR,
	    .offset = offsetof(struct eth_hdr, eth_src)
	},
	{
	    .name = "length",
	    .type = PDU_FTYPE_UINT16,
	    .offset = offsetof(struct eth_hdr, eth_type),
	    .defval = (defval_t []) { {.type = PDU_DEF_LEN_PAYLOAD} }
	},
	{
	    .name = NULL
	}
    },
    .len = sizeof(struct eth_hdr)
};

struct llc_hdr {
    uint8_t llc_dsap;
    uint8_t llc_ssap;
    uint8_t llc_ctrl;
} __attribute__((__packed__));

static const pdu_t pdu_llc = {
    .name = "llc",
    .description = "802.2 Logical Link Control",
    .documented_in = "IEEE Std 802.2",
    .fields = (field_t []) {
	{
	    .name = "ssap",
	    .type = PDU_FTYPE_UINT8,
	    .offset = offsetof(struct llc_hdr, llc_ssap)
	},
	{
	    .name = "dsap",
	    .type = PDU_FTYPE_UINT8,
	    .offset = offsetof(struct llc_hdr, llc_dsap)
	},
	{
	    .name = "ctrl",
	    .type = PDU_FTYPE_UINT8,
	    .offset = offsetof(struct llc_hdr, llc_ctrl)
	},
	{
	    .name = NULL
	}
    },
    .len = sizeof(struct llc_hdr)
};

struct snap_hdr {
    uint8_t snap_oui[3];
    uint16_t snap_code;
} __attribute__((__packed__));

static const pdu_t pdu_snap = {
    .name = "snap",
    .description = "802.2 SNAP",
    .documented_in = "IEEE Std 802.2",
    .fields = (field_t []) {
	{
	    .name = "oui",
	    .type = PDU_FTYPE_DATA,
	    .offset = offsetof(struct snap_hdr, snap_oui),
	    .size = 3
	},
	{
	    .name = "code",
	    .type = PDU_FTYPE_UINT16,
	    .offset = offsetof(struct snap_hdr, snap_code)
	},
	{
	    .name = NULL
	}
    },
    .len = sizeof(struct snap_hdr)
};

void
_pb_register_ether(void)
{
    _pb_register_protocol(&pdu_ether);
    _pb_register_protocol(&pdu_802dot3);
    _pb_register_protocol(&pdu_llc);
    _pb_register_protocol(&pdu_snap);
}
