package latexDraw.ui.dialog;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.GridLayout;
import java.awt.Toolkit;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.ClipboardOwner;
import java.awt.datatransfer.StringSelection;
import java.awt.datatransfer.Transferable;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.*;

import latexDraw.lang.LaTeXDrawLang;
import latexDraw.util.LaTeXDrawResources;


/** 
 * This class defines a frame which allows to show, to the user, the
 * exception which has occurred in a frame. The user can send the report
 * of the exception to the developer of the LaTeXDraw Project
 *<br>
 * This file is part of LaTeXDraw<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 
 * 01/20/06<br>
 * @author Arnaud BLOUIN<br>
 * @version 1.9<br>
 */
public class ExceptionFrameDialog extends JFrame implements ActionListener, ClipboardOwner
{
	private static final long serialVersionUID = 1L;

	/** The exception which much be displayed */
	Exception exception;
	
	/** The label of the button "ok" */
	public static final String LABEL_OK  = AbstractParametersFrame.LABEL_OK;
	
	public static final String LABEL_COPY = "copy"; //$NON-NLS-1$
	
	private JEditorPane editor;
	
	protected JLabel info;
	
	
	
	/**
	 * The constructor by default
	 */
	public ExceptionFrameDialog(Exception e)
	{
		try
		{
			info = new JLabel();
			exception = e;
			
	 		setIconImage(LaTeXDrawResources.errIcon.getImage());
			
			editor = new JEditorPane();
			JScrollPane scroller = new JScrollPane(); 
		 	JViewport vp = scroller.getViewport();
		 	StackTraceElement[] stack = e.getStackTrace();
		 	int i, size = stack.length;
		 	String msg=""; //$NON-NLS-1$
		 	
			editor.setEditable(false);
		 	vp.add(editor); 
		 	
		 	for(i=0; i<size;i++)
		 		msg += "   at "+stack[i].getClassName() +"."+stack[i].getMethodName()+"("+  //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		 		stack[i].getFileName() +":"+stack[i].getLineNumber()+")\n"; //$NON-NLS-1$ //$NON-NLS-2$
		 	
		 	editor.setText(e.toString()+"\n"+msg); //$NON-NLS-1$
		 	
			JPanel bPanel = new JPanel(new GridLayout(1, 2));
			JButton okB = new JButton(LABEL_OK);
			okB.setActionCommand(LABEL_OK);
			okB.addActionListener(this);
			bPanel.add(okB);
			
			JToolBar toolbar = new JToolBar();
			JButton copyB = new JButton(LaTeXDrawResources.copyIcon);
			
			copyB.setActionCommand(LABEL_COPY);
			copyB.setName(LABEL_COPY);
			copyB.addActionListener(this);
			
			toolbar.setFloatable(false);
			toolbar.add(copyB);
			
			JPanel pInfo = new JPanel();
			info.setText(LaTeXDrawLang.getDialogFrameString("ExceptionFrameDialog.1") //$NON-NLS-1$
						+e.toString()+LaTeXDrawLang.getDialogFrameString("ExceptionFrameDialog.2"));//$NON-NLS-1$
			pInfo.add(info); 
			
			setTitle(LaTeXDrawLang.getDialogFrameString("ExceptionFrameDialog.3")); //$NON-NLS-1$
			
			JPanel panel = new JPanel(new BorderLayout());
			panel.add(scroller, BorderLayout.CENTER);
			panel.add(pInfo, BorderLayout.NORTH);
			
			getContentPane().setLayout(new BorderLayout());
			getContentPane().add(panel, BorderLayout.CENTER);
			getContentPane().add(toolbar, BorderLayout.NORTH);
			getContentPane().add(bPanel, BorderLayout.SOUTH);
			setResizable(true);
			pack();
		}
		catch(Exception ex)
		{
			ex.printStackTrace(); 
			JOptionPane.showMessageDialog(this, 
		 		   LaTeXDrawLang.getDialogFrameString("ExceptionFrameDialog.4") + ex.toString(),  //$NON-NLS-1$
		 		   LaTeXDrawLang.getDialogFrameString("ExceptionFrameDialog.5"), JOptionPane.WARNING_MESSAGE);//$NON-NLS-1$
		}
	}
	
	
	
	public void setInfoLabel(String infoLabel) {
		if(infoLabel!=null)
			info.setText(infoLabel);
	}
	
	
	
	@Override
	public void setVisible(boolean visible)
	{		
		if(visible)
		{
			Dimension dim = Toolkit.getDefaultToolkit().getScreenSize();
			setSize(600, 400);
			setLocation(dim.width/2-getWidth()/2, dim.height/2-getHeight()/2);
		}
		
		super.setVisible(visible);
	}
	
	
	
	
	
	/**
	 * Allows to display a frame which contains the exception and which
	 * allows to send the exception log the LaTeXDraw developer
	 * @param e The exception which has occurred
	 */
	public static void showExceptionDialog(Exception e)
	{
		ExceptionFrameDialog f = new ExceptionFrameDialog(e);
		f.editor.setCaretPosition(0);
		f.setVisible(true);
	}



	public void actionPerformed(ActionEvent e) 
	{
		Object o = e.getSource();
		
		if(o instanceof JButton)
		{
			String msg = ((JButton)o).getActionCommand();
			
			if(msg.equals(LABEL_OK))
			{
				setVisible(false);
				return ;
			}
			
			if(msg.equals(LABEL_COPY))
			{
				Clipboard cb = getToolkit().getSystemClipboard();
				cb.setContents(new StringSelection(editor.getText()), this);
				return ;
			}
		}
	}


	public void lostOwnership(Clipboard clipboard, Transferable contents)
	{
		// Nothing to do.
	}



	public JEditorPane getEditor() {
		return editor;
	}
}
