//
//  ========================================================================
//  Copyright (c) 1995-2015 Mort Bay Consulting Pty. Ltd.
//  ------------------------------------------------------------------------
//  All rights reserved. This program and the accompanying materials
//  are made available under the terms of the Eclipse Public License v1.0
//  and Apache License v2.0 which accompanies this distribution.
//
//      The Eclipse Public License is available at
//      http://www.eclipse.org/legal/epl-v10.html
//
//      The Apache License v2.0 is available at
//      http://www.opensource.org/licenses/apache2.0.php
//
//  You may elect to redistribute this code under either of these licenses.
//  ========================================================================
//

package org.eclipse.jetty.osgi.boot;

import static org.ops4j.pax.exam.CoreOptions.*;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

import junit.framework.Assert;

import org.eclipse.jetty.client.ContentExchange;
import org.eclipse.jetty.client.HttpClient;
import org.eclipse.jetty.client.HttpExchange;
import org.eclipse.jetty.http.HttpMethods;
import org.eclipse.jetty.http.HttpStatus;
import org.eclipse.jetty.server.handler.ContextHandler;
import org.eclipse.jetty.toolchain.test.MavenTestingUtils;
import org.junit.Ignore;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.ops4j.pax.exam.CoreOptions;
import org.ops4j.pax.exam.Inject;
import org.ops4j.pax.exam.Option;
import org.ops4j.pax.exam.container.def.PaxRunnerOptions;
import org.ops4j.pax.exam.junit.Configuration;
import org.ops4j.pax.exam.junit.JUnit4TestRunner;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;

/**
 * TestJettyOSGiBootContextAsService
 * 
 * Tests deployment of a ContextHandler as an osgi Service.
 * 
 * Tests the ServiceContextProvider.
 * 
 */
@RunWith( JUnit4TestRunner.class )
public class JettyOSGiBootContextAsService
{
    private static final boolean LOGGING_ENABLED = false;
    private static final boolean REMOTE_DEBUGGING = false;
    
    @Inject
    BundleContext bundleContext = null;

    @Configuration
    public static Option[] configure()
    {
    	ArrayList<Option> options = new ArrayList<Option>();
    	options.addAll(TestJettyOSGiBootCore.provisionCoreJetty());
        options.add(CoreOptions.bootDelegationPackages("org.xml.sax", "org.xml.*",
                                                       "org.w3c.*", "javax.xml.*"));
       
    	   File base = MavenTestingUtils.getBasedir();
           File src = new File (base, "src");
           File tst = new File (src, "test");
           File config = new File (tst, "config");
           
    	
    	// Enable Logging
    	if(LOGGING_ENABLED) {
    	    options.addAll(Arrays.asList(options(
                // install log service using pax runners profile abstraction (there are more profiles, like DS)
        	// logProfile(),
        	// this is how you set the default log level when using pax logging (logProfile)
        	systemProperty( "org.ops4j.pax.logging.DefaultServiceLog.level" ).value( "INFO" )
    	    )));
    	}
    	
    	// Remote JDWP Debugging
    	if(REMOTE_DEBUGGING) {
    	    options.addAll(Arrays.asList(options(
    	        // this just adds all what you write here to java vm argumenents of the (new) osgi process.
    	        PaxRunnerOptions.vmOption( "-Xrunjdwp:transport=dt_socket,server=y,suspend=y,address=5006" )
    	    )));
    	}

    	// Standard Options
    	   options.addAll(Arrays.asList(options(
    	                                    PaxRunnerOptions.vmOption("-Djetty.port=9876 -Djetty.home="+config.getAbsolutePath()+" -D" + OSGiServerConstants.MANAGED_JETTY_XML_CONFIG_URLS + 
    	                                        "=etc/jetty.xml;etc/jetty-deployer.xml;etc/jetty-selector.xml;etc/jetty-testrealm.xml"),

            /* orbit deps */
            mavenBundle().groupId( "org.eclipse.jetty.orbit" ).artifactId( "javax.servlet.jsp" ).versionAsInProject(),
            mavenBundle().groupId( "org.eclipse.jetty.orbit" ).artifactId( "javax.servlet.jsp.jstl" ).versionAsInProject(),
            mavenBundle().groupId( "org.eclipse.jetty.orbit" ).artifactId( "javax.el" ).versionAsInProject(),
            mavenBundle().groupId( "org.eclipse.jetty.orbit" ).artifactId( "com.sun.el" ).versionAsInProject(),
    	    mavenBundle().groupId( "org.eclipse.jetty.orbit" ).artifactId( "org.apache.jasper.glassfish" ).versionAsInProject(),
            mavenBundle().groupId( "org.eclipse.jetty.orbit" ).artifactId( "org.apache.taglibs.standard.glassfish" ).versionAsInProject(),
            mavenBundle().groupId( "org.eclipse.jetty.orbit" ).artifactId( "org.eclipse.jdt.core" ).versionAsInProject(),
    	    
    	    /* jetty-osgi deps */
    	    mavenBundle().groupId( "org.eclipse.jetty.osgi" ).artifactId( "jetty-osgi-boot" ).versionAsInProject().start(),
            mavenBundle().groupId( "org.eclipse.jetty.osgi" ).artifactId( "jetty-osgi-boot-jsp" ).versionAsInProject().start(),

            //a bundle that registers a webapp as a service for the jetty osgi core to pick up and deploy
            mavenBundle().groupId( "org.eclipse.jetty.osgi" ).artifactId( "test-jetty-osgi-context" ).versionAsInProject().start()
            // mavenBundle().groupId( "org.eclipse.equinox.http" ).artifactId( "servlet" ).versionAsInProject().start()     
        )));
    	
    	return options.toArray(new Option[options.size()]);
    }

    /**
     * You will get a list of bundles installed by default
     * plus your testcase, wrapped into a bundle called pax-exam-probe
     */
    @Test
    public void listBundles() throws Exception
    {
    	Map<String,Bundle> bundlesIndexedBySymbolicName = new HashMap<String, Bundle>();
        for( Bundle b : bundleContext.getBundles() )
        {
        	bundlesIndexedBySymbolicName.put(b.getSymbolicName(), b);
        	System.err.println("Got " + b.getSymbolicName() + " " + b.getVersion().toString() + " " + b.getState());
        }
        
        Bundle osgiBoot = bundlesIndexedBySymbolicName.get("org.eclipse.jetty.osgi.boot");
        Assert.assertNotNull("Could not find the org.eclipse.jetty.osgi.boot bundle", osgiBoot);
        Assert.assertTrue(osgiBoot.getState() == Bundle.ACTIVE);
        
        Bundle testWebBundle = bundlesIndexedBySymbolicName.get("org.eclipse.jetty.osgi.testcontext");
        Assert.assertNotNull("Could not find the org.eclipse.jetty.test-jetty-osgi-context.jar bundle", testWebBundle);
        Assert.assertTrue("The bundle org.eclipse.jetty.testcontext is not correctly resolved", testWebBundle.getState() == Bundle.ACTIVE);
        
        //now test the context
        HttpClient client = new HttpClient();
        client.setConnectorType(HttpClient.CONNECTOR_SELECT_CHANNEL);
        try
        {
            client.start();
            
            ContentExchange getExchange = new ContentExchange();
            getExchange.setURL("http://127.0.0.1:9876/acme/index.html");
            getExchange.setMethod(HttpMethods.GET);
     
            client.send(getExchange);
            int state = getExchange.waitForDone();
            Assert.assertEquals("state should be done", HttpExchange.STATUS_COMPLETED, state);
            
            String content = null;
            int responseStatus = getExchange.getResponseStatus();
            Assert.assertEquals(HttpStatus.OK_200, responseStatus);
            if (responseStatus == HttpStatus.OK_200) {
                content = getExchange.getResponseContent();
            }
            Assert.assertTrue(content.indexOf("<h1>Test OSGi Context</h1>") != -1);
        }
        finally
        {
            client.stop();
        }
        
        ServiceReference[] refs = bundleContext.getServiceReferences(ContextHandler.class.getName(), null);
        Assert.assertNotNull(refs);
        Assert.assertEquals(1,refs.length);
        String[] keys = refs[0].getPropertyKeys();
        if (keys != null)
        {
            for (String k:keys)
                System.err.println("service property: "+k+", "+refs[0].getProperty(k));
        }
        ContextHandler ch = (ContextHandler)bundleContext.getService(refs[0]);
        Assert.assertEquals("/acme", ch.getContextPath());
        
        testWebBundle.stop();
        
        //Check you can see CONTEXT DESTROYED on stderr. TODO: think of a better way to communicate this to the test
    }

	
}
