/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2012 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iartifactcorrector.h"


#include "ierror.h"


#include <vtkPolyDataMapper.h>
#include <vtkRenderer.h>
#include <vtkTimerLog.h>

//
//  Templates
//
#include "igenericfiltertemplate.h"


bool iArtifactCorrector::mIsEnabled = true;


iArtifactCorrector::iArtifactCorrector(iViewSubject *vo) : iGenericPolyDataToPolyDataFilter<vtkDepthSortPolyData>(vo,1,true,false)
{
	mTimer = vtkTimerLog::New(); IERROR_ASSERT(mTimer);
	mLastAllocatedTime = 0.0;

	this->SetDepthSortModeToFirstPoint();
}


iArtifactCorrector::~iArtifactCorrector()
{
	mTimer->Delete();
}


void iArtifactCorrector::Enable(bool s)
{
	mIsEnabled = s;
}


bool iArtifactCorrector::HasArtifacts(vtkRenderer *ren, vtkProperty *prop)
{
	//
	//  By default do nothing
	//
	return false;
}


void iArtifactCorrector::ProduceOutput()
{
	mTimer->StartTimer();
	this->ExecuteParent();
	mTimer->StopTimer();
#ifdef I_DEBUG
	iConsole::PrintDebugMessage("Executing iArtifactCorrector...");
#endif
}



void iArtifactCorrector::CorrectArtifacts(vtkRenderer *ren, vtkProperty *prop, vtkPolyDataMapper *mapper, double time)
{
	if(!mIsEnabled) return;

#ifdef I_DEBUG
	iConsole::PrintDebugMessage("iArtifactCorrector: alloc="+iString::FromNumber(time)+"  exec="+iString::FromNumber(mTimer->GetElapsedTime()));
#endif

	if(mTimer->GetElapsedTime()>0.5*time && time<mLastAllocatedTime*3)
	{
		//
		//  Skip correcting for LOD-style rendering
		//
		mLastAllocatedTime = time;
		return;
	}

	mLastAllocatedTime = time;

	this->SetCamera(ren->GetActiveCamera());
	if(this->HasArtifacts(ren,prop))
	{
		this->vtkDepthSortPolyData::SetInput(mapper->GetInput());
		this->Update();
		mapper->GetInput()->ShallowCopy(this->GetOutput());
	}
}

