{-# LANGUAGE OverloadedStrings, RecordWildCards #-}

-- hot.hs: hOpenPGP Tool
-- Copyright © 2012-2014  Clint Adams
--
-- vim: softtabstop=4:shiftwidth=4:expandtab
--
-- This program is free software: you can redistribute it and/or modify
-- it under the terms of the GNU Affero General Public License as
-- published by the Free Software Foundation, either version 3 of the
-- License, or (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU Affero General Public License for more details.
--
-- You should have received a copy of the GNU Affero General Public License
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.

import Paths_hopenpgp_tools (version)
import HOpenPGP.Tools.Common (banner, versioner, warranty)
import HOpenPGP.Tools.Parser (parsePExp)
import qualified Codec.Encryption.OpenPGP.ASCIIArmor as AA
import Codec.Encryption.OpenPGP.ASCIIArmor.Types (Armor(..), ArmorType(..))
import Codec.Encryption.OpenPGP.Serialize ()
import Codec.Encryption.OpenPGP.Types
import Control.Applicative ((<$>), (<*>), optional, pure)
import Control.Error.Util (note)
import Control.Monad.IO.Class (MonadIO, liftIO)
import Control.Monad.Trans.Resource (runResourceT)
import qualified Data.ByteString as B
import qualified Data.ByteString.Lazy as BL
import Data.Conduit (($=), ($$), Sink)
import Data.Conduit.Cereal (conduitGet, conduitPut)
import qualified Data.Conduit.Binary as CB
import qualified Data.Conduit.List as CL
import Data.Conduit.OpenPGP.Filter (conduitPktFilter, FilterPredicates(RPFilterPredicate))
import Data.Serialize (get, put)
import Data.Serialize.Get (Get)
import Data.Monoid ((<>))
import Data.Version (showVersion)

import System.IO (stdin, stderr, stdout, Handle, hFlush, hPutStrLn, hSetBuffering, BufferMode(..))

import Options.Applicative.Builder (argument, command, footerDoc, headerDoc, help, helpDoc, info, long, metavar, option, eitherReader, prefs, progDesc, showHelpOnError, str, strOption, subparser)
import Options.Applicative.Extra (customExecParser, helper)
import Options.Applicative.Types (Parser)
import Text.PrettyPrint.ANSI.Leijen ((<+>), hardline, hPutDoc, list, softline, text)

data Command = DumpC | DeArmorC | ArmorC ArmoringOptions | FilterC FilteringOptions

data ArmoringOptions = ArmoringOptions {
    comment :: Maybe String
  , armortype :: ArmorType
}

data FilteringOptions = FilteringOptions {
    fExpression :: String
}

doDump :: IO ()
doDump = runResourceT $ CB.sourceHandle stdin $= conduitGet (get :: Get Pkt) $$ printer

-- Print every input value to standard output.
printer :: (Show a, MonadIO m) => Sink a m ()
printer = CL.mapM_ (liftIO . print)

doDeArmor :: IO ()
doDeArmor = do
    a <- runResourceT $ CB.sourceHandle stdin $$ CL.consume
    case AA.decode (B.concat a) of
        Left e -> hPutStrLn stderr $ "Failure to decode ASCII Armor:" ++ e
        Right msgs -> BL.putStr $ BL.concat (map (\(Armor _ _ bs) -> bs) msgs)

doArmor :: ArmoringOptions -> IO ()
doArmor ArmoringOptions{..} = do
    m <- runResourceT $ CB.sourceHandle stdin $$ CL.consume
    let a = Armor armortype (("Version", "hot " ++ showVersion version):maybe [] (\x -> [("Comment", x)]) comment ) (BL.fromChunks m)
    BL.putStr $ AA.encodeLazy [a]

armorTypes :: [(String, ArmorType)]
armorTypes = [
    ("message", ArmorMessage)
  , ("pubkeyblock", ArmorPublicKeyBlock)
  , ("privkeyblock", ArmorPrivateKeyBlock)
  , ("signature", ArmorSignature)
  ]

armorTypeReader :: String -> Either String ArmorType
armorTypeReader = note "unknown armor type" . flip lookup armorTypes

doFilter :: FilteringOptions -> IO ()
doFilter fo = runResourceT $ CB.sourceHandle stdin $= conduitGet (get :: Get Pkt) $= conduitPktFilter (parseExpressions fo) $= conduitPut put $$ CB.sinkHandle stdout

aoP :: Parser ArmoringOptions
aoP = ArmoringOptions
    <$> optional (strOption (long "comment" <> metavar "COMMENT" <> help "ASCII armor Comment field"))
    <*> option (eitherReader armorTypeReader) (long "armor-type" <> metavar "ARMORTYPE" <> armortypeHelp)
    where
        armortypeHelp = helpDoc . Just $ text "ASCII armor type" <> softline <> list (map (text . fst) armorTypes)

foP :: Parser FilteringOptions
foP = FilteringOptions
    <$> argument str ( metavar "EXPRESSION" <> filterTargetHelp )
    where
        filterTargetHelp = helpDoc . Just $ text "packet filter expression" <+> softline <> "see source for current syntax"

dispatch :: Command -> IO ()
dispatch c = (banner' stderr >> hFlush stderr) >> dispatch' c
    where
        dispatch' DumpC = doDump
        dispatch' DeArmorC = doDeArmor
        dispatch' (ArmorC o) = doArmor o
        dispatch' (FilterC o) = doFilter o

main :: IO ()
main = do
    hSetBuffering stderr LineBuffering
    customExecParser (prefs showHelpOnError) (info (helper <*> versioner <*> cmd) (headerDoc (Just (banner "hot")) <> progDesc "hOpenPGP OpenPGP-message Tool" <> footerDoc (Just (warranty "hot")))) >>= dispatch

cmd :: Parser Command
cmd = subparser
  ( command "armor" (info ( ArmorC <$> aoP ) ( progDesc "Armor stdin to stdout" ))
 <> command "dearmor" (info ( pure DeArmorC ) ( progDesc "Dearmor stdin to stdout" ))
 <> command "dump" (info ( pure DumpC ) ( progDesc "Dump OpenPGP packets from stdin" ))
 <> command "filter" (info ( FilterC <$> foP ) ( progDesc "Filter some packets from stdin to stdout" ))
  )

banner' :: Handle -> IO ()
banner' h = hPutDoc h (banner "hot" <> hardline <> warranty "hot" <> hardline)

parseExpressions :: FilteringOptions -> FilterPredicates
parseExpressions FilteringOptions{..} = RPFilterPredicate (parseE fExpression)
    where
         parseE e = either (error . ("filter parse error: "++)) id (parsePExp e)
