/*  GFC-UI: GTK+ Foundation Classes (User Interface Library)
 *  Copyright (C) 2004 The GFC Development Team.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/// @file gfc/gtk/combobox.hh
/// @brief A GtkComboBox C++ wrapper interface.
///
/// Provides ComboBox, a widget used to choose from a list of items. 

#ifndef GFC_GTK_COMBO_BOX_HH
#define GFC_GTK_COMBO_BOX_HH

#ifndef GFC_GTK_BIN_HH
#include <gfc/gtk/bin.hh>
#endif

#ifndef GFC_GTK_CELL_LAYOUT_HH
#include <gfc/gtk/celllayout.hh>
#endif

#ifndef __GTK_COMBO_BOX_H__
#include <gtk/gtkcombobox.h>
#endif

namespace GFC {

namespace Gtk {

/// @class ComboBox combobox.hh gfc/gtk/combobox.hh
/// @brief A GtkComboBox C++ wrapper class.
///
/// The ComboBox widget is used to choose an item from a list of items. 

class ComboBox : public Bin, public CellLayout
{
	friend class G::Object;

	ComboBox(const ComboBox&);
	ComboBox& operator=(const ComboBox&);

protected:
/// @name Constructors
/// @{

	explicit ComboBox(GtkComboBox *combo_box, bool owns_reference = false);
	///< Construct a new ComboBox from an existing GtkComboBox.
	///< @param combo_box A pointer to a GtkComboBox.
	///< @param owns_reference Set false if the initial reference count is floating, set true if it's not.
	///<
	///< <BR>The <EM>combo_box</EM> can be a newly created GtkComboBox or an existing
	///< GtkComboBox (see G::Object::Object).

/// @}
/// @name Signal Prototypes
/// @{

	typedef G::Signal<void> ChangedSignalType;
	typedef G::SignalProxy<TypeInstance, ChangedSignalType> ChangedSignalProxy;
	static const ChangedSignalType changed_signal;
	///< Changed signal (see sig_changed()). Calls a slot with the signature:
	///< @code
	///< void function();
	///< @endcode

/// @}

public:
/// @name Constructors
/// @{

	ComboBox();
	///< Constructs a new combo box.
		
	ComboBox(TreeModel& model);
	///< Constructs a new combo box with the model initialized to <EM>model</EM>.
	///< @param model A Gtk::TreeModel. 
	
	virtual ~ComboBox();
	///< Destructor.

/// @}
/// @name Accessors
/// @{

	GtkComboBox* gtk_combo_box() const;
	///< Get a pointer to the GtkComboBox structure.

	operator GtkComboBox* () const;
	///< Conversion operator; safely converts an ComboBox to a GtkComboBox pointer.

	int get_active() const;
	///< Gets the index of the currently active item, or -1 if there's no active item.
	///< @return An integer which is the index of the currently active item, or -1 if there's no active item.
	 
	bool get_active_iter(TreeIter& iter) const;
	///< Sets <EM>iter</EM> to point to the current active item, if it exists.
	///< @param iter A Gtk::TreeIter. 
	///< @return <EM>true</EM> if <EM>iter</EM> was set to point to the active item. 	

	TreeModel* get_model() const;
	///< Gets the Gtk::TreeModel which is acting as data source for the combo box.
	///< @return A Gtk::TreeModel.

/// @}
/// @name Methods
/// @{

	void set_wrap_width(int width);
	///< Sets the wrap width of the combo box to be <EM>width</EM>. 
	///< @param width The preferred number of columns.
	///<	
	///< <BR>The wrap width is basically the preferred number of columns when
	///< you want the popup to be layed out in a table.

	void set_row_span_column(int row_span);
	///< Sets the column with row span information for the combo box to be <EM>row_span</EM>. 
	///< @param row_span A column in the model passed during construction.
	///<	
	///< <BR>The row span column contains integers which indicate how many rows an item should span.
	
	void set_column_span_column(int column_span);
	///< Sets the column with column span information for the combo box to be <EM>column_span</EM>. 
	///< @param column_span A column in the model passed during construction. 
	///<	
	///< <BR>The column span column contains integers which indicate how many columns an item should span.

	void set_active(int index);
	///< Sets the active item of the combo box to be the item at index.
	///< @param index An index in the model passed during construction, or -1 to have no active item. 
		
	void set_active_iter(const TreeIter& iter);
	///< Sets the current active item to be the one referenced by iter. 
	///< @param iter The Gtk::TreeIter. 
	///< 	
	///< <BR><EM>iter</EM> must correspond to a path of depth one.

	void set_model(TreeModel& model);
	///< Sets the model used by the combo box to be <EM>model</EM>.
	///< @param model A Gtk::TreeModel. 
	///<	 
	///< <BR>Will unset a previously set model (if applicable).

	void append_text(const char *text);
	void append_text(const String& text);
	///< Appends a string to the list of strings stored in the combo box. 
	///< @param text A text string. 
	///<	
	///< <BR><B>Note:</B> You can only use this function with combo boxes constructed with create_text().

	void prepend_text(const char *text);
	void prepend_text(const String& text);
	///< Prepends a string to the list of strings stored in the combo box. 
	///< @param text A text string. 
	///<	
	///< <BR><B>Note:</B> You can only use this function with combo boxes constructed with create_text().

	void insert_text(int position, const char *text);
	void insert_text(int position, const String& text);
	///< Inserts a string at <EM>position</EM> in the list of strings stored in the combo box. 
	///< @param position An index to insert text. 
	///< @param text A text string. 
	///<	
	///< <BR><B>Note:</B> You can only use this function with combo boxes constructed with create_text().

	void remove_text(int position);
	///< Removes the string at <EM>position</EM> from the combo box. 
	///< @param position The index of the item to remove. 
	///<	
	///< <BR><B>Note:</B>you can only use this function with combo boxes constructed with create_text().

	void popup();
	///< Pops up the menu or dropdown list of the combo box. This method is mostly intended
	///< for use by accessibility technologies; applications should have little use for it.
	
	void popdown();
	///< Hides the menu or dropdown list of the combo box. This method is mostly intended
	///< for use by accessibility technologies; applications should have little use for it.

/// @}
/// @name Signal Proxies
/// @{

	const ChangedSignalProxy sig_changed();
	///< Connect to the changed_signal; emitted when the active item in the combo box is changed.

/// @}
};

/// @class ComboBoxText combobox.hh gfc/gtk/combobox.hh
/// @brief A Text Gtk::ComboBox class.
///
/// ComboBoxText is a text combo box, which is a Gtk::ComboBox just displaying strings.
/// If you use this text combo box, you should only manipulate its data source with the
/// following convenience methods: append_text(), insert_text(), prepend_text() and
/// remove_text().

class ComboBoxText : public ComboBox
{
	ComboBoxText(const ComboBoxText&);
	ComboBoxText& operator=(const ComboBoxText&);

public:
/// @name Constructors
/// @{

	ComboBoxText();
	///< Constructs a new text combo box.
	
	virtual ~ComboBoxText();
	///< Destructor.
	
/// @}
};
		
} // namespace Gtk

} // namespace GFC

#include <gfc/gtk/inline/combobox.inl>

#endif // GFC_GTK_COMBO_BOX_HH

