/*
 * $Id: sig_usb_bus.c,v 1.1 2013-05-12 14:54:14 vrsieh Exp $
 *
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "config.h"

#include <assert.h>
#include <stdio.h>

#include "glue.h"

#include "sig_usb_bus.h"

int
sig_usb_bus_reset_set(
	struct sig_usb_bus *b,
	void *s,
	int val
)
{
	unsigned int nr;

	assert(b);
	assert(b->type == SIG_GEN_USB_BUS);

	for (nr = 0; nr < b->member_count; nr++) {
		if (! b->member[nr].f
		 || ! b->member[nr].f->reset_set
		 || b->member[nr].s == s) {
			continue;
		}
		b->member[nr].f->reset_set(b->member[nr].s, val);
	}
	return 0;
}

int
sig_usb_bus_speed_set(
	struct sig_usb_bus *b,
	void *s,
	int val
)
{
	unsigned int nr;

	assert(b);
	assert(b->type == SIG_GEN_USB_BUS);

	for (nr = 0; nr < b->member_count; nr++) {
		if (! b->member[nr].f
		 || ! b->member[nr].f->speed_set
		 || b->member[nr].s == s) {
			continue;
		}
		b->member[nr].f->speed_set(b->member[nr].s, val);
	}
	return 0;
}

int
sig_usb_bus_send_token(
	struct sig_usb_bus *b,
	void *s,
	int pid,
	int addr,
	int endp
)
{
	unsigned int nr;

	assert(b);
	assert(b->type == SIG_GEN_USB_BUS);

	for (nr = 0; nr < b->member_count; nr++) {
		if (! b->member[nr].f
		 || ! b->member[nr].f->recv_token
		 || b->member[nr].s == s) {
			continue;
		}
		b->member[nr].f->recv_token(b->member[nr].s,
				pid, addr, endp);
	}
	return 0;
}

int
sig_usb_bus_send_sof(
	struct sig_usb_bus *b,
	void *s,
	int frame_num
)
{
	unsigned int nr;

	assert(b);
	assert(b->type == SIG_GEN_USB_BUS);

	for (nr = 0; nr < b->member_count; nr++) {
		if (! b->member[nr].f
		 || ! b->member[nr].f->recv_sof
		 || b->member[nr].s == s) {
			continue;
		}
		b->member[nr].f->recv_sof(b->member[nr].s, frame_num);
	}
	return 0;
}

int
sig_usb_bus_send_data(
	struct sig_usb_bus *b,
	void *s,
	int pid,
	unsigned int length,
	uint8_t *data,
	uint16_t crc16
)
{
	unsigned int nr;

	assert(b);
	assert(b->type == SIG_GEN_USB_BUS);

	for (nr = 0; nr < b->member_count; nr++) {
		if (! b->member[nr].f
		 || ! b->member[nr].f->recv_data
		 || b->member[nr].s == s) {
			continue;
		}
		b->member[nr].f->recv_data(b->member[nr].s,
				pid, length, data, crc16);
	}
	return 0;
}

int
sig_usb_bus_send_handshake(struct sig_usb_bus *b, void *s, int pid)
{
	unsigned int nr;

	assert(b);
	assert(b->type == SIG_GEN_USB_BUS);

	for (nr = 0; nr < b->member_count; nr++) {
		if (! b->member[nr].f
		 || ! b->member[nr].f->recv_handshake
		 || b->member[nr].s == s) {
			continue;
		}
		b->member[nr].f->recv_handshake(b->member[nr].s, pid);
	}
	return 0;
}

void
sig_usb_bus_connect(
	struct sig_usb_bus *b,
	void *s,
	const struct sig_usb_bus_funcs *f
)
{
	assert(b);
	assert(b->type == SIG_GEN_USB_BUS);
	assert(b->member_count
			< sizeof(b->member) / sizeof(b->member[0]));

	b->member[b->member_count].s = s;
	b->member[b->member_count].f = f;
	b->member_count++;
}

static void
sig_usb_bus_s0_reset_set(void *_f, int val)
{
	struct sig_usb_bus_merge *f = _f;

	sig_usb_bus_reset_set(f->s1, f, val);
}

static void
sig_usb_bus_s0_speed_set(void *_f, int val)
{
	struct sig_usb_bus_merge *f = _f;

	sig_usb_bus_speed_set(f->s1, f, val);
}

static void
sig_usb_bus_s0_recv_token(void *_f, int pid, int addr, int endp)
{
	struct sig_usb_bus_merge *f = _f;

	sig_usb_bus_send_token(f->s1, f, pid, addr, endp);
}

static void
sig_usb_bus_s0_recv_sof(void *_f, int frame_num)
{
	struct sig_usb_bus_merge *f = _f;

	sig_usb_bus_send_sof(f->s1, f, frame_num);
}

static void
sig_usb_bus_s0_recv_data(
	void *_f,
	int pid,
	unsigned int length,
	uint8_t *data,
	uint16_t crc16
)
{
	struct sig_usb_bus_merge *f = _f;

	sig_usb_bus_send_data(f->s1, f, pid, length, data, crc16);
}

static void
sig_usb_bus_s0_recv_handshake(void *_f, int pid)
{
	struct sig_usb_bus_merge *f = _f;

	sig_usb_bus_send_handshake(f->s1, f, pid);
}

static void
sig_usb_bus_s1_reset_set(void *_f, int val)
{
	struct sig_usb_bus_merge *f = _f;

	sig_usb_bus_reset_set(f->s0, f, val);
}

static void
sig_usb_bus_s1_speed_set(void *_f, int val)
{
	struct sig_usb_bus_merge *f = _f;

	sig_usb_bus_speed_set(f->s0, f, val);
}

static void
sig_usb_bus_s1_recv_token(void *_f, int pid, int addr, int endp)
{
	struct sig_usb_bus_merge *f = _f;

	sig_usb_bus_send_token(f->s0, f, pid, addr, endp);
}

static void
sig_usb_bus_s1_recv_sof(void *_f, int frame_num)
{
	struct sig_usb_bus_merge *f = _f;

	sig_usb_bus_send_sof(f->s0, f, frame_num);
}

static void
sig_usb_bus_s1_recv_data(
	void *_f,
	int pid,
	unsigned int length,
	uint8_t *data,
	uint16_t crc16
)
{
	struct sig_usb_bus_merge *f = _f;

	sig_usb_bus_send_data(f->s0, f, pid, length, data, crc16);
}

static void
sig_usb_bus_s1_recv_handshake(void *_f, int pid)
{
	struct sig_usb_bus_merge *f = _f;

	sig_usb_bus_send_handshake(f->s0, f, pid);
}

struct sig_usb_bus_merge *
sig_usb_bus_merge(
	struct sig_usb_bus *s0,
	struct sig_usb_bus *s1
)
{
	static const struct sig_usb_bus_funcs s0_funcs = {
		.reset_set = sig_usb_bus_s0_reset_set,
		.speed_set = sig_usb_bus_s0_speed_set,
		.recv_token = sig_usb_bus_s0_recv_token,
		.recv_sof = sig_usb_bus_s0_recv_sof,
		.recv_data = sig_usb_bus_s0_recv_data,
		.recv_handshake = sig_usb_bus_s0_recv_handshake,
	};
	static const struct sig_usb_bus_funcs s1_funcs = {
		.reset_set = sig_usb_bus_s1_reset_set,
		.speed_set = sig_usb_bus_s1_speed_set,
		.recv_token = sig_usb_bus_s1_recv_token,
		.recv_sof = sig_usb_bus_s1_recv_sof,
		.recv_data = sig_usb_bus_s1_recv_data,
		.recv_handshake = sig_usb_bus_s1_recv_handshake,
	};
	struct sig_usb_bus_merge *m;

	m = shm_alloc(sizeof(*m));
	assert(m);

	m->s0 = s0;
	sig_usb_bus_connect(s0, m, &s0_funcs);
	m->s1 = s1;
	sig_usb_bus_connect(s1, m, &s1_funcs);

	return m;
}

void
sig_usb_bus_split(struct sig_usb_bus_merge *m)
{
	fixme();
}

struct sig_usb_bus *
sig_usb_bus_create(const char *name)
{
	struct sig_usb_bus *b;

	b = shm_alloc(sizeof(*b));
	assert(b);

	b->type = SIG_GEN_USB_BUS;
	b->member_count = 0;

	return b;
}

void
sig_usb_bus_destroy(struct sig_usb_bus *b)
{
	assert(b);
	assert(b->type == SIG_GEN_USB_BUS);

	shm_free(b);
}

void
sig_usb_bus_suspend(struct sig_usb_bus *b, FILE *fp)
{
	generic_suspend(b, sizeof(*b), fp);
}

void
sig_usb_bus_resume(struct sig_usb_bus *b, FILE *fp)
{
	generic_resume(b, sizeof(*b), fp);
}
