/*
 * $Id: sig_sound.c,v 1.53 2013-05-11 20:58:41 vrsieh Exp $
 *
 * Copyright (C) 2004-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "config.h"

#include <assert.h>
#include <stdio.h>

#include "glue.h"

#include "sig_sound.h"

static void
_sig_sound_send(struct sig_sound *b, void *s, int16_t *samples)
{
	unsigned int nr;

	for (nr = 0; nr < b->nmembers; nr++) {
		void (*func)(void *, int16_t *);

		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->samples_set;
		if (func) {
			func(b->member[nr].s, samples);
		}
	}
}

static void
sig_sound_event(void *_b)
{
	struct sig_sound *b = _b;
	const unsigned long long delta = TIME_HZ / SIG_SOUND_RATE;
	unsigned long long now;

	now = time_virt();

	while (b->time < now) {
		int16_t val;

		if (! b->on) {
			/* Generate '0'. */
			val = 0x8000;
		} else if (b->period == (unsigned long long) -1) {
			/* Generate '1'. */
			val = 0x7fff;
		} else {
			/* Use frequency. */
			if (b->phase < b->period / 2) {
				/* Generate '0'. */
				val = 0x8000;
			} else {
				/* Generate '1'. */
				val = 0x7fff;
			}

			b->phase += delta;
			while (b->period <= b->phase) {
				b->phase -= b->period;
			}
		}
		b->sample[b->nsamples] = val;
		b->nsamples++;

		if (SIG_SOUND_RATE / SIG_SOUND_PROCESS_FREQ <= b->nsamples) {
			/* (void *) 0 - FIXME */
			_sig_sound_send(b, (void *) 0, b->sample);
			b->nsamples = 0;
		}

		b->time += delta;
	}

	time_call_at(b->time + TIME_HZ / SIG_SOUND_PROCESS_FREQ,
			sig_sound_event, b);
}

void
sig_sound_attr_set(
	struct sig_sound *b,
	void *s,
	unsigned int on,
	unsigned long long period
)
{
	if (! b->attr_mode) {
		/* Switch to attribute mode. */
		b->attr_mode = 1;

		b->time = time_virt();

		b->phase = 0;
		b->nsamples = 0;

		time_call_at(b->time + TIME_HZ / SIG_SOUND_PROCESS_FREQ,
				sig_sound_event, b);
	}

	b->on = on;
	b->period = period;
}

void
sig_sound_samples_set(
	struct sig_sound *b,
	void *s,
	int16_t *samples
)
{
	assert(! b->attr_mode);

	_sig_sound_send(b, s, samples);
}

void
sig_sound_connect(
	struct sig_sound *b,
	void *s,
	const struct sig_sound_funcs *f
)
{
	assert(b);
	assert(b->type == SIG_GEN_SOUND);
	assert(b->nmembers < sizeof(b->member) / sizeof(b->member[0]));

	b->member[b->nmembers].s = s;
	b->member[b->nmembers].f = f;
	b->nmembers++;
}

static void
sig_sound_s0_samples_set(void *_f, int16_t *samples)
{
	struct sig_sound_merge *f = _f;
	
	sig_sound_samples_set(f->s1, f, samples);
}

static void
sig_sound_s1_samples_set(void *_f, int16_t *samples)
{
	struct sig_sound_merge *f = _f;
	
	sig_sound_samples_set(f->s0, f, samples);
}

struct sig_sound_merge *
sig_sound_merge(struct sig_sound *s0, struct sig_sound *s1)
{
	static const struct sig_sound_funcs s0_funcs = {
		.samples_set = sig_sound_s0_samples_set,
	};
	static const struct sig_sound_funcs s1_funcs = {
		.samples_set = sig_sound_s1_samples_set,
	};
	struct sig_sound_merge *m;

	m = shm_alloc(sizeof(*m));
	assert(m);

	m->s0 = s0;
	m->s1 = s1;
	sig_sound_connect(s0, m, &s0_funcs);
	sig_sound_connect(s1, m, &s1_funcs);

	return m;
}

void
sig_sound_split(struct sig_sound_merge *m)
{
	fixme();
}

struct sig_sound *
sig_sound_create(const char *name)
{
	struct sig_sound *b;

	b = shm_alloc(sizeof(*b));
	assert(b);

	b->type = SIG_GEN_SOUND;
	b->attr_mode = 0;
	b->nmembers = 0;

	return b;
}

void
sig_sound_destroy(struct sig_sound *b)
{
	assert(b);
	assert(b->type == SIG_GEN_SOUND);

	shm_free(b);
}

void
sig_sound_suspend(struct sig_sound *b, FILE *fp)
{
	generic_suspend(b, sizeof(*b), fp);
}

void
sig_sound_resume(struct sig_sound *b, FILE *fp)
{
	generic_resume(b, sizeof(*b), fp);
}
