/*
 * $Id: sig_scsi_bus.c,v 1.23 2013-05-10 07:20:18 vrsieh Exp $
 *
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "config.h"

#include <assert.h>
#include <stdio.h>

#include "glue.h"

#include "sig_scsi_bus.h"

int
sig_scsi_bus_phase_reselect(struct sig_scsi_bus *b, void *s, uint32_t id)
{
	unsigned int nr;
	int status;

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			return 0;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		status = b->member[nr].f->phase_reselect(b->member[nr].s,
				id);
		if (status == 0) {
			continue;
		} else {
			return status;
		}
	}
	return 0; /* not reached */
}

int
sig_scsi_bus_phase_select(struct sig_scsi_bus *b, void *s, uint32_t id)
{
	unsigned int nr;
	int status;

	assert(b->state == SIG_SCSI_BUS_PHASE_FREE);

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			return 0;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		status = b->member[nr].f->phase_select(b->member[nr].s,
				id);
		if (status == 0) {
			continue;
		} else {
			return status;
		}
	}
	return 0; /* not reached */
}

void
sig_scsi_bus_phase_msg_out(struct sig_scsi_bus *b, void *s)
{
	unsigned int nr;

	if (b->state == SIG_SCSI_BUS_PHASE_MSG_OUT) {
		return;
	}
	b->state = SIG_SCSI_BUS_PHASE_MSG_OUT;

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			return;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		b->member[nr].f->phase_msg_out(b->member[nr].s);
	}
	return;
}


void
sig_scsi_bus_phase_msg_in(struct sig_scsi_bus *b, void *s)
{
	unsigned int nr;

	if (b->state == SIG_SCSI_BUS_PHASE_MSG_IN) {
		return;
	}
	b->state = SIG_SCSI_BUS_PHASE_MSG_IN;

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			return;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		b->member[nr].f->phase_msg_in(b->member[nr].s);
	}
	return;
}

void
sig_scsi_bus_phase_command(struct sig_scsi_bus *b, void *s)
{
	unsigned int nr;

	if (b->state == SIG_SCSI_BUS_PHASE_COMMAND) {
		return;
	}
	b->state = SIG_SCSI_BUS_PHASE_COMMAND;

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			return;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		b->member[nr].f->phase_command(b->member[nr].s);
	}
	return;

}

void
sig_scsi_bus_phase_data_out(struct sig_scsi_bus *b, void *s)
{
	unsigned int nr;

	if (b->state == SIG_SCSI_BUS_PHASE_DATA_OUT) {
		return;
	}
	b->state = SIG_SCSI_BUS_PHASE_DATA_OUT;

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			return;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		b->member[nr].f->phase_data_out(b->member[nr].s);
	}
	return;
}

void
sig_scsi_bus_phase_data_in(struct sig_scsi_bus *b, void *s)
{
	unsigned int nr;

	if (b->state == SIG_SCSI_BUS_PHASE_DATA_IN) {
		return;
	}
	b->state = SIG_SCSI_BUS_PHASE_DATA_IN;

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			return;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		b->member[nr].f->phase_data_in(b->member[nr].s);
	}
	return;
}

void
sig_scsi_bus_phase_status(struct sig_scsi_bus *b, void *s)
{
	unsigned int nr;

	if (b->state == SIG_SCSI_BUS_PHASE_STATUS) {
		return;
	}
	b->state = SIG_SCSI_BUS_PHASE_STATUS;

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			return;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		b->member[nr].f->phase_status(b->member[nr].s);
	}
	return;
}

void
sig_scsi_bus_phase_free(struct sig_scsi_bus *b, void *s)
{
	unsigned int nr;

	if (b->state == SIG_SCSI_BUS_PHASE_FREE) {
		return;
	}
	b->state = SIG_SCSI_BUS_PHASE_FREE;

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			return;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		b->member[nr].f->phase_free(b->member[nr].s);
	}
	return;
}

void
sig_scsi_bus_atn_set(struct sig_scsi_bus *b, void *s, unsigned int val)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			return;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		b->member[nr].f->atn_set(b->member[nr].s,
				val);
	}
	return;
}


void
sig_scsi_bus_want_recv(struct sig_scsi_bus *b, void *s, unsigned long count)
{
	unsigned int nr;

	assert(b->state == SIG_SCSI_BUS_PHASE_MSG_OUT
	    || b->state == SIG_SCSI_BUS_PHASE_COMMAND
	    || b->state == SIG_SCSI_BUS_PHASE_DATA_OUT);

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			return;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		b->member[nr].f->want_recv(b->member[nr].s,count);
	}
	return; /* not reached */
}

unsigned long
sig_scsi_bus_send(
	struct sig_scsi_bus *b, 
	void *s, 
	const uint8_t *buf, 
	unsigned long bufsize
)
{
	unsigned int nr;
	int status;

	assert(b->state == SIG_SCSI_BUS_PHASE_MSG_OUT
	    || b->state == SIG_SCSI_BUS_PHASE_COMMAND
	    || b->state == SIG_SCSI_BUS_PHASE_DATA_OUT);

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			return 0;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		status = b->member[nr].f->send(b->member[nr].s,
				buf,bufsize);
	
		/* we do not know whether 0 means "transfer finnished"
		 * or "call not for me", so we have to check the
		 * remaining functions in case of 0! */
		if (status) {
			return status;
		} else {
			continue;
		}
	}
	return 0; /*never reached */

}

void
sig_scsi_bus_want_send(struct sig_scsi_bus *b, void *s, unsigned long count)
{
	unsigned int nr;

	assert(b->state == SIG_SCSI_BUS_PHASE_MSG_IN
	    || b->state == SIG_SCSI_BUS_PHASE_STATUS
	    || b->state == SIG_SCSI_BUS_PHASE_DATA_IN);

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			return;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		b->member[nr].f->want_send(b->member[nr].s,count);
	}
	return; /* not reached */
}

unsigned long
sig_scsi_bus_recv(
	struct sig_scsi_bus *b, 
	void *s, 
	uint8_t *buf, 
	unsigned long bufsize
)
{
	unsigned int nr;
	int status;

	assert(b->state == SIG_SCSI_BUS_PHASE_MSG_IN
	    || b->state == SIG_SCSI_BUS_PHASE_STATUS
	    || b->state == SIG_SCSI_BUS_PHASE_DATA_IN);

	for (nr = 0; ; nr++) {
		if (nr == b->member_count) {
			return 0;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		status = b->member[nr].f->recv(b->member[nr].s,
				buf,bufsize);
		/* we do not know whether 0 means "transfer finnished"
		 * or "call not for me", so we have to check the
		 * remaining functions in case of 0! */
		if (status) {
			return status;
		} else {
			continue;
		}
	
	}
	return 0; /* not reached */
}

void
sig_scsi_bus_connect(
	struct sig_scsi_bus *b,
	void *s,
	const struct sig_scsi_bus_funcs *f
)
{
	assert(b->member_count < sizeof(b->member) / sizeof(b->member[0]));
	assert(b->type == SIG_GEN_SCSI_BUS);

	b->member[b->member_count].s = s;
	b->member[b->member_count].f = f;
	b->member_count++;
}

struct sig_scsi_bus_merge *
sig_scsi_bus_merge(struct sig_scsi_bus *s0, struct sig_scsi_bus *s1)
{
	fixme();
	return NULL;
}

void
sig_scsi_bus_split(struct sig_scsi_bus_merge *m)
{
	fixme();
}

struct sig_scsi_bus *
sig_scsi_bus_create(const char *name)
{
	struct sig_scsi_bus *bus;

	bus = shm_alloc(sizeof(struct sig_scsi_bus));
	assert(bus);

	bus->type = SIG_GEN_SCSI_BUS;
	bus->member_count = 0;

	bus->state = SIG_SCSI_BUS_PHASE_FREE;

	return bus;
}

void
sig_scsi_bus_destroy(struct sig_scsi_bus *sig)
{
	assert(sig);
	assert(sig->type == SIG_GEN_SCSI_BUS);

	shm_free(sig);
}

void
sig_scsi_bus_suspend(struct sig_scsi_bus *b, FILE *fSig)
{
	size_t size = sizeof(*b);
	
	generic_suspend(b, size, fSig);
}

void
sig_scsi_bus_resume(struct sig_scsi_bus *b, FILE *fSig)
{
	size_t size = sizeof(*b);
	
	generic_resume(b, size, fSig);
}
