/*
 * $Id: sig_gen_bus.h,v 1.46 2013-05-06 13:21:53 vrsieh Exp $
 *
 * Copyright (C) 2004-2013 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <inttypes.h>

/* Should be configurable! FIXME VOSSI */
#define SIG_HOST_BUS_HZ     (60*1000*1000)

#include "sig_gen.h"

#ifdef NEED_LOCK
#include "glue-spinlock.h"
#endif

#ifdef NEED_SMM
#define INT_STATE_C	unsigned int state,
#else
#define INT_STATE_C
#endif

#define BUS_HASH_SIZE	(1 << 12)

enum bus_(cycle) {
	BUS_(C0R) = 0,
	BUS_(C0W),
	BUS_(C1R),
	BUS_(C1W),
	BUS_(IOR),
	BUS_(IOW),
	BUS_(MR),
	BUS_(MW),
	BUS_(INTA),
};

struct bus_(funcs) {
#ifdef NEED_ADDR_TYPE
	int (*addr_type)(void *s,
			INT_STATE_C bus_addr_t addr, unsigned int type);
	int (*read_data)(void *s, unsigned int bs, bus_data_t *valp);
	int (*write_data)(void *s, unsigned int bs, bus_data_t val);
#endif /* NEED_ADDR_TYPE */
#ifdef NEED_C0
	int (*c0r)(void *s, bus_addr_t addr, unsigned int bs, bus_data_t *valp);
	int (*c0w)(void *s, bus_addr_t addr, unsigned int bs, bus_data_t val);
#endif /* NEED_C0 */
#ifdef NEED_C1
	int (*c1r)(void *s, bus_addr_t addr, unsigned int bs, bus_data_t *valp);
	int (*c1w)(void *s, bus_addr_t addr, unsigned int bs, bus_data_t val);
#endif /* NEED_C1 */
#ifdef NEED_IO_LEN
	int (*inb)(void *s, unsigned char *valp, bus_addr_t port);
	int (*inw)(void *s, unsigned short *valp, bus_addr_t port);
	int (*outb)(void *s, unsigned char val, bus_addr_t port);
	int (*outw)(void *s, unsigned short val, bus_addr_t port);
#endif /* NEED_IO_LEN */
#ifdef NEED_IO_BS
	int (*ior)(void *s, bus_addr_t port, unsigned int bs, bus_data_t *valp);
	int (*iow)(void *s, bus_addr_t port, unsigned int bs, bus_data_t val);

	int (*ior_info)(void *s, bus_addr_t port, unsigned int bs,
		int (**cfp)(void *, bus_addr_t, unsigned int, bus_data_t *),
		void **csp);
	int (*iow_info)(void *s, bus_addr_t port, unsigned int bs,
		int (**cfp)(void *, bus_addr_t, unsigned int, bus_data_t),
		void **csp);

	void (*ior_info_flush)(void *s, bus_addr_t port, unsigned int bs);
	void (*iow_info_flush)(void *s, bus_addr_t port, unsigned int bs);
#endif /* NEED_IO_BS */
#ifdef NEED_MEM_LEN
	int (*readb)(void *s, bus_addr_t addr, uint8_t *valp);
	int (*readw)(void *s, bus_addr_t addr, uint16_t *valp);
	int (*readl)(void *s, bus_addr_t addr, uint32_t *valp);
	int (*writeb)(void *s, bus_addr_t addr, uint8_t val);
	int (*writew)(void *s, bus_addr_t addr, uint16_t val);
	int (*writel)(void *s, bus_addr_t addr, uint32_t val);
#endif /* NEED_MEM_LEN */
#ifdef NEED_MEM_BS
	int (*mr)(void *s, INT_STATE_C bus_addr_t addr,
			unsigned int bs, bus_data_t *valp);
	int (*mw)(void *s, INT_STATE_C bus_addr_t addr,
			unsigned int bs, bus_data_t val);
#ifdef NEED_X
	int (*mx)(void *s, INT_STATE_C bus_addr_t addr,
			unsigned int bs, bus_data_t *valp);
#endif /* NEED_X */
#endif /* NEED_MEM_BS */
#ifdef NEED_MEM_MAP
	int (*map_r_check)(void *s, INT_STATE_C bus_addr_t addr);
	int (*map_r)(void *s, INT_STATE_C bus_addr_t addr,
			int (**cfp)(void *, bus_addr_t, unsigned int, bus_data_t *),
			void **csp,
			char **haddr_p);
	int (*map_w_check)(void *s, INT_STATE_C bus_addr_t addr);
	int (*map_w)(void *s, INT_STATE_C bus_addr_t addr,
			int (**cfp)(void *, bus_addr_t, unsigned int, bus_data_t),
			void **csp,
			char **haddr_p);
#ifdef NEED_X
	int (*map_x_check)(void *s, INT_STATE_C bus_addr_t addr);
	int (*map_x)(void *s, INT_STATE_C bus_addr_t addr,
			int (**cfp)(void *, bus_addr_t, unsigned int, bus_data_t *),
			void **csp,
			char **haddr_p);
#endif
	void (*unmap)(void *s, bus_addr_t addr, bus_addr_t len);
#endif /* NEED_MEM_MAP */
#ifdef NEED_INTA
	int (*inta_addr)(void *s);
	int (*inta_data)(void *s, uint8_t *valp);
#endif /* NEED_INTA */
};

struct bus {
	enum sig_gen_type type;
	struct {
		void *s;
		const struct bus_(funcs) *f;
	} member[32];
	unsigned int member_count;

#ifdef NEED_IO_BS
	struct bus_(io) {
		struct bus_(io) *lru_prev;
		struct bus_(io) *lru_next;

		struct bus_(io) *hash_prev;
		struct bus_(io) *hash_next;

		enum {
			BUS_(TYPE_IOR),
			BUS_(TYPE_IOW),
		} type;

		bus_addr_t port;
		unsigned int bs;
		int (*ior)(void *s,
			bus_addr_t port, unsigned int bs, bus_data_t *valp);
		int (*iow)(void *s,
			bus_addr_t port, unsigned int bs, bus_data_t val);
		void *s;
	} io[4096];
	struct bus_(io) *io_lru_first;
	struct bus_(io) *io_lru_last;
	struct bus_(io) *io_hash_first[BUS_HASH_SIZE];
	struct bus_(io) *io_hash_last[BUS_HASH_SIZE];
#endif /* NEED_IO_BS */

#ifdef NEED_MEM_BS
	struct bus_(map) {
		struct bus_(map) *lru_prev;
		struct bus_(map) *lru_next;

		struct bus_(map) *hash_prev;
		struct bus_(map) *hash_next;

		enum {
			BUS_(TYPE_MR),
			BUS_(TYPE_MW),
#ifdef NEED_X
			BUS_(TYPE_MX),
#endif
		} type;

#ifdef NEED_SMM
		unsigned int state;
#endif
		bus_addr_t addr;
		const bus_data_t *mr_map;
		bus_data_t *mw_map;
#ifdef NEED_X
		bus_data_t *mx_map;
#endif
		int (*mr_func)(void *, bus_addr_t, unsigned int, bus_data_t *);
		void *mr_cpssp;
		int (*mw_func)(void *, bus_addr_t, unsigned int, bus_data_t);
		void *mw_cpssp;
#ifdef NEED_X
		int (*mx_func)(void *, bus_addr_t, unsigned int, bus_data_t *);
		void *mx_cpssp;
#endif
	} map[4096];
	struct bus_(map) *map_lru_first;
	struct bus_(map) *map_lru_last;
	struct bus_(map) *map_hash_first[BUS_HASH_SIZE];
	struct bus_(map) *map_hash_last[BUS_HASH_SIZE];
#endif /* NEED_MEM_BS */
#ifdef NEED_LOCK
	spinlock_t cache_lock;
#endif
};

struct bus_(merge) {
	struct bus *s0;
	struct bus *s1;
};

#ifdef NEED_ADDR_TYPE
extern int
bus_(addr_type)(struct bus *b, void *s,
		INT_STATE_C bus_addr_t addr, unsigned int type);
extern int
bus_(read_data)(struct bus *b, void *s,
		unsigned int bs, bus_data_t *valp);
extern int
bus_(write_data)(struct bus *b, void *s,
		unsigned int bs, bus_data_t val);
#endif /* NEED_ADDR_TYPE */
#ifdef NEED_C0
extern int
bus_(c0r)(struct bus *b, void *s,
		bus_addr_t addr, unsigned int bs, bus_data_t *valp);
extern int
bus_(c0w)(struct bus *b, void *s,
		bus_addr_t addr, unsigned int bs, bus_data_t val);
#endif /* NEED_C0 */
#ifdef NEED_C1
extern int
bus_(c1r)(struct bus *b, void *s,
		bus_addr_t addr, unsigned int bs, bus_data_t *valp);
extern int
bus_(c1w)(struct bus *b, void *s,
		bus_addr_t addr, unsigned int bs, bus_data_t val);
#endif /* NEED_C1 */
#ifdef NEED_IO_LEN
extern int
bus_(inb)(struct bus *b, void *s,
		unsigned char *valp, bus_addr_t port);
extern int
bus_(inw)(struct bus *b, void *s,
		unsigned short *valp, bus_addr_t port);
extern int
bus_(outb)(struct bus *b, void *s,
		unsigned char val, bus_addr_t port);
extern int
bus_(outw)(struct bus *b, void *s,
		unsigned short val, bus_addr_t port);
#endif /* NEED_IO_LEN */
#ifdef NEED_IO_BS
extern int
bus_(ior)(struct bus *b, void *s,
		bus_addr_t port, unsigned int bs, bus_data_t *valp);
extern int
bus_(iow)(struct bus *b, void *s,
		bus_addr_t port, unsigned int bs, bus_data_t val);

extern int
bus_(ior_info)(struct bus *b, void *s,
		bus_addr_t port, unsigned int bs,
		int (**cf)(void *, bus_addr_t, unsigned int, bus_data_t *),
		void **cs);
extern int
bus_(iow_info)(struct bus *b, void *s,
		bus_addr_t port, unsigned int bs,
		int (**cf)(void *, bus_addr_t, unsigned int, bus_data_t),
		void **cs);

extern void
bus_(ior_info_flush)(struct bus *b, void *s,
		bus_addr_t port, unsigned int bs);
extern void
bus_(iow_info_flush)(struct bus *b, void *s,
		bus_addr_t port, unsigned int bs);
#endif /* NEED_IO_BS */
#ifdef NEED_MEM_LEN
extern int
bus_(readb)(struct bus *b, void *s,
		bus_addr_t addr, uint8_t *valp);
extern int
bus_(readw)(struct bus *b, void *s,
		bus_addr_t addr, uint16_t *valp);
extern int
bus_(readl)(struct bus *b, void *s,
		bus_addr_t addr, uint32_t *valp);
extern int
bus_(writeb)(struct bus *b, void *s,
		bus_addr_t addr, uint8_t val);
extern int
bus_(writew)(struct bus *b, void *s,
		bus_addr_t addr, uint16_t val);
extern int
bus_(writel)(struct bus *b, void *s,
		bus_addr_t addr, uint32_t val);
#endif /* NEED_MEM_LEN */
#ifdef NEED_MEM_BS
extern int
bus_(mr)(struct bus *b, void *s,
		INT_STATE_C bus_addr_t addr, unsigned int bs, bus_data_t *valp);
extern int
bus_(mw)(struct bus *b, void *s,
		INT_STATE_C bus_addr_t addr, unsigned int bs, bus_data_t val);
#ifdef NEED_X
extern int
bus_(mx)(struct bus *b, void *s,
		INT_STATE_C bus_addr_t addr, unsigned int bs, bus_data_t *valp);
#endif /* NEED_X */
#endif /* NEED_MEM_BS */
#ifdef NEED_MEM_MAP
extern int
bus_(map_r_check)(struct bus *b, void *s, INT_STATE_C bus_addr_t addr);
extern int
bus_(map_r)(struct bus *b, void *s,
		INT_STATE_C bus_addr_t addr,
		int (**cfp)(void *, bus_addr_t, unsigned int, bus_data_t *),
		void **csp,
		char **haddr_p);
extern int
bus_(map_w_check)(struct bus *b, void *s, INT_STATE_C bus_addr_t addr);
extern int
bus_(map_w)(struct bus *b, void *s,
		INT_STATE_C bus_addr_t addr,
		int (**cfp)(void *, bus_addr_t, unsigned int, bus_data_t),
		void **csp,
		char **haddr_p);
#ifdef NEED_X
extern int
bus_(map_x_check)(struct bus *b, void *s, INT_STATE_C bus_addr_t addr);
extern int
bus_(map_x)(struct bus *b, void *s,
		INT_STATE_C bus_addr_t addr,
		int (**cfp)(void *, bus_addr_t, unsigned int, bus_data_t *),
		void **csp,
		char **haddr_p);
#endif /* NEED_X */

extern void
bus_(unmap)(struct bus *b, void *s,
		bus_addr_t addr, bus_addr_t len);
#endif /* NEED_MEM_MAP */
#ifdef NEED_INTA
extern int
bus_(inta_addr)(struct bus *b, void *s);
extern int
bus_(inta_data)(struct bus *b, void *s,
		uint8_t *valp);
#endif /* NEED_INTA */

extern void
bus_(connect)(struct bus *b, void *s, const struct bus_(funcs) *f);

extern struct bus_(merge) *
bus_(merge)(
	struct bus *s0,
	struct bus *s1
);
extern void
bus_(split)(struct bus_(merge) *m);

extern struct bus *
bus_(create)(const char *name);
extern void
bus_(destroy)(struct bus *b);

extern void
bus_(suspend)(struct bus *b, FILE *fSig);
extern void
bus_(resume)(struct bus *b, FILE *fSig);

#undef INT_STATE_C
