/* $Id: apm.c,v 1.1 2013-04-22 16:24:21 vrsieh Exp $ 
 *
 * Copyright (C) 2004-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifdef CONFIG_APM_SUPPORT

#include "build_config.h"
#include "compiler.h"

/* =========================== RUNTIME ============================= */
#if defined(RUNTIME_RM) || defined(RUNTIME_PM)

#include "compiler.h"
#include "bios.lds.h"
#include "io.h"
#include "var.h"
#include "ptrace.h"
#include "entry.h"
#include "apm.h"

/*
 * APM Function 5300: installation check
 *
 * In:  AX	= 5300h
 *      BX	= power device id
 *
 * Out: AH	= major version (BCD)
 *	AL	= minor version (BCD)
 *	BX	= 504Dh ("PM")
 *	CX	= Flags
 *	CF	= clear if successful (returned AH=00h)
 *	CF	= set on error
 */
/* This function **must** be inline! */
static inline __attribute__((__always_inline__)) void
bios_15_5300(struct regs *regs)
{
	AX = 0x0102;	/* 1.2 */
	BX = 0x504D;	/* PM */
	CX = 0x0003;	/* 16+32 bit support */
	F &= ~(1 << 0);	/* no error */
}

/*
 * APM Function 5303: connect 32 bit PM interface
 *
 * In:  AX	= 5303h
 *      BX	= power device id
 *
 * Out:	AX	= real-mode segment base address of protected-mode 32-bit code segment
 *	EBX	= offset of entry point
 *	CX	= real-mode segment base address of protected-mode 16-bit code segment
 *	DX	= real-mode segment base address of protected-mode 16-bit data segment
 *	CF	= clear if successful
 *	CF	= set on error
 * APM v1.1:
 *	SI	= APM BIOS code segment length
 *	DI	= APM BIOS data segment length
 */
/* This function **must** be inline! */
static inline __attribute__((__always_inline__)) void
bios_15_5303(struct regs *regs)
{
	unsigned short apm_pm_seg;
	unsigned short entry_off;

	apm_pm_seg = (unsigned long) vma_rt_pm_apm >> 4;
	entry_off = (unsigned long) bios_15_53xx_entry - apm_pm_seg * 16;

	EBX = 0;
	EBX = entry_off;
	AX = apm_pm_seg;
	CX = apm_pm_seg;
	DX = apm_pm_seg;
	DI = 0x0000;
	SI = 0x0000;
	F &= ~(1 << 0); /* no error */

}

/*
 * APM Function 5304: disconnect interface
 *
 * In:  AX      = 5304h
 *      BX	= power device id
 *
 * Out:	AH	= status
 *	CF	= clear if successful (returned AH=00h)
 *	CF	= set on error
 */
/* This function **must** be inline! */
static inline __attribute__((__always_inline__)) void
bios_15_5304(struct regs *regs)
{
	AH = 0x00;
	F &= ~(1 << 0);	/* no error */
}

/*
 * APM Function 5305: cpu idle
 *
 * In:  AX      = 5305h
 *
 * Out:	AH	= status
 *	CF	= clear if successful (returned AH=00h)
 *	CF	= set on error
 */
/* This function **must** be inline! */
static inline __attribute__((__always_inline__)) void
bios_15_5305(struct regs *regs)
{
	asm volatile (
		"sti\n"
		"hlt\n"
	);

	AH = 0x00;
	F &= ~(1 << 0);	/* no error */
}

/*
 * APM Function 5306: cpu busy
 *
 * In:  AX      = 5306h
 *
 * Out:	AH	= status
 *	CF	= clear if successful (returned AH=00h)
 *	CF	= set on error
 */
/* This function **must** be inline! */
static inline __attribute__((__always_inline__)) void
bios_15_5306(struct regs *regs)
{
	/* Nothing to do... */

	AH = 0x00;
	F &= ~(1 << 0);	/* no error */
}

/*
 * APM Function 5307: set power state
 *
 * In:  AX	= 5307h
 *      BX	= power device id
 *	CX	= power state
 *
 * Out: AH	= status
 *	CF	= clear if successful (returned AH=00h)
 *	CF	= set on error
 */
/* This function **must** be inline! */
static inline __attribute__((__always_inline__)) void
bios_15_5307(struct regs *regs)
{
	/* BX = 0x0001 -> all devices manages by BIOS */
	/* CX = 0x0003 -> off */
	if (CX == 0x0003 && BX == 0x0001) {
		/* use PMCNTRL register of 82371AB */
		/* FIXME knilch: make shure PM PCI device is enabled and
		 *               I/O base address is right by looking at
		 *               config space */
		outb(0, 0x4005); /* set suspend type 0: soft power off */
		outb((1 << 5), 0x4005); /* suspend enable */
		/* system is now powered down. */
	}
	F &= ~(1 << 0);	/* no error */
}

/* This function **must** be inline! */
static inline __attribute__((__always_inline__)) void
bios_15_530a(struct regs *regs)
{
	if (BH == 0x80) {
		SI = 0x0000;	/* no batteries installed */
	}
	BH = 0x00;		/* off-line */
	BL = 0x03;		/* charging */
	CH = 80;		/* No system battery */
	CL = 100;		/* 100% charged */
	DX = 0x8000 + 600;	/* battery life = 600 minutes*/
	F &= ~(1 << 0);		/* no error */
}

/* This function **must** be inline! */
static inline __attribute__((__always_inline__)) void
bios_15_530b(struct regs *regs)
{
	AX = 0x8000;	/* no event pending */
	F |= 1 << 0;	/* error */
}

/*
 * APM Function 530e: get driver version
 *
 * In:  AX	= 530eh
 *      BX	= 0000h
 *	CH	= APM driver version (major, BCD)
 *	CL	= APM driver version (minor, BCD)
 *
 * Out:	AH	= APM connection version (major, BCD)
 *	AL	= APM connection version (minor, BCD)
 *	CF	= clear if successful (returned AH=00h)
 *	CF	= set on error
 */
/* This function **must** be inline! */
static inline __attribute__((__always_inline__)) void
bios_15_530e(struct regs *regs)
{
	AX = 0x0102;		/* version 1.2 */
	F &= ~(1 << 0);		/* no error */
}

/*
 * APM Function 530f: engage/disengage power management
 *
 * In:	AX	= 530fh
 *	BX	= Device ID
 *	CX	= 0: disengage
 *		  1: engage
 * Out:	AH	= Error code
 * 	CF	= clear if successful (returned AH=00h)
 * 		= set on error
 */
/* This function **must** be inline! */
static inline __attribute__((__always_inline__)) void
bios_15_530f(struct regs *regs)
{
	/* FIXME VOSSI */
	AH = 0x00;
	F &= ~(1 << 0);		/* no error */
}

/* This function **must** be inline! */
static inline __attribute__((__always_inline__)) void
func_53xx(struct regs *regs)
{
	if (AL == 0x00) {
		/* APM BIOS installation check */
		bios_15_5300(regs);

	} else if (AL == 0x03) {
		/* APM 32 bit connect */
		bios_15_5303(regs);

	} else if (AL == 0x04) {
		/* APM disconnect */
		bios_15_5304(regs);

	} else if (AL == 0x05) {
		/* APM cpu idle */
		bios_15_5305(regs);

	} else if (AL == 0x06) {
		/* APM cpu busy */
		bios_15_5306(regs);

	} else if (AL == 0x07) {
		/* APM set power state */
		bios_15_5307(regs);

	} else if (AX == 0x530a) {
		/* APM get power status */
		bios_15_530a(regs);

	} else if (AX == 0x530b) {
		/* APM get event */
		bios_15_530b(regs);

	} else if (AL == 0x0e) {
		/* APM get driver version */
		bios_15_530e(regs);

	} else if (AL == 0x0f) {
		/* APM engage/disengage */
		bios_15_530e(regs);

	} else {
		AH = 0x86;		/* APM not present */
		F |= 1 << 0;		/* no error */
	}
}

#endif /* RUNTIME_PM || RUNTIME_RM */
/* ==================== REAL-MODE RUNTIME ==================== */
#ifdef RUNTIME_RM

CODE16;

void
bios_15_53xx(struct regs *regs)
{
	func_53xx(regs);
}

#endif /* RUNTIME_RM */
/* ==================== PROTECTED MODE RUNTIME ==================== */
#ifdef RUNTIME_PM

CODE32;

__attribute((section(".text.apm"))) void
bios_15_53xx_pm(struct regs *regs)
{
	func_53xx(regs);
}

#endif /* RUNTIME_PM */

#endif /* CONFIG_APM_SUPPORT */
