/*
 * Copyright (C) 2015 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "xml.h"

static struct xml_attr *
xml_attr_elem_parse(FILE *fp)
{
	char tag[1024];
	char val[64*1024];
	int c;
	int x;
	struct xml_attr *act;

	c = fgetc(fp);
	while (c == ' '
	    || c == '\t'
	    || c == '\n'
	    || c == '\r') {
		c = fgetc(fp);
	}
	ungetc(c, fp);

	if (c == EOF
	 || c == '/'
	 || c == '>') {
		return NULL;
	}

	/*
	 * Read tag.
	 */
	x = 0;
	c = fgetc(fp);
	while (('0' <= c && c <= '9')
	    || ('A' <= c && c <= 'Z')
	    || ('a' <= c && c <= 'z')
	    || c == ':'
	    || c == '-'
	    || c == '_') {
		tag[x++] = c;
		c = fgetc(fp);
	}
	ungetc(c, fp);
	tag[x] = '\0';

	/*
	 * Read '='.
	 */
	c = fgetc(fp);
	assert(c == '=');

	/*
	 * Read '"'.
	 */
	c = fgetc(fp);
	assert(c == '"');

	/*
	 * Read value.
	 */
	x = 0;
	c = fgetc(fp);
	while (c != EOF
	    && c != '"') {
		val[x++] = c;
		c = fgetc(fp);
	}
	ungetc(c, fp);
	val[x] = '\0';

	/*
	 * Read '"'.
	 */
	c = fgetc(fp);
	assert(c == '"');

	act = malloc(sizeof(*act));
	assert(act);

	act->tag = strdup(tag);
	assert(act->tag);
	act->val = strdup(val);
	assert(act->val);

	return act;
}

static void
xml_attr_parse(FILE *fp, struct xml_attr **firstp, struct xml_attr **lastp)
{
	struct xml_attr *first;
	struct xml_attr *last;
	struct xml_attr *act;

	first = NULL;
	last = NULL;

	while ((act = xml_attr_elem_parse(fp)) != NULL) {
		act->prev = last;
		act->next = NULL;
		if (act->prev) {
			act->prev->next = act;
		} else {
			first = act;
		}
		last = act;
	}

	*firstp = first;
	*lastp = last;
}

static void
xml_attr_free(struct xml_attr *first, struct xml_attr *last)
{
	struct xml_attr *act;

	while (first) {
		act = first;
		first = act->next;
		if (act->next) {
			act->next->prev = NULL;
		} else {
			last = NULL;
		}

		free(act->tag);
		free(act->val);
		free(act);
	}

	assert(! first);
	assert(! last);
}

static struct xml *
xml_elem_parse(FILE *fp)
{
	int c;
	int type;
	char string[1024*1024];
	int x;
	struct xml *act;
	struct xml_attr *attr_first;
	struct xml_attr *attr_last;

again:	;
	c = fgetc(fp);
	while (c == ' '
	    || c == '\t'
	    || c == '\n'
	    || c == '\r') {
		c = fgetc(fp);
	}
	ungetc(c, fp);

	c = fgetc(fp);
	switch (c) {
	case EOF:
		/*
		 * End of file.
		 */
		act = NULL;
		break;

	case '<':
		/*
		 * Start/End tag.
		 */
		c = fgetc(fp);
		switch (c) {
		case '?': /* <?xml version="1.0"?> */
		case '!': /* <!DOCTYPE UnivIS SYSTEM "http://univis.uni-erlangen.de/univis.dtd"> */
			c = fgetc(fp);
			while (c != EOF
			    && c != '\n') {
				c = fgetc(fp);
			}
			goto again;
		case '/':
			type = XML_END;
			break;
		default:
			ungetc(c, fp);
			type = XML_START;
			break;
		}

		x = 0;
		c = fgetc(fp);
		while (('0' <= c && c <= '9')
		    || ('A' <= c && c <= 'Z')
		    || ('a' <= c && c <= 'z')
		    || c == ':'
		    || c == '-'
		    || c == '_') {
			string[x++] = c;
			c = fgetc(fp);
		}
		ungetc(c, fp);
		string[x] = '\0';

		if (type == XML_START) {
			xml_attr_parse(fp, &attr_first, &attr_last);
		} else {
			attr_first = NULL;
			attr_last = NULL;
		}

		c = fgetc(fp);
		if (c == '/') {
			type = XML_START_END;
			c = fgetc(fp);
		}
		assert(c == '>');

		act = malloc(sizeof(*act));
		assert(act);

		act->type = type;
		act->start.string = strdup(string);
		assert(act->start.string);
		act->start.attr_first = attr_first;
		act->start.attr_last = attr_last;
		break;

	default:
		/*
		 * Text
		 */
		x = 0;
		while (c != EOF
		    && c != '<') {
			string[x++] = c;
			c = fgetc(fp);
		}
		ungetc(c, fp);
		string[x] = '\0';

		act = malloc(sizeof(*act));
		assert(act);

		act->type = XML_TEXT;
		act->text.string = strdup(string);
		assert(act->text.string);
		break;
	}

#if 0
	if (act == NULL) {
		fprintf(stderr, "EOF\n");
	} else {
		struct xml_attr *attr;

		switch (act->type) {
		case XML_START:
		case XML_START_END:
			fprintf(stderr, "<%s", act->start.string);
			for (attr = act->start.attr_first;
			    attr;
			    attr = attr->next) {
				fprintf(stderr, " %s=\"%s\"", attr->tag, attr->val);
			}
			if (act->type == XML_START_END) {
				fprintf(stderr, " /");
			}
			fprintf(stderr, ">\n");
			break;
		case XML_END:
			fprintf(stderr, "</%s>\n", act->end.string);
			break;
		case XML_TEXT:
			fprintf(stderr, "%s\n", act->text.string);
			break;
		default:
			assert(0);
		}
	}
#endif

	return act;
}

/*forward*/ static struct xml *
xml_parse_list(FILE *fp, struct xml **firstp, struct xml **lastp);
/*forward*/ static void
xml_free_list(struct xml *first, struct xml *last);

static struct xml *
xml_parse_single(FILE *fp)
{
	struct xml *act;
	struct xml *end;

	act = xml_elem_parse(fp);
	switch (act->type) {
	case XML_START:
		end = xml_parse_list(fp, &act->start.child_first, &act->start.child_last);
		if (strcmp(act->start.string, end->end.string) != 0) {
			fprintf(stderr, "XML tag/end tag mismatch: %s <-> %s.\n",
					act->start.string,
					end->end.string);
			exit(1);
		}
		free(end->end.string);
		free(end);
		break;
	break;
	case XML_START_END:
		act->start.child_first = NULL;
		act->start.child_last = NULL;
		break;
	case XML_END:
		break;
	case XML_TEXT:
		break;
	}
	return act;
}

static void
xml_free_single(struct xml *act)
{
	if (act->type == XML_START) {
		xml_free_list(act->start.child_first, act->start.child_last);
	}

	switch (act->type) {
	case XML_START:
	case XML_START_END:
		free(act->start.string);
		break;
	case XML_END:
		assert(0); /* Mustn't happen. */
	case XML_TEXT:
		free(act->text.string);
		break;
	}
}

static struct xml *
xml_parse_list(FILE *fp, struct xml **firstp, struct xml **lastp)
{
	struct xml *first;
	struct xml *last;
	struct xml *act;

	first = NULL;
	last = NULL;

	for (;;) {
		act = xml_parse_single(fp);
		if (act->type == XML_END) {
			break;
		}
		act->prev = last;
		act->next = NULL;
		if (act->prev) {
			act->prev->next = act;
		} else {
			first = act;
		}
		last = act;
	}

	*firstp = first;
	*lastp = last;

	return act;
}

static void
xml_free_list(struct xml *first, struct xml *last)
{
	while (first) {
		struct xml *act;

		act = first;
		first = act->next;
		if (act->next) {
			act->next->prev = NULL;
		} else {
			last = NULL;
		}

		switch (act->type) {
		case XML_START:
		case XML_START_END:
			free(act->start.string);
			xml_attr_free(act->start.attr_first, act->start.attr_last);
			break;
		case XML_END:
			free(act->end.string);
			break;
		case XML_TEXT:
			free(act->text.string);
			break;
		}
		free(act);
	}

	assert(! first);
	assert(! last);
}

struct xml *
xml_parse(FILE *fp)
{
	return xml_parse_single(fp);
}

void
xml_write(FILE *fp, struct xml *x0)
{
	struct xml *x1;
	struct xml_attr *tv;

	switch (x0->type) {
	case XML_START:
		if (! x0->start.child_first) {
			goto start_end;
		}
		fprintf(fp, "<%s", x0->start.string);
		for (tv = x0->start.attr_first; tv; tv = tv->next) {
			fprintf(fp, " %s=\"%s\"", tv->tag, tv->val);
		}
		fprintf(fp, ">\n");
		for (x1 = x0->start.child_first; x1; x1 = x1->next) {
			xml_write(fp, x1);
		}
		fprintf(fp, "</%s>\n", x0->start.string);
		break;
	case XML_START_END:
	start_end:;
		fprintf(fp, "<%s", x0->start.string);
		for (tv = x0->start.attr_first; tv; tv = tv->next) {
			fprintf(fp, " %s=\"%s\"", tv->tag, tv->val);
		}
		fprintf(fp, "/>\n");
		break;
	case XML_TEXT:
		fprintf(fp, "%s", x0->text.string);
		break;
	default:
		assert(0);
	}
}

void
xml_append(struct xml *parent, struct xml *child)
{
	child->prev = parent->start.child_last;
	child->next = NULL;
	if (child->prev) {
		child->prev->next = child;
	} else {
		parent->start.child_first = child;
	}
	parent->start.child_last = child;
}

struct xml *
xml_alloc(const char *tag)
{
	struct xml *xml;

	xml = malloc(sizeof(*xml));
	assert(xml);
	memset(xml, 0, sizeof(*xml));

	xml->type = XML_START;
	xml->start.string = strdup(tag);
	assert(xml->start.string);
	xml->start.attr_first = NULL;
	xml->start.attr_last = NULL;
	xml->start.child_first = NULL;
	xml->start.child_last = NULL;

	return xml;
}

struct xml *
xml_alloc_text(const char *text)
{
	struct xml *xml;

	xml = malloc(sizeof(*xml));
	assert(xml);
	memset(xml, 0, sizeof(*xml));

	xml->type = XML_TEXT;
	xml->text.string = strdup(text);
	assert(xml->text.string);

	return xml;
}

void
xml_free(struct xml *act)
{
	xml_free_single(act);
}

void
xml_attr_set(struct xml *e, const char *tag, const char *val)
{
	struct xml_attr *tv;

	for (tv = e->start.attr_first; ; tv = tv->next) {
		if (! tv) {
			/* New tag. */
			tv = malloc(sizeof(*tv));
			assert(tv);
			tv->tag = strdup(tag);
			assert(tv->tag);
			tv->val = strdup(val);
			assert(tv->val);

			tv->prev = e->start.attr_last;
			tv->next = NULL;
			if (tv->prev) {
				tv->prev->next = tv;
			} else {
				e->start.attr_first = tv;
			}
			e->start.attr_last = tv;
			break;
		}
		if (strcmp(tv->tag, tag) == 0) {
			/* Old tag. */
			free(tv->val);
			tv->val = strdup(val);
			assert(tv->val);
			break;
		}
	}
}

const char *
xml_attr_lookup(struct xml *e, const char *tag)
{
	struct xml_attr *tv;

	for (tv = e->start.attr_first; ; tv = tv->next) {
		if (! tv) {
			return NULL;
		}
		if (strcmp(tv->tag, tag) == 0) {
			return tv->val;
		}
	}
}
