/* $Id$ 
 *
 * Copyright (C) 2004-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <string.h>
#include <stdio.h>

#include "glue-audio_format.h"

/* Some limits */
#define MIN_S16 -32650
#define MAX_S16  32650
#define MIN_U8   0
#define MAX_U8   255
#define MIN_S8  -128
#define MAX_S8   127

#define PI 3.1415926536
#define MAX(x,y) (((x) > (y)) ? (x) : (y))
#define MIN(x,y) (((x) < (y)) ? (x) : (y))

/* pow(10.0, decibel / 20.0)
 * for decibel = -48 ... 15
 */

const double pow10_dB_table[] = {
	0.003981,
	0.004467,
	0.005012,
	0.005623,
	0.006310,
	0.007079,
	0.007943,
	0.008913,
	0.010000,
	0.011220,
	0.012589,
	0.014125,
	0.015849,
	0.017783,
	0.019953,
	0.022387,
	0.025119,
	0.028184,
	0.031623,
	0.035481,
	0.039811,
	0.044668,
	0.050119,
	0.056234,
	0.063096,
	0.070795,
	0.079433,
	0.089125,
	0.100000,
	0.112202,
	0.125893,
	0.141254,
	0.158489,
	0.177828,
	0.199526,
	0.223872,
	0.251189,
	0.281838,
	0.316228,
	0.354813,
	0.398107,
	0.446684,
	0.501187,
	0.562341,
	0.630957,
	0.707946,
	0.794328,
	0.891251,
	1.000000,
	1.122018,
	1.258925,
	1.412538,
	1.584893,
	1.778279,
	1.995262,
	2.238721,
	2.511886,
	2.818383,
	3.162278,
	3.548134,
	3.981072,
	4.466836,
	5.011872
};


/* FIXME JOSEF .... return number of bytes */
int
ss_change_volume(char *data, int len, int format, int channels, int * decibel, int * mute)
{
	int i;
	int gain[2];
	short *data16; /* 16bit data ptr */


	assert(data != 0);
	assert(0 < len);
	assert(0 <= channels && channels <= 3);
	assert(decibel != 0);

	/* 1 dB = 20 * log_10(Ampl1 / Ampl2) and P1 = Ampl1^2 */
	for (i = 0; i < (sizeof(gain) / sizeof(int)); i++) {
		/* gain[i] = 256.0 * pow(10.0, decibel[i] / 20.0); */
		gain[i] = 256 * pow10_dB_table[decibel[i] + 48];
	}

	switch(format) {
	/* unsigned byte */
	case AFMT_U8:
		for (i = 0; i < len; i += channels) {
			int c;

			for (c = 0; c < channels; c++) {
				int x;
				if (! mute[c]) {
					x = gain[c] * ((unsigned char) data[i + c] - 128);
					x >>= 8;
					if(MAX_S8 < x) {
						x = MAX_S8;
					} else if (x < MIN_S8) {
						x = MIN_S8;
					}
					data[i + c] = x + 128;
				} else {
					data[i + c] = 0;
				}
			}
		}
		break;

	/* signed shorts */
	case AFMT_S16_NE:
		data16 = (short*) data;
		for (i = 0; i < len / sizeof(short); i += channels) {
			int c;

			for (c = 0; c < channels; c++) {
				int x;
				if (! mute[c]) {
					x = gain[c] * data16[i + c];
					x >>= 8;
					if(MAX_S16 < x) {
						x = MAX_S16;
					} else if (x < MIN_S16) {
						x = MIN_S16;
					}
					data16[i + c] = x;
				} else {
					data16[i + c] = 0;
				}
			}
		}
		break;

	default:
		assert(0);
		break;
	};

	return 0;
}


int
ss_gen_rect(void *buffer, int len, int channels, int format, int rate, 
		double *phase, int frequency)
{
	int i;
	int samples_per_cycle;
	int bytes_per_cycle;
	int bytes_per_sample;
	double radians_per_sample;

	assert(buffer != 0);
	assert(len != 0);
	assert(0 < channels && channels < 3);
	assert(0 <= rate);
	assert(phase != 0);
	assert(0 < frequency);

	samples_per_cycle = (double)rate / (double)frequency + 0.5;
	switch (format) {
	case AFMT_U8:
		bytes_per_sample = channels * 1; 
		break;
	case AFMT_S16_NE:
		bytes_per_sample = channels * 2;
		break;
	default:
		assert(0);
	}
	bytes_per_cycle = samples_per_cycle * bytes_per_sample;
	radians_per_sample = (2 * PI) / samples_per_cycle; 

	for (i = 0; (i + bytes_per_sample - 1) < len; i += bytes_per_sample) {
		double sigval;

		/* Write sinus value in buffer */
		if (*phase < PI) {
			sigval = 0;
		} else {
			sigval = 1;
		}
		if (format == AFMT_U8) {
			/* FIXME JOSEF
			 * Use uint_8, something fixed */
			unsigned char val8;
			unsigned char *buf;

			val8 = sigval * MAX_U8; 
			buf = (char *) buffer;
			if (channels == 1) {
				buf[i] = val8;
			} else {
				buf[i] = val8;
				buf[i+1] = val8;
			}
		} else if (format == AFMT_S16_NE) {
			int val16;
			short *buf;

			if (sigval) {
				val16 = MAX_S16;
			} else {
				val16 = MIN_S16;
			}
			buf = (short *) buffer;
			if (channels == 1) {
				buf[i / 2] = val16;
			} else {
				buf[i / 2] = val16;
				buf[i / 2 + 1] = val16;
			}

		} else {
			/* Not supported format */
			assert(0);
		}


		/* Update phase */
		*phase += radians_per_sample;

		/* Check if new cycle begins */
		if ((2 * PI) < *phase) {
			*phase = 0.0;
			/* FIXME JOSEF ... cut this */
			/* If no whole cycle fits in buffer */
#if 0
			if ((len - i) < bytes_per_cycle) {
				return i + bytes_per_sample;
			}
#endif
		}
	}

	return i;
}


int 
ss_change_format(void *in_buf, int in_len, void *out_buf, int out_len, int in_format, int out_format) {

	if (in_format == out_format) {
		int nr_bytes;

		nr_bytes = MIN(in_len, out_len);
		memcpy(out_buf, in_buf, nr_bytes);
		return nr_bytes;

	} else if (in_format == AFMT_U8 && out_format == AFMT_S16_NE) {
		unsigned char *inb8;
		short *outb16;
		int i;

		inb8 = (unsigned char *) in_buf;
		outb16 = (short *) out_buf;

		for (i = 0; i < MIN(in_len, out_len / 2); i++) {
			outb16[i] = (((short) inb8[i]) + MIN_S8) * (MAX_S16 / MAX_S8);
		}

		return i * 2;
	} else {
		/* Not supported yet */
		assert(0);
	}

	return -1;
}

int 
ss_change_channels(void *in_buf, int in_len, void *out_buf, int out_len, int in_channels, int out_channels) {

	if (in_channels == out_channels) {
		int nr_bytes;

		nr_bytes = MIN(in_len, out_len);
		memcpy(out_buf, in_buf, nr_bytes);
		return nr_bytes;

	} else if (in_channels == 1 && out_channels == 2) {
		short *inb16;
		short *outb16;
		int i;
		int new_samples;

		inb16 = (short *) in_buf;
		outb16 = (short *) out_buf;
		new_samples = MIN((in_len / sizeof(short)) * 2, out_len / sizeof(short));

		for (i = 0; i < new_samples; i += 2) {
			outb16[i] = outb16[i + 1] = inb16[i / 2];
		}
		return new_samples * sizeof(short);
	} else {
		/* Not supported yet */
		assert(0);
	}

	return -1;
}


int 
ss_change_rate(void *in_buf, int in_len, void *out_buf, int out_len, int in_rate, int out_rate, int channels) {

	assert(in_buf != 0);
	assert(in_len != 0);
	assert(out_buf != 0);
	assert(out_len != 0);
	assert(in_rate > 0);
	assert(out_rate > 0);
	assert(channels > 0);


	if (in_rate == out_rate) {
		int nr_bytes;

		nr_bytes = MIN(in_len, out_len);
		memcpy(out_buf, in_buf, nr_bytes);
		return nr_bytes;

	} else if (in_rate < out_rate) {
		short *inb16;
		short *outb16;
		double sample_ratio;
		int nr_new_samples; /* Number of new short-stereo-samples */
		int nr_old_samples; /* Number of input short-stereo-samples */
		int i;
		int c;

		nr_old_samples = in_len / (sizeof(short) * channels);
		sample_ratio = (double) out_rate / (double) in_rate; 
		nr_new_samples = MIN((in_len / sizeof(short)) * sample_ratio,
				out_len / sizeof(short));
		nr_new_samples /= channels;

		for (c = 0; c < channels; c++) {
			inb16 = ((short *) in_buf); 
			outb16 = ((short *) out_buf);

			/* Along one sample track in the output buffer*/
			for (i = 0; i < nr_new_samples; i++) {
				double oldsample_index; 
				short value;
				double m;
				short y0;
				short y1;
				int x0;
				int x1;

				/* Which is the corresponding index in
				 * the old sample data. */
				oldsample_index = i / sample_ratio;
				x0 = (int) oldsample_index;

#if 0
				fprintf(stderr, "oldsampleindex: %f\n", oldsample_index);
#endif
				/* Reaching the end, where no two values
				 * for interpolation are available ? */
				if (nr_old_samples - 1 <= x0) {
					/* Assign the last sample value */
					value = inb16[c + channels * (int) oldsample_index];
#if 0
					fprintf(stderr, "end reached\n");
#endif
				} else {
					/* Interpolate linearly two neighbor
					 * sample values from input buffer.
					 * L=y0+(y1-y0)/(x1-x0) * (x-x0) */

					x1 = x0 + 1;
					y0 = inb16[c + channels * x0];
					y1 = inb16[c + channels * x1];
					m = (y1 - y0) / ((x1 - x0));
					value = (y0 + m * (oldsample_index - x0)) + 0.5;

					/* Without interpolation: */
					/* value = inb16[c + channels * (int) oldsample_index]; */
				}
				outb16[c + i * channels] = value;
			}
		}

		return nr_new_samples * channels * sizeof(short);

	} else {
		/* Not supported yet */
		assert(0);
	}

	return -1;
}

int
ss_gen_silence(void * buffer, int len, int channels, int format)
{
	int i;
	int c;
	short * sbuf;
	int samples;
	int sample_size;
	assert(buffer != 0);
	assert(0 < len);
	assert(0 < channels);
	assert(format == AFMT_S16_NE);
	sample_size = channels * sizeof_format(format);
	samples = len / sample_size;
	switch(format) {
	case AFMT_S16_NE:
		sbuf = (short *) buffer;
		for (c = 0; c < channels; c++) {
			for (i = 0; i < samples * 2; i++) {
				sbuf[i * channels + c] = 0;
			}
		}
		break;
	default:
		assert (0);
		break;
	}
	return samples * sample_size;
}



