/* $Id: simsetup_expect.l,v 1.16 2013-02-21 11:37:55 vrsieh Exp $
 *
 * Scanner for simulation.setup files.
 *
 * Copyright (C) 2005-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

%option nounput

%{
#include "config.h"

#include <assert.h>
#include <ctype.h>
#include <stdlib.h>
#include <search.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include "env.h"
#include "list.h"
#include "glue-shm.h"
#include "simsetup.h"
#include "simsetup_expect.h"
#include "structure.h"

static int line = 1;
static const char *file_name;

static struct structure_component *current = NULL;

#define YY_SKIP_YYWRAP
#define	YY_DECL	int yylex(const char *current_config_dir)

static int
simsetup_expect_wrap(void)
{
	return 1;
}

%}

%x	MAPPING
%x	OPTIONS

ws			[ \t]
letter			[a-zA-Z_]
alpha			({letter}|[0-9])
path_key		(\/|{alpha}|,|\.|\-|\_|#|$|\*|:)
ip_address		[0-9]+\.[0-9]+\.[0-9]+\.[0-9]+

%%

<*>#.*$ {}
<*>{ws}+ {}

<*>^\[options\] {
	BEGIN(OPTIONS);
}

<OPTIONS>{letter}+((({ws}+)|({ws}*=)){ws}*{path_key}+)? {
	char	*option, *value;

	option=yytext;
	for (value=yytext; value<&yytext[yyleng]; value++) {
		if (*value==' ' || *value == '\t' || *value == '=') {
			*value++ = '\0';
			break;
		}
		if (*value == 0) {
			break;
		}
	}
	if (*value) {
		for (; value<&yytext[yyleng]; value++) {
			if (*value!=' ' && *value != '\t' && *value != '=')
				break;
		}
	}
	value = env_replace(value);
	if (! strcmp(option, "index")) {
		simsetup.index = atoi(value);
	} else if (! strcmp(option, "vhdl_model")) {
		simsetup.vhdl_model = shm_strdup(value);
	} else if (! strcmp(option, "base_entity")) {
		simsetup.base_entity = shm_strdup(value);
	} else if (! strcmp(option, "deterministic")) {
		simsetup.deterministic = atoi(value);
	} else if (! strcmp(option, "multithreaded")) {
		simsetup.multithreaded = atoi(value);
	} else if (! strcmp(option, "interactive")) {
		simsetup.interactive = atoi(value);
	} else if (! strcmp(option, "simplesetup")) {
		simsetup.simplesetup = atoi(value);
	} else if (! strcmp(option, "cow")) {
		simsetup.cow = atoi(value);
	} else if (! strcmp(option, "sparse")) {
		simsetup.sparse = atoi(value);
	} else if (! strcmp(option, "create")) {
		simsetup.create = atoi(value);
	} else if (! strcmp(option, "gui")) {
		simsetup.gui = shm_strdup(value);
	} else {
		fprintf(stderr, "illegal option %s in config file", option);
		exit(1);
	}
}

<*>^\[\/{path_key}+\] {
	BEGIN(MAPPING);
	yytext[yyleng - 1] = 0;	/* delete ] from string */
	list_Foreach(&simsetup.comps, current) {
		if (strcmp(current->name, yytext + 1) == 0) {
			break;
		}
	}
	if (! current) {
		char *name;

		name = shm_strdup(yytext + 1);
		current = MEM_NEW(struct structure_component);
		memset(current, 0, sizeof(*current));
		current->name = name;
		list_push(&simsetup.comps, current);
	}
}

<MAPPING>{path_key}+(({ws}+)|({ws}*=)){ws}*(({path_key}+)|(\"[^\"]*\")) {
	char		*key, *value;
	char		*replaced;
	  
	key=yytext;
	for (value=yytext; value<&yytext[yyleng]; value++) {
		if (*value==' ' || *value == '\t' || *value == '=') {
			*value = 0;
			break;
		}
	}
	value++;
	for (; value<&yytext[yyleng]; value++) {
		if (*value!=' ' && *value != '\t' && *value != '=')
			break;
	}

	if (*value == '"') {
		replaced = value + 1;
		assert(replaced[strlen(replaced) - 1] == '"');
		replaced[strlen(replaced) - 1] = 0;
	} else {
		replaced = env_replace(value);
	}

	if (0 < strlen(replaced)) {
		char *nkey, *nval;
		struct structure_keyval	*m;

		nkey = shm_strdup(key);
		nval = shm_strdup(replaced);
		m = MEM_NEW(struct structure_keyval);
		m->name = nkey;
		m->value = nval;
		list_push(&current->simsetups, m);
	}
}

<*>\n	{ line++; }

<*>. {
	fprintf(stderr, "illegal character (%s) in config-file for directory\n\n\t%s, line %d\n",
		yytext, current_config_dir, line);
	exit(1);
}

<*><<EOF>> {
	return 0;
}

%%

void
simsetup_init(const char *filename)
{
	FILE *input;

	/*
	 * get simulation setup: vhdl-file name, base entity, index,
	 * host mappings (setup node list before).
	 */
	input = fopen(filename, "r");
	if (! input) {
		fprintf(stderr, "can't open config file '%s'\n", filename);
		exit(1);
	}

	simsetup.index = 0x4000;

	list_init(&simsetup.comps);

	simsetup.vhdl_model = NULL;
	simsetup.base_entity = NULL;
	simsetup.deterministic = 0;
	simsetup.multithreaded = 1;
	simsetup.interactive = 0;
	simsetup.cow = 0;
	simsetup.sparse = 0;
	simsetup.create = 0;
	simsetup.gui = NULL;

	file_name = filename;

	yyrestart(input);
	if (yylex(filename)) {
		fprintf(stderr, "parsing config file %s failed", filename);
		exit(1);
	}

#ifndef HAVE_PTHREAD
    if (simsetup.multithreaded) {
        simsetup.multithreaded = 0;
    }
#endif

	fclose(input);
}
