/* $Id: glue-ppm.c,v 1.2 2009-01-27 17:06:40 potyra Exp $ 
 *
 * Copyright (C) 2008-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <inttypes.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "glue.h"
#include "glue-ppm.h"

static bool
read_next_word(FILE *fp, char *dest, size_t len, const char *delims)
{
	int key;
	int i;
	
	i = 0;
	do {
		key = fgetc(fp);
		if (key == EOF) {
			return false;
		}
	} while (strchr(delims, key));

	do {
		if (key == EOF) {
			dest[i] = '\0';
			return true;
		}
		dest[i] = key;
		i++;
		key = fgetc(fp);
	} while (!strchr(delims, key) && i < len - 1);

	dest[i] = '\0';
	return true;
}

/*
 * Lines with trailing spaces and a "#" are comments, except the line
 * "# ignore <n>" which needs to be interpreted.
 */
static int
is_comment(const char *buf)
{
	return ((strspn(buf, " #") > 0) &&
		(strchr(buf, '#') != NULL) &&
		strncmp(buf, "# ignore ", strlen("# ignore")));
}

static bool
read_next_line(FILE *fp, char *dst, size_t len)
{
	bool b;
	do {
		b = read_next_word(fp, dst, len, "\n");
		if (! b) {
			return false;
		}
	} while (is_comment(dst));

	return true;
}

bool
ppm_read(
	uint32_t **dest,
	int *w,
	int *h,
	const char *filename
)
{
	FILE *fp = NULL;
	char id[20] = {'\0','\0','\0','\0','\0','\0','\0','\0','\0','\0',
		'\0','\0','\0','\0','\0','\0','\0','\0','\0','\0'};
	char buf[1024];
	int _w;
	int _h;
	int x, y;
	int colors;
	int ignore_color;
	int mem;
	uint32_t *saved_dest;

	assert(dest);
	assert(w);
	assert(h);

	fp = fopen(filename, "r");
	if (fp == NULL) {
		fprintf(stderr, "\npatternm_read_pnm_file: can't open file %s!\n", 
			filename );
		return false;
	}

	/* magic "P3" */
	if (read_next_line(fp, buf, sizeof(buf)) == -1) {
		fclose(fp);
		return false;
	}
	sscanf(buf, "%s\n", id);

	/* colors to ignore "# ignore <n>" */
	if (read_next_line(fp, buf, sizeof(buf)) == -1) {
		fclose(fp);
		return false;
	}
	if (strncmp(buf, "# ignore", strlen("# ignore")) == 0) {
		sscanf(buf, "# ignore %d\n", &ignore_color);

		/* width and height */
		if ((read_next_line(fp, buf, sizeof(buf)) == -1 )){
			fclose(fp);
			return false;
		}
	} else {
		ignore_color = -1;
	}
	sscanf(buf, "%d %d\n", &_w, &_h);

	/* maximum color */
	if (read_next_line(fp, buf, sizeof(buf)) == -1) {
		fclose(fp);
		return false;
	}
	sscanf(buf, "%d\n", &colors);
	
	if (strcmp(id, "P3")) {
		fprintf(stderr, "\npatternm_read_pnm_file: Wrong file format!");
		fclose(fp);
		return false;
	}

	assert(_w != 0 && _h != 0);

	mem = sizeof(uint32_t) * _w * _h;
	*dest = (uint32_t *) shm_alloc(mem);
	assert(*dest != NULL);

	saved_dest = *dest;
	
	for (y = 0; y < _h; y++) {
		for (x = 0; x < _w; x++) {
			uint8_t r, g, b;
			uint32_t color;

			if (read_next_word(fp, id, sizeof(id), " \n") == false) {
				fclose(fp);
				return false;
			}
			r = atoi(id) & 0xFF;

			if (read_next_word(fp, id, sizeof(id), " \n") == false) {
				fclose(fp);
				return false;
			}
			g = atoi(id) & 0xFF;

			if (read_next_word(fp, id, sizeof(id), " \n") == false) {
				fclose(fp);
				return false;
			}
			b = atoi(id) & 0xFF;

			color = (r << 16) + (g << 8) + b;

			if (color == ignore_color) {
				**dest = (ALPHA_TRANSPARENT << 24) + color;
			} else {
				**dest = (ALPHA_OPAQUE << 24) + color;
			}
			
			(*dest)++;
		}
	}

	*w = _w;
	*h = _h;
	*dest = saved_dest;

	fclose(fp);

	return true;
}
