/*
 * $Id: glue-audio_macosx.c,v 1.14 2009-01-28 11:00:33 potyra Exp $
 *
 * Mac OS X audio output driver for FAUmachine
 *
 * Derived from MPlayer (libao2 - ao_macosx).
 *
 * Copyright (c) 2007-2009 FAUmachine Team.
 * Copyright (c) MPlayer Team.
 * Original Copyright (c) 2000 Timothy J. Wood
 *
 *  libao is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or 
 *  (at your option) any later version.
 *
 *  libao is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with libao; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

/*
 * The MacOS X CoreAudio framework doesn't mesh as simply as some
 * simpler frameworks do. This is due to the fact that CoreAudio pulls
 * audio samples rather than having them pushed at it (which is nice
 * when you are wanting to do good buffering of audio). 
 */

#include <CoreServices/CoreServices.h>
#include <AudioUnit/AudioUnit.h>
#include <AudioToolbox/AudioToolbox.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <math.h>
#include <sys/time.h>
#include <mach/mach_time.h>
#include <inttypes.h>
#include <pthread.h>

#include "config.h"

#include "glue-audio_internal.h"

static ao_info_t info =
{
	"Darwin/Mac OS X native audio output",
	"macosx",
	"Timothy J. Wood & Dan Christiansen & Chris Roccati",
	""
};

LIBAO_EXTERN(macosx)

static ao_data_t ao_data={0,0,0,0,OUTBURST,-1,0};

static double timebase_ratio;

typedef struct ao_macosx_s
{
	/* AudioUnit */
	AudioUnit theOutputUnit;
	int packetSize;
	int paused;

	/* Ring-buffer */
	/* does not need explicit synchronization */
	unsigned char *buffer;
	unsigned int buffer_len;
	unsigned int buf_read_pos;
	unsigned int buf_write_pos;
} ao_macosx_t;

static ao_macosx_t *ao = NULL;

/**
 * \brief return number of buffered bytes
 *    may only be called by playback thread
 * \return minimum number of buffered bytes, value may change between
 *    two immediately following calls, and the real number of buffered bytes
 *    might actually be larger!
 */
static int
buf_used(void)
{
	int used = ao->buf_write_pos - ao->buf_read_pos;
	if(used < 0) used += ao->buffer_len;
	return used;
}

/**
 * \brief add data to ringbuffer
 */
static int
write_buffer(unsigned char * data, int len)
{
	int first_len, space;

	first_len = ao->buffer_len - ao->buf_write_pos;
	space = get_space();
	if (len > space) len = space;
	if (first_len > len) first_len = len;

	// till end of buffer
	memcpy (&ao->buffer[ao->buf_write_pos], data, first_len);
	if (len > first_len) { // we have to wrap around
		// remaining part from beginning of buffer
		memcpy (ao->buffer, &data[first_len], len - first_len);
	}
	ao->buf_write_pos = (ao->buf_write_pos + len) % ao->buffer_len;
	return len;
}

/**
 * \brief remove data from ringbuffer
 */
static int
read_buffer(unsigned char * data, int len)
{
	int first_len, buffered;

	first_len = ao->buffer_len - ao->buf_read_pos;
	buffered = buf_used();
	if (len > buffered) len = buffered;
	if (first_len > len) first_len = len;

	// till end of buffer
	memcpy (data, &ao->buffer[ao->buf_read_pos], first_len);
	if (len > first_len) { // we have to wrap around
		// remaining part from beginning of buffer
		memcpy (&data[first_len], ao->buffer, len - first_len);
	}
	ao->buf_read_pos = (ao->buf_read_pos + len) % ao->buffer_len;
	return len;
}

OSStatus
theRenderProc(void *inRefCon, AudioUnitRenderActionFlags *inActionFlags,
		const AudioTimeStamp *inTimeStamp, UInt32 inBusNumber,
		UInt32 inNumFrames, AudioBufferList *ioData)
{
	int amt;

	amt = read_buffer((unsigned char *)ioData->mBuffers[0].mData, inNumFrames * ao->packetSize);
	if(amt == 0) {
		/* stop callback */
		if(AudioOutputUnitStop(ao->theOutputUnit) == noErr) {
			ao->paused=1;
		}
	}
	ioData->mBuffers[0].mNumberChannels = GLUE_AUDIO_CHANNELS;
	ioData->mBuffers[0].mDataByteSize = amt;

	return noErr;
}

static int
init(void)
{
	AudioStreamBasicDescription inDesc;
	ComponentDescription desc; 
	Component comp; 
	AURenderCallbackStruct renderCallback;
	OSErr err;
	UInt32 size, maxBufferBytes;
	struct mach_timebase_info timebase;
	int aoIsCreated = ao != NULL;

	if (!aoIsCreated)
		ao = malloc(sizeof(ao_macosx_t));

	// Build Description for the input format
	inDesc.mSampleRate=GLUE_AUDIO_RATE;
	inDesc.mFormatID=kAudioFormatLinearPCM;
	inDesc.mChannelsPerFrame=GLUE_AUDIO_CHANNELS;
	switch(GLUE_AUDIO_FORMAT){
		case AFMT_U8:
			inDesc.mBitsPerChannel=8;
			inDesc.mFormatFlags = kAudioFormatFlagIsPacked;
			break;
		case AFMT_S8:
			inDesc.mBitsPerChannel=8;
			inDesc.mFormatFlags = kAudioFormatFlagIsSignedInteger|kAudioFormatFlagIsPacked;
			break;
		case AFMT_S16_LE:
			inDesc.mBitsPerChannel=16;
			inDesc.mFormatFlags = kAudioFormatFlagIsSignedInteger|kAudioFormatFlagIsPacked;
			break;
		case AFMT_S16_BE:
			inDesc.mBitsPerChannel=16;
			inDesc.mFormatFlags |= kAudioFormatFlagIsBigEndian;
			break;
		case AFMT_U16_LE:
			inDesc.mBitsPerChannel=16;
			inDesc.mFormatFlags = kAudioFormatFlagIsPacked;
			break;
		case AFMT_U16_BE:
			inDesc.mBitsPerChannel=16;
			inDesc.mFormatFlags = kAudioFormatFlagIsPacked;
			inDesc.mFormatFlags |= kAudioFormatFlagIsBigEndian;
			break;
		case AFMT_S32_LE:
			inDesc.mBitsPerChannel=32;
			inDesc.mFormatFlags = kAudioFormatFlagIsSignedInteger|kAudioFormatFlagIsPacked;
			break;
		case AFMT_S32_BE:
			inDesc.mBitsPerChannel=32;
			inDesc.mFormatFlags = kAudioFormatFlagIsSignedInteger|kAudioFormatFlagIsPacked;
			inDesc.mFormatFlags |= kAudioFormatFlagIsBigEndian;
			break;
		default:
			return 0;
			break;
	}

	if(GLUE_AUDIO_FORMAT==AFMT_FLOAT) {
		/* float */
		inDesc.mFormatFlags = kAudioFormatFlagIsFloat|kAudioFormatFlagIsPacked;
	}

	inDesc.mFramesPerPacket = 1;
	ao->packetSize = inDesc.mBytesPerPacket = inDesc.mBytesPerFrame = inDesc.mFramesPerPacket*GLUE_AUDIO_CHANNELS*(inDesc.mBitsPerChannel/8);

	if (!aoIsCreated) {
		desc.componentType = kAudioUnitType_Output;
		desc.componentSubType = kAudioUnitSubType_DefaultOutput;
		desc.componentManufacturer = kAudioUnitManufacturer_Apple;
		desc.componentFlags = 0;
		desc.componentFlagsMask = 0;

		/* finds an component that meets the desc spec's */
		comp = FindNextComponent(NULL, &desc);
		if(comp == NULL) {
			return 0;
		}

		/* gains access to the services provided by the component */
		err = OpenAComponent(comp, &(ao->theOutputUnit));
		if(err != noErr) {
			return 0;
		}

		/* initialize AudioUnit */
		err = AudioUnitInitialize(ao->theOutputUnit);
		if(err != noErr) {
			return 0;
		}
	}

	size =  sizeof(AudioStreamBasicDescription);
	err = AudioUnitSetProperty(ao->theOutputUnit, kAudioUnitProperty_StreamFormat, kAudioUnitScope_Input, 0, &inDesc, size);
	if(err != noErr) {
		return 0;
	}

	size = sizeof(UInt32);
	err = AudioUnitGetProperty(ao->theOutputUnit, kAudioDevicePropertyBufferSize, kAudioUnitScope_Input, 0, &maxBufferBytes, &size);
	if(err != noErr) {
		return 0;
	}
	maxBufferBytes *= 4;
	size = sizeof(UInt32);
	err = AudioUnitSetProperty(ao->theOutputUnit, kAudioDevicePropertyBufferSize, kAudioUnitScope_Input, 0, &maxBufferBytes, size);
	if(err != noErr) {
		return 0;
	}

	ao_data.samplerate = GLUE_AUDIO_RATE;
	ao_data.channels = inDesc.mChannelsPerFrame;
	ao_data.bps = ao_data.samplerate * inDesc.mBytesPerFrame;
	ao_data.outburst = maxBufferBytes;
	ao_data.buffersize = ao_data.bps;

	ao->buffer_len = ao_data.bps;
	ao->buffer = aoIsCreated ?
		realloc(ao->buffer,ao->buffer_len)
		: malloc(ao->buffer_len);
	if(ao->buffer == NULL) {
		return 0;
	}

	renderCallback.inputProc = theRenderProc;
	renderCallback.inputProcRefCon = 0;
	err = AudioUnitSetProperty(ao->theOutputUnit,
			kAudioUnitProperty_SetRenderCallback,
			kAudioUnitScope_Input, 0, &renderCallback,
			sizeof(AURenderCallbackStruct));
	if (err != noErr) {
		return 0;
	}

	/* initialize timer */
	mach_timebase_info(&timebase);
	timebase_ratio = (double)timebase.numer
		/ (double)timebase.denom 
		* (double)1e-9;

	/* stop callback */
	if(AudioOutputUnitStop(ao->theOutputUnit) == noErr) {
		ao->paused=1;
	}

	/* reset ring-buffer state */
	ao->buf_read_pos=0;
	ao->buf_write_pos=0;

	return 1;
}

static int
play(void * output_samples,int num_bytes)
{
	int wrote=write_buffer(output_samples, num_bytes);

	if(ao->paused && wrote > 0) {
		/* start callback */
		if(AudioOutputUnitStart(ao->theOutputUnit) == noErr) {
			ao->paused=0;
		}
	}

	return wrote;
}

/* return available space */
static int
get_space(void)
{
	return ao->buffer_len - buf_used();
}

/* return delay until audio is played */
static float
get_delay(void)
{
	// inaccurate, should also contain the data buffered e.g. by the OS
	return ((float)buf_used())/((float)ao_data.bps);
}

/* unload plugin and deregister from coreaudio */
static void
uninit(int immed)
{
	if (!immed) {
		uint64_t deadline = ((buf_used()/ao_data.bps) / timebase_ratio) + mach_absolute_time();
		mach_wait_until(deadline);
	}

	AudioOutputUnitStop(ao->theOutputUnit);
	AudioUnitUninitialize(ao->theOutputUnit);
	CloseComponent(ao->theOutputUnit);

	free(ao->buffer);
	free(ao);
	ao = NULL;
}
