/*
 * $Id: usb2serial.c,v 1.31 2013-05-09 21:51:49 vrsieh Exp $ 
 *
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "glue.h"

#include "lib_usb.h"
#include "usb_device_descriptor.h"

#include "usb2serial.h"

/* ********************* DEFINITIONS ************************** */

#define SENDBUFFER_SIZE	1024
#define RINGBUFFER_SIZE 1024

#define DEBUG        0

/*----------------------------DEBUG-------------------------------------------*/
/* binary OR these in U2S_DEBUGMASK */
#define U2S_DEBUG_WARNINGS              (1 << 0)
#define U2S_DEBUG_CIM                   (1 << 1)
#define U2S_DEBUG_DUMP_PACKETS          (1 << 2)
#define U2S_DEBUG_STATE_TRANSITIONS     (1 << 3)
#define U2S_DEBUG_WELCOME_MESSAGES      (1 << 4)
#define U2S_DEBUG_DUMP_SERIAL_DATA      (1 << 5)
#define U2S_DEBUG_STATISTICS            (1 << 6)

#if DEBUG
#define U2S_DEBUGMASK (U2S_DEBUG_WARNINGS | U2S_DEBUG_STATE_TRANSITIONS | U2S_DEBUG_DUMP_SERIAL_DATA)
#else /* DEBUG */
#define U2S_DEBUGMASK 0x0000
#endif /* DEBUG */

#define U2S_LOG_TYPE                    "usb_to_serial"
#define U2S_LOG_NAME                    ""

/* endpoint numbers for PL2303 */
/* encoding like used in device requests (with bit 7 for in/out) */
#define USB_ENDPT_INTERRUPT_IN	0x81
#define USB_ENDPT_BULK_OUT	0x02
#define USB_ENDPT_BULK_IN	0x83

/* encoding like used in token packets */
#define USB_ENDPT_NR_INTERRUPT_IN	(USB_ENDPT_INTERRUPT_IN & 0x0F)
#define USB_ENDPT_NR_BULK_OUT		(USB_ENDPT_BULK_OUT & 0x0F)
#define USB_ENDPT_NR_BULK_IN		(USB_ENDPT_BULK_IN & 0x0F)

/* endpoint FIFO sizes */
#define USB_PACKETSIZE_CONTROL0		8
#define USB_PACKETSIZE_INTERRUPT_IN	10
#define USB_PACKETSIZE_BULK_OUT		64
#define USB_PACKETSIZE_BULK_IN		64

#if U2S_DEBUGMASK & U2S_DEBUG_STATE_TRANSITIONS
static unsigned char state_names[][20] = {
	"invalid",
	"OFF",
	"ATTACHED",
	"POWERED",
	"DEFAULT",
	"ADDRESS",
	"CONFIGURED" };
#endif

/* *********************** MACROS ***************************** */
#if U2S_DEBUGMASK & U2S_DEBUG_STATE_TRANSITIONS
#define SET_USB_DEVSTATE(cpssp,newstate) \
	if (cpssp->state != (newstate)) { \
		int oldstate = cpssp->state; \
		cpssp->state = (newstate); \
	faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME, \
		"state transition: %s => %s\n", \
		state_names[oldstate], \
		state_names[(newstate)]); \
	}
#else
#define SET_USB_DEVSTATE(cpssp,newstate) (cpssp->state = (newstate))
#endif

#define MIN(a,b) ((a) < (b) ? (a) : (b))

/* ****************** GLOBAL VARIABLES ************************ */

struct ringbuffer {
	unsigned long buffer_size;
	unsigned char *buffer;

	unsigned long head, tail;

	unsigned char full;
	int overwrite;
};

static const struct usb_device_descriptor device_descriptor_pl2303 = {
	.bLength = 18,
	.bDescriptorType = USB_DESCRIPTOR_TYPE_DEVICE,
	.bcdUSB = 0x0110,
	.bDeviceClass = 0,
	.bDeviceSubClass = 0,
	.bDeviceProtocol = 0,
	.bMaxPacketSize0 = 8,
	.idVendor = 0x067b,
	.idProduct = 0x2303,
	.bcdDevice = 0x0202,
	.iManufacturer = 0,
	.iProduct = 0,
	.iSerialNumber = 0,
	.bNumConfigurations = 1
};

static const struct __attribute__ ((__packed__)) usb_combined_descriptor {
	struct usb_configuration_descriptor conf0;
	struct usb_interface_descriptor if0;
	struct usb_endpoint_descriptor ep1;
	struct usb_endpoint_descriptor ep2;
	struct usb_endpoint_descriptor ep3;
} combined_descriptor_pl2303 = {
	.conf0 = {
		.bLength = 9,
		.bDescriptorType = USB_DESCRIPTOR_TYPE_CONFIGURATION,
		.wTotalLength = 39,
		.bNumInterfaces = 1,
		.bConfigurationValue = 1,
		.iConfiguration = 0,
		.bmAttributes = 0xa0,	/* remote wakeup */
		.MaxPower = 50		/* 100mA */
	},
	.if0 = {
		.bLength = 9,
		.bDescriptorType = USB_DESCRIPTOR_TYPE_INTERFACE,
		.bInterfaceNumber = 0,
		.bAlternateSetting = 0,
		.bNumEndpoints = 3,
		.bInterfaceClass = 255,
		.bInterfaceSubClass = 0,
		.bInterfaceProtocol = 0,
		.iInterface = 0
	},
	.ep1 = {
		.bLength = 7,
		.bDescriptorType = USB_DESCRIPTOR_TYPE_ENDPOINT,
		.bEndpointAddress = 0x81,
		.bmAttributes = 3,
		.wMaxPacketSize = 10,
		.bInterval = 1,
	},
	.ep2 = {
		.bLength = 7,
		.bDescriptorType = USB_DESCRIPTOR_TYPE_ENDPOINT,
		.bEndpointAddress = 0x02,
		.bmAttributes = 2,
		.wMaxPacketSize = 64,
		.bInterval = 0,
	},
	.ep3 = {
		.bLength = 7,
		.bDescriptorType = USB_DESCRIPTOR_TYPE_ENDPOINT,
		.bEndpointAddress = 0x83,
		.bmAttributes = 2,
		.wMaxPacketSize = 64,
		.bInterval = 0,
	}
};

struct cpssp {
	struct sig_serial *port_serial;
	struct sig_std_logic *port_usb_power;
	struct sig_usb_bus *port_usb_main;

	/* cf. USB Spec. 1.1, chapter 9.1 */
	enum {
		STATE_OFF = 1,
		STATE_ATTACHED,
		STATE_POWERED,
		STATE_DEFAULT,
		STATE_ADDRESS,
		STATE_CONFIGURED
	} state;

	/* device address */
	unsigned char addr;

	/* stage of current control transfer */
	struct {
		enum {
			STAGE_NONE,
			STAGE_SETUP,
			STAGE_DATA,
			STAGE_STATUS
		} stage;

		struct {
			unsigned char bmRequestType;
			unsigned char bRequest;
			unsigned short wValue;
			unsigned short wIndex;
			unsigned short wLength;
		} request;

		unsigned char data[SENDBUFFER_SIZE];
		unsigned long data_length;
		unsigned long data_sent;
	} control_information;

	/* pipe state */
	struct {
		/* DATA0/DATA1 protocol toggle */
		unsigned char data_toggle;
		unsigned char stalled;
		unsigned char last_data_acked;
		unsigned char fifo[USB_PACKETSIZE_BULK_IN];
		unsigned int fifo_datalen;
	} control, interrupt_in, bulk_in, bulk_out;

	/* information about last received token */
	struct last_token {
		unsigned char pid;
		unsigned char addr, endp;
	} last_token;

	struct ringbuffer serial_buffer;

#if U2S_DEBUGMASK & U2S_DEBUG_STATISTICS
	struct statistics {
		unsigned long token_count_total,
			      token_count,
			      token_count_ctrl,
			      token_count_int_in,
			      token_count_bulk_in,
			      token_count_bulk_out;
	} statistics;
#endif
};

/* ******************** IMPLEMENTATION ************************* */

static void
ringbuffer_create(struct ringbuffer *rb, unsigned long size, int overwrite)
{
	assert(rb);

	rb->buffer = shm_alloc(size);
	assert(rb->buffer);

	rb->buffer_size = size;
	rb->head = rb->tail = 0;

	rb->full = 0;
	rb->overwrite = overwrite;
}

static void
ringbuffer_destroy(struct ringbuffer *rb)
{
	assert(rb);

	/* also works if buffer is NULL */
	shm_free(rb->buffer);
}

static int
ringbuffer_put_byte(struct ringbuffer *rb, unsigned char c)
{
	if (rb->full) {
		if (!rb->overwrite) {
			return -1;
		} else {
			rb->head++;
		}
	}

	rb->buffer[rb->tail++] = c;
	rb->tail %= rb->buffer_size;

	if (rb->tail == rb->head) {
		rb->full = 1;
	}

	return 0;
}

/*
 * returns the number of bytes stored, which is between 0 and size
 */
static unsigned long
ringbuffer_put(
	struct ringbuffer *rb,
	const unsigned char *data,
	unsigned long size
)
{
	unsigned long i;

	assert(rb);
	assert(rb->buffer);
	assert(data);

	for (i = 0; i < size; i++) {
		if (ringbuffer_put_byte(rb, data[i])) {
			break;
		}
	}

	return i;
}

static int
ringbuffer_get_byte(struct ringbuffer *rb, unsigned char *c)
{
	if (!rb->full && rb->head == rb->tail) {
		return -1;
	}

	*c = rb->buffer[rb->head++];
	rb->head %= rb->buffer_size;

	rb->full = 0;

	return 0;
}

/*
 * returns the number of bytes retrieved, which is between 0 and size
 */
static unsigned long
ringbuffer_get(
	struct ringbuffer *rb,
	unsigned char *data,
	unsigned long size
)
{
	unsigned long i;

	assert(rb);
	assert(rb->buffer);
	assert(data);

	for (i = 0; i < size; i++) {
		if (ringbuffer_get_byte(rb, &data[i])) {
			break;
		}
	}

	return i;
}

#if U2S_DEBUGMASK & U2S_DEBUG_DUMP_PACKETS
static void
usb_debug_data(unsigned char *data, unsigned int length)
{
	int i;
	if (0 < length) {
		faum_cont(FAUM_LOG_DEBUG, " ( ");

		for (i = 0; i < length && i < 8; i++) {
			faum_cont(FAUM_LOG_DEBUG, "%02X ", data[i]);
		}

		if (i != length) {
			faum_cont(FAUM_LOG_DEBUG, "... ");
		}

		faum_cont(FAUM_LOG_DEBUG, ")");
	}

	faum_cont(FAUM_LOG_DEBUG, "\n");
}
#endif

/* USB signaling wrappers for debugging purposes */
static void
usb_send_handshake(struct cpssp * cpssp, unsigned char pid)
{
#if U2S_DEBUGMASK & U2S_DEBUG_DUMP_PACKETS
	faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME,
			"sending %s handshake\n",
			cim_usb_debug_pid2str(pid));
#endif
	sig_usb_bus_send_handshake(cpssp->port_usb_main, cpssp, pid);
}

static void
usb_send_data(
	struct cpssp * cpssp,
	unsigned char pid,
	unsigned char *data,
	unsigned int length
)
{
#if U2S_DEBUGMASK & U2S_DEBUG_DUMP_PACKETS
	faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME,
			"sending %s with %d bytes",
			cim_usb_debug_pid2str(pid), length);
	usb_debug_data(data, length);
#endif
	sig_usb_bus_send_data(cpssp->port_usb_main, cpssp,
			pid, length, data, 0);
}

static void
usb_reset_set(void *_cpssp, int val)
{
	struct cpssp * cpssp = (struct cpssp *) _cpssp;

	/* RESET */
	cpssp->addr = 0;
	SET_USB_DEVSTATE(cpssp, STATE_DEFAULT);
}

/* handle IN token on control endpoint (0) */
static void
usb_handle_packet_usb_token_in_control(struct cpssp * cpssp)
{
	if (cpssp->control_information.stage == STAGE_DATA
	    && (cpssp->control_information.request.bmRequestType
		& USB_DEV_REQ_DEVICE_TO_HOST)) {
		/* data stage of a device-to-host transfer */
		usb_send_data(cpssp,
				cpssp->control.data_toggle,
				cpssp->control_information.data
				+ cpssp->control_information.data_sent,
				MIN(USB_PACKETSIZE_CONTROL0,
				    cpssp->control_information.data_length
				    - cpssp->control_information.data_sent));
	} else if (cpssp->control_information.stage == STAGE_STATUS
		   || (cpssp->control_information.stage == STAGE_DATA
		       && !(cpssp->control_information.request.bmRequestType
			    & USB_DEV_REQ_DEVICE_TO_HOST))) {
		/* status stage or end of data stage of host-to-device transfer */
		/* send zero byte status stage packet */
		cpssp->control_information.stage = STAGE_STATUS;
		usb_send_data(cpssp, USB_PID_DATA1, NULL, 0);
	}
}

/* handle IN token on interrupt in endpoint */
static void
usb_handle_packet_usb_token_in_interrupt_in(struct cpssp * cpssp)
{
	/* TODO: ? */
	/* usb_send_handshake(cpssp, USB_PID_NAK); */

	static char data[10] = {
		'\xa1',
		'\x20',
		'\x00',
		'\x00',
		'\x00',
		'\x00',
		'\x02',
		'\x00',
		'\x82',
		'\x00'
	};

	usb_send_data(cpssp,
			cpssp->interrupt_in.data_toggle,
			data,
			10);
}

/* handle IN token on bulk in endpoint */
static void
usb_handle_packet_usb_token_in_bulk_in(struct cpssp * cpssp)
{
	/* answer with bytes from serial buffer */

	/* only fetch new data if previous packet was ACKed! */
	if (cpssp->bulk_in.last_data_acked) {
		cpssp->bulk_in.fifo_datalen =
			ringbuffer_get(&cpssp->serial_buffer,
					cpssp->bulk_in.fifo,
					USB_PACKETSIZE_BULK_IN);
	} else {
#if U2S_DEBUGMASK & U2S_DEBUG_DUMP_PACKETS
		faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME,
				"last data packet not acked, sending same data again!\n");
#endif
	}

	if (cpssp->bulk_in.fifo_datalen == 0) {
		cpssp->bulk_in.last_data_acked = 1;
		usb_send_handshake(cpssp, USB_PID_NAK);
	} else {
		cpssp->bulk_in.last_data_acked = 0;
		usb_send_data(cpssp,
				cpssp->bulk_in.data_toggle,
				cpssp->bulk_in.fifo,
				cpssp->bulk_in.fifo_datalen);
	}
}

static void
usb_handle_packet_usb_token(
	void *_cpssp,
	int pid,
	int addr,
	int endp
)
{
	struct cpssp * cpssp = (struct cpssp *) _cpssp;

	if (cpssp->state < STATE_DEFAULT) {
		/* unable to handle USB packets in these states */
		return;
	}

	/* unknown PID? */
	if (pid != USB_PID_SETUP
	    && pid != USB_PID_IN
	    && pid != USB_PID_OUT) {
		faum_log(FAUM_LOG_ERROR, U2S_LOG_TYPE, U2S_LOG_NAME,
				"unknown PID %02X\n", pid);
		return;
	}

	/* save for later use */
	cpssp->last_token.pid = pid;
	cpssp->last_token.addr = addr;
	cpssp->last_token.endp = endp;

#if U2S_DEBUGMASK & U2S_DEBUG_DUMP_PACKETS
	faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME,
			"received %s token to %02d:%02d%s\n",
			cim_usb_debug_pid2str(pid), addr, endp,
			cpssp->addr == addr ? " (this is me!)" : "");
#endif

#if U2S_DEBUGMASK & U2S_DEBUG_STATISTICS
	cpssp->statistics.token_count_total++;
#endif

	/* ignore tokens to other addresses */
	if (cpssp->addr != addr) {
		return;
	}

#if U2S_DEBUGMASK & U2S_DEBUG_STATISTICS
	cpssp->statistics.token_count++;

	switch (endp) {
	case 0: cpssp->statistics.token_count_ctrl++; break;
	case USB_ENDPT_NR_INTERRUPT_IN: cpssp->statistics.token_count_int_in++; break;
	case USB_ENDPT_NR_BULK_OUT: cpssp->statistics.token_count_bulk_out++; break;
	case USB_ENDPT_NR_BULK_IN: cpssp->statistics.token_count_bulk_in++; break;
	}
#endif

	if (pid == USB_PID_IN) {
		if (endp == 0) {
			usb_handle_packet_usb_token_in_control(cpssp);
			return;
		}

		if (cpssp->state != STATE_CONFIGURED) {
			/* we must not accept packets to other endpoints
			 * but 0 if not in CONFIGURED state */
			return;
		}

		switch (endp) {
		case USB_ENDPT_NR_INTERRUPT_IN:
			usb_handle_packet_usb_token_in_interrupt_in(cpssp);
			break;
		case USB_ENDPT_NR_BULK_IN:
			usb_handle_packet_usb_token_in_bulk_in(cpssp);
			break;
		case USB_ENDPT_NR_BULK_OUT:
#if U2S_DEBUGMASK & U2S_DEBUG_DUMP_PACKETS
			faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME,
					"IN token cannot go into an OUT endpoint\n");
#endif
			/* FIXME: stall bulk out pipe instead? */
			cpssp->control.stalled = 1;
			usb_send_handshake(cpssp, USB_PID_STALL);
			break;
		default:
#if U2S_DEBUGMASK & U2S_DEBUG_DUMP_PACKETS
			faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME,
					"IN token for unknown endpoint\n");
#endif
			cpssp->control.stalled = 1;
			usb_send_handshake(cpssp, USB_PID_STALL);
		}
	} else if (pid == USB_PID_SETUP
		   && endp == 0) {
		/* initialize stage information */
		cpssp->control_information.stage = STAGE_SETUP;
		cpssp->control.data_toggle = USB_PID_DATA0;
	} else if (pid == USB_PID_OUT
		   && endp == 0
		   && cpssp->control_information.stage == STAGE_DATA
		   && (cpssp->control_information.request.bmRequestType
		       & USB_DEV_REQ_DEVICE_TO_HOST)) {
		/* advance stage */
		cpssp->control_information.stage = STAGE_STATUS;
		cpssp->control.data_toggle = USB_PID_DATA1;
	}
}

/* invert DATA0 <-> DATA1 */
static unsigned char
usb_toggle_data(unsigned char toggle)
{
	if (toggle == USB_PID_DATA0) {
		return USB_PID_DATA1;
	} else if (toggle == USB_PID_DATA1) {
		return USB_PID_DATA0;
	} else {
		faum_log(FAUM_LOG_ERROR, U2S_LOG_TYPE, U2S_LOG_NAME,
				"%s: invalid data toggle %X\n",
				__FUNCTION__, toggle);
		return USB_PID_DATA0;
	}
}

/* handle DATA0/DATA1 packet for control endpoint (0) */
static void
usb_handle_packet_usb_data_control(
 	struct cpssp * cpssp,
	int pid,
	unsigned int length,
	uint8_t *data
)
{
	if (pid != cpssp->control.data_toggle) {
#if U2S_DEBUGMASK & U2S_DEBUG_DUMP_PACKETS
		faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME,
				"data packet with wrong data toggle (%s; expected %s)\n",
				cim_usb_debug_pid2str(pid),
				cim_usb_debug_pid2str(cpssp->control.data_toggle));
#endif
		usb_send_handshake(cpssp, USB_PID_ACK);
		return;
	}

	if (cpssp->control_information.stage == STAGE_NONE) {
#if U2S_DEBUGMASK & U2S_DEBUG_DUMP_PACKETS
		faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME,
				"not in any control transfer stage, ignoring data packet\n");
#endif
		cpssp->control.stalled = 1;
		usb_send_handshake(cpssp, USB_PID_STALL);
		return;
	}

	if (cpssp->control_information.stage == STAGE_SETUP) {
		if (length != 8) {
#if U2S_DEBUGMASK & U2S_DEBUG_DUMP_PACKETS
			faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME,
					"ignoring device request with %d != 8 bytes length\n",
					length);
#endif
			cpssp->control.stalled = 1;
			usb_send_handshake(cpssp, USB_PID_STALL);
			return;
		}

		cpssp->control_information.request.bmRequestType =
			data[0];
		cpssp->control_information.request.bRequest =
			data[1];
		cpssp->control_information.request.wValue =
			data[2] + (data[3] << 8);
		cpssp->control_information.request.wIndex =
			data[4] + (data[5] << 8);
		cpssp->control_information.request.wLength =
			data[6] + (data[7] << 8);

		if ((cpssp->control_information.request.bmRequestType
		     & USB_DEV_REQ_TYPE) != USB_DEV_REQ_TYPE_STANDARD) {
			/* we have a problem now: undocumented vendor specific request type */
			if (cpssp->control_information.request.bmRequestType
			    & USB_DEV_REQ_DEVICE_TO_HOST) {
				unsigned short _length =
					cpssp->control_information.request.wLength;
				int i;
#if U2S_DEBUGMASK & U2S_DEBUG_DUMP_PACKETS
				faum_log(FAUM_LOG_WARNING, U2S_LOG_TYPE, U2S_LOG_NAME,
						"answering non-standard device request with zero-filled packet\n");
#endif
				for (i = 0; i < _length; i++) {
					cpssp->control_information.data[i] = 0;
				}
				cpssp->control_information.data_length = _length;
				cpssp->control_information.data_sent = 0;
			} else {
#if U2S_DEBUGMASK & U2S_DEBUG_DUMP_PACKETS
				faum_log(FAUM_LOG_WARNING, U2S_LOG_TYPE, U2S_LOG_NAME,
						"acknowledging non-standard device request\n");
#endif
			}
			/* FIXME: ugly coding style! */
			goto advance_stage;
		}

#if U2S_DEBUGMASK & U2S_DEBUG_DUMP_PACKETS
		faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME,
				"%s default request received\n",
				cim_usb_debug_defaultrequest2str(
					cpssp->control_information.request.bRequest));
#endif

		/* is this standard request implemented? */
		switch (cpssp->control_information.request.bRequest) {

			/* GET_STATUS not implemented */

		case USB_DEV_REQ_SET_FEATURE:
		case USB_DEV_REQ_CLEAR_FEATURE:
			/* check sanity */
			if ((cpssp->control_information.request.bmRequestType != 0
			     && cpssp->control_information.request.bmRequestType != 2)
			    || (cpssp->control_information.request.bmRequestType == 0
				&& (cpssp->control_information.request.wValue != USB_FEATURE_DEVICE_REMOTE_WAKEUP
				    || cpssp->control_information.request.wIndex != 0))
			    || (cpssp->control_information.request.bmRequestType == 2
				&& (cpssp->control_information.request.wValue != USB_FEATURE_ENDPOINT_STALL
				    || (cpssp->control_information.request.wIndex != 0
					&& cpssp->control_information.request.wIndex != USB_ENDPT_INTERRUPT_IN
					&& cpssp->control_information.request.wIndex != USB_ENDPT_BULK_OUT
					&& cpssp->control_information.request.wIndex != USB_ENDPT_BULK_IN)))
			    || cpssp->control_information.request.wLength != 0) {
#if U2S_DEBUGMASK & U2S_DEBUG_DUMP_PACKETS
				faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME,
						"ignoring invalid %s request\n",
						cim_usb_debug_defaultrequest2str(
							cpssp->control_information.request.bRequest));
#endif
				cpssp->control.stalled = 1;
				usb_send_handshake(cpssp, USB_PID_STALL);
				return;
			}

			if (cpssp->control_information.request.bmRequestType == 0) {
#if U2S_DEBUGMASK & U2S_DEBUG_DUMP_PACKETS
				faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME,
						"DEVICE_REMOTE_WAKEUP");
#endif
			} else {
				int value;
#if U2S_DEBUGMASK & U2S_DEBUG_DUMP_PACKETS
				faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME,
						"ENDPOINT_STALL on EP %02X",
						cpssp->control_information.request.wIndex);
#endif
				value = cpssp->control_information.request.bRequest
					== USB_DEV_REQ_SET_FEATURE ? 1 : 0;

				switch (cpssp->control_information.request.wIndex) {
				case 0:
					cpssp->control.stalled = value;
					cpssp->control.data_toggle = USB_PID_DATA0;
					cpssp->control.last_data_acked = 1;
					break;
				case USB_ENDPT_INTERRUPT_IN:
					cpssp->interrupt_in.stalled = value;
					cpssp->interrupt_in.data_toggle = USB_PID_DATA0;
					cpssp->interrupt_in.last_data_acked = 1;
					break;
				case USB_ENDPT_BULK_OUT:
					cpssp->bulk_out.stalled = value;
					cpssp->bulk_out.data_toggle = USB_PID_DATA0;
					cpssp->bulk_out.last_data_acked = 1;
					break;
				case USB_ENDPT_BULK_IN:
					cpssp->bulk_in.stalled = value;
					cpssp->bulk_in.data_toggle = USB_PID_DATA0;
					cpssp->bulk_in.last_data_acked = 1;
					break;
				}
			}

#if U2S_DEBUGMASK & U2S_DEBUG_DUMP_PACKETS
			if (cpssp->control_information.request.bRequest
			    == USB_DEV_REQ_SET_FEATURE) {
				faum_cont(FAUM_LOG_DEBUG, " SET\n");
			} else {
				faum_cont(FAUM_LOG_DEBUG, " CLEARED\n");
			}
#endif
			break;

		case USB_DEV_REQ_SET_ADDRESS:
			/* check sanity */
			if (cpssp->control_information.request.bmRequestType != 0
			    || cpssp->control_information.request.wIndex != 0
			    || cpssp->control_information.request.wLength != 0) {
#if U2S_DEBUGMASK & U2S_DEBUG_DUMP_PACKETS
				faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME,
						"ignoring invalid %s request\n",
						cim_usb_debug_defaultrequest2str(
							cpssp->control_information.request.bRequest));
#endif
				cpssp->control.stalled = 1;
				usb_send_handshake(cpssp, USB_PID_STALL);
				return;
			}
#if U2S_DEBUGMASK & U2S_DEBUG_DUMP_PACKETS
			faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME,
					"(new device address %02d, not valid until status stage completes)\n",
					cpssp->control_information.request.wValue & 0x00FF);
#endif
			/* address gets valid after status stage, not right now! */
			break;

		case USB_DEV_REQ_GET_DESCRIPTOR:
			{
				unsigned char descriptor_type =
					cpssp->control_information.request.wValue >> 8;
				unsigned char descriptor_index =
					cpssp->control_information.request.wValue & 0x00FF;

				/* check sanity */
				if (cpssp->control_information.request.bmRequestType != 0x80
				    || (descriptor_type != USB_DESCRIPTOR_TYPE_DEVICE
					&& descriptor_type != USB_DESCRIPTOR_TYPE_CONFIGURATION
					&& descriptor_type != USB_DESCRIPTOR_TYPE_STRING)) {
#if U2S_DEBUGMASK & U2S_DEBUG_DUMP_PACKETS
					faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME,
							"ignoring invalid %s request\n",
							cim_usb_debug_defaultrequest2str(
								cpssp->control_information.request.bRequest));
#endif
					cpssp->control.stalled = 1;
					usb_send_handshake(cpssp, USB_PID_STALL);
					return;
				}
#if U2S_DEBUGMASK & U2S_DEBUG_DUMP_PACKETS
				faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME,
						"requested descriptor is %s (index %d, langid %d) with %d bytes length\n",
						cim_usb_debug_descriptortype2str(descriptor_type),
						descriptor_index,
						cpssp->control_information.request.wIndex,
						cpssp->control_information.request.wLength);
#endif
				if (descriptor_type == USB_DESCRIPTOR_TYPE_DEVICE
				    && descriptor_index == 0) {

					memcpy(cpssp->control_information.data,
							&device_descriptor_pl2303,
							sizeof(device_descriptor_pl2303));

					cpssp->control_information.data_length =
						MIN(sizeof(device_descriptor_pl2303),
							cpssp->control_information.request.wLength);
					cpssp->control_information.data_sent = 0;

				} else if (descriptor_type == USB_DESCRIPTOR_TYPE_CONFIGURATION
					   && descriptor_index == 0) {

					memcpy(cpssp->control_information.data,
							&combined_descriptor_pl2303,
							sizeof(combined_descriptor_pl2303));

					cpssp->control_information.data_length =
						MIN(sizeof(combined_descriptor_pl2303),
							cpssp->control_information.request.wLength);
					cpssp->control_information.data_sent = 0;

				} else {
					cpssp->control.stalled = 1;
					usb_send_handshake(cpssp, USB_PID_STALL);
					return;
				}
			}

			break;

		/* SET_DESCRIPTOR not implemented (optional) */

		case USB_DEV_REQ_GET_CONFIGURATION:
			{
				unsigned char configuration;

				/* check sanity */
				if (cpssp->control_information.request.bmRequestType != 0x80
				    || cpssp->control_information.request.wIndex != 0
				    || cpssp->control_information.request.wLength != 1) {
#if U2S_DEBUGMASK & U2S_DEBUG_DUMP_PACKETS
					faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME,
							"ignoring invalid %s request\n",
							cim_usb_debug_defaultrequest2str(
								cpssp->control_information.request.bRequest));
#endif
					cpssp->control.stalled = 1;
					usb_send_handshake(cpssp, USB_PID_STALL);
					return;
				}
				/* this is easy, we only have one configuration */
				configuration =
					cpssp->state == STATE_CONFIGURED ? 1 : 0;
#if U2S_DEBUGMASK & U2S_DEBUG_DUMP_PACKETS
				faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME,
						"telling we have configuration %d\n",
						configuration);
#endif
				/* send one byte answer */
				cpssp->control_information.data[0] = configuration;
				cpssp->control_information.data_length = 1;
				cpssp->control_information.data_sent = 0;
				break;
			}

		case USB_DEV_REQ_SET_CONFIGURATION:
			{
				unsigned char configuration;

				/* check sanity */
				if (cpssp->control_information.request.bmRequestType != 0
				    || cpssp->control_information.request.wIndex != 0
				    || cpssp->control_information.request.wLength != 0) {
#if U2S_DEBUGMASK & U2S_DEBUG_DUMP_PACKETS
					faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME,
							"ignoring invalid %s request\n",
							cim_usb_debug_defaultrequest2str(
								cpssp->control_information.request.bRequest));
#endif
					cpssp->control.stalled = 1;
					usb_send_handshake(cpssp, USB_PID_STALL);
					return;
				}
				configuration =
					cpssp->control_information.request.wValue & 0x00FF;
#if U2S_DEBUGMASK & U2S_DEBUG_DUMP_PACKETS
				faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME,
						"setting configuration %d\n", configuration);
#endif
				/* FIXME: maybe we may only transit to CONFIGURED state
				 * after status stage, but I don't care at the moment
				 */
				if (cpssp->state == STATE_ADDRESS
				    && configuration == 1) {
					/* configured */
					SET_USB_DEVSTATE(cpssp, STATE_CONFIGURED);

					/* reset endpoint state */
					cpssp->control.data_toggle = USB_PID_DATA0;
					cpssp->control.last_data_acked = 1;
					cpssp->interrupt_in.data_toggle = USB_PID_DATA0;
					cpssp->interrupt_in.last_data_acked = 1;
					cpssp->bulk_out.data_toggle = USB_PID_DATA0;
					cpssp->bulk_out.last_data_acked = 1;
					cpssp->bulk_in.data_toggle = USB_PID_DATA0;
					cpssp->bulk_in.last_data_acked = 1;
				} else if (cpssp->state == STATE_CONFIGURED
					   && configuration == 0) {
					/* deconfigured */
					SET_USB_DEVSTATE(cpssp, STATE_ADDRESS);
				} else {
#if U2S_DEBUGMASK & U2S_DEBUG_DUMP_PACKETS
					faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME,
							"cannot move to CONFIGURED state, not yet in ADDRESS state\n");
#endif
				}
				break;
			}

		case USB_DEV_REQ_GET_INTERFACE:
			/* check sanity */
			if (cpssp->control_information.request.bmRequestType != 0x81
			    || cpssp->control_information.request.wValue != 0
			    || cpssp->control_information.request.wLength != 1) {
#if U2S_DEBUGMASK & U2S_DEBUG_DUMP_PACKETS
				faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME,
						"ignoring invalid %s request\n",
						cim_usb_debug_defaultrequest2str(
							cpssp->control_information.request.bRequest));
#endif
				cpssp->control.stalled = 1;
				usb_send_handshake(cpssp, USB_PID_STALL);
				return;
			}
			/* this is easy, we only have one alternate setting for
			 * the one interface */
#if U2S_DEBUGMASK & U2S_DEBUG_DUMP_PACKETS
			faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME,
					"interface %d, telling we have AlternateSetting %d\n",
					cpssp->control_information.request.wIndex & 0x00FF,
					0);
#endif
			/* send one byte answer */
			cpssp->control_information.data[0] = 0;
			cpssp->control_information.data_length = 1;
			cpssp->control_information.data_sent = 0;
			break;

		case USB_DEV_REQ_SET_INTERFACE:
			{
				unsigned char alternate_setting;

				/* check sanity */
				if (cpssp->control_information.request.bmRequestType != 0x01
				    || cpssp->control_information.request.wLength != 0) {
#if U2S_DEBUGMASK & U2S_DEBUG_DUMP_PACKETS
					faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME,
							"ignoring invalid %s request\n",
							cim_usb_debug_defaultrequest2str(
								cpssp->control_information.request.bRequest));
#endif
					cpssp->control.stalled = 1;
					usb_send_handshake(cpssp, USB_PID_STALL);
					return;
				}
				alternate_setting =
					cpssp->control_information.request.wValue & 0x00FF;
#if U2S_DEBUGMASK & U2S_DEBUG_DUMP_PACKETS
				faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME,
						"interface %d, setting AlternateSetting %d\n",
						cpssp->control_information.request.wIndex,
						alternate_setting);
#endif

				/* reset endpoint state */
				cpssp->control.data_toggle = USB_PID_DATA0;
				cpssp->control.last_data_acked = 1;
				cpssp->bulk_out.data_toggle = USB_PID_DATA0;
				cpssp->bulk_out.last_data_acked = 1;
				cpssp->bulk_in.data_toggle = USB_PID_DATA0;
				cpssp->bulk_in.last_data_acked = 1;

				break;
			}

			/* SYNCH_FRAME not implemented (optional) */

		default:
			faum_log(FAUM_LOG_WARNING, U2S_LOG_TYPE, U2S_LOG_NAME,
					"ignoring standard device request %X\n",
					cpssp->control_information.request.bRequest);
			cpssp->control.stalled = 1;
			usb_send_handshake(cpssp, USB_PID_STALL);
			return;
		}

advance_stage:
		/* advance stage */
		/* length == 0 indicates there is no data stage (USB spec., 9.3.5) */
		if (cpssp->control_information.request.wLength == 0)
			cpssp->control_information.stage = STAGE_STATUS;
		else {
			cpssp->control_information.stage = STAGE_DATA;
		}
	} else if (cpssp->control_information.stage == STAGE_DATA) {
		/* only vendor-specific host-to-device requests -- just ACK */
	} else if (cpssp->control_information.stage == STAGE_STATUS) {
		if (cpssp->control_information.request.bmRequestType
		    & USB_DEV_REQ_DEVICE_TO_HOST) {
			/* this should be the zero byte status stage packet */
			if (length != 0) {
#if U2S_DEBUGMASK & U2S_DEBUG_DUMP_PACKETS
				faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME,
						"ignoring status stage data packet with %d > 0 bytes length\n",
						length);
#endif
				cpssp->control.stalled = 1;
				usb_send_handshake(cpssp, USB_PID_STALL);
				return;
			}

			cpssp->control_information.stage = STAGE_NONE;
		} else {
#if U2S_DEBUGMASK & U2S_DEBUG_DUMP_PACKETS
			faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME,
					"ignoring status stage data packet in host-to-device request\n");
#endif
			cpssp->control.stalled = 1;
			usb_send_handshake(cpssp, USB_PID_STALL);
			return;
		}
	}

	cpssp->control.data_toggle =
		usb_toggle_data(cpssp->control.data_toggle);
	usb_send_handshake(cpssp, USB_PID_ACK);
}

/* handle DATA0/DATA1 packet for bulk out endpoint */
static void
usb_handle_packet_usb_data_bulk_out(
 	struct cpssp * cpssp,
	int pid,
	unsigned int length,
	uint8_t *data
)
{
	/* TODO: does the real PL2303 behave like this? */

	/* check data toggle */
	if (pid != cpssp->bulk_out.data_toggle) {
#if U2S_DEBUGMASK & U2S_DEBUG_DUMP_PACKETS
		faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME,
				"data packet with wrong data toggle (%s; expected %s)\n",
				cim_usb_debug_pid2str(pid),
				cim_usb_debug_pid2str(cpssp->bulk_out.data_toggle));
#endif
		usb_send_handshake(cpssp, USB_PID_ACK);
		return;
	}

#if U2S_DEBUGMASK & U2S_DEBUG_DUMP_PACKETS
	faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME,
			"pushing %d bytes of data to serial interface\n",
			length);
#endif
	while (0 < length) {
		sig_serial_send(cpssp->port_serial, cpssp, *data++);
		length--;
	}

	cpssp->bulk_out.data_toggle =
		usb_toggle_data(cpssp->bulk_out.data_toggle);
	usb_send_handshake(cpssp, USB_PID_ACK);
}

/* handle DATA0/DATA1 packet */
static void
usb_handle_packet_usb_data(
	void *_cpssp,
	int pid,
	unsigned int length,
	uint8_t *data,
	uint16_t crc16
)
{
	struct cpssp * cpssp = (struct cpssp *) _cpssp;

	if (cpssp->state < STATE_DEFAULT) {
		/* unable to handle USB packets in these states */
		return;
	}

	if (cpssp->addr != cpssp->last_token.addr) {
#if U2S_DEBUGMASK & U2S_DEBUG_DUMP_PACKETS
		faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME,
				"data packet not meant for me\n");
#endif
		return;
	}
#if U2S_DEBUGMASK & U2S_DEBUG_DUMP_PACKETS
	faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME,
			"%s packet received with %d bytes",
			cim_usb_debug_pid2str(pid), length);
	usb_debug_data(data, length);
#endif

	if (cpssp->last_token.endp == 0) {
		usb_handle_packet_usb_data_control(cpssp, pid, length, data);
		return;
	}

	if (cpssp->state != STATE_CONFIGURED) {
		/* we must not accept packets to other endpoints
		 * but 0 if not in CONFIGURED state */
		/* FIXME: stall target pipe instead? */
		cpssp->control.stalled = 1;
		usb_send_handshake(cpssp, USB_PID_STALL);
		return;
	}

	switch (cpssp->last_token.endp) {
	case USB_ENDPT_NR_INTERRUPT_IN:
	case USB_ENDPT_NR_BULK_IN:
#if U2S_DEBUGMASK & U2S_DEBUG_DUMP_PACKETS
		faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME,
				"data packet cannot go into an IN endpoint\n");
#endif
		/* FIXME: stall bulk in pipe instead? */
		cpssp->control.stalled = 1;
		usb_send_handshake(cpssp, USB_PID_STALL);
		break;
	case USB_ENDPT_NR_BULK_OUT:
		usb_handle_packet_usb_data_bulk_out(cpssp, pid, length, data);
		break;
	default:
#if U2S_DEBUGMASK & U2S_DEBUG_DUMP_PACKETS
		faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME,
				"data packet for unknown endpoint\n");
#endif
		cpssp->control.stalled = 1;
		usb_send_handshake(cpssp, USB_PID_STALL);
	}
}

/* handle handshake packet for control endpoint (0) */
static void
usb_handle_packet_usb_handshake_control(struct cpssp * cpssp, int pid)
{
	if (cpssp->control_information.stage == STAGE_DATA
	    && (cpssp->control_information.request.bmRequestType
		& USB_DEV_REQ_DEVICE_TO_HOST)) {
		/* data stage of a device-to-host transfer, data packet handshake */
		if (pid == USB_PID_ACK) {
			cpssp->control_information.data_sent +=
				MIN(USB_PACKETSIZE_CONTROL0,
						cpssp->control_information.data_length
						- cpssp->control_information.data_sent);
			cpssp->control.data_toggle =
				usb_toggle_data(cpssp->control.data_toggle);
			cpssp->control.last_data_acked = 1;
		}

	} else if (cpssp->control_information.stage == STAGE_STATUS
		   && !(cpssp->control_information.request.bmRequestType
			& USB_DEV_REQ_DEVICE_TO_HOST)) {
		/* status stage of a host-to-device transfer, zero data packet handshake */
		if (cpssp->control_information.request.bRequest
		    == USB_DEV_REQ_SET_ADDRESS) {
			/* new address gets valid now */
			cpssp->addr = cpssp->control_information.request.wValue & 0x0F;
#if U2S_DEBUGMASK & U2S_DEBUG_DUMP_PACKETS
			faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME,
					"new device address %02d is valid now\n",
					cpssp->addr);
#endif
			if (cpssp->addr == 0) {
				SET_USB_DEVSTATE(cpssp, STATE_DEFAULT);
			} else {
				SET_USB_DEVSTATE(cpssp, STATE_ADDRESS);
			}
		}
		cpssp->control_information.stage = STAGE_NONE;
		cpssp->control.data_toggle = USB_PID_DATA0;
		cpssp->control.last_data_acked = 1;
	}
	/* TODO: lots more to do here! */
}

/* handle handshake packet for interrupt in endpoint */
static void
usb_handle_packet_usb_handshake_interrupt_in(struct cpssp * cpssp, int pid)
{
	if (pid == USB_PID_ACK) {
		cpssp->interrupt_in.data_toggle =
			usb_toggle_data(cpssp->interrupt_in.data_toggle);
		cpssp->interrupt_in.last_data_acked = 1;
	}
}

/* handle handshake packet for bulk in endpoint */
static void
usb_handle_packet_usb_handshake_bulk_in(struct cpssp * cpssp, int pid)
{
	if (pid == USB_PID_ACK) {
		cpssp->bulk_in.data_toggle =
			usb_toggle_data(cpssp->bulk_in.data_toggle);
		cpssp->bulk_in.last_data_acked = 1;
	}
}

/* handle handshake packet for bulk out endpoint */
static void
usb_handle_packet_usb_handshake_bulk_out(struct cpssp * cpssp, int pid)
{
	if (pid == USB_PID_ACK) {
		cpssp->bulk_out.data_toggle =
			usb_toggle_data(cpssp->bulk_out.data_toggle);
		cpssp->bulk_out.last_data_acked = 1;
	}
}

/* handle handshake packet */
static void
usb_handle_packet_usb_handshake(void *_cpssp, int pid)
{
	struct cpssp * cpssp = (struct cpssp *) _cpssp;

	if (cpssp->state < STATE_DEFAULT) {
		/* unable to handle USB packets in these states */
		return;
	}

	if (cpssp->addr != cpssp->last_token.addr) {
#if U2S_DEBUGMASK & U2S_DEBUG_DUMP_PACKETS
		faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME,
				"handshake packet not meant for me\n");
#endif
		return;
	}
#if U2S_DEBUGMASK & U2S_DEBUG_DUMP_PACKETS
	faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME,
			"%s packet received\n",
			cim_usb_debug_pid2str(pid));
#endif

	if (cpssp->last_token.endp == 0) {
		usb_handle_packet_usb_handshake_control(cpssp, pid);
		return;
	}

	if (cpssp->state != STATE_CONFIGURED) {
		/* we must not accept packets to other endpoints
		 * but 0 if not in CONFIGURED state */
		cpssp->control.stalled = 1;
		usb_send_handshake(cpssp, USB_PID_STALL);
		return;
	}

	switch (cpssp->last_token.endp) {
	case USB_ENDPT_NR_INTERRUPT_IN:
		usb_handle_packet_usb_handshake_interrupt_in(cpssp, pid);
		break;
	case USB_ENDPT_NR_BULK_IN:
		usb_handle_packet_usb_handshake_bulk_in(cpssp, pid);
		break;
	case USB_ENDPT_NR_BULK_OUT:
		usb_handle_packet_usb_handshake_bulk_out(cpssp, pid);
		break;
	default:
#if U2S_DEBUGMASK & U2S_DEBUG_DUMP_PACKETS
		faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME,
				"data packet for unknown endpoint\n");
#endif
		cpssp->control.stalled = 1;
		usb_send_handshake(cpssp, USB_PID_STALL);
	}
}

/* push serial data to USB */
void 
usb_push_data(struct cpssp * cpssp, const char *data, unsigned int length)
{
	ringbuffer_put(&cpssp->serial_buffer, data, length);
}

#if 0
/* reconfigure CIM */
void
usb_reconfig(void)
{
	int new_peer_count;
#if U2S_DEBUGMASK & U2S_DEBUG_CIM
	faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME,
		"reconfiguring USB bridge\n");
#endif
	cim_reconfig(dci);
	new_peer_count = cim_get_peer_count(&config->bcc_usb);

	if (new_peer_count > 0
	 && (state.state == STATE_OFF
	  || state.state == STATE_ATTACHED)) {
#if U2S_DEBUGMASK & U2S_DEBUG_CIM
		faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME,
			"telling USB we are bus powered and a fullspeed device\n");
#endif
		SET_USB_DEVSTATE(STATE_ATTACHED);
		cim_usb_send_linestate(dci, 
			USB_MSG_LINESTATE_TYPE_BUS_POWERED,
			USB_SPEED_FULL, 0);
	} else if (new_peer_count == 0
	        && state.state != STATE_OFF) {
		SET_USB_DEVSTATE(STATE_OFF);
#if U2S_DEBUGMASK & U2S_DEBUG_CIM
		faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME,
			"detached from USB bus\n");
#endif
	}
}
#endif

/* reset USB state machine to defaults */
static void
usb_reset_state(struct cpssp * cpssp)
{
	cpssp->addr = 0;
	cpssp->control.data_toggle = USB_PID_DATA0;
	cpssp->control.stalled = 0;
	cpssp->control.last_data_acked = 1;
	cpssp->last_token.pid = 0xFF;
	cpssp->last_token.addr = 0xFF;
	cpssp->last_token.endp = 0xFF;
	SET_USB_DEVSTATE(cpssp, STATE_OFF);
}

static void
serial_recv(void *_cpssp, uint8_t byte)
{
	struct cpssp * cpssp = (struct cpssp *) _cpssp;

	usb_push_data(cpssp, &byte, sizeof(byte));
}

static void
usb_power_set(void *_cpssp, unsigned int val)
{
	struct cpssp * cpssp = (struct cpssp *) _cpssp;

	if (val) {
		usb_reset_state(cpssp);
		SET_USB_DEVSTATE(cpssp, STATE_POWERED);

#if U2S_DEBUGMASK & U2S_DEBUG_CIM
		faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME,
				"answering we are bus powered and a fullspeed device\n");
#endif

		/*
		 * Low speed devices only support control and interrupt
		 * endpoints; we have bulk endpoints, therefore we
		 * _must_ be a fullspeed device.
		 */

		/* Send answer: we are bus powered now. */
		sig_usb_bus_speed_set(cpssp->port_usb_main, cpssp,
				USB_SPEED_FULL);

	} else {
#if U2S_DEBUGMASK & U2S_DEBUG_CIM
		if (cpssp->state != STATE_OFF
		    && cpssp->state != STATE_ATTACHED) {
			faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME,
					"no power on USB anymore, powering down\n");
		}
#endif
		SET_USB_DEVSTATE(cpssp, STATE_ATTACHED);
	}
}

void *
usb2serial_create(
	const char *name,
	struct sig_manage *port_manage,
	struct sig_serial *port_serial,
	struct sig_usb_conn *port_usb
)
{
	static const struct sig_serial_funcs serial_funcs = {
		.recv = serial_recv,
	};
	static const struct sig_std_logic_funcs usb_power_funcs = {
		.boolean_or_set = usb_power_set,
	};
	static const struct sig_usb_bus_funcs usb_main_funcs = {
		.reset_set = usb_reset_set,
		.speed_set = NULL,
		.recv_token = usb_handle_packet_usb_token,
		.recv_sof = NULL,
		.recv_data = usb_handle_packet_usb_data,
		.recv_handshake = usb_handle_packet_usb_handshake,
	};
	struct cpssp *cpssp;

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);
	memset(cpssp, 0, sizeof(*cpssp));

	assert(sizeof(device_descriptor_pl2303) == 18);
	assert(sizeof(combined_descriptor_pl2303) == 39);

	ringbuffer_create(&cpssp->serial_buffer, RINGBUFFER_SIZE, 1);
	cpssp->state = 0;
	usb_reset_state(cpssp);

	cpssp->port_serial = port_serial;
	cpssp->port_usb_power = port_usb->power;
	cpssp->port_usb_main = port_usb->bus;

	/* Call */
	sig_serial_connect(port_serial, cpssp, &serial_funcs);
	sig_usb_bus_connect(port_usb->bus, cpssp, &usb_main_funcs);

	/* Out */

	/* In */
	sig_std_logic_connect_in(port_usb->power, cpssp, &usb_power_funcs);

	return cpssp;
}

void
usb2serial_destroy(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	ringbuffer_destroy(&cpssp->serial_buffer);

#if U2S_DEBUGMASK & U2S_DEBUG_STATISTICS
	faum_log(FAUM_LOG_DEBUG, U2S_LOG_TYPE, U2S_LOG_NAME,
			"token count on USB = %lu\n"
			"token count for me = %lu\n"
			"token count CTL0 = %lu\n"
			"token count INT_IN = %lu\n"
			"token count BULK_IN = %lu\n"
			"token count BULK_OUT = %lu\n",
			cpssp->statistics.token_count_total,
			cpssp->statistics.token_count,
			cpssp->statistics.token_count_ctrl,
			cpssp->statistics.token_count_int_in,
			cpssp->statistics.token_count_bulk_in,
			cpssp->statistics.token_count_bulk_out);
#endif

	shm_free(cpssp);
}

void
usb2serial_suspend(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;
	
	generic_suspend(cpssp, sizeof(*cpssp), fComp);
	
	if (fwrite(cpssp->serial_buffer.buffer, cpssp->serial_buffer.buffer_size, 1, fComp) != 1){
		fprintf(stderr, "fwrite error\n");
	}
}

void
usb2serial_resume(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;
	
	generic_resume(cpssp, sizeof(*cpssp), fComp);
	
	if (fread(cpssp->serial_buffer.buffer, cpssp->serial_buffer.buffer_size, 1, fComp) != 1){
		fprintf(stderr, "fread error\n");
	}
}
