/*
 * $Id: pata_disk_template.c,v 1.31 2013/06/26 09:43:21 vrsieh Exp $ 
 *
 * Copyright (C) 2013 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/* Debugging Options */
#define DEBUG_CONTROL_FLOW	0
#define DEBUG_INTERRUPT		0
#define DEBUG_OPCODES		0

#define RESERVED_SECTORS		20
#define RESERVED_SECTORS_THRESHOLD	1

#include "config.h"

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "conv_gen.h"
#include "glue.h"
#include "std-ide.h"

#include "sig_boolean.h"
#include "sig_integer.h"

#define STATE

struct cpssp {
	struct sig_ide_bus *port_ide;

	struct {
		unsigned int unit;

		volatile uint8_t error;   /* error reg    (read only)  */
		volatile uint8_t features;/* features reg (write only) */
		volatile uint8_t nsector; /* sector count reg */
		volatile uint8_t sector;  /* sector reg */
		volatile uint16_t cyl;    /* cylinder reg */
		volatile uint8_t select;  /* select reg */
		volatile uint8_t command; /* command reg */
		uint8_t control;          /* device control reg */
	} common;

#define NAME		core
#define NAME_(x)	core_ ## x
#define SNAME		"core"
#include "arch_ata_disk.c"
#undef NAME
#undef NAME_
#undef SNAME

#define NAME		iface
#define NAME_(x)	iface_ ## x
#define SNAME		"iface"
#include "arch_pata_ata.c"
#undef NAME
#undef NAME_
#undef SNAME
};

#undef STATE

#define BEHAVIOR

static void
iface_dmarq_out_set(struct cpssp *cpssp, unsigned int val)
{
	sig_ide_bus_dmarq_set(cpssp->port_ide, val);
}

static void
iface_irqrq_out_set(struct cpssp *cpssp, unsigned int val)
{
	sig_ide_bus_irq(cpssp->port_ide, cpssp, val);
}

/*forward*/ static void
iface_dma_in(struct cpssp *cpssp, unsigned int count);
/*forward*/ static void
iface_dma_out(struct cpssp *cpssp, unsigned int count);
/*forward*/ static void
iface_pio_in(struct cpssp *cpssp, unsigned int count, int irq);
/*forward*/ static void
iface_pio_out(struct cpssp *cpssp, unsigned int count, int irq);
/*forward*/ static void
iface_done_io(struct cpssp *cpssp);
/*forward*/ static void
iface_done_cmd(struct cpssp *cpssp, int irq);

/*forward*/ static void
core_data_in(struct cpssp *cpssp, uint16_t *valp);
/*forward*/ static void
core_data_out(struct cpssp *cpssp, uint16_t val);
/*forward*/ static void
core_command(struct cpssp *cpssp);
/*forward*/ static void
core_soft_reset(struct cpssp *cpssp);

static void
core_dma_in(struct cpssp *cpssp, unsigned int count)
{
	iface_dma_in(cpssp, count);
}

static void
core_dma_out(struct cpssp *cpssp, unsigned int count)
{
	iface_dma_out(cpssp, count);
}

static void
core_pio_in(struct cpssp *cpssp, unsigned int count, int irq)
{
	iface_pio_in(cpssp, count, irq);
}

static void
core_pio_out(struct cpssp *cpssp, unsigned int count, int irq)
{
	iface_pio_out(cpssp, count, irq);
}

static void
core_done_io(struct cpssp *cpssp)
{
	iface_done_io(cpssp);
}

static void
core_done_cmd(struct cpssp *cpssp, int irq)
{
	iface_done_cmd(cpssp, irq);
}

static void
iface_data_in(struct cpssp *cpssp, uint16_t *valp)
{
	core_data_in(cpssp, valp);
}

static void
iface_data_out(struct cpssp *cpssp, uint16_t val)
{
	core_data_out(cpssp, val);
}

static void
iface_command(struct cpssp *cpssp)
{
	core_command(cpssp);
}

static void
iface_soft_reset(struct cpssp *cpssp)
{
	core_soft_reset(cpssp);
}

#define NAME		core
#define NAME_(x)	core_ ## x
#define SNAME		"core"
#include "arch_ata_disk.c"
#undef NAME
#undef NAME_
#undef SNAME

#define DEBUG			0
#define PACKET_INTERFACE	0
#define NAME		iface
#define NAME_(x)	iface_ ## x
#define SNAME		"iface"
#include "arch_pata_ata.c"
#undef NAME
#undef NAME_
#undef SNAME
#undef PACKET_INTERFACE
#undef DEBUG

static void
COMP_(power_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	core_power_set(cpssp, val);
	iface_power_set(cpssp, val);
}

static void
COMP_(reset)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	core_reset_set(cpssp, val);
	iface_reset_set(cpssp, val);
}

static int
COMP_(inw)(void *_cpssp, unsigned short port, uint16_t *valp)
{
	struct cpssp *cpssp = _cpssp;

	return iface_inw(cpssp, port, valp);
}

static void
COMP_(outw)(void *_cpssp, unsigned short port, uint16_t val)
{
	struct cpssp *cpssp = _cpssp;

	iface_outw(cpssp, port, val);
}

static int
COMP_(dmainw)(void *_cpssp, uint16_t *valp)
{
	struct cpssp *cpssp = _cpssp;

	return iface_dmainw(cpssp, valp);
}

static void
COMP_(dmaoutw)(void *_cpssp, uint16_t val)
{
	struct cpssp *cpssp = _cpssp;

	iface_dmaoutw(cpssp, val);
}

void *
COMP_(create)(
	const char *name,
	const char *image,
#ifndef CYLINDERS
	const char *cylinders,
	const char *heads,
	const char *sectors,
	const char *size,
#endif
	const char *unit,
	struct sig_manage *port_manage,
	struct sig_molex_ide_power_conn *port_power,
	struct sig_ide_bus *port_ide
)
{
	static const struct sig_std_logic_funcs power_funcs = {
		.boolean_or_set = COMP_(power_set),
	};
	static const struct sig_ide_bus_funcs ide_funcs = {
		.reset = COMP_(reset),
		.inw = COMP_(inw),
		.outw = COMP_(outw),
		.dmainw = COMP_(dmainw),
		.dmaoutw = COMP_(dmaoutw),
	};
	struct cpssp *cpssp;

	assert(unit);

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);

	cpssp->common.unit = strtoul(unit, NULL, 0);

	core_create(cpssp, name, image,
#ifndef CYLINDERS
			cylinders, heads, sectors, size,
#endif
			port_manage);
	iface_create(cpssp);

	/* Call */
	cpssp->port_ide = port_ide;
	sig_ide_bus_connect(port_ide, cpssp, &ide_funcs);

	/* In */
	sig_std_logic_connect_in(port_power->_plus_5V, cpssp, &power_funcs);

	return cpssp;
}

void
COMP_(destroy)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	core_destroy(cpssp);
	iface_destroy(cpssp);

	shm_free(cpssp);
}

void
COMP_(suspend)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;

	generic_suspend(&cpssp->common, sizeof(cpssp->common), fp);
	core_suspend(cpssp, fp);
	iface_suspend(cpssp, fp);
}

void
COMP_(resume)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;

	generic_resume(&cpssp->common, sizeof(cpssp->common), fp);
	core_resume(cpssp, fp);
	iface_resume(cpssp, fp);
}

#undef BEHAVIOR
