/*
 * Copyright (C) 2003-2014 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#define DEBUG	0

#include "config.h"

#include <netinet/ip.h>
#include <arpa/inet.h>
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "glue-shm.h"
#include "glue-suspend.h"

#define INCLUDE
#include "arch_eth_iface.c"
#include "arch_arp.c"
#include "arch_route_ip.c"
#include "arch_local_ip.c"
#include "arch_udp.c"
#include "arch_tcp.c"
#include "arch_dhcp.c"
#include "arch_slirp.c"
#undef INCLUDE

#include "network_router.h"

#define COMP_(x) network_router_ ## x

struct cpssp {
	struct sig_eth *port_eth;

#define STATE
#define NAME		slirp
#include "arch_slirp.c"
#undef NAME
#define NAME		dhcp
#include "arch_dhcp.c"
#undef NAME
#define NAME		tcp
#include "arch_tcp.c"
#undef NAME
#define NAME		udp
#include "arch_udp.c"
#undef NAME
#define NAME		local_ip
#include "arch_local_ip.c"
#undef NAME
#define NAME		route_ip
#include "arch_route_ip.c"
#undef NAME
#define NAME		arp
#include "arch_arp.c"
#undef NAME
#define NAME		eth
#include "arch_eth_iface.c"
#undef NAME
#undef STATE
};

/* ------------------------------------------------------------------ */
/* dhcp */

/*forward*/ static void
udp_from_dhcp(struct cpssp *cpssp, const unsigned char *buf, unsigned int len);

static void
dhcp_reply(void *cpssp, const unsigned char *buf, unsigned int len)
{
	udp_from_dhcp(cpssp, buf, len);
}

#define BEHAVIOR
#define NAME		dhcp
#define NAME_(x)	dhcp_ ## x
#include "arch_dhcp.c"
#undef NAME_
#undef NAME
#undef BEHAVIOR

/* ------------------------------------------------------------------ */
/* uip */

/*forward*/ static void
local_ip_from_udp(void *cpssp, const unsigned char *buf, unsigned int len);

static void
udp_to_local_ip(void *cpssp, const unsigned char *buf, unsigned int len)
{
	local_ip_from_udp(cpssp, buf, len);
}

static void
udp_to_dhcp(void *cpssp, const unsigned char *buf, unsigned int len)
{
	dhcp_request(cpssp, buf, len);
}

#define BEHAVIOR
#define NAME		udp
#define NAME_(x)	udp_ ## x
#include "arch_udp.c"
#undef NAME_
#undef NAME
#undef BEHAVIOR

/* ------------------------------------------------------------------ */
/* tcp */

/*forward*/ static void
local_ip_from_tcp(void *cpssp, const unsigned char *buf, unsigned int len);

static void
tcp_to_local_ip(void *cpssp, const unsigned char *buf, unsigned int len)
{
	local_ip_from_tcp(cpssp, buf, len);
}

#define BEHAVIOR
#define NAME		tcp
#define NAME_(x)	tcp_ ## x
#include "arch_tcp.c"
#undef NAME_
#undef NAME
#undef BEHAVIOR

/* ------------------------------------------------------------------ */
/* local ip */

/*forward*/ static void
route_ip_from_local_ip(
	struct cpssp *cpssp,
	const unsigned char *buf,
	unsigned int len
);

static void
local_ip_to_route_ip(void *cpssp, const unsigned char *buf, unsigned int len)
{
	route_ip_from_local_ip(cpssp, buf, len);
}

static void
local_ip_to_udp(struct cpssp *cpssp, const unsigned char *buf, unsigned int len)
{
	udp_from_local_ip(cpssp, buf, len);
}

static void
local_ip_to_tcp(struct cpssp *cpssp, const unsigned char *buf, unsigned int len)
{
	tcp_from_local_ip(cpssp, buf, len);
}

#define BEHAVIOR
#define NAME		local_ip
#define NAME_(x)	local_ip_ ## x
#include "arch_local_ip.c"
#undef NAME_
#undef NAME
#undef BEHAVIOR

/* ------------------------------------------------------------------ */
/* slirp ip */

/*forward*/ static void
route_ip_from_default_ip(
	struct cpssp *cpssp,
	const unsigned char *buf,
	unsigned int len
);

static void
slirp_recv(void *cpssp, unsigned char *buf, int len)
{
	route_ip_from_default_ip(cpssp, buf, len);
}

#define BEHAVIOR
#define NAME		slirp
#define NAME_(x)	slirp_ ## x
#include "arch_slirp.c"
#undef NAME_
#undef NAME
#undef BEHAVIOR

/* ------------------------------------------------------------------ */

/*forward*/ static void
arp_from_route_ip(
	struct cpssp *cpssp,
	const unsigned char *buf,
	unsigned int len
);

static void
route_ip_to_arp(void *cpssp, const unsigned char *buf, unsigned int len)
{
	arp_from_route_ip(cpssp, buf, len);
}

static void
route_ip_to_local_ip(void *cpssp, const unsigned char *buf, unsigned int len)
{
	local_ip_from_route_ip(cpssp, buf, len);
}

static void
route_ip_to_default_ip(void *cpssp, const unsigned char *buf, unsigned int len)
{
	slirp_send(cpssp, buf, len);
}

#define BEHAVIOR
#define NAME		route_ip
#define NAME_(x)	route_ip_ ## x
#include "arch_route_ip.c"
#undef NAME_
#undef NAME
#undef BEHAVIOR

/* ------------------------------------------------------------------ */
/* arp layer for eth interface -- handle lookup of mac/ip */

/*forward*/ static void
eth_from_arp(struct cpssp *cpssp, const unsigned char *buf, int len);

static void
arp_to_eth(struct cpssp *cpssp, const unsigned char *buf, int len)
{
	eth_from_arp(cpssp, buf, len);
}

#define BEHAVIOR
#define NAME		arp
#define NAME_(x)	arp_ ## x
#include "arch_arp.c"
#undef NAME_
#undef NAME
#undef BEHAVIOR

/* ------------------------------------------------------------------ */
/* eth layer -- low level packet handling */

/*forward*/ static void
COMP_(send)(
	void *cpssp,
	const unsigned char *buf,
	unsigned int len
);

static void
eth_to_net(struct cpssp *cpssp, const unsigned char *buf, int len)
{
	COMP_(send)(cpssp, buf, len);
}

static void
eth_to_arp(void *cpssp, const unsigned char *buf, unsigned int len)
{
	arp_from_eth(cpssp, buf, len);
}

#define BEHAVIOR
#define NAME		eth
#define NAME_(x)	eth_ ## x
#include "arch_eth_iface.c"
#undef NAME_
#undef NAME
#undef BEHAVIOR

/* ------------------------------------------------------------------ */
/* interface to main.c */

static void
COMP_(recv)(void *cpssp, const void *buf, unsigned int bufsize)
{
	eth_from_net(cpssp, buf, bufsize);
}

static void
COMP_(send)(
	void *_cpssp,
	const unsigned char *buf,
	unsigned int bufsize
)
{
	struct cpssp *cpssp = _cpssp;

	if (bufsize < 60) {
		static unsigned char b[60];

		memcpy(b, buf, bufsize);
		buf = b;
		bufsize = 60;
	}
	sig_eth_send(cpssp->port_eth, cpssp, buf, bufsize);
}

void *
COMP_(create)(
	const char *name,
	const char *mac,
	const char *ip,
	const char *first_ip,
	const char *netmask,
	const char *broadcast,
	const char *default_router,
	const char *dns_server,
	struct sig_manage *manage,
	struct sig_serial *port_console,
	struct sig_eth_conn *port_eth
)
{
	static char mac_default[] = "00:01:02:03:04:01";
	static char ip_default[] = "10.0.0.1";
	static const struct sig_eth_funcs eth_funcs = {
		.recv = COMP_(recv),
	};
	struct cpssp *cpssp;
	uint8_t mac_bin[6];
	struct in_addr ip_bin;
	struct in_addr first_ip_bin;
	struct in_addr netmask_bin;
	struct in_addr broadcast_bin;
	struct in_addr default_router_bin;
	struct in_addr dns_server_bin;
	unsigned int i;

	if (! mac
	 || strcmp(mac, "00:00:00:00:00:00") == 0) {
		mac = mac_default;
	}
	if (! ip) {
		ip = ip_default;
	}
	if (! first_ip) {
		first_ip = "10.0.0.2";
	}
	if (! netmask) {
		netmask = "255.255.255.0";
	}
	if (! broadcast) {
		broadcast = "10.0.0.255";
	}
	if (! default_router) {
		default_router = "10.0.0.1";
	}
	if (! dns_server
	 || ! *dns_server) {
		/*
		 * Use a DNS server from /etc/resolv.conf as default.
		 */
		static char server[32];
		FILE *fp;
		char line[1024];
		int ret;

		fp = fopen("/etc/resolv.conf", "r");
		if (! fp) {
			fprintf(stderr, "Warning: Can't open \"/etc/resolv.conf\"!\n");
			strcpy(server, "131.188.3.73");
			goto done;
		} else {
			for (;;) {
				if (! fgets(line, sizeof(line), fp)) {
					ret = fclose(fp);
					assert(0 <= ret);
					fprintf(stderr, "Warning: Can't find nameserver entry in \"/etc/resolv.conf\"!\n");
					strcpy(server, "131.188.3.73");
					goto found;
				}
				if (strncmp(line, "nameserver", strlen("nameserver")) == 0) {
					/* FIXME */
					strcpy(server, line + strlen("nameserver") + 1);
					if (strchr(server, '\n')) {
						*strchr(server, '\n') = '\0';
					}
					break;
				}
			}
		found:	;
			ret = fclose(fp);
			assert(0 <= ret);
		}
	done:	;
		dns_server = server;
	}

	for (i = 0; i < 6; i++) {
		mac_bin[i] = strtoul(&mac[i * 3], NULL, 16);
	}

	if (DEBUG) {
		fprintf(stderr, "%s:\n", __FUNCTION__);
		fprintf(stderr, "ip=%s\n", ip);
		fprintf(stderr, "first_ip=%s\n", first_ip);
		fprintf(stderr, "netmask=%s\n", netmask);
		fprintf(stderr, "broadcast=%s\n", broadcast);
		fprintf(stderr, "default_router=%s\n", default_router);
		fprintf(stderr, "dns_server=%s\n", dns_server);
	}

	inet_aton(ip, &ip_bin);
	inet_aton(first_ip, &first_ip_bin);
	inet_aton(netmask, &netmask_bin);
	inet_aton(broadcast, &broadcast_bin);
	inet_aton(default_router, &default_router_bin);
	inet_aton(dns_server, &dns_server_bin);

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);
	memset(cpssp, 0, sizeof(*cpssp)); /* FIXME */

	eth_create(cpssp, mac_bin);
	arp_create(cpssp, mac_bin, ntohl(ip_bin.s_addr));
	route_ip_create(cpssp, ntohl(ip_bin.s_addr), ntohl(netmask_bin.s_addr),
			ntohl(broadcast_bin.s_addr));
	local_ip_create(cpssp);
	tcp_create(cpssp);
	udp_create(cpssp);
	slirp_create(cpssp);
	dhcp_create(cpssp,
		ntohl(ip_bin.s_addr),
		ntohl(first_ip_bin.s_addr), ntohl(netmask_bin.s_addr),
		ntohl(broadcast_bin.s_addr), ntohl(default_router_bin.s_addr),
		ntohl(dns_server_bin.s_addr));

	/* Call */
	cpssp->port_eth = port_eth->inout;
	sig_eth_connect(port_eth->inout, cpssp, &eth_funcs);

	if (ip == ip_default) {
		ip_default[sizeof(ip_default) - 2]++;
	}
	if (mac == mac_default) {
		mac_default[sizeof(mac_default) - 2]++;
	}

	return cpssp;
}

void
COMP_(destroy)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	dhcp_destroy(cpssp);
	slirp_destroy(cpssp);
	udp_destroy(cpssp);
	tcp_destroy(cpssp);
	local_ip_destroy(cpssp);
	route_ip_destroy(cpssp);
	arp_destroy(cpssp);
	eth_destroy(cpssp);

	shm_free(cpssp);
}

void
COMP_(suspend)(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;
	
	generic_suspend(cpssp, sizeof(*cpssp), fComp);
}

void
COMP_(resume)(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;
	
	int saveslirp_fd = cpssp->slirp.slirp_fd;
	pid_t saveslirp_pid = cpssp->slirp.slirp_pid;
	pid_t savegrep_pid = cpssp->slirp.grep_pid;
	
	generic_resume(cpssp, sizeof(*cpssp), fComp);
	
	cpssp->slirp.slirp_fd = saveslirp_fd;
	cpssp->slirp.slirp_pid = saveslirp_pid;
	cpssp->slirp.grep_pid = savegrep_pid;
}
