/*
 * $Id: mb_ibm_at_reset.c,v 1.15 2013-04-05 11:12:50 vrsieh Exp $ 
 *
 * Copyright (C) 2008-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>

#include "glue.h"

#include "mb_ibm_at_reset.h"

struct cpssp {
	/* Ports */
	struct sig_std_logic *port_n_button;
	unsigned int state_n_button;
	unsigned int state_n_kbc_reset;
	struct sig_std_logic *port_n_reset;
	
	/* State */
	unsigned int reset_triggered;
};

static void
mb_ibm_at_reset_reset_update(struct cpssp *cpssp)
{
	unsigned int reset;

	reset = cpssp->reset_triggered
		| ! cpssp->state_n_button
		| ! cpssp->state_n_kbc_reset;

	sig_std_logic_or_set(cpssp->port_n_reset, cpssp, ! reset);
}

static void
mb_ibm_at_reset_event(void *_cpssp)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	cpssp->reset_triggered = 0;
	mb_ibm_at_reset_reset_update(cpssp);
}

static void
mb_ibm_at_reset_power_set(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	sig_std_logic_set(cpssp->port_n_button, cpssp,
			val ? SIG_STD_LOGIC_H : SIG_STD_LOGIC_L);

	if (val) {
		cpssp->reset_triggered = 1;
		mb_ibm_at_reset_reset_update(cpssp);
		time_call_after(TIME_HZ / 10, mb_ibm_at_reset_event, cpssp);
	} else {
		/* Remove timer - FIXME */
	}
}

static void
mb_ibm_at_reset_n_button_set(void *_cpssp, unsigned int n_val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	switch (n_val) {
	case SIG_STD_LOGIC_0:
	case SIG_STD_LOGIC_L: n_val = 0; break;
	case SIG_STD_LOGIC_1:
	case SIG_STD_LOGIC_H: n_val = 1; break;
	default: /* FIXME */ return;
	}

	cpssp->state_n_button = n_val;
	mb_ibm_at_reset_reset_update(cpssp);
}

static void
mb_ibm_at_reset_n_kbc_reset_set(void *_cpssp, unsigned int n_val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;
	
	cpssp->state_n_kbc_reset = n_val;
	mb_ibm_at_reset_reset_update(cpssp);
}

void *
mb_ibm_at_reset_create(
	const char *name,
	struct sig_manage *port_manage,
	struct sig_std_logic *port_power,
	struct sig_std_logic *port_button_hash_,
	struct sig_boolean *port_kbc_reset_hash_,
	struct sig_std_logic *port_reset_hash_
)
{
	static const struct sig_std_logic_funcs power_funcs = {
		.boolean_or_set = mb_ibm_at_reset_power_set,
	};
	static const struct sig_std_logic_funcs n_button_funcs = {
		.std_logic_set = mb_ibm_at_reset_n_button_set,
	};
	static const struct sig_boolean_funcs n_kbc_reset_funcs = {
		.set = mb_ibm_at_reset_n_kbc_reset_set,
	};
	struct cpssp *cpssp;

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);

	/* Out */
	cpssp->port_n_button = port_button_hash_;
	cpssp->state_n_button = 0;
	sig_std_logic_connect_out(port_button_hash_, cpssp, SIG_STD_LOGIC_L);

	cpssp->port_n_reset = port_reset_hash_;
	sig_std_logic_connect_out(port_reset_hash_, cpssp, SIG_STD_LOGIC_0);

	/* In */
	sig_std_logic_connect_in(port_power, cpssp, &power_funcs);
	sig_std_logic_connect_in(port_button_hash_, cpssp, &n_button_funcs);
	sig_boolean_connect_in(port_kbc_reset_hash_, cpssp, &n_kbc_reset_funcs);

	return cpssp;
}

void
mb_ibm_at_reset_destroy(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	shm_free(cpssp);
}

void
mb_ibm_at_reset_suspend(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;
	
	generic_suspend(cpssp, sizeof(*cpssp), fComp);
}

void
mb_ibm_at_reset_resume(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;
	
	generic_resume(cpssp, sizeof(*cpssp), fComp);
}
