/*
 * Copyright (C) 2015 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <math.h>
#include <stdio.h>

#include "glue.h"

#include "harvester.h"

#define COMP_(x)	harvester_ ## x

#define CAPACITY	1.0 /* F */
#define MIN_VOLTAGE	3.5 /* V */
#define MAX_VOLTAGE	5.0 /* V */

#define FREQ		128

struct cpssp {
	/* Config */
	double capacity;

	/* Ports */
	struct sig_std_logic *port_voltage;
	struct sig_std_logic *port_output;
	struct sig_std_logic *port_input;
	unsigned int state_input_voltage;
	unsigned int state_input_current;

	/* State */
	double energy;

	unsigned long long time;
};

static void
COMP_(input_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	cpssp->state_input_voltage = SIG_mV(val);
	cpssp->state_input_current = SIG_mA(val);
}

static void
COMP_(event)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;
	double e1, e2, ein, eout;
	double v;

	/* Calculate energy we could accumulate in our capacitor. */
	e1 = 0.5 * cpssp->capacity * MAX_VOLTAGE * MAX_VOLTAGE - cpssp->energy;
	assert(0.0 <= e1);

	/* Calculate energy we could obtain from input. */
	e2 = (double) cpssp->state_input_voltage / 1000.0
			* (double) cpssp->state_input_current / 1000.0
			/ FREQ;

	/* Calculate energy we obtain and accumulate. */
	if (e1 < e2) {
		ein = e1;
	} else {
		ein = e2;
	}

	/* Consume energy from input and fill up capacitor. */
	cpssp->energy += ein;
	sig_std_logic_consume(cpssp->port_input, cpssp, ein);

	/* Consumed energy */
	eout = sig_std_logic_supply(cpssp->port_output, cpssp);

{
	unsigned long long t;
	t = time_virt();
	fprintf(stderr, "ENERGY %lu.%03lu %e %e %e\n",
			t / TIME_HZ, (t % TIME_HZ) * 1000 / (1ULL<<32),
			cpssp->energy, ein, eout);
}

	/* Supply output with energy for last interval. */
	cpssp->energy -= eout;

	/* Calculate voltage output. */
	v = sqrt(cpssp->energy / 0.5 / cpssp->capacity);
	/* Voltage Divider */
	v /= 2.0;
	assert(-0.001 <= v && v <= MAX_VOLTAGE + 0.001);
	sig_std_logic_set(cpssp->port_voltage, cpssp,
			SIG_COMB((int) (v * 1000.0 + 0.5), -1));

	/* Set output for next interval. */
	if (cpssp->energy < 0.5 * cpssp->capacity * MIN_VOLTAGE * MIN_VOLTAGE) {
		/* If energy is low switch off output. */
		sig_std_logic_set(cpssp->port_output, cpssp, SIG_STD_LOGIC_0);
	} else {
		/* If energy is high switch on output. */
		sig_std_logic_set(cpssp->port_output, cpssp, SIG_COMB(3300, -1));
	}

	cpssp->time += TIME_HZ / FREQ;
	time_call_at(cpssp->time, COMP_(event), cpssp);
}

void *
COMP_(create)(
	const char *name,
	const char *capacity,
	const char *energy,
	struct sig_manage *port_manage,
	struct sig_std_logic *port_voltage,
	struct sig_std_logic *port_output,
	struct sig_std_logic *port_input
)
{
	static const struct sig_std_logic_funcs input_funcs = {
		.std_logic_set = COMP_(input_set),
	};
	struct cpssp *cpssp;

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);

	if (capacity) {
		cpssp->capacity = atof(capacity) / 1000.0;
	} else {
		cpssp->capacity = CAPACITY;
	}

	if (energy) {
		cpssp->energy = atof(energy) / 1000.0;
	} else {
		cpssp->energy = 0.0;
	}

	/* Out */
	cpssp->port_voltage = port_voltage;
	sig_std_logic_connect_out(port_voltage, cpssp, SIG_STD_LOGIC_0);
	cpssp->port_output = port_output;
	sig_std_logic_connect_out(port_output, cpssp, SIG_STD_LOGIC_0);

	/* In */
	cpssp->state_input_voltage = 0;
	cpssp->state_input_current = 0;
	cpssp->port_input = port_input;
	sig_std_logic_connect_in(port_input, cpssp, &input_funcs);

	/* Time */
	cpssp->time = 0;
	time_call_at(cpssp->time, COMP_(event), cpssp);

	return cpssp;
}

void
COMP_(destroy)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	shm_free(cpssp);
}

void
COMP_(suspend)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;

	generic_suspend(cpssp, sizeof(*cpssp), fp);
}

void
COMP_(resume)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;

	generic_resume(cpssp, sizeof(*cpssp), fp);
}
