/*
 * $Id: chassis.c,v 1.46 2013-05-10 21:02:10 vrsieh Exp $ 
 *
 * Copyright (C) 2006-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "glue.h"

#include "chassis.h"

#define COMP "chassis"

struct cpssp {
	struct sig_std_logic *port_n_power_button;

	struct sig_std_logic *port_n_reset_button;

	struct sig_boolean *port_opt_power_led;

	struct sig_boolean *port_opt_ide_led;

	struct sig_sound *port_speaker;
	struct sig_sound *port_audio_speaker;
};

static void
chassis_power_button_set(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (val) {
		sig_std_logic_set(cpssp->port_n_power_button, cpssp, SIG_STD_LOGIC_0);
	} else {
		sig_std_logic_set(cpssp->port_n_power_button, cpssp, SIG_STD_LOGIC_Z);
	}
}

static void
chassis_reset_button_set(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (val) {
		sig_std_logic_set(cpssp->port_n_reset_button, cpssp, SIG_STD_LOGIC_0);
	} else {
		sig_std_logic_set(cpssp->port_n_reset_button, cpssp, SIG_STD_LOGIC_Z);
	}
}

static void
chassis_power_led_set(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	sig_boolean_set(cpssp->port_opt_power_led, cpssp, val);
}

static void
chassis_ide_led_set(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	sig_boolean_set(cpssp->port_opt_ide_led, cpssp, val);
}

static void
chassis_speaker_samples_set(void *_cpssp, int16_t *buf)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	sig_sound_samples_set(cpssp->port_audio_speaker, cpssp, buf);
}

void *
chassis_create(
	const char *name,
	struct sig_manage *port_manage,
	struct sig_boolean *port_mech_power_button,
	struct sig_std_logic *port_n_power_button,
	struct sig_boolean *port_mech_reset_button,
	struct sig_std_logic *port_n_reset_button,
	struct sig_std_logic_conn *port_power_led,
	struct sig_boolean *port_opt_power_led,
	struct sig_std_logic *port_ide_led,
	struct sig_boolean *port_opt_ide_led,
	struct sig_sound *port_speaker,
	struct sig_sound *port_audio_speaker
)
{
	static const struct sig_boolean_funcs power_button_funcs = {
		.set = chassis_power_button_set,
	};
	static const struct sig_boolean_funcs reset_button_funcs = {
		.set = chassis_reset_button_set,
	};
	static const struct sig_std_logic_funcs power_led_funcs = {
		.boolean_or_set = chassis_power_led_set,
	};
	static const struct sig_std_logic_funcs ide_led_funcs = {
		.boolean_or_set = chassis_ide_led_set,
	};
	static const struct sig_sound_funcs speaker_funcs = {
		.samples_set = chassis_speaker_samples_set,
	};
	struct cpssp *cpssp;

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);

	/* Call */
	/* Out */
	cpssp->port_n_power_button = port_n_power_button;
	sig_std_logic_connect_out(port_n_power_button, cpssp, SIG_STD_LOGIC_Z);

	cpssp->port_n_reset_button = port_n_reset_button;
	sig_std_logic_connect_out(port_n_reset_button, cpssp, SIG_STD_LOGIC_Z);

	cpssp->port_opt_power_led = port_opt_power_led;
	sig_boolean_connect_out(port_opt_power_led, cpssp, 0);

	cpssp->port_opt_ide_led = port_opt_ide_led;
	sig_boolean_connect_out(port_opt_ide_led, cpssp, 0);

	cpssp->port_audio_speaker = port_audio_speaker;

	/* In */
	sig_boolean_connect_in(port_mech_power_button, cpssp, &power_button_funcs);

	sig_boolean_connect_in(port_mech_reset_button, cpssp, &reset_button_funcs);

	sig_std_logic_connect_in(port_power_led->sig, cpssp, &power_led_funcs);

	sig_std_logic_connect_in(port_ide_led, cpssp, &ide_led_funcs);

	cpssp->port_speaker = port_speaker;
	sig_sound_connect(port_speaker, cpssp, &speaker_funcs);

	return cpssp;
}

void
chassis_destroy(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	shm_free(cpssp);
}

void
chassis_suspend(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;
	
	generic_suspend(cpssp, sizeof(*cpssp), fComp);
}

void
chassis_resume(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;
	
	generic_resume(cpssp, sizeof(*cpssp), fComp);
}
