/*
 * $Id: arch_sata_ata.c,v 1.24 2013/06/24 12:07:28 vrsieh Exp $ 
 *
 * Copyright (C) 2013 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifdef STATE

struct {
	enum {
		STATE_IDLE,
		STATE_FILLED,
		STATE_BUSY,
		STATE_DMA_IN,
		STATE_DMA_OUT,
		STATE_PIO_IN,
		STATE_PIO_OUT,
	} state;

	unsigned int irq;

	unsigned int count;
} NAME;

#endif /* STATE */

#ifdef BEHAVIOR

static void
NAME_(dmarq_set)(struct cpssp *cpssp, unsigned int val)
{
	NAME_(dmarq_out_set)(cpssp, val);
}

static void
NAME_(irqrq_update)(struct cpssp *cpssp)
{
	unsigned int val;

	if (cpssp->NAME.irq
	 && ((cpssp->common.control >> 1) & 1) == 0) {
		val = 1;
	} else {
		val = 0;
	}

	NAME_(irqrq_out_set)(cpssp, val);
}

static void
NAME_(irqrq_set)(struct cpssp *cpssp, unsigned int val)
{
	cpssp->NAME.irq = val;

	NAME_(irqrq_update)(cpssp);
}

static void
NAME_(dma_out)(struct cpssp *cpssp, unsigned int count)
{
	if (DEBUG) {
		fprintf(stderr, "%s/%p\n", __FUNCTION__, cpssp);
	}

	cpssp->NAME.state = STATE_DMA_OUT;
	cpssp->NAME.count = count;

	NAME_(dmarq_set)(cpssp, 1);
}

static void
NAME_(dma_in)(struct cpssp *cpssp, unsigned int count)
{
	if (DEBUG) {
		fprintf(stderr, "%s/%p\n", __FUNCTION__, cpssp);
	}

	cpssp->NAME.state = STATE_DMA_IN;
	cpssp->NAME.count = count;

	NAME_(dmarq_set)(cpssp, 1);
}

static void
NAME_(pio_out)(struct cpssp *cpssp, unsigned int count, int irq)
{
	if (DEBUG) {
		fprintf(stderr, "%s/%p: irq=%d\n", __FUNCTION__, cpssp, irq);
	}

	cpssp->NAME.state = STATE_PIO_OUT;
	cpssp->NAME.count = count;

	if (irq) {
		NAME_(irqrq_set)(cpssp, 1);
	}
}

static void
NAME_(pio_in)(struct cpssp *cpssp, unsigned int count, int irq)
{
	if (DEBUG) {
		fprintf(stderr, "%s/%p: irq=%d\n", __FUNCTION__, cpssp, irq);
	}

	cpssp->NAME.state = STATE_PIO_IN;
	cpssp->NAME.count = count;

	if (irq) {
		NAME_(irqrq_set)(cpssp, 1);
	}
}

static void
NAME_(done_io)(struct cpssp *cpssp)
{
	if (DEBUG) {
		fprintf(stderr, "%s/%p\n", __FUNCTION__, cpssp);
	}

	switch (cpssp->NAME.state) {
	case STATE_IDLE:
		assert(0);
	case STATE_FILLED:
		cpssp->NAME.state = STATE_IDLE;
		break;
	case STATE_BUSY:
		assert(0);
	case STATE_DMA_IN:
	case STATE_DMA_OUT:
	case STATE_PIO_IN:
	case STATE_PIO_OUT:
		cpssp->NAME.state = STATE_BUSY;
		break;
	}
}

static void
NAME_(done_cmd)(struct cpssp *cpssp, int irq)
{
	if (DEBUG) {
		fprintf(stderr, "%s/%p: irq=%d\n", __FUNCTION__, cpssp, irq);
	}

	switch (cpssp->NAME.state) {
	case STATE_IDLE:
		cpssp->NAME.state = STATE_IDLE; /* In case of reset. */
		break;
	case STATE_FILLED:
		cpssp->NAME.state = STATE_IDLE; /* In case of reset. */
		break;
	case STATE_BUSY:
		cpssp->NAME.state = STATE_IDLE;
		break;
	case STATE_DMA_IN:
	case STATE_DMA_OUT:
		assert(0);
	case STATE_PIO_IN:
		cpssp->NAME.state = STATE_FILLED;
		break;
	case STATE_PIO_OUT:
		assert(0);
	}

	if (irq) {
		NAME_(irqrq_set)(cpssp, 1);
	}
}

static uint8_t
NAME_(status)(struct cpssp *cpssp)
{
	uint8_t status;

	switch (cpssp->NAME.state) {
	case STATE_IDLE:
		status = READY_STAT;
		break;
	case STATE_FILLED:
		status = READY_STAT | DRQ_STAT;
		break;
	case STATE_BUSY:
		status = BUSY_STAT;
		break;
	case STATE_DMA_IN:
	case STATE_DMA_OUT:
	case STATE_PIO_IN:
	case STATE_PIO_OUT:
		status = DRQ_STAT;
#if PACKET_INTERFACE == 0
		status |= READY_STAT;
#endif
	}
	switch (cpssp->NAME.state) {
	case STATE_IDLE:
	case STATE_FILLED:
		if (cpssp->common.error & ~1) {
			status |= ERR_STAT;
		}
		break;
	case STATE_BUSY:
	case STATE_DMA_IN:
	case STATE_DMA_OUT:
	case STATE_PIO_IN:
	case STATE_PIO_OUT:
		break;
	}
#if PACKET_INTERFACE == 0
	status |= SEEK_STAT;
#endif

	return status;
}

static int
NAME_(inw)(struct cpssp *cpssp, unsigned short port, uint16_t *valp)
{
	switch (port) {
	case 0:
		switch (cpssp->NAME.state) {
		case STATE_PIO_IN:
		case STATE_FILLED:
			NAME_(data_in)(cpssp, valp);
			break;
		case STATE_IDLE:
		case STATE_BUSY:
		case STATE_DMA_IN:
		case STATE_DMA_OUT:
		case STATE_PIO_OUT:
#if 0
			fprintf(stderr, "WARNING: %s: state=%d.\n",
					__FUNCTION__, cpssp->NAME.state);
#endif
			*valp = 0;
			break;
		}
		break;
	case 1:
		*valp = cpssp->common.error;
		break;
	case 2:
		*valp = cpssp->common.nsector;
		break;
	case 3:
		*valp = cpssp->common.sector;
		break;
	case 4:
		*valp = (cpssp->common.cyl >> 0) & 0xff;
		break;
	case 5:
		*valp = (cpssp->common.cyl >> 8) & 0xff;
		break;
	case 6:
		*valp = cpssp->common.select | 0xa0;
		break;
	case 7:
		*valp = NAME_(status)(cpssp);
		NAME_(irqrq_set)(cpssp, 0);
		break;
	case 8 + 6:
		*valp = NAME_(status)(cpssp);
		break;
	default:
		assert(0);
	}

	if (DEBUG) {
		fprintf(stderr, "%s/%p: port=0x%x, val=0x%04x\n",
				__FUNCTION__, cpssp, port, *valp);
	}

	return 0;
}

static void
NAME_(outw)(struct cpssp *cpssp, unsigned short port, uint16_t val)
{
	if (DEBUG) {
		fprintf(stderr, "%s/%p: port=0x%x, val=0x%04x\n",
				__FUNCTION__, cpssp, port, val);
	}

	switch (port) {
	case 0: /* See below. */
		break;
	case 1:
		cpssp->common.features = val;
		return;
	case 2:
		cpssp->common.nsector = val;
		return;
	case 3:
		cpssp->common.sector = val;
		return;
	case 4:
		cpssp->common.cyl
			= (cpssp->common.cyl & 0xff00) | (val << 0);
		return;
	case 5:
		cpssp->common.cyl
			= (val << 8) | (cpssp->common.cyl & 0x00ff);
		return;
	case 6:
		cpssp->common.select = val & ~0xa0;
		NAME_(irqrq_update)(cpssp);
		return;
	case 7:
		break; /* See below. */
	case 8 + 6:
		cpssp->common.control = val;
		if (val & 0x04) {
			NAME_(soft_reset)(cpssp);
		}
		return;
	}

	switch (port) {
	case 0:
		switch (cpssp->NAME.state) {
		case STATE_IDLE:
		case STATE_FILLED:
		case STATE_BUSY:
		case STATE_DMA_IN:
		case STATE_DMA_OUT:
		case STATE_PIO_IN:
			fprintf(stderr, "WARNING: %s: state=%d.\n",
					__FUNCTION__, cpssp->NAME.state);
			break;
		case STATE_PIO_OUT:
			NAME_(data_out)(cpssp, val);
			break;
		}
		break;
	case 7:
		switch (cpssp->NAME.state) {
		case STATE_IDLE:
		case STATE_FILLED:
		case STATE_PIO_IN:
		case STATE_PIO_OUT:
			cpssp->NAME.state = STATE_BUSY;
			NAME_(irqrq_set)(cpssp, 0);
			cpssp->common.command = val;
			NAME_(command)(cpssp);
			break;
		case STATE_BUSY:
		case STATE_DMA_IN:
		case STATE_DMA_OUT:
			fprintf(stderr, "WARNING: %s: state=%d.\n",
					__FUNCTION__, cpssp->NAME.state);
			break;
		}
		break;
	default:
		assert(0);
	}
}

static int
NAME_(dmainw)(struct cpssp *cpssp, uint16_t *valp)
{
	NAME_(data_in)(cpssp, valp);
	if (cpssp->NAME.state != STATE_DMA_IN) {
		NAME_(dmarq_set)(cpssp, 0);
	}

	if (DEBUG) {
		fprintf(stderr, "%s/%p: val=0x%04x\n",
				__FUNCTION__, cpssp, *valp);
	}

	return 0;
}

static void
NAME_(dmaoutw)(struct cpssp *cpssp, uint16_t val)
{
	if (DEBUG) {
		fprintf(stderr, "%s/%p: val=0x%04x\n",
				__FUNCTION__, cpssp, val);
	}

	NAME_(data_out)(cpssp, val);
	if (cpssp->NAME.state != STATE_DMA_OUT) {
		NAME_(dmarq_set)(cpssp, 0);
	}
}

static void
NAME_(power_set)(struct cpssp *cpssp, unsigned int val)
{
}

static void
NAME_(reset_set)(struct cpssp *cpssp, unsigned int val)
{
	cpssp->NAME.state = STATE_IDLE;
	cpssp->NAME.irq = 0;
	NAME_(irqrq_update)(cpssp);
}

static void
NAME_(create)(struct cpssp *cpssp)
{
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
}

static void
NAME_(suspend)(struct cpssp *cpssp, FILE *fp)
{
	generic_suspend(&cpssp->NAME, sizeof(cpssp->NAME), fp);
}

static void
NAME_(resume)(struct cpssp *cpssp, FILE *fp)
{
	generic_resume(&cpssp->NAME, sizeof(cpssp->NAME), fp);
}

#endif /* BEHAVIOR */
