/* $Id: arch_gen_mouse.c,v 1.1 2013-05-13 18:03:35 vrsieh Exp $ 
 *
 * Copyright (C) 2008-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#if !defined(MOUSE_TYPE) || (MOUSE_TYPE != 0 && MOUSE_TYPE != 3 && MOUSE_TYPE != 4)
#error MOUSE_TYPE must be one of 0, 3, 4
#endif

#ifdef STATE

struct {
	int state_clkrunning;

	unsigned int state_button1;
	unsigned int state_button2;
	unsigned int state_button3;
	unsigned int state_button4;
	unsigned int state_button5;

	int change_x;
	int change_y;
	int change_z;
	unsigned int change_button1;
	unsigned int change_button2;
	unsigned int change_button3;
	unsigned int change_button4;
	unsigned int change_button5;
	unsigned int latest_button1;
	unsigned int latest_button2;
	unsigned int latest_button3;
	unsigned int latest_button4;
	unsigned int latest_button5;

	uint8_t outbuf[32];
	unsigned int outhead;
	unsigned int outtail;
	unsigned int outcount;
	unsigned int outrunning;

	uint8_t command;

#define MOUSE_STATUS_REMOTE	0x40
#define MOUSE_STATUS_ENABLED	0x20
#define MOUSE_STATUS_SCALE21	0x10
	uint8_t status;

	uint8_t resolution;
	uint8_t sample_rate;
	uint8_t wrap;

#define TYPE_PS2		0
#define TYPE_WHEEL		3
#define TYPE_WHEEL_5BUTTON	4
	uint8_t active_type;

#define STATE_NONE	0
#define STATE_200	1
#define STATE_200_100	2
#define STATE_200_200	3
	uint8_t detect_state;
} NAME;

#endif /* STATE */

#ifdef BEHAVIOUR

#include <assert.h>
#include <stdio.h>
#include <string.h>

#include "glue-log.h"
#include "glue-main.h"

static void
NAME_(send2)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;
	uint8_t byte;

	assert(cpssp->NAME.outrunning);
	assert(cpssp->NAME.outcount);

	byte = cpssp->NAME.outbuf[cpssp->NAME.outtail];
	cpssp->NAME.outtail = (cpssp->NAME.outtail + 1)
			% sizeof(cpssp->NAME.outbuf);
	cpssp->NAME.outcount--;
	cpssp->NAME.outrunning = 0;

	NAME_(send)(cpssp, byte);

	if (cpssp->NAME.outcount
	 && cpssp->NAME.state_clkrunning) {
		/* Sending/receiving a byte using 9600 Baud takes about 1ms. */
		cpssp->NAME.outrunning = 1;
		time_call_after(TIME_HZ / 1000, NAME_(send2), cpssp);
	}
}

static void
NAME_(send1)(struct cpssp *cpssp, uint8_t byte)
{
	if (! cpssp->NAME.outcount
	 && cpssp->NAME.state_clkrunning) {
		/* Sending/receiving a byte using 9600 Baud takes about 1ms. */
		cpssp->NAME.outrunning = 1;
		time_call_after(TIME_HZ / 1000, NAME_(send2), cpssp);
	}

	cpssp->NAME.outbuf[cpssp->NAME.outhead] = byte;
	cpssp->NAME.outhead = (cpssp->NAME.outhead + 1)
			% sizeof(cpssp->NAME.outbuf);
	cpssp->NAME.outcount++;
}

/*
 * FIXME: we are sending valid data packets
 * even in case of overflows; this seems to
 * work just fine, but maybe it would be better
 * to send packets with overflow bits set!?!
 */
static void
NAME_(send_packet)(
	struct cpssp *cpssp,
	unsigned int button0,
	unsigned int button1,
	unsigned int button2,
	unsigned int button3,
	unsigned int button4,
	int wheel0,
	int wheel1,
	int wheel2
)
{
	uint8_t packet[4] = { 0, 0, 0, 0 };

	packet[0] = (1 << 3); /* Always '1' */
	if (button0) packet[0] |= (1 << 0);
	if (button1) packet[0] |= (1 << 2);
	if (button2) packet[0] |= (1 << 1);

	if (wheel0 < 0) packet[0] |= (1 << 4); /* sign bit is in byte 0 */
	if (wheel0 < -256 || wheel0 > 255) {
		/* overflow: assume highest possible abs. value */
		packet[1] = (wheel0 < 0) ? 0 : 0xff;
	} else {
		packet[1] = wheel0 & 0xff;
	}

	if (wheel1 < 0) packet[0] |= (1 << 5); /* sign bit is in byte 0 */
	if (wheel1 < -256 || wheel1 > 255) {
		/* overflow: see above */
		packet[2] = (wheel1 < 0) ? 0 : 0xff;
	} else {
		packet[2] = wheel1 & 0xff;
	}

	if (cpssp->NAME.active_type == TYPE_WHEEL) {
		/* wheel mouse support only:
		 * fourth byte of response contains only z movement */
		if (wheel2 < -128 || wheel2 > 127) {
			/* overflow: see above */
			packet[3] = (wheel2 < 0) ? 0x80 : 0x7f;
		} else {
			packet[3] = wheel2 & 0xff;
		}

	} else if (cpssp->NAME.active_type == TYPE_WHEEL_5BUTTON) {
		/* wheel + 5 button mouse support:
		 * fourth byte contains z movement in lower 4 bits,
		 * buttons 4,5 in bits 4,5, resp. */

		if (wheel2 < -8 || wheel2 > 7) {
			/* overflow: see above */
			packet[3] |= (wheel2 < 0) ? 0x08 : 0x07;
		} else {
			packet[3] |= wheel2 & 0x0f;
		}
		if (button3) packet[3] |= (1 << 4);
		if (button4) packet[3] |= (1 << 5);
	}

	/*
	 * Send bytes.
	 */
	NAME_(send1)(cpssp, packet[0]);
	NAME_(send1)(cpssp, packet[1]);
	NAME_(send1)(cpssp, packet[2]);
	if (cpssp->NAME.active_type == TYPE_WHEEL
	 || cpssp->NAME.active_type == TYPE_WHEEL_5BUTTON) {
		NAME_(send1)(cpssp, packet[3]);
	}
}

static void
NAME_(report)(struct cpssp *cpssp, int force_send)
{
	int do_send;
	int delta_x;
	int delta_y;
	int delta_z;

	do_send = cpssp->NAME.change_x
	       || cpssp->NAME.change_y
	       || cpssp->NAME.change_z;

	delta_x = cpssp->NAME.change_x;
	cpssp->NAME.change_x = 0;

	delta_y = cpssp->NAME.change_y;
	cpssp->NAME.change_y = 0;

	delta_z = cpssp->NAME.change_z;
	cpssp->NAME.change_z = 0;

	if (cpssp->NAME.change_button1) {
		cpssp->NAME.state_button1 ^= 1;
		cpssp->NAME.status ^= (1 << 2);
		cpssp->NAME.change_button1--;
		do_send = 1;
	}
	if (cpssp->NAME.change_button2) {
		cpssp->NAME.state_button2 ^= 1;
		cpssp->NAME.status ^= (1 << 0);
		cpssp->NAME.change_button2--;
		do_send = 1;
	}
	if (cpssp->NAME.change_button3) {
		cpssp->NAME.state_button3 ^= 1;
		cpssp->NAME.status ^= (1 << 1);
		cpssp->NAME.change_button3--;
		do_send = 1;
	}
	if (cpssp->NAME.change_button4) {
		cpssp->NAME.state_button4 ^= 1;
		cpssp->NAME.change_button4--;
		do_send = 1;
	}
	if (cpssp->NAME.change_button5) {
		cpssp->NAME.state_button5 ^= 1;
		cpssp->NAME.change_button5--;
		do_send = 1;
	}

	if (force_send
	 || (do_send
	  && (cpssp->NAME.status & MOUSE_STATUS_ENABLED))) {
		NAME_(send_packet)(
				cpssp,
				cpssp->NAME.state_button1,
				cpssp->NAME.state_button2,
				cpssp->NAME.state_button3,
				cpssp->NAME.state_button4,
				cpssp->NAME.state_button5,
				delta_x,
				delta_y,
				delta_z);
	}
}

static void
NAME_(set_sample_rate)(struct cpssp *cpssp, uint8_t rate)
{
	/* FIXME: this really happens but is not defined!!!
	 *        -> Ignore it for now */
	if(rate == 0) return;

	cpssp->NAME.sample_rate = rate;

	/*
	 * If rate is set to 200, 100, 80 mouse is switched
	 * to Wheel-mode (if supported).
	 *
	 * If rate is set to 200, 200, 80 mouse is switched
	 * to Wheel+5-Button-mode (if supported).
	 */
	switch (cpssp->NAME.detect_state) {
	case STATE_NONE:
		switch (rate) {
		case 200:
			cpssp->NAME.detect_state = STATE_200;
			break;
		default:
			cpssp->NAME.detect_state = STATE_NONE;
			break;
		}
		break;
	case STATE_200:
		switch (rate) {
		case 100:
			cpssp->NAME.detect_state = STATE_200_100;
			break;
		case 200:
			cpssp->NAME.detect_state = STATE_200_200;
			break;
		default:
			cpssp->NAME.detect_state = STATE_NONE;
			break;
		}
		break;
	case STATE_200_100:
		switch (rate) {
		case 80:
#if TYPE_WHEEL <= MOUSE_TYPE
			cpssp->NAME.active_type = TYPE_WHEEL;
#endif
			cpssp->NAME.detect_state = STATE_NONE;
			break;
		default:
			cpssp->NAME.detect_state = STATE_NONE;
			break;
		}
		break;
	case STATE_200_200:
		switch (rate) {
		case 80:
#if MOUSE_TYPE == TYPE_WHEEL_5BUTTON
			cpssp->NAME.active_type = TYPE_WHEEL_5BUTTON;
#endif
			cpssp->NAME.detect_state = STATE_NONE;
			break;
		default:
			cpssp->NAME.detect_state = STATE_NONE;
			break;
		}
		break;
	}
}

static void
NAME_(reset_counters)(struct cpssp *cpssp)
{
	cpssp->NAME.change_x = 0;
	cpssp->NAME.change_y = 0;
	cpssp->NAME.change_z = 0;

	cpssp->NAME.change_button1 = 0;
	cpssp->NAME.change_button2 = 0;
	cpssp->NAME.change_button3 = 0;
	cpssp->NAME.change_button4 = 0;
	cpssp->NAME.change_button5 = 0;

	cpssp->NAME.state_button1 = cpssp->NAME.latest_button1;
	cpssp->NAME.state_button2 = cpssp->NAME.latest_button2;
	cpssp->NAME.state_button3 = cpssp->NAME.latest_button3;
	cpssp->NAME.state_button4 = cpssp->NAME.latest_button4;
	cpssp->NAME.state_button5 = cpssp->NAME.latest_button5;

	/* update button bits in status register */
	cpssp->NAME.status &= ~0x07;
	if(cpssp->NAME.state_button1) cpssp->NAME.status |= (1 << 2);
	if(cpssp->NAME.state_button2) cpssp->NAME.status |= (1 << 0);
	if(cpssp->NAME.state_button3) cpssp->NAME.status |= (1 << 1);
}

static void
NAME_(clk)(void *_cpssp)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	NAME_(report)(cpssp, 0);
	time_call_after(TIME_HZ / cpssp->NAME.sample_rate, NAME_(clk), cpssp);
}

static void
NAME_(handle_command)(struct cpssp *cpssp, uint8_t value)
{
	if (cpssp->NAME.command == 0
	 && cpssp->NAME.wrap
	 && value != 0xec       /* Reset wrap mode */
	 && value != 0xff) {    /* Reset */
		/*
		 * Wrap mode.
		 *
		 * Just echo byte.
		 */
		NAME_(send1)(cpssp, value);

	} else if (cpssp->NAME.command == 0) {
		/*
		 * Non-wrap mode.
		 * Command phase.
		 */
		NAME_(send1)(cpssp, 0xfa);      /* ACK */

		switch (value) {
		case 0xe6: /* Set 1:1 scaling. */
			cpssp->NAME.status &= ~MOUSE_STATUS_SCALE21;
			break;

		case 0xe7: /* Set 2:1 scaling */
			cpssp->NAME.status |= MOUSE_STATUS_SCALE21;
			break;

		case 0xe8: /* Set resolution. */
			cpssp->NAME.command = value;
			break;

		case 0xe9: /* Get extended status. */
			NAME_(send1)(cpssp, cpssp->NAME.status);
			NAME_(send1)(cpssp, cpssp->NAME.resolution);
			NAME_(send1)(cpssp, cpssp->NAME.sample_rate);
			NAME_(reset_counters)(cpssp);
			break;

		case 0xea: /* Set stream. */
			NAME_(reset_counters)(cpssp);
			cpssp->NAME.status &= ~MOUSE_STATUS_REMOTE;
			break;

		case 0xeb: /* Read data (poll). */
			NAME_(report)(cpssp,1);
			break;

		case 0xec: /* Reset wrap mode. */
			cpssp->NAME.wrap = 0;
			NAME_(reset_counters)(cpssp);
			if ((cpssp->NAME.status & MOUSE_STATUS_ENABLED)
					&& (cpssp->NAME.status & MOUSE_STATUS_REMOTE) == 0) {
				time_call_after(TIME_HZ / cpssp->NAME.sample_rate, NAME_(clk), cpssp);
			}
			break;

		case 0xee: /* Set wrap mode. */
			time_call_delete(NAME_(clk), cpssp);
			NAME_(reset_counters)(cpssp);
			cpssp->NAME.wrap = 1;
			break;

		case 0xf0: /* Set remote mode. */
			time_call_delete(NAME_(clk), cpssp);
			cpssp->NAME.status |= MOUSE_STATUS_REMOTE;
			cpssp->NAME.status &= ~MOUSE_STATUS_ENABLED;
			NAME_(report)(cpssp,1);
			NAME_(reset_counters)(cpssp);
			break;

		case 0xf2: /* Get type. */
			NAME_(reset_counters)(cpssp);
			NAME_(send1)(cpssp, (uint8_t) cpssp->NAME.active_type);
			break;

		case 0xf3: /* Set sample rate. */
			cpssp->NAME.command = value;
			break;

		case 0xf4: /* Enable data reporting. */
			NAME_(reset_counters)(cpssp);
			if((cpssp->NAME.status & MOUSE_STATUS_REMOTE) == 0
					&& (cpssp->NAME.status & MOUSE_STATUS_ENABLED) == 0) {
				cpssp->NAME.status |= MOUSE_STATUS_ENABLED;
				time_call_after(TIME_HZ / cpssp->NAME.sample_rate, NAME_(clk), cpssp);
			}
			break;

		case 0xf5: /* Disable data reporting. */
			time_call_delete(NAME_(clk), cpssp);
			NAME_(reset_counters)(cpssp);
			cpssp->NAME.status &= ~MOUSE_STATUS_ENABLED;
			break;

		case 0xf6: /* Set defaults. */
			time_call_delete(NAME_(clk), cpssp);
			cpssp->NAME.sample_rate = 100;
			cpssp->NAME.resolution = 2;
			cpssp->NAME.status = 0;
			NAME_(reset_counters)(cpssp);
			break;

		case 0xff: /* Reset. */
			time_call_delete(NAME_(clk), cpssp);
			cpssp->NAME.sample_rate = 100;
			cpssp->NAME.resolution = 2;
			cpssp->NAME.status = 0;
			cpssp->NAME.active_type = TYPE_PS2;
			cpssp->NAME.detect_state = STATE_NONE;
			NAME_(reset_counters)(cpssp);
			NAME_(send1)(cpssp, 0xaa);
			NAME_(send1)(cpssp, (uint8_t) cpssp->NAME.active_type);
			break;

		default:
			faum_log(FAUM_LOG_WARNING, SNAME, "",
				"%s: value=0x%02x\n", __FUNCTION__, value);
			break;
		}

	} else {
		/*
		 * Non-wrap mode.
		 * Parameter phase.
		 */
		NAME_(send1)(cpssp, 0xfa);      /* ACK */

		switch (cpssp->NAME.command) {
		case 0xe8: /* Set resolution. */
			cpssp->NAME.resolution = value;
			NAME_(reset_counters)(cpssp);
			break;

		case 0xf3: /* Set sample rate. */
			NAME_(set_sample_rate)(cpssp, value);
			NAME_(reset_counters)(cpssp);
			break;

		default:
			assert(0);
			/*NOTREACHED*/
		}

		cpssp->NAME.command = 0;
	}
}

static void
NAME_(power_set)(struct cpssp *cpssp, unsigned int val)
{
	/* FIXME */
}

static void
NAME_(recv)(struct cpssp *cpssp, uint8_t byte)
{
	NAME_(handle_command)(cpssp, byte);
}

static void
NAME_(clkrunning)(struct cpssp *cpssp, int val)
{
	cpssp->NAME.state_clkrunning = val;

	if (! cpssp->NAME.state_clkrunning) {
		if (cpssp->NAME.outrunning) {
			/* Cancel running transmission. */
			time_call_delete(NAME_(send2), cpssp);
			cpssp->NAME.outrunning = 0;
		}
	} else {
		if (! cpssp->NAME.outrunning
		 && cpssp->NAME.outcount) {
			/* Start new transmission. */
			cpssp->NAME.outrunning = 1;
			time_call_after(TIME_HZ / 1000, NAME_(send2), cpssp);
		}
	}
}

static void
NAME_(deltax_set)(struct cpssp *cpssp, int val)
{
	cpssp->NAME.change_x += val;
}

static void
NAME_(deltay_set)(struct cpssp *cpssp, int val)
{
	cpssp->NAME.change_y += val;
}

static void
NAME_(deltaz_set)(struct cpssp *cpssp, int val)
{
	cpssp->NAME.change_z += val;
}

static void
NAME_(button1_set)(struct cpssp *cpssp, unsigned int val)
{
	cpssp->NAME.latest_button1 = val;
	cpssp->NAME.change_button1++;
}

static void
NAME_(button2_set)(struct cpssp *cpssp, unsigned int val)
{
	cpssp->NAME.latest_button2 = val;
	cpssp->NAME.change_button2++;
}

static void
NAME_(button3_set)(struct cpssp *cpssp, unsigned int val)
{
	cpssp->NAME.latest_button3 = val;
	cpssp->NAME.change_button3++;
}

static void
NAME_(button4_set)(struct cpssp *cpssp, unsigned int val)
{
	cpssp->NAME.latest_button4 = val;
	cpssp->NAME.change_button4++;
}

static void
NAME_(button5_set)(struct cpssp *cpssp, unsigned int val)
{
	cpssp->NAME.latest_button5 = val;
	cpssp->NAME.change_button5++;
}

static void
NAME_(init)(struct cpssp *cpssp)
{
	memset(&cpssp->NAME, 0, sizeof(cpssp->NAME)); /* FIXME */

	cpssp->NAME.sample_rate = 100;
	cpssp->NAME.resolution = 2;
	cpssp->NAME.status = 0;
	cpssp->NAME.active_type = TYPE_PS2;
	cpssp->NAME.detect_state = STATE_NONE;
}

static void
NAME_(create)(struct cpssp *cpssp)
{
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
}

#endif /* BEHAVIOUR */
