/*
 * $Id: arch_gen_atapi.c,v 1.39 2013/06/24 09:40:26 vrsieh Exp $ 
 *
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/* This will turn on debug output. */
#define DEBUG_CONTROL_FLOW	0


#ifdef STATE

struct {
	/*
	 * State
	 */
	unsigned char pio_mode;
	unsigned char mdma_mode;
	unsigned char udma_mode;

#define SEND_MASK       (0 << 0) /* means: send me data */
#define COMMAND_MASK    (1 << 0) /* means: send me a command packet */
#define IN_MASK         (1 << 1) /* means: read result bytes */

#define nsector		nsector
#define irq_status	nsector
#define bytecount	cyl

	unsigned int cmd_flag;
	unsigned int data_flag;
	unsigned int msg_flag;
	unsigned int in_flag;
	unsigned int req_count;

	/*volatile*/ uint16_t buf[0x10000];
	volatile unsigned int head;
	volatile unsigned int tail;
	volatile unsigned int count;
} NAME;

#endif /* STATE */

#ifdef BEHAVIOR

static void
NAME_(phase_free)(struct cpssp *cpssp)
{
	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: called()\n", __FUNCTION__);
	}

	cpssp->common.irq_status = IN_MASK | COMMAND_MASK;
	NAME_(done_cmd)(cpssp, 1);
}

static void
NAME_(phase_cmd)(struct cpssp *cpssp)
{
	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: called()\n", __FUNCTION__);
	}

	cpssp->NAME.cmd_flag = 1;
	cpssp->NAME.data_flag = 0;
	cpssp->NAME.msg_flag = 0;
	cpssp->NAME.in_flag = 0;
}

static void
NAME_(phase_status)(struct cpssp *cpssp)
{
	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: called()\n", __FUNCTION__);
	}

	cpssp->NAME.cmd_flag = 1;
	cpssp->NAME.data_flag = 0;
	cpssp->NAME.msg_flag = 0;
	cpssp->NAME.in_flag = 1;
}

static void
NAME_(phase_data_in)(struct cpssp *cpssp)
{
	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: called()\n", __FUNCTION__);
	}

	cpssp->NAME.cmd_flag = 0;
	cpssp->NAME.data_flag = 1;
	cpssp->NAME.msg_flag = 0;
	cpssp->NAME.in_flag = 1;
}

static void
NAME_(phase_data_out)(struct cpssp *cpssp)
{
	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: called()\n", __FUNCTION__);
	}

	cpssp->NAME.cmd_flag = 0;
	cpssp->NAME.data_flag = 1;
	cpssp->NAME.msg_flag = 0;
	cpssp->NAME.in_flag = 0;
}

static void
NAME_(phase_msg_in)(struct cpssp *cpssp)
{
	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: called()\n", __FUNCTION__);
	}

	cpssp->NAME.cmd_flag = 0;
	cpssp->NAME.data_flag = 0;
	cpssp->NAME.msg_flag = 1;
	cpssp->NAME.in_flag = 1;
}

static void
NAME_(phase_msg_out)(struct cpssp *cpssp)
{
	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: called()\n", __FUNCTION__);
	}

	cpssp->NAME.cmd_flag = 0;
	cpssp->NAME.data_flag = 0;
	cpssp->NAME.msg_flag = 1;
	cpssp->NAME.in_flag = 0;
}

static void
NAME_(reset)(struct cpssp *cpssp);

static void
NAME_(read_param_callback)(struct cpssp *cpssp)
{
	NAME_(done_cmd)(cpssp, 0);
}

static void
NAME_(write_param_callback)(struct cpssp *cpssp)
{
	NAME_(command)(cpssp);
}

static void
NAME_(should_recv)(struct cpssp *cpssp, unsigned long count)
{
	uint8_t byte;

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: called(count=%lu)\n", __FUNCTION__, count);
	}

	assert(cpssp->NAME.in_flag);

	cpssp->NAME.req_count = count;

	if (cpssp->NAME.cmd_flag) {
		/* Status Phase */
		cpssp->NAME.req_count = NAME_(recv)(cpssp, &byte, sizeof(byte));
		cpssp->common.error = byte << 4;

		assert(cpssp->NAME.req_count == 0);

	} else if (cpssp->NAME.data_flag) {
		/* Data In Phase */
		if (cpssp->NAME.req_count < cpssp->common.bytecount) {
			cpssp->common.bytecount = cpssp->NAME.req_count;
		}

		cpssp->common.irq_status = IN_MASK;

		if (cpssp->common.features & 1) {
			NAME_(dma_in)(cpssp, count);
		} else {
			NAME_(pio_in)(cpssp, count, 1);
		}
	
	} else if (cpssp->NAME.msg_flag) {
		/* Message In Phase */
		cpssp->NAME.req_count = NAME_(recv)(cpssp, &byte, sizeof(byte));

		assert(byte == 0x00); /* Completion Byte */
		assert(cpssp->NAME.req_count == 0);

	} else {
		assert(0); /* Mustn't happen. */
	}
}

static void
NAME_(data_in)(struct cpssp *cpssp, uint16_t *valp)
{
	if (cpssp->common.command == WIN_PACKETCMD) {
		assert(cpssp->NAME.in_flag);

		if (cpssp->NAME.req_count) {
			cpssp->NAME.req_count = NAME_(recv)(cpssp,
					valp, sizeof(*valp));

			if (cpssp->NAME.req_count == 0) {
				NAME_(done_io)(cpssp);
			}
		} else {
			goto report_end;
		}

	} else {
		assert(cpssp->NAME.tail + 1 <= sizeof(cpssp->NAME.buf));

		if (1 <= cpssp->NAME.count) {
			*valp = cpssp->NAME.buf[cpssp->NAME.tail];
			cpssp->NAME.tail++;
			cpssp->NAME.count--;

			if (cpssp->NAME.count == 0) {
				NAME_(done_io)(cpssp);
				NAME_(read_param_callback)(cpssp);
			}
		} else {
			/*
			 * Some programs / OSes seem to read beyond
			 * the end of the transferred block:
			 * return an "undefined" value (e.g. zero).
			 */
			static int reported = 0;

		report_end:;
			if (! reported) {
				fprintf(stderr, "%s: Reading beyond end of data block.\n",
						__FUNCTION__);
				reported = 1;
			}

			*valp = 0x0000;
		}
	}
}

static void
NAME_(should_send)(struct cpssp *cpssp, unsigned long count)
{
	uint8_t byte;

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: called(count=%lu)\n", __FUNCTION__, count);
	}

	assert(! cpssp->NAME.in_flag);

	cpssp->NAME.req_count = count;

	if (cpssp->NAME.cmd_flag) {
		/* Command Phase */
		cpssp->common.irq_status = COMMAND_MASK;
		NAME_(pio_out)(cpssp, count, 0);

	} else if (cpssp->NAME.data_flag) {
		/* Data Out Phase */
		if (cpssp->NAME.req_count < cpssp->common.bytecount) {
			cpssp->common.bytecount = cpssp->NAME.req_count;
		}

		cpssp->common.irq_status = 0;

		if (cpssp->common.features & 1) {
			NAME_(dma_out)(cpssp, count);
		} else {
			NAME_(pio_out)(cpssp, count, 1);
		}

	} else if (cpssp->NAME.msg_flag) {
		/* Message Out Phase */
		byte = 0x80; /* Identify message: use LUN 0. */
		cpssp->NAME.req_count = NAME_(send)(cpssp, &byte, sizeof(byte));
		assert(cpssp->NAME.req_count == 0);

	} else {
		assert(0); /* Mustn't happen. */
	}
}

static void
NAME_(data_out)(struct cpssp *cpssp, uint16_t val)
{
	if (cpssp->common.command == WIN_PACKETCMD) {
		assert(! cpssp->NAME.in_flag);
		assert(cpssp->NAME.req_count);

		cpssp->NAME.req_count = NAME_(send)(cpssp, &val,
				sizeof(val));

		if (cpssp->NAME.req_count == 0) {
			NAME_(done_io)(cpssp);
		}

	} else {
		assert(cpssp->NAME.head + 1
			<= sizeof(cpssp->NAME.buf) / sizeof(cpssp->NAME.buf[0]));
		assert(1 <= cpssp->NAME.count);

		cpssp->NAME.buf[cpssp->NAME.head] = val;
		cpssp->NAME.head++;
		cpssp->NAME.count--;

		if (cpssp->NAME.count == 0) {
			NAME_(done_io)(cpssp);
			NAME_(write_param_callback)(cpssp);
		}
	}
}

static void
NAME_(set_signature)(struct cpssp *cpssp)
{
	cpssp->common.error = 0x01; /* FreeBSD-7.0 (and others?) needs it... */
	cpssp->common.irq_status = COMMAND_MASK;
	cpssp->common.sector = 0x01;
	cpssp->common.bytecount = 0xEB14;
	cpssp->common.select &= 0x10;
}

static void
NAME_(reset)(struct cpssp *cpssp)
{
	cpssp->NAME.pio_mode = 0;
	cpssp->NAME.mdma_mode = 0;
	cpssp->NAME.udma_mode = 0;

	cpssp->NAME.head = 0;
	cpssp->NAME.tail = 0;
	cpssp->NAME.count = 0;

	scsi_gen_cdrom_reset(cpssp);

	NAME_(set_signature)(cpssp);
	cpssp->common.features = 0;
	cpssp->common.control = 0;
	NAME_(done_cmd)(cpssp, 0);
}

static void
NAME_(abort)(struct cpssp *cpssp)
{
	cpssp->common.error = 1 << 2; /* Abort */
	NAME_(done_cmd)(cpssp, 1);
}

static void
NAME_(dummy)(struct cpssp *cpssp)
{
	NAME_(done_cmd)(cpssp, 1);
}

static void
NAME_(nop)(struct cpssp *cpssp)
{
	cpssp->NAME.tail = 0;
	cpssp->NAME.head = 0;
	cpssp->NAME.count = 0;

	cpssp->common.error = 1 << 2; /* Abort */
	NAME_(done_cmd)(cpssp, 1);
}

static void
NAME_(execute_device_diagnostic)(struct cpssp *cpssp)
{
	cpssp->NAME.tail = 0;
	cpssp->NAME.head = 0;
	cpssp->NAME.count = 0;

	NAME_(set_signature)(cpssp);

	NAME_(done_cmd)(cpssp, 1);
}

/* Adapted from QEMU */

static void
padstr(char *str, const char *src, int len)
{
	int i, v;

	for(i = 0; i < len; i++) {
		if (*src) {
			v = *src++;
		} else {
			v = ' ';
		}
		*(char *)((long)str ^ 1) = v;
		str++;
	}
}

static void
NAME_(pidentify)(struct cpssp *cpssp)
{
	unsigned short *p = (unsigned short *) cpssp->NAME.buf;

	memset(p, 0, 512);
	p[0] = 0x85c0; /* ATAPI removable media CDROM, 50 us DRQ, 12b packet */
	padstr((char *) (p + 10), "FAUMDVD001", 20);	/* Serial No. */
	padstr((char *) (p + 23), "1.0", 8);		/* Firmware rev. */
#if CD_WRITER_SUPPORT && CD_WRITER_RW_SUPPORT
	padstr((char *) (p + 27), "FAUmachine ATAPI CD/DVD-RW", 40);
#elif CD_WRITER_SUPPORT
	padstr((char *) (p + 27), "FAUmachine ATAPI CD/DVD-R", 40);
#else
	padstr((char *) (p + 27), "FAUmachine ATAPI CD/DVD-ROM", 40);
#endif
	p[49] |= (1 << 11); /* IORDY supported */
	p[49] |= (1 << 10); /* IORDY may be disabled */
 	p[49] |= (1 <<  9); /* LBA supported */
#if 1
	p[49] |= (1 <<  8); /* DMA supported */
#endif

	p[51] = (cpssp->NAME.pio_mode << 8) | 0x00;
	p[53] = 0x0006; /* word 88 and word 64-70 are valid */
#if 1
	p[63] = (cpssp->NAME.mdma_mode << 8) | 0x07; /* Highbyte: Currently selected DMA mode. lowbyte: supported DMA modes */
#endif
	p[64] = 0x0003; /* Supported PIO Modes */
	/* DMA and PIO timings */
	p[65] = 0x0078;
	p[66] = 0x0078;
	p[67] = 0x0078;
	p[68] = 0x0078;
	p[80] = 0x001E; /* We support ATA/ATAPI 1-4 */
	p[82] = (1 << 9) /* Device Reset supported */ | (1 << 4); /* PACKET Command feature set */
	p[83] = (1 << 14); /* This should be set to One */
	p[84] = (1 << 14); /* This should be set to One */
	/* FIXME: Make this configurable: Which feature is enabled?! */
	p[85] = (1 << 9) /* Device Reset supported */ | (1 << 4); /* PACKET Command feature set */
	p[87] = (1 << 14); /* This should be set to One */
#if 1
	p[88] = (cpssp->NAME.udma_mode << 8) | 0x07;
#endif

	cpssp->NAME.tail = 0;
	cpssp->NAME.head = 256;
	cpssp->NAME.count = 256;
	NAME_(pio_in)(cpssp, cpssp->NAME.count, 1);
}

static void
NAME_(setfeatures)(struct cpssp *cpssp)
{
	switch(cpssp->common.features) {
	case 0x02: /* Enable write cache */
	case 0x82: /* Disable write cache */
		/* FIXME --tg 21:04 05-01-25 */
		break;

	case 0x03: /* Set transfer mode */
		switch (cpssp->common.nsector & 0xF8) {
		case 0x08: /* PIO Mode */
			cpssp->NAME.pio_mode = cpssp->common.nsector & 0x07;
			break;

	/* Multiword- *or* Ultra- DMA is selected */

		case 0x20: /* Multiword DMA Mode */
			cpssp->NAME.mdma_mode = cpssp->common.nsector & 0x07;
			cpssp->NAME.udma_mode = 0;
			break;
		case 0x40: /* UDMA Mode */
			cpssp->NAME.udma_mode = cpssp->common.nsector & 0x07;
			cpssp->NAME.mdma_mode = 0;
			break;
		}
		break;

	case 0x55: /* Disable read look-ahead. */
	case 0xaa: /* Enable read look-ahead. */
		/* Nothing to do (yet)... */
		break;

	default:
		fprintf(stderr, "%s: Warning: tried to set feature: 0x%02x: not implemented\n",
				__FUNCTION__, cpssp->common.features);
		NAME_(abort)(cpssp);
		break;
	}

	NAME_(done_cmd)(cpssp, 1);
}

static void
NAME_(device_reset)(struct cpssp *cpssp)
{
	NAME_(set_signature)(cpssp);
	NAME_(done_cmd)(cpssp, 0);
}

static void
NAME_(command)(struct cpssp *cpssp)
{
#if DEBUGPCOM
	fprintf(stderr, "%s: ", __FUNCTION__);
	if (cpssp->common.command == WIN_NOP) {
		fprintf(stderr, "WIN_NOP\n");
	} else if (cpssp->common.command == WIN_EXECUTE_DEVICE_DIAGNOSTIC) {
		fprintf(stderr, "WIN_EXECUTE_DEVICE_DIAGNOSTIC\n");
	} else if (cpssp->common.command == WIN_PACKETCMD) {
		fprintf(stderr, "WIN_PACKETCMD\n");
	} else if (cpssp->common.command == WIN_PIDENTIFY) {
		fprintf(stderr, "WIN_PIDENTIFY\n");
	} else if (cpssp->common.command == WIN_STANDBYNOW1) {
		fprintf(stderr, "WIN_STANDBYNOW1\n");
	} else if (cpssp->common.command == WIN_IDLEIMMEDIATE) {
		fprintf(stderr, "WIN_IDLEIMMEDIATE\n");
	} else if (cpssp->common.command == WIN_SLEEPNOW1) {
		fprintf(stderr, "WIN_SLEEPNOW1\n");
	} else if (cpssp->common.command == WIN_SRST) {
		fprintf(stderr, "WIN_SRST\n");
	} else if (cpssp->common.command == WIN_IDENTIFY) {
		fprintf(stderr, "WIN_IDENTIFY\n");
	} else if (cpssp->common.command == WIN_SETFEATURES) {
		fprintf(stderr, "WIN_SETFEATURES\n");
	} else if (cpssp->common.command == WIN_CHECKPOWERMODE1) {
		fprintf(stderr, "WIN_CHECKPOWERMODE1\n");
	} else if (cpssp->common.command == WIN_CHECKPOWERMODE1) {
		fprintf(stderr, "WIN_CHECKPOWERMODE1\n");
	} else {
		fprintf(stderr, "command 0x%02x\n", cpssp->common.command);
	}
#endif

	cpssp->common.error = 0x00;

	switch (cpssp->common.command) {
	case WIN_NOP: /* 0x00, mandatory */
		/* ATAPI-7, 164 */
		NAME_(nop)(cpssp);
		break;

	case WIN_READ: /* 0x20, mandatory */
		NAME_(set_signature)(cpssp);
		NAME_(abort)(cpssp);
		break;

	case WIN_EXECUTE_DEVICE_DIAGNOSTIC: /* 0x90, mandatory */
		/* ATAPI-7, 106 */
		NAME_(execute_device_diagnostic)(cpssp);
		break;

	case WIN_PACKETCMD: /* 0xa0, mandatory */
		/* ATAPI-7, 166 */
		NAME_(phase_select)(cpssp);
		break;

	case WIN_PIDENTIFY: /* ATAPI Identify Device, 0xa1, mandatory */
		/* do we need ide_gen_cdrom_set_signature(cpssp) here? */
		NAME_(pidentify)(cpssp);
		break;

	case WIN_STANDBYNOW1: /* standby immediate, 0xe0, mandatory */
		fixme();
#if 0
		NAME_(dummy)(cpssp);
#endif
		NAME_(done_cmd)(cpssp, 1);
		break;

	case WIN_IDLEIMMEDIATE: /* 0xe1, mandatory */
		NAME_(dummy)(cpssp);
		NAME_(done_cmd)(cpssp, 1);
		break;

	case WIN_SLEEPNOW1: /* 0xe6, mandatory */
		fixme();
#if 0
		NAME_(dummy)(cpssp);
#endif
		NAME_(done_cmd)(cpssp, 1);
		break;

	case WIN_SRST: /* ATAPI soft reset, 0x08, mandatory */
		NAME_(device_reset)(cpssp);
		break;

	case WIN_IDENTIFY: /* 0xec, shall be aborted */
		NAME_(abort)(cpssp);
		break;

	case WIN_SETFEATURES: /* 0xef, mandatory */
		NAME_(setfeatures)(cpssp);
		break;

	case WIN_CHECKPOWERMODE1: /* 0xe5, mandatory */
		fixme();
		break;

	default:
		/* FIXME */
		fprintf(stderr, "%s: %s: unknown step 0x%02x\n", __FILE__,
				__FUNCTION__, cpssp->common.command);
		NAME_(abort)(cpssp);
	}
}

static void
NAME_(power_set)(struct cpssp *cpssp, unsigned int val)
{
	if (val) {
		/* Power On Event */
		NAME_(reset)(cpssp);
	}
}

static void
NAME_(reset_set)(struct cpssp *cpssp, unsigned int val)
{
	if (! val) {
		NAME_(reset)(cpssp);
	}
}

static void
NAME_(soft_reset)(struct cpssp *cpssp)
{
	NAME_(reset)(cpssp);
}

static void
NAME_(create)(struct cpssp *cpssp)
{
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
}

static void
NAME_(suspend)(struct cpssp *cpssp, FILE *fp)
{
	generic_suspend(&cpssp->NAME, sizeof(cpssp->NAME), fp);
}

static void
NAME_(resume)(struct cpssp *cpssp, FILE *fp)
{
	generic_resume(&cpssp->NAME, sizeof(cpssp->NAME), fp);
}

#endif /* BEHAVIOR */

#undef DEBUG_CONTROL_FLOW
