/*
 * $Id: mem_template_gui_gtk.c,v 1.7 2013-05-06 19:15:42 vrsieh Exp $
 *
 * Copyright (C) 2003-2013 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <gtk/gtk.h>

#include "system.h"
#include "glue-gui.h"
#include "glue-gui-gtk.h"
#include "glue-gui-gtk-fi.h"
#include "glue.h"

#include "sig_boolean.h"

#if (DATA_BITS == 64)
/* FIXME */
#define bus_addr_t	uint64_t
#elif (ADDR_BITS <= 32)
#define bus_addr_t	uint32_t
#else
#error "Bad Address Size."
#endif

#if (BS_BITS != 1 && BS_BITS != 2 && BS_BITS != 4 && BS_BITS != 8)
#error "Bad BS Size."
#endif

#if (DATA_BITS == 8)
#include "sig_cs8.h"
#define bus_data_t	uint8_t
#define sig_bus		sig_cs8
#define sig_bus_funcs	sig_cs8_funcs
#define sig_bus_connect	sig_cs8_connect
#define sig_bus_unmap	sig_cs8_unmap
#elif (DATA_BITS == 16)
#include "sig_cs16.h"
#define bus_data_t	uint16_t
#define sig_bus		sig_cs16
#define sig_bus_funcs	sig_cs16_funcs
#define sig_bus_connect	sig_cs16_connect
#define sig_bus_unmap	sig_cs16_unmap
#elif (DATA_BITS == 32)
#include "sig_cs32.h"
#define bus_data_t	uint32_t
#define sig_bus		sig_cs32
#define sig_bus_funcs	sig_cs32_funcs
#define sig_bus_connect	sig_cs32_connect
#define sig_bus_unmap	sig_cs32_unmap
#elif (DATA_BITS == 64)
#include "sig_cs64.h"
#define bus_data_t	uint64_t
#define sig_bus		sig_cs64
#define sig_bus_funcs	sig_cs64_funcs
#define sig_bus_connect	sig_cs64_connect
#define sig_bus_unmap	sig_cs64_unmap
#else
#error "Bad Data Size."
#endif

#define MEM_SIZE	((1 << ADDR_BITS) * (DATA_BITS / 8))

enum fault {
	FAULT_BIT_FLIP,
	FAULT_STUCK_AT_0,
	FAULT_STUCK_AT_1,
	FAULT_COUPLING,
};

struct cpssp {
	char name[1024];

	unsigned long size;

	GtkWidget *gui_fi;

	int comp_id;

	struct {
		enum fault type;
		unsigned int addr0;
		unsigned int bit0;
		unsigned int addr1;
		unsigned int bit1;

		int sig_id;
		void *sig_ptr;
	} fault[256];
	unsigned int nfaults;
};

/*
 * Simulator Callbacks
 */

/*
 * GUI Callbacks
 */
static void
COMP_(gui_error_event)(
	struct cpssp *cpssp,
	enum fault type,
	unsigned int addr0,
	unsigned int bit0,
	unsigned int addr1,
	unsigned int bit1,
	unsigned int val
)
{
	unsigned int i;
	char path[1024];

	for (i = 0; ; i++) {
		if (i == cpssp->nfaults) {
			/* Not Found */
			char port[256];

			assert(cpssp->nfaults < sizeof(cpssp->fault) / sizeof(cpssp->fault[0])); /* FIXME */

			cpssp->fault[i].type = type;
			cpssp->fault[i].addr0 = addr0;
			cpssp->fault[i].bit0 = bit0;
			cpssp->fault[i].addr1 = addr1;
			cpssp->fault[i].bit1 = bit1;

			snprintf(path, sizeof(path)-1, "%s:fault%u", cpssp->name, i);
			path[sizeof(path)-1] = '\0';
			cpssp->fault[i].sig_id = system_sig_create("boolean", path);
			assert(0 <= cpssp->fault[i].sig_id);
			cpssp->fault[i].sig_ptr = system_sig_get(cpssp->fault[i].sig_id);
			assert(cpssp->fault[i].sig_ptr);

			switch (type) {
			case FAULT_BIT_FLIP:
				sprintf(port, "bitflip/%u/%u",
						addr0, bit0);
				break;
			case FAULT_STUCK_AT_0:
				sprintf(port, "stuck_at_0/%u/%u",
						addr0, bit0);
				break;
			case FAULT_STUCK_AT_1:
				sprintf(port, "stuck_at_1/%u/%u",
						addr0, bit0);
				break;
			case FAULT_COUPLING:
				sprintf(port, "coupling/%u/%u/%u/%u",
						addr0, bit0, addr1, bit1);
				break;
			default:
				assert(0); /* Cannot happen. */
			}
			system_comp_connect(cpssp->comp_id, port,
					cpssp->fault[i].sig_id);
			cpssp->nfaults++;
			break;
		}
		if (cpssp->fault[i].type == type
		 && cpssp->fault[i].addr0 == addr0
		 && cpssp->fault[i].bit0 == bit0
		 && cpssp->fault[i].addr1 == addr1
		 && cpssp->fault[i].bit1 == bit1) {
			/* Found */
			break;
		}
	}

	sig_boolean_set(cpssp->fault[i].sig_ptr, cpssp, val);
}

static void
COMP_(gui_bit_flip_event)(GtkWidget *w, gpointer _fault, gpointer _cpssp)
{
	struct fault {
		unsigned int state;
		const char *loc0[4];
		const char *loc1[4];
	} *fault = (struct fault *) _fault;
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	COMP_(gui_error_event)(cpssp, FAULT_BIT_FLIP,
			strtoul(fault->loc0[0], NULL, 0),
			strtoul(fault->loc0[1], NULL, 0),
			0, 0,
			fault->state);

	free(fault);
}

static void
COMP_(gui_stuck_at_0_event)(GtkWidget *w, gpointer _fault, gpointer _cpssp)
{
	struct fault {
		unsigned int state;
		const char *loc0[4];
		const char *loc1[4];
	} *fault = (struct fault *) _fault;
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	COMP_(gui_error_event)(cpssp, FAULT_STUCK_AT_0,
			strtoul(fault->loc0[0], NULL, 0),
			strtoul(fault->loc0[1], NULL, 0),
			0, 0,
			fault->state);

	free(fault);
}

static void
COMP_(gui_stuck_at_1_event)(GtkWidget *w, gpointer _fault, gpointer _cpssp)
{
	struct fault {
		unsigned int state;
		const char *loc0[4];
		const char *loc1[4];
	} *fault = (struct fault *) _fault;
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	COMP_(gui_error_event)(cpssp, FAULT_STUCK_AT_1,
			strtoul(fault->loc0[0], NULL, 0),
			strtoul(fault->loc0[1], NULL, 0),
			0, 0, fault->state);

	free(fault);
}

static void
COMP_(gui_coupling_event)(GtkWidget *w, gpointer _fault, gpointer _cpssp)
{
	struct fault {
		unsigned int state;
		const char *loc0[4];
		const char *loc1[4];
	} *fault = (struct fault *) _fault;
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	COMP_(gui_error_event)(cpssp, FAULT_COUPLING,
			strtoul(fault->loc0[0], NULL, 0),
			strtoul(fault->loc0[1], NULL, 0),
			strtoul(fault->loc1[0], NULL, 0),
			strtoul(fault->loc1[1], NULL, 0),
			fault->state);

	free(fault);
}

void *
COMP_(gui_gtk_create)(
	unsigned int page,
	const char *name,
	struct sig_manage *manage,
	struct sig_std_logic *port_power,
	struct sig_bus *port_cs,
	struct sig_mem_bus *port_bus
)
{
	struct cpssp *cpssp;
	GtkWidget *w;

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);

	system_name_push(name);

	strcpy(cpssp->name, system_path());

	cpssp->comp_id = system_comp_lookup(name);

	cpssp->size = MEM_SIZE;

	cpssp->gui_fi = gtk_vbox_new(FALSE, 1);

	w = gui_gtk_fi_pattern_new(
		"Bit Flip", "Trigger",
		2, "Address", 0ULL, (unsigned long long) cpssp->size - 1, NULL, "Bit", 0ULL, 7ULL, NULL,
		0);
	g_signal_connect(G_OBJECT(w), "fi-change",
			G_CALLBACK(COMP_(gui_bit_flip_event)), cpssp);

	gtk_widget_show(w);
	gtk_box_pack_start(GTK_BOX(cpssp->gui_fi), w, FALSE, FALSE, 1);

	w = gui_gtk_fi_pattern_new(
		"Stuck-At-0", "Boolean",
		2, "Address", 0ULL, (unsigned long long) cpssp->size - 1, NULL, "Bit", 0ULL, 7ULL, NULL,
		0);
	g_signal_connect(G_OBJECT(w), "fi-change",
			G_CALLBACK(COMP_(gui_stuck_at_0_event)), cpssp);

	gtk_widget_show(w);
	gtk_box_pack_start(GTK_BOX(cpssp->gui_fi), w, FALSE, FALSE, 1);

	w = gui_gtk_fi_pattern_new(
		"Stuck-At-1", "Boolean",
		2, "Address", 0ULL, (unsigned long long) cpssp->size - 1, NULL, "Bit", 0ULL, 7ULL, NULL,
		0);
	g_signal_connect(G_OBJECT(w), "fi-change",
			G_CALLBACK(COMP_(gui_stuck_at_1_event)), cpssp);

	gtk_widget_show(w);
	gtk_box_pack_start(GTK_BOX(cpssp->gui_fi), w, FALSE, FALSE, 1);

	w = gui_gtk_fi_pattern_new(
		"Coupling", "Boolean",
		2, "Address", 0ULL, (unsigned long long) cpssp->size - 1, NULL, "Bit", 0ULL, 7ULL, NULL,
		2, "Address", 0ULL, (unsigned long long) cpssp->size - 1, NULL, "Bit", 0ULL, 7ULL, NULL);
	g_signal_connect(G_OBJECT(w), "fi-change",
			G_CALLBACK(COMP_(gui_coupling_event)), cpssp);

	gtk_widget_show(w);
	gtk_box_pack_start(GTK_BOX(cpssp->gui_fi), w, FALSE, FALSE, 1);

	gui_gtk_comp_add(page, SCOMP, name, NULL, FALSE, FALSE, cpssp->gui_fi);

	system_name_pop();

	return cpssp;
}

void
COMP_(gui_gtk_destroy)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	shm_free(cpssp);
}

void
COMP_(gui_gtk_suspend)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;
	
	generic_suspend(cpssp, sizeof(*cpssp), fp);
}

void
COMP_(gui_gtk_resume)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;
	
	GtkWidget *savegui_fi = cpssp->gui_fi;
	
	generic_resume(cpssp, sizeof(*cpssp), fp);
	
	cpssp->gui_fi = savegui_fi;
	
	fprintf(stderr, "\t\t\ttoDo: Handle GtkWidget gui_fi and faults\n", 0);
}
