/*
 * Copyright (C) 2014 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/*
 * For register descriptions see:
 * "KL02 Sub-Family Reference Manual"
 * "Document Number: KL02P32M48SF0RM"
 * "Rev 3.1, July 2013"
 */

#include <assert.h>
#include <stdio.h>

#include "glue.h"

#define INCLUDE
#include "arch_arm_cpu.c"
#include "arch_arm_nvic.c"
#include "arch_arm_systick.c"
#include "arch_kinetis_adc.c"
#include "arch_kinetis_cmp.c"
#if defined(KL46Z)
#include "arch_kinetis_dac.c"
#endif
#if defined(KL46Z)
#include "arch_kinetis_dma.c"
#include "arch_kinetis_dmamux.c"
#endif
#include "arch_gen_flash.c"
#include "arch_kinetis_fmc.c"
#include "arch_kinetis_ftfa.c"
#include "arch_kinetis_gpio.c"
#include "arch_kinetis_i2c.c"
#include "arch_kinetis_llwu.c"
#include "arch_kinetis_mcg.c"
#include "arch_kinetis_mcm.c"
#include "arch_kinetis_osc.c"
#if defined(KL46Z)
#include "arch_kinetis_pit.c"
#endif
#include "arch_kinetis_pmc.c"
#include "arch_kinetis_port.c"
#include "arch_kinetis_sim.c"
#include "arch_kinetis_rcm.c"
#if defined(KL46Z)
#include "arch_kinetis_slcd.c"
#endif
#include "arch_kinetis_smc.c"
#include "arch_kinetis_spi.c"
#include "arch_gen_sram.c"
#include "arch_kinetis_tpm.c"
#include "arch_kinetis_uart.c"
#undef INCLUDE

#define POWER_addsub			0.7e-9
#define POWER_addsubsp			1.1e-9
#define POWER_alu			0.7e-9
#define POWER_branchconditional		1.2e-9
#define POWER_branchunconditional	0.9e-9
#define POWER_cps			1.0e-9 /* FIXME */
#define POWER_extend			1.1e-9
#define POWER_hint			1.0e-9 /* FIXME */
#define POWER_hireg			0.7e-9
#define POWER_immediate			1.0e-9
#define POWER_inst32			2.7e-9
#define POWER_lea			0.6e-9
#define POWER_memimmediate		1.3e-9
#define POWER_memmultiple		1.7e-9
#define POWER_mempcrel			2.0e-9
#define POWER_memreg			1.1e-9
#define POWER_memsprel			1.3e-9
#define POWER_pushpop			1.5e-9
#define POWER_reverse			1.0e-9 /* FIXME */
#define POWER_shift			0.8e-9
#define POWER_wfi			0.94e-9 /* 45mW * (1/48 MHz) */

#define POWER_adc_read			2104.94e-9

#define POWER_cmp_set			75.84e-9

#define POWER_i2c_read			85.38e-9
#define POWER_i2c_write			84.95e-9

#define POWER_static			0.625e-9

struct cpssp {
	struct process process;
	double energy;

	int power_state;
	unsigned int count_1kHz;
	unsigned int count_system_clk;

	struct sig_std_logic *port_vdd;
	unsigned int state_vdd;

	struct sig_std_logic *port_pta[32];
	struct sig_std_logic *port_ptb[32];
#if defined(KL46Z)
	struct sig_std_logic *port_ptc[32];
	struct sig_std_logic *port_ptd[32];
	struct sig_std_logic *port_pte[32];
#endif

#define STATE

#define NAME		matrix
#define NAME_(x)	matrix_ ## x
#if defined(KL02Z)
#include "chip_kl02z32m_matrix.c"
#elif defined(KL46Z)
#include "chip_kl46z256m_matrix.c"
#endif
#undef NAME_
#undef NAME

#define NAME		adc0
#define NAME_(x)	adc0_ ## x
#if defined(KL46Z)
#define CONFIG_DMA	1
#else
#define CONFIG_DMA	0
#endif
#include "arch_kinetis_adc.c"
#undef CONFIG_DMA
#undef NAME_
#undef NAME

#define NAME		cmp0
#define NAME_(x)	cmp0_ ## x
#define CONFIG_WINDOW	0
#define CONFIG_SAMPLE	0
#include "arch_kinetis_cmp.c"
#undef CONFIG_SAMPLE
#undef CONFIG_WINDOW
#undef NAME_
#undef NAME

#if defined(KL46Z)
#define NAME		dac0
#define NAME_(x)	dac0_ ## x
#include "arch_kinetis_dac.c"
#undef NAME_
#undef NAME
#endif

#if defined(KL46Z)
#define NAME		dma
#define NAME_(x)	dma_ ## x
#include "arch_kinetis_dma.c"
#undef NAME_
#undef NAME

#define NAME		dmamux
#define NAME_(x)	dmamux_ ## x
#include "arch_kinetis_dmamux.c"
#undef NAME_
#undef NAME
#endif /* KL46Z */

#define NAME		flash
#define NAME_(x)	flash_ ## x
#include "arch_gen_flash.c"
#undef NAME_
#undef NAME

#define NAME		fmc
#define NAME_(x)	fmc_ ## x
#include "arch_kinetis_fmc.c"
#undef NAME_
#undef NAME

#define NAME		ftfa
#define NAME_(x)	ftfa_ ## x
#include "arch_kinetis_ftfa.c"
#undef NAME_
#undef NAME

#define NAME		pta
#define NAME_(x)	pta_ ## x
#include "arch_kinetis_gpio.c"
#undef NAME_
#undef NAME

#define NAME		ptb
#define NAME_(x)	ptb_ ## x
#include "arch_kinetis_gpio.c"
#undef NAME_
#undef NAME

#if defined(KL46Z)
#define NAME		ptc
#define NAME_(x)	ptc_ ## x
#include "arch_kinetis_gpio.c"
#undef NAME_
#undef NAME

#define NAME		ptd
#define NAME_(x)	ptd_ ## x
#include "arch_kinetis_gpio.c"
#undef NAME_
#undef NAME

#define NAME		pte
#define NAME_(x)	pte_ ## x
#include "arch_kinetis_gpio.c"
#undef NAME_
#undef NAME
#endif /* KL46Z */

#define NAME		i2c0
#define NAME_(x)	i2c0_ ## x
#if defined(KL46Z)
#define CONFIG_DMA	1
#else
#define CONFIG_DMA	0
#endif
#include "arch_kinetis_i2c.c"
#undef CONFIG_DMA
#undef NAME_
#undef NAME

#define NAME		i2c1
#define NAME_(x)	i2c1_ ## x
#if defined(KL46Z)
#define CONFIG_DMA	1
#else
#define CONFIG_DMA	0
#endif
#include "arch_kinetis_i2c.c"
#undef CONFIG_DMA
#undef NAME_
#undef NAME

#define NAME		llwu
#define NAME_(x)	llwu_ ## x
#include "arch_kinetis_llwu.c"
#undef NAME_
#undef NAME

#define NAME		mcg
#define NAME_(x)	mcg_ ## x
#include "arch_kinetis_mcg.c"
#undef NAME_
#undef NAME

#define NAME		mcm
#define NAME_(x)	mcm_ ## x
#include "arch_kinetis_mcm.c"
#undef NAME_
#undef NAME

#define NAME		osc
#define NAME_(x)	osc_ ## x
#include "arch_kinetis_osc.c"
#undef NAME_
#undef NAME

#if defined(KL46Z)
#define NAME		pit
#define NAME_(x)	pit_ ## x
#define N		2
#include "arch_kinetis_pit.c"
#undef N
#undef NAME_
#undef NAME
#endif /* KL46Z */

#define NAME		pmc
#define NAME_(x)	pmc_ ## x
#include "arch_kinetis_pmc.c"
#undef NAME_
#undef NAME

#define NAME		porta
#define NAME_(x)	porta_ ## x
#if defined(KL46Z)
#define CONFIG_NALTS	8
#else
#define CONFIG_NALTS	4
#endif
#include "arch_kinetis_port.c"
#undef NAME_
#undef NAME

#define NAME		portb
#define NAME_(x)	portb_ ## x
#if defined(KL46Z)
#define CONFIG_NALTS	8
#else
#define CONFIG_NALTS	4
#endif
#include "arch_kinetis_port.c"
#undef NAME_
#undef NAME

#if defined(KL46Z)
#define NAME		portc
#define NAME_(x)	portc_ ## x
#define CONFIG_NALTS	8
#include "arch_kinetis_port.c"
#undef NAME_
#undef NAME

#define NAME		portd
#define NAME_(x)	portd_ ## x
#define CONFIG_NALTS	8
#include "arch_kinetis_port.c"
#undef NAME_
#undef NAME

#define NAME		porte
#define NAME_(x)	porte_ ## x
#define CONFIG_NALTS	8
#include "arch_kinetis_port.c"
#undef NAME_
#undef NAME
#endif /* KL46Z */

#define NAME		rcm
#define NAME_(x)	rcm_ ## x
#include "arch_kinetis_rcm.c"
#undef NAME_
#undef NAME

#define NAME		sim
#define NAME_(x)	sim_ ## x
#include "arch_kinetis_sim.c"
#undef NAME_
#undef NAME

#define NAME		spi0
#define NAME_(x)	spi0_ ## x
#include "arch_kinetis_spi.c"
#undef NAME_
#undef NAME

#if defined(KL46Z)
#define NAME		spi1
#define NAME_(x)	spi1_ ## x
#include "arch_kinetis_spi.c"
#undef NAME_
#undef NAME
#endif /* KL46Z */

#if defined(KL46Z)
#define NAME		lcd
#define NAME_(x)	lcd_ ## x
#include "arch_kinetis_slcd.c"
#undef NAME_
#undef NAME
#endif /* KL46Z */

#define NAME		smc
#define NAME_(x)	smc_ ## x
#include "arch_kinetis_smc.c"
#undef NAME_
#undef NAME

#define NAME		sram
#define NAME_(x)	sram_ ## x
#include "arch_gen_sram.c"
#undef NAME_
#undef NAME

#define NAME		lptmr0
#define NAME_(x)	lptmr0_ ## x
#include "arch_kinetis_lptmr.c"
#undef NAME_
#undef NAME

#define NAME		tpm0
#define NAME_(x)	tpm0_ ## x
#include "arch_kinetis_tpm.c"
#undef NAME_
#undef NAME

#define NAME		tpm1
#define NAME_(x)	tpm1_ ## x
#include "arch_kinetis_tpm.c"
#undef NAME_
#undef NAME

#if defined(KL46Z)
#define NAME		tpm2
#define NAME_(x)	tpm2_ ## x
#include "arch_kinetis_tpm.c"
#undef NAME_
#undef NAME
#endif /* KL46Z */

#define NAME		uart0
#define NAME_(x)	uart0_ ## x
#if defined(KL46Z)
#define CONFIG_DMA	1
#else
#define CONFIG_DMA	0
#endif
#include "arch_kinetis_uart.c"
#undef CONFIG_DMA
#undef NAME_
#undef NAME

#if defined(KL46Z)
#define NAME		uart1
#define NAME_(x)	uart1_ ## x
#define CONFIG_DMA	1
#include "arch_kinetis_uart.c"
#undef CONFIG_DMA
#undef NAME_
#undef NAME

#define NAME		uart2
#define NAME_(x)	uart2_ ## x
#define CONFIG_DMA	1
#include "arch_kinetis_uart.c"
#undef CONFIG_DMA
#undef NAME_
#undef NAME
#endif /* KL46Z */

#define NAME		arm_nvic
#define NAME_(x)	arm_nvic_ ## x
#define CONFIG_INTLINESNUM	32
#include "arch_arm_nvic.c"
#undef CONFIG_INTLINESNUM
#undef NAME_
#undef NAME

#define NAME		arm_systick
#define NAME_(x)	arm_systick_ ## x
#include "arch_arm_systick.c"
#undef NAME_
#undef NAME

#define NAME		arm_cpu
#define NAME_(x)	arm_cpu_ ## x
#define CONFIG_DEBUG	0
#define CONFIG_VERSION	6
#define CONFIG_ARM	0
#define CONFIG_THUMB	1
#define CONFIG_E	0
#define CONFIG_M	1
#include "arch_arm_cpu.c"
#undef CONFIG_M
#undef CONFIG_E
#undef CONFIG_THUMB
#undef CONFIG_ARM
#undef CONFIG_VERSION
#undef CONFIG_DEBUG
#undef NAME_
#undef NAME

#undef STATE
};

/*forward*/ static void
CHIP_(switch_mr)(
	struct cpssp *cpssp,
	uint32_t addr,
	unsigned int bs,
	uint32_t *valp
);
/*forward*/ static void
CHIP_(switch_mw)(
	struct cpssp *cpssp,
	uint32_t addr,
	unsigned int bs,
	uint32_t val
);

/*
 * Forward declarations arm_cpu.
 */
/*forward*/ static void
arm_cpu_SCB_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
arm_cpu_SCB_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);
/*forward*/ static void
arm_cpu_clk(struct cpssp *cpssp);
/*forward*/ static void
arm_cpu_irq_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
arm_cpu_reset(struct cpssp *cpssp);

/*
 * Forward declarations arm_nvic.
 */
/*forward*/ static void
arm_nvic_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
arm_nvic_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);
/*forward*/ static void
arm_nvic_irq_N_set(struct cpssp *cpssp, int n, unsigned int val);
/*forward*/ static void
arm_nvic_systick_pending_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
arm_nvic_reset(struct cpssp *cpssp);

/*
 * Forward declarations arm_systick.
 */
/*forward*/ static void
arm_systick_exttick(struct cpssp *cpssp);
/*forward*/ static void
arm_systick_clk(struct cpssp *cpssp);
/*forward*/ static void
arm_systick_reset(struct cpssp *cpssp);

/*
 * Forward declarations matrix.
 */
/*forward*/ static void
matrix_portaNM_in_set(struct cpssp *cpssp, int n, int m, unsigned int val);
/*forward*/ static void
matrix_portbNM_in_set(struct cpssp *cpssp, int n, int m, unsigned int val);
#if defined(KL46Z)
/*forward*/ static void
matrix_portcNM_in_set(struct cpssp *cpssp, int n, int m, unsigned int val);
/*forward*/ static void
matrix_portdNM_in_set(struct cpssp *cpssp, int n, int m, unsigned int val);
/*forward*/ static void
matrix_porteNM_in_set(struct cpssp *cpssp, int n, int m, unsigned int val);
#endif

/*
 * Forward declarations ADC0.
 */
/*forward*/ static void
adc0_se_inN_set(struct cpssp *cpssp, int n, unsigned int val);
#if defined(KL46Z)
/*forward*/ static void
adc0_dp_inN_set(struct cpssp *cpssp, int n, unsigned int val);
/*forward*/ static void
adc0_dm_inN_set(struct cpssp *cpssp, int n, unsigned int val);
/*forward*/ static void
adc0_se4a_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
adc0_se4b_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
adc0_se5a_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
adc0_se5b_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
adc0_se6a_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
adc0_se6b_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
adc0_se7a_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
adc0_se7b_in_set(struct cpssp *cpssp, unsigned int val);
#endif
/*forward*/ static void
adc0_refN_set(struct cpssp *cpssp, int n, unsigned int val);
/*forward*/ static void
adc0_clk0(struct cpssp *cpssp);
/*forward*/ static void
adc0_clk1(struct cpssp *cpssp);
/*forward*/ static void
adc0_reset(struct cpssp *cpssp);

/*
 * Forward declarations CMP0.
 */
/*forward*/ static void
cmp0_in_inN_set(struct cpssp *cpssp, int n, unsigned int val);
/*forward*/ static void
cmp0_clk(struct cpssp *cpssp);
/*forward*/ static void
cmp0_reset(struct cpssp *cpssp);

#if defined(KL46Z)
/*
 * Forward declarations DAC0.
 */
/*forward*/ static void
dac0_refN_set(struct cpssp *cpssp, unsigned int n, unsigned int val);
/*forward*/ static void
dac0_reset(struct cpssp *cpssp);
#endif

#if defined(KL46Z)
/*
 * Forward declarations DMA.
 */
/*forward*/ static void
dma_requestN_set(struct cpssp *cpssp, unsigned int n, unsigned int val);
/*forward*/ static void
dma_clk(struct cpssp *cpssp);
/*forward*/ static void
dma_reset(struct cpssp *cpssp);

/*
 * Forward declarations DMAMUX.
 */
/*forward*/ static void
dmamux_request_inN_set(struct cpssp *cpssp, unsigned int n, unsigned int val);
/*forward*/ static void
dmamux_trigger_inN_set(struct cpssp *cpssp, unsigned int n, unsigned int val);
/*forward*/ static void
dmamux_reset(struct cpssp *cpssp);
#endif

/*
 * Forward declarations FLASH.
 */
/*forward*/ static void
flash_read(struct cpssp *cpssp, uint32_t addr, uint32_t *valp);
/*forward*/ static void
flash_erase(struct cpssp *cpssp, uint32_t addr, uint32_t len);
/*forward*/ static void
flash_write(struct cpssp *cpssp, uint32_t addr, uint32_t val);

/*
 * Forward declarations FMC.
 */
/*forward*/ static void
fmc_reset(struct cpssp *);

/*
 * Forward declarations FTFA.
 */
/*forward*/ static void
ftfa_clk(struct cpssp *cpssp);
/*forward*/ static void
ftfa_reset(struct cpssp *);

/*
 * Forward declarations for PTA.
 */
/*forward*/ static void
pta_inN_set(struct cpssp *cpssp, int n, unsigned int);
/*forward*/ static void
pta_reset(struct cpssp *);

/*
 * Forward declarations for PTB.
 */
/*forward*/ static void
ptb_inN_set(struct cpssp *cpssp, int n, unsigned int);
/*forward*/ static void
ptb_reset(struct cpssp *);

#if defined(KL46Z)
/*
 * Forward declarations for PTC.
 */
/*forward*/ static void
ptc_inN_set(struct cpssp *cpssp, int n, unsigned int);
/*forward*/ static void
ptc_reset(struct cpssp *);

/*
 * Forward declarations for PTD.
 */
/*forward*/ static void
ptd_inN_set(struct cpssp *cpssp, int n, unsigned int);
/*forward*/ static void
ptd_reset(struct cpssp *);

/*
 * Forward declarations for PTE.
 */
/*forward*/ static void
pte_inN_set(struct cpssp *cpssp, int n, unsigned int);
/*forward*/ static void
pte_reset(struct cpssp *);
#endif

/*
 * Forward declarations for I2C0.
 */
/*forward*/ static void
i2c0_sda_in_set(struct cpssp *, unsigned int val);
/*forward*/ static void
i2c0_scl_in_set(struct cpssp *, unsigned int val);
/*forward*/ static void
i2c0_clk(struct cpssp *);
/*forward*/ static void
i2c0_reset(struct cpssp *);

/*
 * Forward declarations for I2C1.
 */
/*forward*/ static void
i2c1_sda_in_set(struct cpssp *, unsigned int val);
/*forward*/ static void
i2c1_scl_in_set(struct cpssp *, unsigned int val);
/*forward*/ static void
i2c1_clk(struct cpssp *);
/*forward*/ static void
i2c1_reset(struct cpssp *);

/*
 * Forward declarations for LLWU.
 */
/*forward*/ static void
llwu_moduleN_set(struct cpssp *cpssp, int n, unsigned int val);
/*forward*/ static void
llwu_p_inN_set(struct cpssp *cpssp, int n, unsigned int val);
/*forward*/ static void
llwu_reset(struct cpssp *);

/*
 * Forward declarations for LPTMR.
 */
/*forward*/ static void
lptmr0_clk0(struct cpssp *cpssp);
/*forward*/ static void
lptmr0_clk1(struct cpssp *cpssp);
/*forward*/ static void
lptmr0_clk2(struct cpssp *cpssp);
/*forward*/ static void
lptmr0_clk3(struct cpssp *cpssp);
/*forward*/ static void
lptmr0_alt_inN_set(struct cpssp *cpssp, int n, unsigned int val);
/*forward*/ static void
lptmr0_reset(struct cpssp *);

/*
 * Forward declarations for MCG
 */
/*forward*/ static void
mcg_oscclk(struct cpssp *cpssp);
/*forward*/ static void
mcg_reset(struct cpssp *);

/*
 * Forward declarations for MCM
 */
/*forward*/ static void
mcm_reset(struct cpssp *);

/*
 * Forward declarations for OSC.
 */
/*forward*/ static void
osc_reset(struct cpssp *);

#if defined(KL46Z)
/*
 * Forward declarations for PIT.
 */
/*forward*/ static void
pit_clk(struct cpssp *);
/*forward*/ static void
pit_reset(struct cpssp *);
#endif

/*
 * Forward declarations for PMC.
 */
/*forward*/ static void
pmc_reset(struct cpssp *);

/*
 * Forward declarations for PORTA
 */
/*forward*/ static void
porta_internalNM_in_set(struct cpssp *cpssp, int n, int m, unsigned int);
/*forward*/ static void
porta_reset(struct cpssp *);

/*
 * Forward declarations for PORTB
 */
/*forward*/ static void
portb_internalNM_in_set(struct cpssp *cpssp, int n, int m, unsigned int);
/*forward*/ static void
portb_reset(struct cpssp *);

#if defined(KL46Z)
/*
 * Forward declarations for PORTC
 */
/*forward*/ static void
portc_internalNM_in_set(struct cpssp *cpssp, int n, int m, unsigned int);
/*forward*/ static void
portc_reset(struct cpssp *);

/*
 * Forward declarations for PORTD
 */
/*forward*/ static void
portd_internalNM_in_set(struct cpssp *cpssp, int n, int m, unsigned int);
/*forward*/ static void
portd_reset(struct cpssp *);

/*
 * Forward declarations for PORTE
 */
/*forward*/ static void
porte_internalNM_in_set(struct cpssp *cpssp, int n, int m, unsigned int);
/*forward*/ static void
porte_reset(struct cpssp *);
#endif

/*
 * Forward declarations for RCM.
 */
/*forward*/ static void
rcm_por(struct cpssp *);
/*forward*/ static void
rcm_wdog(struct cpssp *);
/*forward*/ static void
rcm_wakeup(struct cpssp *);

/*
 * Forward declarations for SIM.
 */
/*forward*/ static void
sim_lpo(struct cpssp *cpssp);
/*forward*/ static void
sim_outclkin(struct cpssp *cpssp);
/*forward*/ static void
sim_fllclkin(struct cpssp *cpssp);
#if defined(KL02Z)
/* No PLL */
#elif defined(KL46Z)
/*forward*/ static void
sim_pllclkin(struct cpssp *cpssp);
#else
#error
#endif
/*forward*/ static void
sim_tpm_clkinN_set(struct cpssp *cpssp, int n, unsigned int val);
/*forward*/ static void
sim_reset(struct cpssp *);

/*
 * Forward declarations for SMC.
 */
/*forward*/ static void
smc_stop(struct cpssp *cpssp, int sleepdeep);
/*forward*/ static void
smc_start(struct cpssp *cpssp);
/*forward*/ static void
smc_system_clk_in(struct cpssp *cpssp);
/*forward*/ static void
smc_bus_clk_in(struct cpssp *cpssp);
/*forward*/ static void
smc_reset(struct cpssp *cpssp);

#if defined(KL46Z)
/*
 * Forward declarations for SLCD.
 */
/*forward*/ static void
lcd_clk0(struct cpssp *cpssp);
/*forward*/ static void
lcd_clk1(struct cpssp *cpssp);
/*forward*/ static void
lcd_clk2(struct cpssp *cpssp);
/*forward*/ static void
lcd_reset(struct cpssp *cpssp);
#endif

/*
 * Forward declarations for SPI0.
 */
/*forward*/ static void
spi0_miso_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
spi0_mosi_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
spi0_spsck_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
spi0_ss_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
spi0_reset(struct cpssp *cpssp);

#if defined(KL46Z)
/*
 * Forward declarations for SPI1.
 */
/*forward*/ static void
spi1_miso_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
spi1_mosi_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
spi1_spsck_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
spi1_ss_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
spi1_reset(struct cpssp *cpssp);
#endif

/*
 * Forward declarations for TPM0.
 */
/*forward*/ static void
tpm0_ch_inN_set(struct cpssp *cpssp, int n, unsigned int val);
/*forward*/ static void
tpm0_clock(struct cpssp *cpssp);
/*forward*/ static void
tpm0_reset(struct cpssp *);

/*
 * Forward declarations for TPM1.
 */
/*forward*/ static void
tpm1_ch_inN_set(struct cpssp *cpssp, int n, unsigned int val);
/*forward*/ static void
tpm1_clock(struct cpssp *cpssp);
/*forward*/ static void
tpm1_reset(struct cpssp *);

#if defined(KL46Z)
/*
 * Forward declarations for TPM1.
 */
/*forward*/ static void
tpm2_ch_inN_set(struct cpssp *cpssp, int n, unsigned int val);
/*forward*/ static void
tpm2_clock(struct cpssp *cpssp);
/*forward*/ static void
tpm2_reset(struct cpssp *);
#endif

/*
 * Forward declarations for UART0.
 */
/*forward*/ static void
uart0_rx_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
uart0_clock(struct cpssp *cpssp);
/*forward*/ static void
uart0_reset(struct cpssp *);

#if defined(KL46Z)
/*
 * Forward declarations for UART1.
 */
/*forward*/ static void
uart1_rx_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
uart1_clock(struct cpssp *cpssp);
/*forward*/ static void
uart1_reset(struct cpssp *);

/*
 * Forward declarations for UART2.
 */
/*forward*/ static void
uart2_rx_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
uart2_clock(struct cpssp *cpssp);
/*forward*/ static void
uart2_reset(struct cpssp *);
#endif /* KL46Z */

/*
 * See: "MCU resets".
 */
static void
CHIP_(do_chip_reset)(struct cpssp *cpssp)
{
	/* Early Chip Reset */
	/* FIXME */

	/* Chip Reset */
	arm_cpu_reset(cpssp);
	arm_nvic_reset(cpssp);
	arm_systick_reset(cpssp);
	adc0_reset(cpssp);
#if defined(KL46Z)
	dac0_reset(cpssp);
	dma_reset(cpssp);
	dmamux_reset(cpssp);
#endif
	fmc_reset(cpssp);
	ftfa_reset(cpssp);
	i2c0_reset(cpssp);
	i2c1_reset(cpssp);
	pta_reset(cpssp);
	ptb_reset(cpssp);
#if defined(KL46Z)
	ptc_reset(cpssp);
	ptd_reset(cpssp);
	pte_reset(cpssp);
#endif
	mcm_reset(cpssp);
#if defined(KL46Z)
	pit_reset(cpssp);
#endif
	porta_reset(cpssp);
	portb_reset(cpssp);
#if defined(KL46Z)
	portc_reset(cpssp);
	portd_reset(cpssp);
	porte_reset(cpssp);
#endif
	spi0_reset(cpssp);
#if defined(KL46Z)
	spi1_reset(cpssp);
#endif
	tpm0_reset(cpssp);
	tpm1_reset(cpssp);
#if defined(KL46Z)
	tpm2_reset(cpssp);
#endif
	uart0_reset(cpssp);
#if defined(KL46Z)
	uart1_reset(cpssp);
	uart2_reset(cpssp);
#endif
}

#if 0
static void
CHIP_(chip_reset)(struct cpssp *cpssp)
{
	CHIP_(do_chip_reset)(cpssp);
}
#endif

static void
CHIP_(do_chip_reset_not_vlls)(struct cpssp *cpssp)
{
	/* Reset (part of) SMC, LLWU. */
	smc_reset(cpssp); /* FIXME */
	llwu_reset(cpssp); /* FIXME */

	/* Reset modules remain powered during VLLS mode. */
	cmp0_reset(cpssp);
	osc_reset(cpssp);
#if defined(KL46Z)
	lcd_reset(cpssp);
#endif
	/* FIXME */
}

#if 0
/*
 * Asserts on RESET pin (and ...).
 */
static void
CHIP_(chip_reset_not_vlls)(struct cpssp *cpssp)
{
	CHIP_(do_chip_reset_not_vlls)(cpssp);

	/* Reset other levels. */
	CHIP_(do_chip_reset)(cpssp);
}
#endif

static void
CHIP_(do_chip_por)(struct cpssp *cpssp)
{
	/* Reset Reset Pin Filter */
	/* FIXME */

	/* Reset (part of) SIM and MCG. */
	sim_reset(cpssp); /* FIXME */
	mcg_reset(cpssp); /* FIXME */
}

/*
 * Asserts on VLLS Wakeup (and LVD and POR).
 */
static void
CHIP_(chip_por)(struct cpssp *cpssp)
{
	CHIP_(do_chip_por)(cpssp);

	/* Reset other levels. */
	CHIP_(do_chip_reset)(cpssp);

	rcm_wakeup(cpssp);
}

static void
CHIP_(do_chip_por_not_vlls)(struct cpssp *cpssp)
{
	/* Reset (part of) SMC and SIM. */
	sim_reset(cpssp); /* FIXME */
	smc_reset(cpssp); /* FIXME */

	/* Reset LPTMR. */
	lptmr0_reset(cpssp);
}

#if 0
/*
 * Asserts on LVD (and POR).
 */
static void
CHIP_(chip_por_not_vlls)(struct cpssp *cpssp)
{
	CHIP_(do_chip_por_not_vlls)(cpssp);

	CHIP_(do_chip_por)(cpssp);
	CHIP_(do_chip_reset_not_vlls)(cpssp);
	CHIP_(do_chip_reset)(cpssp);
}
#endif

static void
CHIP_(do_por_only_reset)(struct cpssp *cpssp)
{
	pmc_reset(cpssp);
	/* rtc_reset(cpssp); FIXME */
}

/*
 * Asserts on POR.
 */
static void
CHIP_(por_only_reset)(struct cpssp *cpssp)
{
	CHIP_(do_por_only_reset)(cpssp);

	/* Reset all other modules. */
	CHIP_(do_chip_por_not_vlls)(cpssp);
	CHIP_(do_chip_por)(cpssp);
	CHIP_(do_chip_reset_not_vlls)(cpssp);
	CHIP_(do_chip_reset)(cpssp);

	rcm_por(cpssp);
}

/*
 * Callback for power consumption.
 */
static void
power_consume(struct cpssp *cpssp, double energy)
{
	cpssp->energy += energy;
}

/*
 * Callbacks for matrix.
 */
static void
matrix_portaNM_out_set(struct cpssp *cpssp, int n, int m, unsigned int val)
{
	porta_internalNM_in_set(cpssp, n, m, val);
}

static void
matrix_portbNM_out_set(struct cpssp *cpssp, int n, int m, unsigned int val)
{
	portb_internalNM_in_set(cpssp, n, m, val);
}

#if defined(KL46Z)
static void
matrix_portcNM_out_set(struct cpssp *cpssp, int n, int m, unsigned int val)
{
	portc_internalNM_in_set(cpssp, n, m, val);
}

static void
matrix_portdNM_out_set(struct cpssp *cpssp, int n, int m, unsigned int val)
{
	portd_internalNM_in_set(cpssp, n, m, val);
}

static void
matrix_porteNM_out_set(struct cpssp *cpssp, int n, int m, unsigned int val)
{
	porte_internalNM_in_set(cpssp, n, m, val);
}
#endif

/*
 * Callbacks for NVIC.
 */
static void
arm_nvic_irq_set(struct cpssp *cpssp, unsigned int val)
{
	arm_cpu_irq_set(cpssp, val);
}

/*
 * Callback for SysTick.
 */
static void
arm_systick_irq_set(struct cpssp *cpssp)
{
	arm_nvic_systick_pending_set(cpssp, 1);
}

/*
 * Callbacks for ADC.
 */
#if defined(KL46Z)
static void
adc0_dma_set(struct cpssp *cpssp, unsigned int val)
{
	/* See: DMA request sources - MUX 0 */
	dmamux_request_inN_set(cpssp, 40, val);
}
#endif

static void
adc0_irq_set(struct cpssp *cpssp, unsigned int val)
{
	arm_nvic_irq_N_set(cpssp, 15, val);
}

/*
 * Callbacks for CMP0.
 */
static void
cmp0_irq_set(struct cpssp *cpssp, unsigned int val)
{
	llwu_moduleN_set(cpssp, 1, val);
	arm_nvic_irq_N_set(cpssp, 16, val);
}

static void
cmp0_cout_set(struct cpssp *cpssp, unsigned int val)
{
	lptmr0_alt_inN_set(cpssp, 0, val);
}

#if defined(KL46Z)
/*
 * Callbacks for DAC0.
 */
static void
dac0_irq_set(struct cpssp *cpssp, unsigned int val)
{
	arm_nvic_irq_N_set(cpssp, 25, val);
}
#endif /* KL46Z */

#if defined(KL46Z)
/*
 * Callbacks for DMA.
 */
static void
dma_irqN_set(struct cpssp *cpssp, unsigned int n, unsigned int val)
{
	switch (n) {
	case 0: n = 0; break;
	case 1: n = 1; break;
	case 2: n = 2; break;
	case 3: n = 3; break;
	default:
		assert(0);
	}
	arm_nvic_irq_N_set(cpssp, n, val);
}

static void
dma_out_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	CHIP_(switch_mr)(cpssp, addr, bs, valp);
}

static void
dma_out_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	CHIP_(switch_mw)(cpssp, addr, bs, val);
}
#endif /* KL46Z */

#if defined(KL46Z)
/*
 * Callbacks for DMAMUX.
 */
static void
dmamux_request_outN_set(struct cpssp *cpssp, int n, unsigned int val)
{
	dma_requestN_set(cpssp, n, val);
}
#endif /* KL46Z */

/*
 * Callbacks for FMC.
 */
static void
fmc_flash_read(struct cpssp *cpssp, uint32_t addr, uint32_t *valp)
{
	flash_read(cpssp, addr, valp);
}

/*
 * Callbacks for FTFA.
 */
static void
ftfa_irq_set(struct cpssp *cpssp, unsigned int val)
{
	arm_nvic_irq_N_set(cpssp, 5, val);
}

static void
ftfa_flash_read(struct cpssp *cpssp, uint32_t addr, uint32_t *valp)
{
	flash_read(cpssp, addr, valp);
}

static void
ftfa_flash_erase(struct cpssp *cpssp, uint32_t addr, uint32_t len)
{
	flash_erase(cpssp, addr, len);
}

static void
ftfa_flash_write(struct cpssp *cpssp, uint32_t addr, uint32_t val)
{
	flash_write(cpssp, addr, val);
}

/*
 * Callbacks for I2C0.
 */
static void
i2c0_irq_set(struct cpssp *cpssp, unsigned int val)
{
	arm_nvic_irq_N_set(cpssp, 8, val);
}

#if defined(KL46Z)
static void
i2c0_dma_set(struct cpssp *cpssp, unsigned int val)
{
	/* See: DMA request sources - MUX 0 */
	dmamux_request_inN_set(cpssp, 22, val);
}
#endif

/*
 * Callbacks for I2C1.
 */
static void
i2c1_irq_set(struct cpssp *cpssp, unsigned int val)
{
	arm_nvic_irq_N_set(cpssp, 9, val);
}

#if defined(KL46Z)
static void
i2c1_dma_set(struct cpssp *cpssp, unsigned int val)
{
	/* See: DMA request sources - MUX 0 */
	dmamux_request_inN_set(cpssp, 23, val);
}
#endif

/*
 * Callbacks for LLWU.
 */
static void
llwu_irq_set(struct cpssp *cpssp, unsigned int val)
{
	arm_nvic_irq_N_set(cpssp, 7, val);
}

static void
llwu_start(struct cpssp *cpssp)
{
	smc_start(cpssp);
}

/*
 * Callbacks for MCG.
 */
static void
mcg_mcgirclk(struct cpssp *cpssp)
{
#if defined(KL46Z)
	lcd_clk1(cpssp);
#endif /* KL46Z */
	lptmr0_clk0(cpssp);

	/* See: UART clocking. */
	if (cpssp->sim.uart0src == 0b01) {
		uart0_clock(cpssp);
	}

	if (cpssp->sim.tpmsrc == 0b01) {
		tpm0_clock(cpssp);
		tpm1_clock(cpssp);
#if defined(KL46Z)
		tpm2_clock(cpssp);
#endif /* KL46Z */
	}
}

static void
mcg_mcgoutclk(struct cpssp *cpssp)
{
	sim_outclkin(cpssp);
}

static void
mcg_mcgfllclk(struct cpssp *cpssp)
{
	sim_fllclkin(cpssp);
}

static void
mcg_mcgffclk(struct cpssp *cpssp)
{
	/* FIXME */
}

#if defined(KL02Z)
/* No PLL */
#elif defined(KL46Z)
static void
mcg_mcgpllclk(struct cpssp *cpssp)
{
	sim_pllclkin(cpssp);
}
#else
#error
#endif

/*
 * Callbacks for GPIO.
 */

/*
 * Callbacks for LPTMR
 */
static void
lptmr0_irq_set(struct cpssp *cpssp, unsigned int val)
{
	llwu_moduleN_set(cpssp, 0, val);
	arm_nvic_irq_N_set(cpssp, 28, val);
}

/*
 * Callbacks for OSC
 */
static void
osc_oscclk(struct cpssp *cpssp)
{
	mcg_oscclk(cpssp);
}

static void
osc_oscerclk(struct cpssp *cpssp)
{
	adc0_clk1(cpssp);
#if defined(KL46Z)
	lcd_clk2(cpssp);
#endif
	lptmr0_clk3(cpssp);

	if (cpssp->sim.tpmsrc == 0b10) {
		tpm0_clock(cpssp);
		tpm1_clock(cpssp);
#if defined(KL46Z)
		tpm2_clock(cpssp);
#endif /* KL46Z */
	}

	/* See: UART clocking. */
	if (cpssp->sim.uart0src == 0b10) {
		uart0_clock(cpssp);
	}
}

static void
osc_erclk32k(struct cpssp *cpssp)
{
#if defined(KL46Z)
	lcd_clk0(cpssp);
#endif
	lptmr0_clk2(cpssp);
	/* FIXME */
}

#if defined(KL46Z)
/*
 * Callbacks for PIT.
 */
static void
pit_out0_set(struct cpssp *cpssp, unsigned int val)
{
	dmamux_trigger_inN_set(cpssp, 0, val);
}

static void
pit_out1_set(struct cpssp *cpssp, unsigned int val)
{
	dmamux_trigger_inN_set(cpssp, 1, val);
}

static void
pit_irq_set(struct cpssp *cpssp, unsigned int val)
{
	arm_nvic_irq_N_set(cpssp, 22, val);
}
#endif

/*
 * Callbacks for PMC.
 */
static void
pmc_lpo(struct cpssp *cpssp)
{
	lptmr0_clk1(cpssp);
	sim_lpo(cpssp);
}

/*
 * Callbacks for PORTA.
 */
static void
porta_irq_set(struct cpssp *cpssp, unsigned int val)
{
	arm_nvic_irq_N_set(cpssp, 30, val);
}

static void
porta_internalNM_out_set(struct cpssp *cpssp, int alt, int pin, unsigned int val)
{
	matrix_portaNM_in_set(cpssp, alt, pin, val);
}

static void
porta_pinN_out_set(struct cpssp *cpssp, int n, unsigned int val)
{
	if (cpssp->port_pta[n]) {
		sig_std_logic_set(cpssp->port_pta[n], cpssp, val);
	}
}

/*
 * Callbacks for PORTB
 */
static void
portb_irq_set(struct cpssp *cpssp, unsigned int val)
{
	arm_nvic_irq_N_set(cpssp, 31, val);
}

static void
portb_internalNM_out_set(struct cpssp *cpssp, int alt, int pin, unsigned int val)
{
	matrix_portbNM_in_set(cpssp, alt, pin, val);
}

static void
portb_pinN_out_set(struct cpssp *cpssp, int n, unsigned int val)
{
	if (cpssp->port_ptb[n]) {
		sig_std_logic_set(cpssp->port_ptb[n], cpssp, val);
	}
}

#if defined(KL46Z)
/*
 * Callbacks for PORTC
 */
static void
portc_irq_set(struct cpssp *cpssp, unsigned int val)
{
	arm_nvic_irq_N_set(cpssp, 30, val); /* FIXME */
}

static void
portc_internalNM_out_set(struct cpssp *cpssp, int alt, int pin, unsigned int val)
{
	matrix_portcNM_in_set(cpssp, alt, pin, val);
}

static void
portc_pinN_out_set(struct cpssp *cpssp, int n, unsigned int val)
{
	if (cpssp->port_ptc[n]) {
		sig_std_logic_set(cpssp->port_ptc[n], cpssp, val);
	}
}

/*
 * Callbacks for PORTD
 */
static void
portd_irq_set(struct cpssp *cpssp, unsigned int val)
{
	arm_nvic_irq_N_set(cpssp, 30, val); /* FIXME */
}

static void
portd_internalNM_out_set(struct cpssp *cpssp, int alt, int pin, unsigned int val)
{
	matrix_portdNM_in_set(cpssp, alt, pin, val);
}

static void
portd_pinN_out_set(struct cpssp *cpssp, int n, unsigned int val)
{
	if (cpssp->port_ptd[n]) {
		sig_std_logic_set(cpssp->port_ptd[n], cpssp, val);
	}
}

/*
 * Callbacks for PORTE
 */
static void
porte_irq_set(struct cpssp *cpssp, unsigned int val)
{
	arm_nvic_irq_N_set(cpssp, 30, val); /* FIXME */
}

static void
porte_internalNM_out_set(struct cpssp *cpssp, int alt, int pin, unsigned int val)
{
	matrix_porteNM_in_set(cpssp, alt, pin, val);
}

static void
porte_pinN_out_set(struct cpssp *cpssp, int n, unsigned int val)
{
	if (cpssp->port_pte[n]) {
		sig_std_logic_set(cpssp->port_pte[n], cpssp, val);
	}
}
#endif /* KL46Z */

/*
 * Callbacks for SIM.
 */
static void
sim_coreclk(struct cpssp *cpssp)
{
	smc_system_clk_in(cpssp);
}

static void
sim_busclk(struct cpssp *cpssp)
{
	smc_bus_clk_in(cpssp);
}

static void
sim_xllclkout(struct cpssp *cpssp)
{
	/* See: UART clocking. */
	if (cpssp->sim.uart0src == 0b11) {
		uart0_clock(cpssp);
	}

	if (cpssp->sim.tpmsrc == 0b11) {
		tpm0_clock(cpssp);
		tpm1_clock(cpssp);
#if defined(KL46Z)
		tpm2_clock(cpssp);
#endif /* KL46Z */
	}
}

static void
sim_cop_reset(struct cpssp *cpssp)
{
	rcm_wdog(cpssp);
}

#if defined(KL46Z)
/*
 * Callbacks for SLCD
 */
#endif

/*
 * Callbacks for SMC
 */
static void
smc_system_clk_out(struct cpssp *cpssp)
{
	cpssp->count_system_clk++;
	if (cpssp->count_system_clk == 16) {
		arm_systick_exttick(cpssp);
		cpssp->count_system_clk = 0;
	}

	arm_cpu_clk(cpssp);
	arm_systick_clk(cpssp);
	if (cpssp->sim.i2c1) {
		i2c1_clk(cpssp);
	}
}

static void
smc_bus_clk_out(struct cpssp *cpssp)
{
	adc0_clk0(cpssp);
	cmp0_clk(cpssp);
#if defined(KL46Z)
	dma_clk(cpssp);
#endif /* KL46Z */
	ftfa_clk(cpssp);
	if (cpssp->sim.i2c0) {
		i2c0_clk(cpssp);
	}
#if defined(KL46Z)
	pit_clk(cpssp);
#endif
#if defined(KL46Z)
	/* See: UART clocking. */
	uart1_clock(cpssp);
	uart2_clock(cpssp);
#endif /* KL46Z */
}

static void
smc_power0_set(struct cpssp *cpssp, unsigned int val)
{
	/* FIXME */
}

static void
smc_power1_set(struct cpssp *cpssp, unsigned int val)
{
	/* FIXME */
}

static void
smc_power2_set(struct cpssp *cpssp, unsigned int val)
{
	/* FIXME */
}

static void
smc_power3_set(struct cpssp *cpssp, unsigned int val)
{
	/* FIXME */
}

static void
smc_runm_set(struct cpssp *cpssp, unsigned int val)
{
	/* FIXME */
}

static void
smc_reset_out(struct cpssp *cpssp)
{
	CHIP_(chip_por)(cpssp);
}

/*
 * Callbacks for UART0
 */
static void
uart0_irq_set(struct cpssp *cpssp, unsigned int val)
{
	/* See: Interrupt vector assignments. */
	arm_nvic_irq_N_set(cpssp, 12, val);
}

#if defined(KL46Z)
static void
uart0_dma_receive(struct cpssp *cpssp, unsigned int val)
{
	/* See: DMA request sources - MUX 0 */
	dmamux_request_inN_set(cpssp, 2, val);
}

static void
uart0_dma_transmit(struct cpssp *cpssp, unsigned int val)
{
	/* See: DMA request sources - MUX 0 */
	dmamux_request_inN_set(cpssp, 3, val);
}
#endif /* KL46Z */

#if defined(KL46Z)
/*
 * Callbacks for UART1
 */
static void
uart1_irq_set(struct cpssp *cpssp, unsigned int val)
{
	/* See: Interrupt vector assignments. */
	arm_nvic_irq_N_set(cpssp, 13, val);
}

static void
uart1_dma_receive(struct cpssp *cpssp, unsigned int val)
{
	/* See: DMA request sources - MUX 0 */
	dmamux_request_inN_set(cpssp, 4, val);
}

static void
uart1_dma_transmit(struct cpssp *cpssp, unsigned int val)
{
	/* See: DMA request sources - MUX 0 */
	dmamux_request_inN_set(cpssp, 5, val);
}
#endif /* KL46Z */

#if defined(KL46Z)
/*
 * Callbacks for UART0
 */
static void
uart2_irq_set(struct cpssp *cpssp, unsigned int val)
{
	/* See: Interrupt vector assignments. */
	arm_nvic_irq_N_set(cpssp, 14, val);
}

static void
uart2_dma_receive(struct cpssp *cpssp, unsigned int val)
{
	/* See: DMA request sources - MUX 0 */
	dmamux_request_inN_set(cpssp, 6, val);
}

static void
uart2_dma_transmit(struct cpssp *cpssp, unsigned int val)
{
	/* See: DMA request sources - MUX 0 */
	dmamux_request_inN_set(cpssp, 7, val);
}
#endif /* KL46Z */

#define COUNT_INST(x)	power_consume(cpssp, POWER_ ## x)

#define BEHAVIOR

#define extal_inN_set(cpssp, n, val)
#define extrg_in_in_set(cpssp, val)
#define i2s0_rx_bclk_in_set(cpssp, val)
#define i2s0_rx_fs_in_set(cpssp, val)
#define	i2s0_tx_bclk_in_set(cpssp, val)
#define i2s0_tx_fs_in_set(cpssp, val)
#define irq__inN_set(cpssp, n, val)	llwu_p_inN_set(cpssp, n, val)
#define nmi_b_in_set(cpssp, val)
#define reset_b_in_set(cpssp, val)
#define rtc_clkin_in_set(cpssp, val)

/* Mapping SPI0. */
#define spi0_sck_in_set(cpssp, val)	spi0_spsck_in_set(cpssp, val)
#define spi0_sck_out_set(cpssp, val)	spi0_spsck_out_set(cpssp, val)
#if defined(KL02Z)
#define spi0_ss_b_in_set(cpssp, val)	spi0_ss_in_set(cpssp, val)
#define spi0_ss_b_out_set(cpssp, val)	spi0_ss_out_set(cpssp, val)
#elif defined(KL46Z)
#define spi0_pcso_in_set(cpssp, val)	spi0_ss_in_set(cpssp, val)
#define spi0_pcso_out_set(cpssp, val)	spi0_ss_out_set(cpssp, val)
#endif

#if defined(KL46Z)
/* Mapping SPI1. */
#define spi1_pcso_in_set(cpssp, val)	spi1_ss_in_set(cpssp, val)
#define spi1_pcso_out_set(cpssp, val)	spi1_ss_out_set(cpssp, val)
#define spi1_sck_in_set(cpssp, val)	spi1_spsck_in_set(cpssp, val)
#define spi1_sck_out_set(cpssp, val)	spi1_spsck_out_set(cpssp, val)
#define spi1_pcso_in_set(cpssp, val)	spi1_ss_in_set(cpssp, val)
#define spi1_pcso_out_set(cpssp, val)	spi1_ss_out_set(cpssp, val)
#endif
#define swd_dio_in_set(cpssp, val)
#define tpm_clkin_inN_set(cpssp, n, val) sim_tpm_clkinN_set(cpssp, n, val)
#define tsi0_ch_inN_set(cpssp, n, val)
#define usb_clkin_in_set(cpssp, val)
#define xtal_inN_set(cpssp, n, val)

#define NAME		matrix
#define NAME_(x)	matrix_ ## x
#if defined(KL02Z)
#include "chip_kl02z32m_matrix.c"
#elif defined(KL46Z)
#include "chip_kl46z256m_matrix.c"
#endif
#undef NAME_
#undef NAME

#define NAME		adc0
#define NAME_(x)	adc0_ ## x
#if defined(KL46Z)
#define CONFIG_DMA	1
#else
#define CONFIG_DMA	0
#endif
#include "arch_kinetis_adc.c"
#undef CONFIG_DMA
#undef NAME_
#undef NAME

#define NAME		cmp0
#define NAME_(x)	cmp0_ ## x
#define CONFIG_WINDOW	0
#define CONFIG_SAMPLE	0
#include "arch_kinetis_cmp.c"
#undef CONFIG_SAMPLE
#undef CONFIG_WINDOW
#undef NAME_
#undef NAME

#if defined(KL46Z)
#define NAME		dac0
#define NAME_(x)	dac0_ ## x
#include "arch_kinetis_dac.c"
#undef NAME_
#undef NAME
#endif /* KL46Z */

#if defined(KL46Z)

#define NAME		dma
#define NAME_(x)	dma_ ## x
#include "arch_kinetis_dma.c"
#undef NAME_
#undef NAME

#define NAME		dmamux
#define NAME_(x)	dmamux_ ## x
#include "arch_kinetis_dmamux.c"
#undef NAME_
#undef NAME

#endif

#define NAME		flash
#define NAME_(x)	flash_ ## x
#include "arch_gen_flash.c"
#undef NAME_
#undef NAME

#define NAME		fmc
#define NAME_(x)	fmc_ ## x
#include "arch_kinetis_fmc.c"
#undef NAME_
#undef NAME

#define NAME		ftfa
#define NAME_(x)	ftfa_ ## x
#include "arch_kinetis_ftfa.c"
#undef NAME_
#undef NAME

#define NAME		pta
#define NAME_(x)	pta_ ## x
#include "arch_kinetis_gpio.c"
#undef NAME_
#undef NAME

#define NAME		ptb
#define NAME_(x)	ptb_ ## x
#include "arch_kinetis_gpio.c"
#undef NAME_
#undef NAME

#if defined(KL46Z)

#define NAME		ptc
#define NAME_(x)	ptc_ ## x
#include "arch_kinetis_gpio.c"
#undef NAME_
#undef NAME

#define NAME		ptd
#define NAME_(x)	ptd_ ## x
#include "arch_kinetis_gpio.c"
#undef NAME_
#undef NAME

#define NAME		pte
#define NAME_(x)	pte_ ## x
#include "arch_kinetis_gpio.c"
#undef NAME_
#undef NAME

#endif /* KL46Z */

#define NAME		i2c0
#define NAME_(x)	i2c0_ ## x
#if defined(KL46Z)
#define CONFIG_DMA	1
#else
#define CONFIG_DMA	0
#endif
#include "arch_kinetis_i2c.c"
#undef CONFIG_DMA
#undef NAME_
#undef NAME

#define NAME		i2c1
#define NAME_(x)	i2c1_ ## x
#if defined(KL46Z)
#define CONFIG_DMA	1
#else
#define CONFIG_DMA	0
#endif
#include "arch_kinetis_i2c.c"
#undef CONFIG_DMA
#undef NAME_
#undef NAME

#define NAME		llwu
#define NAME_(x)	llwu_ ## x
#include "arch_kinetis_llwu.c"
#undef NAME_
#undef NAME

#define NAME		lptmr0
#define NAME_(x)	lptmr0_ ## x
#include "arch_kinetis_lptmr.c"
#undef NAME_
#undef NAME

#define NAME		mcg
#define NAME_(x)	mcg_ ## x
#include "arch_kinetis_mcg.c"
#undef NAME_
#undef NAME

#define NAME		mcm
#define NAME_(x)	mcm_ ## x
#include "arch_kinetis_mcm.c"
#undef NAME_
#undef NAME

#define NAME		osc
#define NAME_(x)	osc_ ## x
#include "arch_kinetis_osc.c"
#undef NAME_
#undef NAME

#if defined(KL46Z)
#define NAME		pit
#define NAME_(x)	pit_ ## x
#define N		2
#include "arch_kinetis_pit.c"
#undef N
#undef NAME_
#undef NAME
#endif /* KL46Z */

#define NAME		pmc
#define NAME_(x)	pmc_ ## x
#include "arch_kinetis_pmc.c"
#undef NAME_
#undef NAME

#define NAME		porta
#define NAME_(x)	porta_ ## x
#if defined(KL46Z)
#define CONFIG_NALTS	8
#else
#define CONFIG_NALTS	4
#endif
#include "arch_kinetis_port.c"
#undef NAME_
#undef NAME

#define NAME		portb
#define NAME_(x)	portb_ ## x
#include "arch_kinetis_port.c"
#if defined(KL46Z)
#define CONFIG_NALTS	8
#else
#define CONFIG_NALTS	4
#endif
#undef NAME_
#undef NAME

#if defined(KL46Z)

#define NAME		portc
#define NAME_(x)	portc_ ## x
#define CONFIG_NALTS	8
#include "arch_kinetis_port.c"
#undef NAME_
#undef NAME

#define NAME		portd
#define NAME_(x)	portd_ ## x
#define CONFIG_NALTS	8
#include "arch_kinetis_port.c"
#undef NAME_
#undef NAME

#define NAME		porte
#define NAME_(x)	porte_ ## x
#define CONFIG_NALTS	8
#include "arch_kinetis_port.c"
#undef NAME_
#undef NAME

#endif /* KL46Z */

#define NAME		rcm
#define NAME_(x)	rcm_ ## x
#include "arch_kinetis_rcm.c"
#undef NAME_
#undef NAME

#define NAME		sim
#define NAME_(x)	sim_ ## x
#include "arch_kinetis_sim.c"
#undef NAME_
#undef NAME

#define NAME		spi0
#define NAME_(x)	spi0_ ## x
#include "arch_kinetis_spi.c"
#undef NAME_
#undef NAME

#if defined(KL46Z)
#define NAME		spi1
#define NAME_(x)	spi1_ ## x
#include "arch_kinetis_spi.c"
#undef NAME_
#undef NAME
#endif

#if defined(KL46Z)
#define NAME		lcd
#define NAME_(x)	lcd_ ## x
#include "arch_kinetis_slcd.c"
#undef NAME_
#undef NAME
#endif

#define NAME		smc
#define NAME_(x)	smc_ ## x
#include "arch_kinetis_smc.c"
#undef NAME_
#undef NAME

#define NAME		sram
#define NAME_(x)	sram_ ## x
#include "arch_gen_sram.c"
#undef NAME_
#undef NAME

#define NAME		tpm0
#define NAME_(x)	tpm0_ ## x
#include "arch_kinetis_tpm.c"
#undef NAME_
#undef NAME

#define NAME		tpm1
#define NAME_(x)	tpm1_ ## x
#include "arch_kinetis_tpm.c"
#undef NAME_
#undef NAME

#if defined(KL46Z)
#define NAME		tpm2
#define NAME_(x)	tpm2_ ## x
#include "arch_kinetis_tpm.c"
#undef NAME_
#undef NAME
#endif /* KL46Z */

#define NAME		uart0
#define NAME_(x)	uart0_ ## x
#if defined(KL46Z)
#define CONFIG_DMA	1
#else
#define CONFIG_DMA	0
#endif
#include "arch_kinetis_uart.c"
#undef CONFIG_DMA
#undef NAME_
#undef NAME

#if defined(KL46Z)
#define NAME		uart1
#define NAME_(x)	uart1_ ## x
#define CONFIG_DMA	1
#include "arch_kinetis_uart.c"
#undef CONFIG_DMA
#undef NAME_
#undef NAME

#define NAME		uart2
#define NAME_(x)	uart2_ ## x
#define CONFIG_DMA	1
#include "arch_kinetis_uart.c"
#undef CONFIG_DMA
#undef NAME_
#undef NAME
#endif /* KL46Z */

#define NAME		arm_nvic
#define NAME_(x)	arm_nvic_ ## x
#define CONFIG_INTLINESNUM	32
#include "arch_arm_nvic.c"
#undef CONFIG_INTLINESNUM
#undef NAME_
#undef NAME

#define NAME		arm_systick
#define NAME_(x)	arm_systick_ ## x
#include "arch_arm_systick.c"
#undef NAME_
#undef NAME

#undef BEHAVIOR

static struct {
	void (*ld)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);
	void (*st)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
} CHIP_(gpio_table)[] = {
	[0x0] = { pta_ld, pta_st },
	[0x1] = { ptb_ld, ptb_st },
#if defined(KL46Z)
	[0x2] = { ptc_ld, ptc_st },
	[0x3] = { ptd_ld, ptd_st },
	[0x4] = { pte_ld, pte_st },
#endif
};

static void
CHIP_(gpio_ld)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	unsigned int n;

	addr &= 0x1000 - 1;

	n = addr >> 6;
	if (n < sizeof(CHIP_(gpio_table)) / sizeof(CHIP_(gpio_table)[0])) {
		(*CHIP_(gpio_table)[n].ld)(cpssp, addr, bs, valp);
	} else {
		fprintf(stderr, "WARNING: %s: addr=0x%08lx bs=0x%x\n",
				__FUNCTION__, addr, bs);
		*valp = 0;
	}
}

static void
CHIP_(gpio_st)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	unsigned int n;

	addr &= 0x1000 - 1;

	n = addr >> 6;
	if (n < sizeof(CHIP_(gpio_table)) / sizeof(CHIP_(gpio_table)[0])) {
		(*CHIP_(gpio_table)[n].st)(cpssp, addr, bs, val);
	} else {
		fprintf(stderr, "WARNING: %s: addr=0x%08lx bs=0x%x\n",
				__FUNCTION__, addr, bs);
		assert(0);
	}
}

/*
 * Private Peripheral Bridge
 */
static void
CHIP_(ppb_ld)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	assert(bs == 0b1111); /* Unpredictable otherwise. */

	addr &= 0x100000 - 1;

	switch (addr) {
	case 0x0e000 ... 0x0e00f:
		/* Reserved */
		/* FIXME */
		/* Auxiliary Control Register */
		/* B3-274 */
		assert(0); /* FIXME */
	case 0x0e010 ... 0x0e0ff:
		/* System Timer Register */
		/* B3.3.2 */
		arm_systick_ld(cpssp, addr, valp);
		break;
	case 0x0e100 ... 0x0ecff:
		/* Nested Vectored Interrupt Controller Register */
		/* 3.4.2 */
		arm_nvic_ld(cpssp, addr, bs, valp);
		break;
	case 0x0ed00 ... 0x0ed8f:
		/* System Control and ID Registers */
		/* B3.2.2 */
		arm_cpu_SCB_ld(cpssp, addr, bs, valp);
		break;
#if defined(CONFIG_DEBUG) && CONFIG_DEBUG
	case 0x0edf0 ... 0x0eeff:
		/* Debug */
		assert(0); /* FIXME */
#endif
	default:
		/* Reserved */
		fprintf(stderr, "WARNING: %s: addr=0x%08lx\n",
				__FUNCTION__, addr);
		assert(0); /* FIXME */
	}
}

static void
CHIP_(ppb_st)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	assert(bs == 0b1111); /* Unpredictable otherwise. */

	addr &= 0x100000 - 1;

	switch (addr) {
	case 0x0e000 ... 0x0e00f:
		/* Reserved (0x000-0x00f) */
		/* FIXME */
		/* Auxiliary Control Register */
		/* B3-274 */
		assert(0); /* FIXME */
	case 0x0e010 ... 0x0e0ff:
		/* System Timer Register */
		/* B3.3.2 */
		arm_systick_st(cpssp, addr, val);
		break;
	case 0x0e100 ... 0x0ecff:
		/* Nested Vectored Interrupt Controller Register */
		/* 3.4.2 */
		arm_nvic_st(cpssp, addr, bs, val);
		break;
	case 0x0ed00 ... 0x0ed8f:
		/* System Control Block */
		/* B3.2.2 */
		arm_cpu_SCB_st(cpssp, addr, bs, val);
		break;
#if defined(CONFIG_DEBUG) && CONFIG_DEBUG
	case 0x0edf0 ... 0x0eeff:
		/* Debug */
		assert(0);
#endif
	default:
		/* Reserved */
		fprintf(stderr, "WARNING: %s: addr=0x%08lx val=0x%08lx\n",
				__FUNCTION__, addr, val);
		assert(0); /* FIXME */
	}
}

/*
 * See: Peripheral bridge 0 slot assignments.
 */
static const struct {
	void (*ld)(struct cpssp *, uint32_t, unsigned int, uint32_t *);
	void (*st)(struct cpssp *, uint32_t, unsigned int, uint32_t);
} CHIP_(pbridge_table)[256] = {
#if defined(KL46Z)
	[8] = { dma_ld, dma_st },
#endif
	[15] = { CHIP_(gpio_ld), CHIP_(gpio_st) },
	[32] = { ftfa_ld, ftfa_st },
#if defined(KL46Z)
	[33] = { dmamux_ld, dmamux_st },
#endif
#if defined(KL46Z)
	[55] = { pit_ld, pit_st },
#endif
	[56] = { tpm0_ld, tpm0_st },
	[57] = { tpm1_ld, tpm1_st },
#if defined(KL46Z)
	[58] = { tpm2_ld, tpm2_st },
#endif
	[59] = { adc0_ld, adc0_st },
#if defined(KL46Z)
	[63] = { dac0_ld, dac0_st },
#endif
	[64] = { lptmr0_ld, lptmr0_st },
	[71] = { sim_7ld, sim_7st },
	[72] = { sim_8ld, sim_8st },
	[73] = { porta_ld, porta_st },
	[74] = { portb_ld, portb_st },
#if defined(KL46Z)
	[75] = { portc_ld, portc_st },
	[76] = { portd_ld, portd_st },
	[77] = { porte_ld, porte_st },
#endif
#if defined(KL46Z)
	[83] = { lcd_ld, lcd_st },
#endif
	[100] = { mcg_ld, mcg_st },
	[101] = { osc_ld, osc_st },
	[102] = { i2c0_ld, i2c0_st },
	[103] = { i2c1_ld, i2c1_st },
	[106] = { uart0_ld, uart0_st },
#if defined(KL46Z)
	[107] = { uart1_ld, uart1_st },
	[108] = { uart2_ld, uart2_st },
#endif
	[115] = { cmp0_ld, cmp0_st },
	[118] = { spi0_ld, spi0_st },
#if defined(KL46Z)
	[119] = { spi1_ld, spi1_st },
#endif
	[124] = { llwu_ld, llwu_st },
	[125] = { pmc_ld, pmc_st },
	[126] = { smc_ld, smc_st },
	[127] = { rcm_ld, rcm_st },
	[255] = { CHIP_(gpio_ld), CHIP_(gpio_st) },
};

static void
CHIP_(pbridge_mr)(
	struct cpssp *cpssp,
	uint32_t addr,
	unsigned int bs,
	uint32_t *valp
)
{
	void (*func)(struct cpssp *, uint32_t, unsigned int, uint32_t *);

	assert(! (addr & 3));

	func = CHIP_(pbridge_table)[(addr >> 12) & 0xff].ld;
	if (func) {
		(*func)(cpssp, addr & 0xfff, bs, valp);
	} else {
		fprintf(stderr, "WARNING: %s: addr=0x%08lx bs=0x%x\n",
				__FUNCTION__, addr, bs);
		*valp = 0; assert(0); /* FIXME */
	}
}

static void
CHIP_(pbridge_mw)(
	struct cpssp *cpssp,
	uint32_t addr,
	unsigned int bs,
	uint32_t val
)
{
	void (*func)(struct cpssp *, uint32_t, unsigned int, uint32_t);

	assert(! (addr & 3));

	func = CHIP_(pbridge_table)[(addr >> 12) & 0xff].st;
	if (func) {
		(*func)(cpssp, addr & 0xfff, bs, val);
	} else {
		fprintf(stderr, "WARNING: %s: addr=0x%08lx bs=0x%x val=0x%08lx\n",
				__FUNCTION__, addr, bs, val);
		assert(0); /* FIXME */
	}
}

static void
CHIP_(switch_mx)(
	struct cpssp *cpssp,
	uint32_t addr,
	unsigned int bs,
	uint32_t *valp
)
{
	assert(! (addr & 3));

	if (0x00000000 <= addr && addr < 0x00000000 + FLASH_SIZE) {
		/* Flash */
		fmc_mx(cpssp, addr, bs, valp);
	} else if (0x20000000 - SRAM_SIZE / 4 * 1 <= addr && addr < 0x20000000 + SRAM_SIZE / 4 * 3) {
		/* SRAM */
		sram_ld(cpssp, addr, bs, valp);
	} else {
		assert(0); /* FIXME */
	}
}

static void
CHIP_(switch_mr)(
	struct cpssp *cpssp,
	uint32_t addr,
	unsigned int bs,
	uint32_t *valp
)
{
	assert(! (addr & 3));

	if (0x00000000 <= addr && addr < 0x00000000 + FLASH_SIZE) {
		/* Flash */
		fmc_mr(cpssp, addr, bs, valp);
	} else if (0x20000000 - SRAM_SIZE / 4 * 1 <= addr && addr < 0x20000000 + SRAM_SIZE / 4 * 3) {
		/* SRAM */
		sram_ld(cpssp, addr, bs, valp);
	} else if (0x40000000 <= addr && addr < 0x50000000) {
		CHIP_(pbridge_mr)(cpssp, addr, bs, valp);
	} else if (0xe0000000 <= addr && addr < 0xe0100000) {
		/* PPB */
		assert(0); /* Mustn't happen. */
	} else if (0xf0003000 <= addr && addr < 0xf0004000) {
		/* MCM */
		mcm_ld(cpssp, addr, bs, valp);
	} else {
		fprintf(stderr, "WARNING: %s: addr=0x%08lx bs=0x%x\n",
				__FUNCTION__, addr, bs);
		*valp = 0;
	}
}

static void
CHIP_(switch_mw)(
	struct cpssp *cpssp,
	uint32_t addr,
	unsigned int bs,
	uint32_t val
)
{
	assert(! (addr & 3));

	if (0x00000000 <= addr && addr < 0x00000000 + FLASH_SIZE) {
		/* Flash */
		fmc_mw(cpssp, addr, bs, val);
	} else if (0x20000000 - SRAM_SIZE / 4 * 1 <= addr && addr < 0x20000000 + SRAM_SIZE / 4 * 3) {
		/* SRAM */
		sram_st(cpssp, addr, bs, val);
	} else if (0x40000000 <= addr && addr < 0x50000000) {
		CHIP_(pbridge_mw)(cpssp, addr, bs, val);
	} else if (0xe0000000 <= addr && addr < 0xe0100000) {
		/* PPB */
		assert(0); /* Mustn't happen. */
	} else if (0xf0003000 <= addr && addr < 0xf0004000) {
		/* MCM */
		mcm_st(cpssp, addr, bs, val);
	} else {
		fprintf(stderr, "WARNING: %s: addr=0x%08lx bs=0x%x val=0x%08lx\n",
				__FUNCTION__, addr, bs, val);
	}
}

/*
 * Callbacks for ARM CPU.
 */
static void
arm_cpu_pri_set(struct cpssp *cpssp, unsigned int n, unsigned int val)
{
	arm_nvic_pri_set(cpssp, n, val);
}

static unsigned int
arm_cpu_pri_get(struct cpssp *cpssp, unsigned int n)
{
	return arm_nvic_pri_get(cpssp, n);
}

static void
arm_cpu_primask_set(struct cpssp *cpssp, unsigned int val)
{
	arm_nvic_primask_set(cpssp, val);
}

static unsigned int
arm_cpu_primask_get(struct cpssp *cpssp)
{
	return arm_nvic_primask_get(cpssp);
}

static void
arm_cpu_NMI_pending_set(struct cpssp *cpssp, unsigned int val)
{
	arm_nvic_NMI_pending_set(cpssp, val);
}

static void
arm_cpu_PendSV_pending_set(struct cpssp *cpssp, unsigned int val)
{
	arm_nvic_PendSV_pending_set(cpssp, val);
}

static void
arm_cpu_SVCall_pending_set(struct cpssp *cpssp, unsigned int val)
{
	arm_nvic_SVCall_pending_set(cpssp, val);
}

static void
arm_cpu_SysTick_pending_set(struct cpssp *cpssp, unsigned int val)
{
	arm_nvic_systick_pending_set(cpssp, val);
}

static uint8_t
arm_cpu_exc_ack(struct cpssp *cpssp)
{
	return arm_nvic_exc_ack(cpssp);
}

static void
arm_cpu_exc_eoi(struct cpssp *cpssp, uint8_t vec)
{
	return arm_nvic_exc_eoi(cpssp, vec);
}

static void
arm_cpu_mx(
	struct cpssp *cpssp,
	uint32_t addr,
	unsigned int bs,
	uint32_t *valp
)
{
	if (0xe0000000 <= addr && addr < 0xe0100000) {
		assert(0); /* FIXME */
	} else {
		CHIP_(switch_mx)(cpssp, addr, bs, valp);
	}
}

static void
arm_cpu_mr(
	struct cpssp *cpssp,
	uint32_t addr,
	unsigned int bs,
	uint32_t *valp
)
{
	if (0xe0000000 <= addr && addr < 0xe0100000) {
		CHIP_(ppb_ld)(cpssp, addr, bs, valp);
	} else {
		CHIP_(switch_mr)(cpssp, addr, bs, valp);
	}
}

static void
arm_cpu_mw(
	struct cpssp *cpssp,
	uint32_t addr,
	unsigned int bs,
	uint32_t val
)
{
	if (0xe0000000 <= addr && addr < 0xe0100000) {
		CHIP_(ppb_st)(cpssp, addr, bs, val);
	} else {
		CHIP_(switch_mw)(cpssp, addr, bs, val);
	}
}

static void
arm_cpu_gpio_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	CHIP_(gpio_ld)(cpssp, addr, bs, valp);
}

static void
arm_cpu_gpio_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	CHIP_(gpio_st)(cpssp, addr, bs, val);
}

static void
arm_cpu_stop(struct cpssp *cpssp, int sleepdeep)
{
	smc_stop(cpssp, sleepdeep);
}

#define BEHAVIOR
#define NAME		arm_cpu
#define NAME_(x)	arm_cpu_ ## x
#define CONFIG_DEBUG	0
#define CONFIG_VERSION	6
#define CONFIG_ARM	0
#define CONFIG_THUMB	1
#define CONFIG_E	0
#define CONFIG_M	1
#include "arch_arm_cpu.c"
#undef CONFIG_M
#undef CONFIG_E
#undef CONFIG_THUMB
#undef CONFIG_ARM
#undef CONFIG_VERSION
#undef CONFIG_DEBUG
#undef NAME_
#undef NAME
#undef BEHAVIOR

static void
CHIP_(port_vdda_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;
	
	adc0_refN_set(cpssp, 1, val);
#if defined(KL46Z)
	dac0_refN_set(cpssp, 1, val);
#endif
}

static void
CHIP_(port_vdd_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;
	
	cpssp->state_vdd = 2000 < SIG_mV(val);

	cmp0_vinN_set(cpssp, 1, val);

#if defined(KL02Z)
	/* Vdda internally connected to Vdd. */
	CHIP_(port_vdda_set)(cpssp, val);
#endif
}

static void
CHIP_(port_vrefh_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;
	
	adc0_refN_set(cpssp, 0, val);
	cmp0_vinN_set(cpssp, 0, val);
#if defined(KL46Z)
	dac0_refN_set(cpssp, 0, val);
#endif
}

static void
CHIP_(ptaN_set)(void *_cpssp, int n, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	porta_pinN_in_set(cpssp, n, val);
}

static void
CHIP_(ptbN_set)(void *_cpssp, int n, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	portb_pinN_in_set(cpssp, n, val);
}

#if defined(KL46Z)
static void
CHIP_(ptcN_set)(void *_cpssp, int n, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	portc_pinN_in_set(cpssp, n, val);
}

static void
CHIP_(ptdN_set)(void *_cpssp, int n, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	portd_pinN_in_set(cpssp, n, val);
}

static void
CHIP_(pteN_set)(void *_cpssp, int n, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	porte_pinN_in_set(cpssp, n, val);
}
#endif

static void __attribute__((__noreturn__))
CHIP_(step)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

again:  ;
	if (likely(cpssp->power_state)) {
		if (likely(cpssp->state_vdd)) {
			/* Power-on */
			while (cpssp->process.inst_cnt < cpssp->process.inst_limit) {
				osc_exclk(cpssp);

				mcg_4MHz(cpssp, 1, (8*1000*1000) / (4*1000*1000));

				mcg_32kHz(cpssp, 1, (8*1000*1000) / (32*1024));

				cpssp->count_1kHz++;
				if ((8*1000*1000) / (1*1000) <= cpssp->count_1kHz) {
					cpssp->count_1kHz = 0;
					pmc_1kHz(cpssp);
				}

				power_consume(cpssp, POWER_static);

				cpssp->process.inst_cnt += 1;
			}
		} else {
			/* Power-on => Power-off */
			cpssp->power_state = 0;
			goto again;
		}
	} else {
		if (unlikely(cpssp->state_vdd)) {
			/* Power-off => Power-on */
			cpssp->power_state = 1;
			CHIP_(por_only_reset)(cpssp);

			cpssp->process.inst_cnt += 100;

			goto again;

		} else {
			/* Power-off */
			if (cpssp->process.inst_cnt < cpssp->process.inst_limit) {
				cpssp->process.inst_cnt = cpssp->process.inst_limit;
			}
		}
	}

	sig_std_logic_consume(cpssp->port_vdd, cpssp, cpssp->energy);
	cpssp->energy = 0.0;

	sched_to_scheduler();

	goto again;
}

void *
CHIP_(create)(
	const char *name,
	struct sig_manage *manage,
#if defined(KL02Z)
	struct sig_std_logic *port_ptb6,
	struct sig_std_logic *port_ptb7,
	struct sig_std_logic *port_vdd,
	struct sig_std_logic *port_vrefh,
	struct sig_std_logic *port_vrefl,
	struct sig_std_logic *port_vss,
	struct sig_std_logic *port_pta3,
	struct sig_std_logic *port_pta4,
	struct sig_std_logic *port_pta5,
	struct sig_std_logic *port_pta6,
	struct sig_std_logic *port_ptb8,
	struct sig_std_logic *port_ptb9,
	struct sig_std_logic *port_ptb10,
	struct sig_std_logic *port_ptb11,
	struct sig_std_logic *port_pta7,
	struct sig_std_logic *port_ptb0,
	struct sig_std_logic *port_ptb1,
	struct sig_std_logic *port_ptb2,
	struct sig_std_logic *port_pta8,
	struct sig_std_logic *port_pta9,
	struct sig_std_logic *port_pta10,
	struct sig_std_logic *port_pta11,
	struct sig_std_logic *port_ptb3,
	struct sig_std_logic *port_ptb4,
	struct sig_std_logic *port_ptb5,
	struct sig_std_logic *port_pta12,
	struct sig_std_logic *port_pta13,
	struct sig_std_logic *port_ptb12,
	struct sig_std_logic *port_ptb13,
	struct sig_std_logic *port_pta0,
	struct sig_std_logic *port_pta1,
	struct sig_std_logic *port_pta2
#elif defined(KL46Z)
	struct sig_std_logic *port_pte0,
	struct sig_std_logic *port_pte1,
	struct sig_std_logic *port_pte2,
	struct sig_std_logic *port_pte3,
	struct sig_std_logic *port_pte4,
	struct sig_std_logic *port_pte5,
	struct sig_std_logic *port_pte6,
	struct sig_std_logic *port_vdd0,
	struct sig_std_logic *port_vss0,
	struct sig_std_logic *port_usb0_dp,
	struct sig_std_logic *port_usb0_dm,
	struct sig_std_logic *port_vout33,
	struct sig_std_logic *port_vregin,
	struct sig_std_logic *port_pte16,
	struct sig_std_logic *port_pte17,
	struct sig_std_logic *port_pte18,
	struct sig_std_logic *port_pte19,
	struct sig_std_logic *port_pte20,
	struct sig_std_logic *port_pte21,
	struct sig_std_logic *port_pte22,
	struct sig_std_logic *port_pte23,
	struct sig_std_logic *port_vdda,
	struct sig_std_logic *port_vrefh,
	struct sig_std_logic *port_vrefl,
	struct sig_std_logic *port_vssa,

	struct sig_std_logic *port_pte29,
	struct sig_std_logic *port_pte30,
	struct sig_std_logic *port_pte31,
	struct sig_std_logic *port_vdd1,
	struct sig_std_logic *port_vss1,
	struct sig_std_logic *port_pte24,
	struct sig_std_logic *port_pte25,
	struct sig_std_logic *port_pte26,
	struct sig_std_logic *port_pta0,
	struct sig_std_logic *port_pta1,
	struct sig_std_logic *port_pta2,
	struct sig_std_logic *port_pta3,
	struct sig_std_logic *port_pta4,
	struct sig_std_logic *port_pta5,
	struct sig_std_logic *port_pta6,
	struct sig_std_logic *port_pta7,
	struct sig_std_logic *port_pta12,
	struct sig_std_logic *port_pta13,
	struct sig_std_logic *port_pta14,
	struct sig_std_logic *port_pta15,
	struct sig_std_logic *port_pta16,
	struct sig_std_logic *port_pta17,
	struct sig_std_logic *port_vdd2,
	struct sig_std_logic *port_vss2,
	struct sig_std_logic *port_pta18,

	struct sig_std_logic *port_pta19,
	struct sig_std_logic *port_pta20,
	struct sig_std_logic *port_ptb0,
	struct sig_std_logic *port_ptb1,
	struct sig_std_logic *port_ptb2,
	struct sig_std_logic *port_ptb3,
	struct sig_std_logic *port_ptb7,
	struct sig_std_logic *port_ptb8,
	struct sig_std_logic *port_ptb9,
	struct sig_std_logic *port_ptb10,
	struct sig_std_logic *port_ptb11,
	struct sig_std_logic *port_ptb16,
	struct sig_std_logic *port_ptb17,
	struct sig_std_logic *port_ptb18,
	struct sig_std_logic *port_ptb19,
	struct sig_std_logic *port_ptb20,
	struct sig_std_logic *port_ptb21,
	struct sig_std_logic *port_ptb22,
	struct sig_std_logic *port_ptb23,
	struct sig_std_logic *port_ptc0,
	struct sig_std_logic *port_ptc1,
	struct sig_std_logic *port_ptc2,
	struct sig_std_logic *port_ptc3,
	struct sig_std_logic *port_vss3,
	struct sig_std_logic *port_vll3,

	struct sig_std_logic *port_vll2,
	struct sig_std_logic *port_vll1,
	struct sig_std_logic *port_vcap2,
	struct sig_std_logic *port_vcap1,
	struct sig_std_logic *port_ptc4,
	struct sig_std_logic *port_ptc5,
	struct sig_std_logic *port_ptc6,
	struct sig_std_logic *port_ptc7,
	struct sig_std_logic *port_ptc8,
	struct sig_std_logic *port_ptc9,
	struct sig_std_logic *port_ptc10,
	struct sig_std_logic *port_ptc11,
	struct sig_std_logic *port_ptc12,
	struct sig_std_logic *port_ptc13,
	struct sig_std_logic *port_ptc16,
	struct sig_std_logic *port_ptc17,
	struct sig_std_logic *port_ptc18,
	struct sig_std_logic *port_ptd0,
	struct sig_std_logic *port_ptd1,
	struct sig_std_logic *port_ptd2,
	struct sig_std_logic *port_ptd3,
	struct sig_std_logic *port_ptd4,
	struct sig_std_logic *port_ptd5,
	struct sig_std_logic *port_ptd6,
	struct sig_std_logic *port_ptd7
#endif
)
{
	const static struct sig_std_logic_funcs port_vdd_funcs = {
		.std_logic_set = CHIP_(port_vdd_set),
	};
#if defined(KL46Z)
	const static struct sig_std_logic_funcs port_vdda_funcs = {
		.std_logic_set = CHIP_(port_vdda_set),
	};
#endif
	const static struct sig_std_logic_funcs port_vrefh_funcs = {
		.std_logic_set = CHIP_(port_vrefh_set),
	};
	const static struct sig_std_logic_funcs port_pta_funcs = {
		.set_extN = CHIP_(ptaN_set),
	};
	const static struct sig_std_logic_funcs port_ptb_funcs = {
		.set_extN = CHIP_(ptbN_set),
	};
#if defined(KL46Z)
	const static struct sig_std_logic_funcs port_ptc_funcs = {
		.set_extN = CHIP_(ptcN_set),
	};
	const static struct sig_std_logic_funcs port_ptd_funcs = {
		.set_extN = CHIP_(ptdN_set),
	};
	const static struct sig_std_logic_funcs port_pte_funcs = {
		.set_extN = CHIP_(pteN_set),
	};
#endif
	struct cpssp *cpssp;

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);
	memset(cpssp, 0, sizeof(*cpssp));

	arm_cpu_create(name, cpssp);
	arm_nvic_create(cpssp);
	arm_systick_create(cpssp);
	matrix_create(cpssp);
	adc0_create(cpssp);
	cmp0_create(cpssp);
#if defined(KL46Z)
	dac0_create(cpssp);
	dma_create(cpssp);
	dmamux_create(cpssp);
#endif
	flash_create(cpssp, "flash", "nvram");
	fmc_create(cpssp);
	ftfa_create(cpssp);
	pta_create(cpssp);
	ptb_create(cpssp);
#if defined(KL46Z)
	ptc_create(cpssp);
	ptd_create(cpssp);
	pte_create(cpssp);
#endif
	i2c0_create(cpssp);
	i2c1_create(cpssp);
	llwu_create(cpssp);
	lptmr0_create(cpssp);
	mcg_create(cpssp);
	mcm_create(cpssp);
	osc_create(cpssp);
#if defined(KL46Z)
	pit_create(cpssp);
#endif
	pmc_create(cpssp);
	porta_create(cpssp);
	portb_create(cpssp);
#if defined(KL46Z)
	portc_create(cpssp);
	portd_create(cpssp);
	porte_create(cpssp);
#endif
	rcm_create(cpssp);
	sim_create(cpssp);
#if defined(KL46Z)
	lcd_create(cpssp);
#endif
	smc_create(cpssp);
	spi0_create(cpssp);
#if defined(KL46Z)
	spi1_create(cpssp);
#endif
	sram_create(cpssp);
	tpm0_create(cpssp);
	tpm1_create(cpssp);
#if defined(KL46Z)
	tpm2_create(cpssp);
#endif
	uart0_create(cpssp);
#if defined(KL46Z)
	uart1_create(cpssp);
	uart2_create(cpssp);
#endif /* KL46Z */

	cpssp->process.inst_hz = 8*1000*1000;
	cpssp->energy = 0.0;

	cpssp->power_state = 0;
	cpssp->count_1kHz = 0;
	cpssp->count_system_clk = 0;

	sched_process_init(&cpssp->process, CHIP_(step), cpssp);

#if defined(KL02Z)
	cpssp->port_vdd = port_vdd;
#elif defined(KL46Z)
	cpssp->port_vdd = port_vdd0;
#else
#error
#endif
	sig_std_logic_connect_in(cpssp->port_vdd, cpssp, &port_vdd_funcs);

#define conn(a, n) \
	cpssp->port_pt ## a[n] = port_pt ## a ## n; \
	sig_std_logic_connect_out(port_pt ## a ## n, cpssp, SIG_STD_LOGIC_Z); \
	sig_std_logic_connect_inN(port_pt ## a ## n, cpssp, n, &port_pt ## a ## _funcs);
	/* Port A Pins */
	memset(cpssp->port_pta, 0, sizeof(cpssp->port_pta));
#if defined(KL02Z)
	conn(a, 0) conn(a, 1) conn(a, 2) conn(a, 3)
	conn(a, 4) conn(a, 5) conn(a, 6) conn(a, 7)
	conn(a, 8) conn(a, 9) conn(a, 10) conn(a, 11)
	conn(a, 12) conn(a, 13)
#elif defined(KL46Z)
	conn(a, 0) conn(a, 1) conn(a, 2) conn(a, 3)
	conn(a, 4) conn(a, 5) conn(a, 6) conn(a, 7)
	conn(a, 12) conn(a, 13) conn(a, 14) conn(a, 15)
	conn(a, 16) conn(a, 17) conn(a, 18) conn(a, 19)
	conn(a, 20)
#endif

	/* Port B Pins */
	memset(cpssp->port_ptb, 0, sizeof(cpssp->port_ptb));
#if defined(KL02Z)
	conn(b, 0) conn(b, 1) conn(b, 2) conn(b, 3)
	conn(b, 4) conn(b, 5) conn(b, 6) conn(b, 7)
	conn(b, 8) conn(b, 9) conn(b, 10) conn(b, 11)
	conn(b, 12) conn(b, 13)
#elif defined(KL46Z)
	conn(b, 0) conn(b, 1) conn(b, 2) conn(b, 3)
	conn(b, 7) conn(b, 8) conn(b, 9) conn(b, 10)
	conn(b, 11) conn(b, 16) conn(b, 17) conn(b, 18)
	conn(b, 19) conn(b, 20) conn(b, 21) conn(b, 22)
	conn(b, 23)
#endif
#if defined(KL46Z)
	/* Port C Pins */
	memset(cpssp->port_ptc, 0, sizeof(cpssp->port_ptc));

	conn(c, 0) conn(c, 1) conn(c, 2) conn(c, 3)
	conn(c, 4) conn(c, 5) conn(c, 6) conn(c, 7)
	conn(c, 8) conn(c, 9) conn(c, 10) conn(c, 11)
	conn(c, 12) conn(c, 13) conn(c, 16) conn(c, 17)
	conn(c, 18)
#endif
#if defined(KL46Z)
	/* Port D Pins */
	memset(cpssp->port_ptd, 0, sizeof(cpssp->port_ptd));

	conn(d, 0) conn(d, 1) conn(d, 2) conn(d, 3)
	conn(d, 4) conn(d, 5) conn(d, 6) conn(d, 7)
#endif
#if defined(KL46Z)
	/* Port E Pins */
	memset(cpssp->port_pte, 0, sizeof(cpssp->port_pte));

	conn(e, 0) conn(e, 1) conn(e, 2) conn(e, 3)
	conn(e, 4) conn(e, 5) conn(e, 6) conn(e, 16)
	conn(e, 17) conn(e, 18) conn(e, 19) conn(e, 20)
	conn(e, 21) conn(e, 22) conn(e, 23) conn(e, 24)
	conn(e, 25) conn(e, 26) conn(e, 29) conn(e, 30)
	conn(e, 31)
#endif
#undef conn

#if defined(KL46Z)
	sig_std_logic_connect_in(port_vdda, cpssp, &port_vdda_funcs);
#endif
	sig_std_logic_connect_in(port_vrefh, cpssp, &port_vrefh_funcs);

	return cpssp;
}

void
CHIP_(destroy)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	arm_cpu_destroy(cpssp);
	arm_nvic_destroy(cpssp);
	arm_systick_destroy(cpssp);
	matrix_destroy(cpssp);
	adc0_destroy(cpssp);
	cmp0_destroy(cpssp);
#if defined(KL46Z)
	dac0_destroy(cpssp);
	dma_destroy(cpssp);
	dmamux_destroy(cpssp);
#endif
	flash_destroy(cpssp);
	fmc_destroy(cpssp);
	ftfa_destroy(cpssp);
	pta_destroy(cpssp);
	ptb_destroy(cpssp);
#if defined(KL46Z)
	ptc_destroy(cpssp);
	ptd_destroy(cpssp);
	pte_destroy(cpssp);
#endif
	i2c0_destroy(cpssp);
	i2c1_destroy(cpssp);
	llwu_destroy(cpssp);
	lptmr0_destroy(cpssp);
	mcg_destroy(cpssp);
	mcm_destroy(cpssp);
	osc_destroy(cpssp);
#if defined(KL46Z)
	pit_destroy(cpssp);
#endif
	pmc_destroy(cpssp);
	porta_destroy(cpssp);
	portb_destroy(cpssp);
#if defined(KL46Z)
	portc_destroy(cpssp);
	portd_destroy(cpssp);
	porte_destroy(cpssp);
#endif
	rcm_destroy(cpssp);
	sim_destroy(cpssp);
#if defined(KL46Z)
	lcd_destroy(cpssp);
#endif
	smc_destroy(cpssp);
	spi0_destroy(cpssp);
#if defined(KL46Z)
	spi1_destroy(cpssp);
#endif
	sram_destroy(cpssp);
	tpm0_destroy(cpssp);
	tpm1_destroy(cpssp);
#if defined(KL46Z)
	tpm2_destroy(cpssp);
#endif
	uart0_destroy(cpssp);
#if defined(KL46Z)
	uart1_destroy(cpssp);
	uart2_destroy(cpssp);
#endif /* KL46Z */

	shm_free(cpssp);
}


void
CHIP_(suspend)(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;

	generic_suspend(cpssp, sizeof(*cpssp), fComp);
}

void
CHIP_(resume)(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;

	generic_resume(cpssp, sizeof(*cpssp), fComp);
}
